<?php

if (!defined('ABSPATH')) exit;

if (!class_exists('Buildecom_NotifyProcess')) {
    require_once __DIR__ . '/class-buildecom-notifyprocess.php';
}

require_once __DIR__ . '/interface-buildecom-notfiable.php';

// Load PHPMailer if needed
if (!class_exists('PHPMailer\PHPMailer\PHPMailer')) {
    require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
    require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
    require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
}

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class Buildecom_Notify_Email extends Buildecom_NotifyProcess
{
    /**
     * Email of receiver
     *
     * @var string
     */
    public $email;

    public function __construct()
    {
        $this->statusField = 'email_status';
        $this->body = 'email_body';
        $this->globalTemplate = 'buildecom_email_template';
        $this->notifyConfig = 'buildecom_mail_config';
    }

    /**
     * Send notification
     */
    public function send()
    {
        $message = $this->getMessage();

        if (get_option('buildecom_email_notification') && $message) {
            $mailConfig = buildecom_to_object(get_option($this->notifyConfig));
            $methodName = !empty($mailConfig->name) ? $mailConfig->name : 'php';
            $method = $this->mailMethods($methodName);

            if (method_exists($this, $method)) {
                try {
                    $this->$method();
                } catch (\Throwable $e) {
                    error_log('Email sending failed: ' . $e->getMessage());
                }
            } else {
                error_log("Invalid email sending method: {$methodName}");
            }
        }
    }

    /**
     * Map method name
     */
    protected function mailMethods($name)
    {
        $methods = [
            'php'       => 'sendPhpMail',
            'smtp'      => 'sendSmtpMail',
            'sendgrid'  => 'sendSendGridMail',
            'mailjet'   => 'sendMailjetMail',
        ];
        return isset($methods[$name]) ? $methods[$name] : $methods['php'];
    }

    /**
     * Send using PHP mail()
     */
    protected function sendPhpMail()
    {
        $siteName = get_bloginfo('name');
        $mailFrom = get_option('buildecom_email_from');

        $headers = [];
        $headers[] = "From: $siteName <$mailFrom>";
        $headers[] = "Reply-To: $siteName <$mailFrom>";
        $headers[] = "MIME-Version: 1.0";
        $headers[] = "Content-Type: text/html; charset=utf-8";

        @wp_mail($this->email, $this->subject, $this->finalMessage, $headers);
    }

    /**
     * Send using SMTP via PHPMailer
     */
    protected function sendSmtpMail()
    {
        $mail = new PHPMailer(true);
        $config = buildecom_to_object(get_option($this->notifyConfig));

        try {
            $mail->isSMTP();
            $mail->Host       = $config->host;
            $mail->SMTPAuth   = true;
            $mail->Username   = $config->username;
            $mail->Password   = $config->password;
            $mail->SMTPSecure = ($config->enc === 'ssl') ? PHPMailer::ENCRYPTION_SMTPS : PHPMailer::ENCRYPTION_STARTTLS;
            $mail->Port       = $config->port;
            $mail->CharSet    = 'UTF-8';

            $mail->setFrom(get_option('buildecom_email_from'), get_bloginfo('name'));
            $mail->addAddress($this->email, $this->receiverName);
            $mail->addReplyTo(get_option('buildecom_email_from'), get_bloginfo('name'));

            $mail->isHTML(true);
            $mail->Subject = $this->subject;
            $mail->Body    = $this->finalMessage;

            $mail->send();
        } catch (Exception $e) {
            error_log('SMTP Mail Error: ' . $mail->ErrorInfo);
        }
    }

    /**
     * Send using SendGrid API
     */
    protected function sendSendGridMail()
    {
        if (!class_exists('SendGrid')) {
            error_log('SendGrid library not loaded.');
            return;
        }

        $config = buildecom_to_object(get_option($this->notifyConfig));
        $sendgridMail = new \SendGrid\Mail\Mail();
        $sendgridMail->setFrom(get_option('buildecom_email_from'), get_bloginfo('name'));
        $sendgridMail->setSubject($this->subject);
        $sendgridMail->addTo($this->email, $this->receiverName);
        $sendgridMail->addContent("text/html", $this->finalMessage);

        $sendgrid = new \SendGrid($config->appkey);
        $response = $sendgrid->send($sendgridMail);

        if ($response->statusCode() != 202) {
            error_log('SendGrid Error: ' . $response->body());
        }
    }

    /**
     * Send using Mailjet API
     */
    protected function sendMailjetMail()
    {
        if (!class_exists('Mailjet\Client')) {
            error_log('Mailjet library not loaded.');
            return;
        }

        $config = buildecom_to_object(get_option($this->notifyConfig));
        $mj = new \Mailjet\Client($config->public_key, $config->secret_key, true, ['version' => 'v3.1']);
        $body = [
            'Messages' => [
                [
                    'From' => [
                        'Email' => get_option('buildecom_email_from'),
                        'Name'  => get_bloginfo('name'),
                    ],
                    'To' => [
                        [
                            'Email' => $this->email,
                            'Name'  => $this->receiverName,
                        ]
                    ],
                    'Subject' => $this->subject,
                    'TextPart' => '',
                    'HTMLPart' => $this->finalMessage,
                ]
            ]
        ];
        $mj->post(\Mailjet\Resources::$Email, ['body' => $body]);
    }

    /**
     * Configure before sending
     */
    protected function prevConfiguration()
    {
        if (!empty($this->user)) {
            $this->email = $this->user->email;
            $this->receiverName = $this->user->fullname;
        }
        $this->toAddress = $this->email;
    }
}
