<?php

if (!defined('ABSPATH')) exit;

class BuildecomUser extends WP_User
{
    public function __construct($user_id = false)
    {
        if (! $user_id) {
            throw new Exception(esc_html__('You must provide a $user_id to initiate a user object.', 'buildecom'));
        }

        parent::__construct($user_id);
    }

    public function send_reset_code()
    {
        $allowed_roles = buildecom_get_allowed_roles();
        $allowed       = false;

        foreach ($allowed_roles as $role) {
            if (! in_array($role, (array) $this->roles)) {
                continue;
            }

            $allowed = true;
        }

        if (! $allowed) {
            throw new Exception(esc_html__('You cannot request a password reset for a user with this role.', 'buildecom'));
        }

        $email      = $this->get_email_address();
        $code       = buildecom_generate_password_reset_token();
        $expiration = buildecom_get_new_code_expiration_time();

        $this->save_user_meta(
            'buildecom-password-reset-code',
            array(
                'code'    => $code,
                'expiry'  => $expiration,
                'attempt' => 0,
            )
        );

        return buildecom_send_password_reset_code_email($email, $code, $expiration);
    }

    public function set_new_password($code, $password)
    {

        $code_valid = $this->validate_code($code);

        if (! $code_valid) {
            throw new Exception(esc_html__('There was a problem validating the code.', 'buildecom'));
        }

        $this->delete_user_meta('buildecom-password-reset-code');
        return wp_set_password($password, $this->ID);
    }

    public function validate_code($code)
    {
        $now            = strtotime('now');
        $stored_details = $this->get_user_meta('buildecom-password-reset-code');

        if (! $code) {
            throw new Exception(esc_html__('Please provide a password reset code.', 'buildecom'));
        }

        if (! $stored_details) {
            throw new Exception(esc_html__('You must request a password reset code before you try to set a new password.', 'buildecom'));
        }

        $stored_code = intval($stored_details['code']);
        $code_expiry = $stored_details['expiry'];
        $attempt     = (isset($stored_details['attempt'])) ? $stored_details['attempt'] : 0;
        ++$attempt;
        $attempts_string = '';

        $attempts_max = apply_filters('buildecom_max_attempts', 3);

        if ($code !== $stored_code && $attempts_max !== -1) {

            $stored_details['attempt'] = $attempt;
            $remaining_attempts        = $attempts_max - $attempt;

            $this->save_user_meta('buildecom-password-reset-code', $stored_details);

            $attempts_string = sprintf(
                /* translators: %s: Number of remaining attempts */
                __('You have %s attempts remaining.', 'buildecom'),
                $remaining_attempts
            );

            if ($remaining_attempts <= 0) {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                $attempts_string = esc_html__('You have used the maximum number of attempts allowed. You must request a new code.', 'buildecom');
                $this->delete_user_meta('buildecom-password-reset-code');
            }

            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            throw new Exception(esc_html__('The reset code provided is not valid.', 'buildecom') . esc_html($attempts_string));
        }

        if ($code !== $stored_code) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            throw new Exception(esc_html__('The reset code provided is not valid.', 'buildecom'));
        }

        $expired = true;

        if ($code_expiry === -1) {
            $expired = false;
        }

        if ($now > $code_expiry) {
            $expired = false;
        }

        if (! $expired) {
            $this->delete_user_meta('buildecom-password-reset-code');
            throw new Exception(esc_html__('The reset code provided has expired.', 'buildecom'));
        }

        return true;
    }

    private function get_user_meta($key)
    {
        $value = get_user_meta($this->ID, $key, true);
        return ($value !== '') ? $value : false;
    }

    private function save_user_meta($key, $value)
    {
        return update_user_meta($this->ID, $key, $value);
    }

    private function delete_user_meta($key)
    {
        return delete_user_meta($this->ID, $key);
    }

    private function get_email_address()
    {
        return $this->user_email;
    }
}
