<?php

if (!defined('ABSPATH')) exit;

require_once 'class-api-auth.php';

/**
 * Handle all the api routes for buildecom. Some of them are protected(already protected through `buildecom_check_auth`) 
 * and some of thema are not. 
 */
class Buildecom_Api_Controller
{
    public function tokenStore(WP_REST_Request $request)
    {
        global $wpdb;

        $user_id = $request->get_param('user_id');
        $token   = $request->get_param('token');

        if (empty($user_id) || !is_numeric($user_id)) {
            return buildecom_response_error('validation_error', 'User ID is required and must be an integer.', [], 400);
        }

        if (empty($token)) {
            return buildecom_response_error('validation_error', 'Device token is required.', [], 400);
        }

        $user_id = intval($user_id);
        $token   = sanitize_text_field($token);

        $table_name = $wpdb->prefix . 'buildecom_device_tokens';

        $exists = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE user_id = %d AND device_token = %s",
                $user_id,
                $token
            )
        );

        if ($exists > 0) {
            return buildecom_response_error('token_exists', 'Device token already exists for this user.', [], 409);
        }

        $inserted = $wpdb->insert(
            $table_name,
            [
                'user_id'      => $user_id,
                'device_token' => $token,
            ],
            [
                '%d',
                '%s'
            ]
        );

        if ($inserted === false) {
            return buildecom_response_error('db_error', 'Failed to save device token.', [], 500);
        }

        return buildecom_response_success('token_added', 'Device token added successfully.');
    }

    public function deleteAllCartCoupons()
    {
        $user = $this->getUser();

        if (!$user) {
            return buildecom_response_success('authentication_error', 'User is not authenticated', [], 401);
        }

        if (!WC()->session) {
            WC()->initialize_session();
        } elseif (!WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        if (!WC()->customer) {
            WC()->customer = new WC_Customer($user->ID);
        }

        if (!WC()->cart) {
            WC()->cart = new WC_Cart();
        }

        WC()->cart->remove_coupons();
        WC()->cart->calculate_totals();

        return buildecom_response_success('cart_coupon_deleted', 'All coupons removed successfully');
    }

    public function deleteCartCoupon(WP_REST_Request $request)
    {
        $user = $this->getUser();

        if (!$user) {
            return buildecom_response_error('authentication_error', 'User is not authenticated', [], 401);
        }

        if (!WC()->session) {
            WC()->initialize_session();
        } elseif (!WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        WC()->session->set('customer_id', $user->ID);

        if (!WC()->customer) {
            WC()->customer = new WC_Customer($user->ID);
        }

        if (!WC()->cart) {
            WC()->cart = new WC_Cart();
        }

        WC()->cart->get_cart_from_session();

        $code = $request->get_param('code');

        $code = strtolower(trim(sanitize_text_field($code)));

        if (!$code || empty($code)) {
            return buildecom_response_error('validation_error', 'Coupon code is required', [], 400);
        }

        $applied_coupons = WC()->cart->get_applied_coupons();

        if (!in_array($code, $applied_coupons)) {
            return buildecom_response_error('coupon_not_in_cart', 'Coupon not in cart', [], 404);
        }

        if (!WC()->cart->has_discount($code)) {
            return buildecom_response_error('coupon_not_in_cart', 'Coupon not in cart', [], 404);
        }

        WC()->cart->remove_coupon($code);
        WC()->cart->calculate_totals();

        return buildecom_response_success('coupon_removed', 'Coupon removed successfully');
    }

    public function addcartCoupon(WP_REST_Request $request)
    {
        $user = $this->getUser();

        if (!$user) {
            return buildecom_response_error('authentication_error', 'User is not authenticated', [], 401);
        }

        if (!WC()->session) {
            WC()->initialize_session();
        } elseif (!WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        if (!WC()->customer) {
            WC()->customer = new WC_Customer($user->ID);
        }

        if (!WC()->cart) {
            WC()->cart = new WC_Cart();
        }

        $params = $request->get_json_params();
        $code   = sanitize_text_field($params['code']);

        if (!$code || empty($code)) {
            return buildecom_response_error('validation_error', 'Code is required', [], 400);
        }

        $coupon = new WC_Coupon($code);

        if (!$coupon->get_id() || !$coupon->get_code()) {
            return buildecom_response_error('coupon_not_found', 'Invalid coupon code.', [], 404);
        }

        if ($coupon->get_status() !== 'publish') {
            return buildecom_response_error('coupon_invalid', 'Coupon is not active.', [], 400);
        }

        $applied = WC()->cart->apply_coupon($code);

        if (is_wp_error($applied)) {
            return buildecom_response_error('coupon_error', $applied->get_error_message(), [], 400);
        }

        return buildecom_response_success('success', 'Coupon applied successfully');
    }

 public function checkoutHandler(WP_REST_Request $request)
{
    $user = $this->getUser();
    if (!$user) {
        return buildecom_response_error('authentication_error', 'User is not authenticated', [], 401);
    }

    // Ensure WC context
    if (!WC()->session) {
        WC()->initialize_session();
    } elseif (!WC()->session->has_session()) {
        WC()->session->set_customer_session_cookie(true);
    }
    if (!WC()->customer) {
        WC()->customer = new WC_Customer($user->ID);
    }
    if (!WC()->cart) {
        WC()->cart = new WC_Cart();
    }

    $cart = WC()->cart;
    if ($cart->is_empty()) {
        return buildecom_response_error('cart_empty', 'Cart is empty', [], 400);
    }

    // Basic stock checks
    foreach ($cart->get_cart() as $cart_item) {
        $product = wc_get_product($cart_item['product_id']);
        if (!$product || !$product->is_in_stock()) {
            return buildecom_response_error('out_of_stock', ($product ? $product->get_name() : 'A product') . ' is out of stock.', [], 400);
        }
        if (!$product->has_enough_stock($cart_item['quantity'])) {
            return buildecom_response_error('not_enough_stock', 'Not enough stock for ' . $product->get_name() . '. Only ' . $product->get_stock_quantity() . ' left.', [], 400);
        }
    }

    $posted_data = $request->get_json_params();
    $errors = [];

    // Validate required payload
    if (!isset($posted_data['billing_address']) || !is_array($posted_data['billing_address'])) {
        $errors['billing_address'] = 'Billing address is required.';
    } else {
        $billing = $posted_data['billing_address'];
        if (empty($billing['first_name'])) $errors['first_name'] = 'First name is required.';
        if (empty($billing['last_name']))  $errors['last_name']  = 'Last name is required.';
        if (empty($billing['email']))      $errors['email']      = 'Email is required.';
        elseif (!is_email($billing['email'])) $errors['email']   = 'Invalid email address.';
        if (empty($billing['address']))    $errors['address']    = 'Address is required.';
        if (empty($billing['city']))       $errors['city']       = 'City is required.';
        if (empty($billing['state']))      $errors['state']      = 'State is required.';
        if (empty($billing['postcode']))   $errors['postcode']   = 'Postcode is required.';
        if (empty($billing['country']))    $errors['country']    = 'Country is required.';
    }
    if (empty($posted_data['payment_method'])) {
        $errors['payment_method'] = 'Payment method is required.';
    }
    if (!empty($errors)) {
        return buildecom_response_error('validation_error', array_values($errors), [], 400);
    }

    // Prepare/flatten data Woo expects
    $billing  = $posted_data['billing_address'];
    $shipping = isset($posted_data['shipping_address']) && is_array($posted_data['shipping_address'])
        ? $posted_data['shipping_address']
        : $billing;

    $wc_posted = [
        // Billing
        'billing_first_name' => sanitize_text_field($billing['first_name']),
        'billing_last_name'  => sanitize_text_field($billing['last_name']),
        'billing_company'    => !empty($billing['company']) ? sanitize_text_field($billing['company']) : '',
        'billing_email'      => sanitize_email($billing['email']),
        'billing_phone'      => !empty($billing['phone']) ? sanitize_text_field($billing['phone']) : '',
        'billing_address_1'  => sanitize_text_field($billing['address']),
        'billing_address_2'  => !empty($billing['address_2']) ? sanitize_text_field($billing['address_2']) : '',
        'billing_city'       => sanitize_text_field($billing['city']),
        'billing_state'      => sanitize_text_field($billing['state']),
        'billing_postcode'   => sanitize_text_field($billing['postcode']),
        'billing_country'    => sanitize_text_field($billing['country']),

        // Shipping
        'shipping_first_name' => sanitize_text_field($shipping['first_name']),
        'shipping_last_name'  => sanitize_text_field($shipping['last_name']),
        'shipping_company'    => !empty($shipping['company']) ? sanitize_text_field($shipping['company']) : '',
        'shipping_address_1'  => sanitize_text_field($shipping['address']),
        'shipping_address_2'  => !empty($shipping['address_2']) ? sanitize_text_field($shipping['address_2']) : '',
        'shipping_city'       => sanitize_text_field($shipping['city']),
        'shipping_state'      => sanitize_text_field($shipping['state']),
        'shipping_postcode'   => sanitize_text_field($shipping['postcode']),
        'shipping_country'    => sanitize_text_field($shipping['country']),

        // Checkout meta
        'payment_method' => sanitize_text_field($posted_data['payment_method']),
        'terms'          => 'yes',
        'ship_to_different_address' => !empty($posted_data['ship_to_different_address']) ? '1' : '0',
    ];

    // Persist customer details too (optional, but nice)
    WC()->customer->set_props([
        'first_name' => $wc_posted['billing_first_name'],
        'last_name'  => $wc_posted['billing_last_name'],
        'email'      => $wc_posted['billing_email'],
        'phone'      => $wc_posted['billing_phone'],
        'billing'    => [
            'first_name' => $wc_posted['billing_first_name'],
            'last_name'  => $wc_posted['billing_last_name'],
            'company'    => $wc_posted['billing_company'],
            'address_1'  => $wc_posted['billing_address_1'],
            'address_2'  => $wc_posted['billing_address_2'],
            'city'       => $wc_posted['billing_city'],
            'state'      => $wc_posted['billing_state'],
            'postcode'   => $wc_posted['billing_postcode'],
            'country'    => $wc_posted['billing_country'],
            'email'      => $wc_posted['billing_email'],
            'phone'      => $wc_posted['billing_phone'],
        ],
        'shipping'   => [
            'first_name' => $wc_posted['shipping_first_name'],
            'last_name'  => $wc_posted['shipping_last_name'],
            'company'    => $wc_posted['shipping_company'],
            'address_1'  => $wc_posted['shipping_address_1'],
            'address_2'  => $wc_posted['shipping_address_2'],
            'city'       => $wc_posted['shipping_city'],
            'state'      => $wc_posted['shipping_state'],
            'postcode'   => $wc_posted['shipping_postcode'],
            'country'    => $wc_posted['shipping_country'],
        ],
    ]);
    WC()->customer->save();

    // Totals & chosen payment method must be set before create_order
    WC()->cart->calculate_totals();
    WC()->session->set('chosen_payment_method', $wc_posted['payment_method']);

    // Create order with flat data
    $order_id = WC()->checkout()->create_order($wc_posted);
    if (is_wp_error($order_id)) {
        return buildecom_response_error('order_creation_failed', $order_id->get_error_message(), [], 400);
    }
    $order = wc_get_order($order_id);

    // Attach payment method to order
    $gateways = WC()->payment_gateways()->get_available_payment_gateways();
    $pm = $wc_posted['payment_method'];
    if (empty($gateways[$pm])) {
        return buildecom_response_error('invalid_payment_method', 'Payment method not available.', [], 400);
    }
    $order->set_payment_method($gateways[$pm]);
    $order->set_customer_id($user->ID);
    $order->save();

    // For redirect/intent-based gateways (Stripe/PayPal/etc.) return the secure pay URL.
    // For offline methods (cod/bacs/cheque), process immediately.
    $offline = ['cod', 'bacs', 'cheque'];
    if (in_array($pm, $offline, true)) {
        $result = $gateways[$pm]->process_payment($order_id);
        if (is_wp_error($result) || empty($result['result']) || 'success' !== $result['result']) {
            return buildecom_response_error('payment_processing_error', is_wp_error($result) ? $result->get_error_message() : 'Payment failed.', [], 400);
        }
        // Usually sends to thank you
        $redirect_url = !empty($result['redirect']) ? $result['redirect'] : $order->get_checkout_order_received_url();
    } else {
        // Let Woo/gateway JS handle the flow on the order pay page
        $redirect_url = $order->get_checkout_payment_url(true);
    }

    // Build response
    $response_data = [
        'order_id'      => $order->get_id(),
        'status'        => $order->get_status(),
        'order_key'     => $order->get_order_key(),
        'customer_id'   => $order->get_customer_id(),
        'payment_method'=> $order->get_payment_method(),
        'redirect_url'  => $redirect_url,
        'billing_address' => [
            'first_name' => $order->get_billing_first_name(),
            'last_name'  => $order->get_billing_last_name(),
            'company'    => $order->get_billing_company(),
            'address_1'  => $order->get_billing_address_1(),
            'address_2'  => $order->get_billing_address_2(),
            'city'       => $order->get_billing_city(),
            'state'      => $order->get_billing_state(),
            'postcode'   => $order->get_billing_postcode(),
            'country'    => $order->get_billing_country(),
            'email'      => $order->get_billing_email(),
            'phone'      => $order->get_billing_phone(),
        ],
        'shipping_address' => [
            'first_name' => $order->get_shipping_first_name(),
            'last_name'  => $order->get_shipping_last_name(),
            'company'    => $order->get_shipping_company(),
            'address_1'  => $order->get_shipping_address_1(),
            'address_2'  => $order->get_shipping_address_2(),
            'city'       => $order->get_shipping_city(),
            'state'      => $order->get_shipping_state(),
            'postcode'   => $order->get_shipping_postcode(),
            'country'    => $order->get_shipping_country(),
        ],
    ];

    return new WP_REST_Response($response_data, 200);
}

    public function addItemToCart(WP_REST_Request $request)
    {
        $user = $this->getUser();

        if (!$user) {
            return buildecom_response_error('authentication_error', 'User is not authenticated', [], 401);
        }

        if (!WC()->session) {
            WC()->initialize_session();
        } elseif (!WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        if (!WC()->customer) {
            WC()->customer = new WC_Customer($user->ID);
        }

        if (!WC()->cart) {
            WC()->cart = new WC_Cart();
            WC()->cart->set_session(WC()->session->get_session);
        }

        $params       = $request->get_params();
        $product_id   = isset($params['id']) ? sanitize_text_field($params['id']) : '';
        $quantity     = isset($params['quantity']) ? sanitize_text_field($params['quantity']) : '1';
        $variation_id = isset($params['variation_id']) ? sanitize_text_field($params['variation_id']) : '';

        if (!$user) {
            return buildecom_response_error('authentication_error', 'User is not authenticated', [], 401);
        }

        if (!$product_id || !$quantity) {
            return buildecom_response_error('validation_error', 'Product ID and Quantity are required', [], 400);
        }

        if ($quantity <= 0) {
            return buildecom_response_error('invalid_quantity', 'Quantity must be greater than 0', [], 400);
        }

        $product = wc_get_product($product_id);

        if (!$product || !$product->is_purchasable()) {
            return buildecom_response_error('product_not_purchasable', 'Invalid product or product is not purchasable', [], 400);
        }

        if ($product->is_type('variable') && !$variation_id) {
            return buildecom_response_error('product_variation_not_found', 'No matching variation found', [], 400);
        }

        if ($variation_id) {
            $variation_product = wc_get_product($variation_id);

            if (!$variation_product || !$variation_product->is_type('variation') || $variation_product->get_parent_id() != $product_id) {
                return buildecom_response_error('invalid_variation_id', 'Invalid variation ID', [], 400);
            }

            if (!$variation_product->is_in_stock() || !$variation_product->has_enough_stock($quantity)) {
                return buildecom_response_error('out_of_stock', 'The variation is out of stock', [], 400);
            }
        }

        if (!$product->is_in_stock() || !$product->has_enough_stock($quantity)) {
            return buildecom_response_error('out_of_stock', 'The product is out of stock', [], 400);
        }

        $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity, $variation_id);

        if (!$cart_item_key) {
            return buildecom_response_error('product_not_added', 'Unable to add to cart', [], 400);
        }

        return buildecom_response_success('product_added', 'Product added to cart');
    }

    public function handleWoocommerceReview(WP_REST_Request $request)
    {
        $params     = $request->get_body_params();
        $product_id = isset($params['product_id']) ? sanitize_text_field($params['product_id']) : '';
        $review     = isset($params['review']) ? sanitize_text_field($params['review']) : '';
        $rating     = isset($params['rating']) ? sanitize_text_field($params['rating']) : '';
        $order_id   = isset($params['order_id']) ? sanitize_text_field($params['order_id']) : '';

        $user = $this->getUser();

        if (!$product_id || !$review || !$rating || !$order_id) {
            return buildecom_response_error('validation_error', 'Product ID, Review, Rating and Order ID are required', [], 400);
        }

        if ($rating < 1 || $rating > 5) {
            return buildecom_response_error('validation_error', 'Rating must be in between 1 and 5', [], 400);
        }

        $order = wc_get_order($order_id);

        if (!$order) {
            return buildecom_response_error('order_not_found', 'Order not found for this order ID', [], 404);
        }

        if (!buildecom_has_user_purchased_product($user_id, $product_id, $order_id)) {
            return buildecom_response_error('product_not_purchased', 'You must purchase the product to leave a review.', [], 403);
        }

        if (buildecom_has_user_reviewed_product($user_id, $product_id, $order_id)) {
            return buildecom_response_error('already_reviewed', 'You have already reviewed this product for this order', [], 403);
        }

        // Create the review
        $commentdata = [
            'comment_post_ID'      => $product_id,
            'comment_author'       => $user->display_name,
            'comment_author_email' => $user->user_email,
            'comment_content'      => $review,
            'comment_type'         => 'review',
            'comment_parent'       => 0,
            'user_id'              => $user_id,
            'comment_approved'     => 1,
        ];

        $comment_id = wp_insert_comment($commentdata);

        if ($comment_id) {
            update_comment_meta($comment_id, 'rating', $rating);

            $order = wc_get_order($order_id);

            foreach ($order->get_items() as $item) {
                if ($item->get_product_id() == $product_id && !wc_get_order_item_meta($item->get_id(), '_reviewed', true)) {
                    $item_id = $item->get_id();
                    wc_update_order_item_meta($item_id, '_reviewed', true);
                    wc_update_order_item_meta($item_id, '_review_text', $review);
                    wc_update_order_item_meta($item_id, '_review_rating', $rating);
                    break;
                }
            }

            return buildecom_response_success('review_submitted', 'Review submitted successfully');
        } else {
            return buildecom_response_error('review_failed', 'Unable to submit review', [], 500);
        }
    }

    public function deleteAccount()
    {
        global $wpdb;

        $user_id  = isset($_POST['user_id']) ? intval($_POST['user_id']) : false;
        $password = isset($_POST['password']) ? wp_unslash($_POST['password']) : false;

        $errors = [];

        if (empty($user_id)) {
            $errors['user_id'] = 'User ID is required.';
        }

        if (empty($password)) {
            $errors['password'] = 'Password is required.';
        }

        if (!empty($errors)) {
            return buildecom_response_error('validation_error', $errors, [], 400);
        }

        $user = get_userdata($user_id);

        if (!$user) {
            return buildecom_response_error('user_not_found', ['nouser' => 'User not found'], [], 404);
        }

        if (wp_check_password($password, $user->user_pass, $user->ID)) {
            $wpdb->query($wpdb->prepare("DELETE FROM $wpdb->users WHERE ID = %d", $user->ID));
            $wpdb->query($wpdb->prepare("DELETE FROM $wpdb->usermeta WHERE user_id = %d", $user->ID));

            return buildecom_response_success('user_deleted', ['msg' => 'User deleted successfully']);
        } else {
            return buildecom_response_error('incorrect_password', ['msg' => 'Wrong password'], [], 401);
        }
    }

    public function get_eligible_shipping_methods(WP_REST_Request $request)
    {
        $data = $request->get_json_params();

        if (empty($data['shipping_address'])) {
            return buildecom_response_error('no_shipping_address', 'No shipping address provided', [], 400);
        }

        if (!WC()->session) {
            WC()->initialize_session();
        } elseif (!WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        if (!WC()->customer) {
            WC()->customer = new WC_Customer();
        }

        if (!WC()->cart) {
            WC()->cart = new WC_Cart();
        }

        $shipping_address = $data['shipping_address'];

        WC()->customer->set_shipping_location($shipping_address['country'], $shipping_address['state'], $shipping_address['postcode'], $shipping_address['city']);

        if (isset($data['line_items'])) {
            WC()->cart->empty_cart();
            foreach ($data['line_items'] as $item) {
                if (isset($item['variation_id'])) {
                    WC()->cart->add_to_cart($item['product_id'], $item['quantity'], $item['variation_id']);
                } else {
                    WC()->cart->add_to_cart($item['product_id'], $item['quantity']);
                }
            }
        }

        if (isset($data['coupon_codes'])) {
            foreach ($data['coupon_codes'] as $coupon_code) {
                WC()->cart->apply_coupon($coupon_code);
            }
        }

        $package = array(
            'destination' => array(
                'country'   => $shipping_address['country'],
                'state'     => $shipping_address['state'],
                'postcode'  => $shipping_address['postcode'],
                'city'      => $shipping_address['city'],
                'address'   => $shipping_address['address_1'],
                'address_2' => $shipping_address['address_2'],
            ),
            'contents'        => WC()->cart->get_cart(),
            'contents_cost'   => WC()->cart->get_cart_contents_total(),
            'applied_coupons' => WC()->cart->get_applied_coupons(),
            'user'            => array(
                'ID' => 0,
            ),
        );

        WC()->shipping->calculate_shipping(array($package));

        $rates = [];
        foreach (WC()->shipping()->get_packages() as $package_key => $package) {
            foreach ($package['rates'] as $rate) {
                $rates[$rate->id] = array(
                    'id'        => $rate->get_id(),
                    'label'     => $rate->get_label(),
                    'cost'      => $rate->get_cost(),
                    'taxes'     => $rate->get_taxes(),
                    'method_id' => $rate->get_method_id(),
                );
            }
        }

        if (empty($rates)) {
            return buildecom_response_error('no_methods', 'No shipping methods available', ['shipping' => []], 400);
        }

        return buildecom_response_success('shipping_methods', 'Shipping methods available', ['shipping' => $rates]);
    }

    /**
     * Get the settings of the system
     * @return WP_REST_Response
     */
    public function fetchSettings()
    {
        $settings                                = [];
        $settings['buildecom_cur_text']          = get_option('buildecom_cur_text', null);
        $settings['buildecom_cur_sym']           = get_option('buildecom_cur_sym', null);
        $settings['woocommerce_cur_sym']         = html_entity_decode(get_woocommerce_currency_symbol()) ?? null;
        $settings['woocommerce_cur_text']        = get_woocommerce_currency() ?? null;
        $settings['buildecom_max_product_price'] = $this->get_max_product_price() ?? null;
        $settings['buildecom_policy']            = get_option('buildecom_policy', null);
        $settings['buildecom_terms']             = get_option('buildecom_terms', null);
        $settings['buildecom_about_us']          = get_option('buildecom_about_us', null);

        if ($settings) {
            return buildecom_response_success('settings_found', 'Settings found', [
                'settings' => $settings
            ]);
        } else {
            return buildecom_response_error('settings_not_found', 'No settings found', [
                'settings' => $settings
            ]);
        }
    }

    private function get_max_product_price()
    {
        $args = [
            'post_type'      => 'product',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ];

        $query = new \WP_Query($args);
        $max_price = 0;

        foreach ($query->posts as $product_id) {
            $product = wc_get_product($product_id);

            if ($product->is_type('variable')) {
                $variation_ids = $product->get_children();
                foreach ($variation_ids as $variation_id) {
                    $variation = wc_get_product($variation_id);
                    $price     = floatval($variation->get_price());
                    if ($price > $max_price) {
                        $max_price = $price;
                    }
                }
            } elseif ($product->is_type('grouped')) {
                $children = $product->get_children();
                foreach ($children as $child_id) {
                    $child_product = wc_get_product($child_id);
                    $price         = floatval($child_product->get_price());
                    if ($price > $max_price) {
                        $max_price = $price;
                    }
                }
            } else {
                $price = floatval($product->get_price());
                if ($price > $max_price) {
                    $max_price = $price;
                }
            }
        }

        return $max_price;
    }

    public function fetchTestimonials()
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'buildecom_testimonials';

        $testimonials = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE status = %d ORDER BY id DESC",
                1
            )
        );

        if (!buildecom_check_empty($testimonials)) {
            return buildecom_response_success('testimonials_found', 'Testimonials found', [
                'testimonials' => $testimonials
            ]);
        } else {
            return buildecom_response_error('testimonials_not_found', 'No testimonials found', [
                'testimonials' => []
            ]);
        }
    }

    public function fetchBannerCategories()
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'buildecom_categories';

        $id = isset($_GET['id']) ? intval(wp_unslash($_GET['id'])) : false;

        if ($id) {
            $categories = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM $table_name WHERE status = %d AND id = %d",
                    1,
                    $id
                )
            );
        } else {
            $categories = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM $table_name WHERE status = %d",
                    1
                )
            );
        }

        $categoryData = [];
        foreach ($categories as $singleCategory) {
            $categoryInfo = [];

            foreach ($singleCategory as $key => $category) {
                if ($key == 'product_data' || $key == 'category_data') {
                    $categoryInfo[$key] = json_decode($category);
                } else {
                    $categoryInfo[$key] = $category;
                }
            }
            $categoryData[] = $categoryInfo;
        }

        if (!buildecom_check_empty($categoryData)) {
            return buildecom_response_success('categories_found', 'Categories found', [
                'categories' => $categoryData
            ]);
        } else {
            return buildecom_response_error('categories_not_found', 'No categories found', [
                'categories' => []
            ]);
        }
    }

    /**
     * Fetch all banners
     * @return WP_REST_Response
     */
    function fetchBanners()
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'buildecom_banners';
        $position = isset($_GET['position']) ? sanitize_text_field(wp_unslash($_GET['position'])) : false;
        $banner_category = isset($_GET['banner_category']) ? intval(wp_unslash($_GET['banner_category'])) : false;

        // Fetch banners
        if ($position) {
            $banners = $wpdb->get_results(
                $wpdb->prepare("SELECT * FROM $table_name WHERE status = %d AND position = %s", 1, $position)
            );
        } elseif ($banner_category) {
            $banners = $wpdb->get_results(
                $wpdb->prepare("SELECT * FROM $table_name WHERE status = %d AND banner_category_id = %d", 1, $banner_category)
            );
        } else {
            $banners = $wpdb->get_results("SELECT * FROM $table_name WHERE status = 1");
        }

        $banner_data = [];

        foreach ($banners as $b) {
            $data = [];

            foreach ($b as $key => $value) {
                if ($key === 'product_data') {
                    $decoded = json_decode($value, true);
                    if (!empty($decoded['id'])) {
                        $product = wc_get_product($decoded['id']);
                        if ($product) {
                            $data['product'] = [
                                'id'    => $product->get_id(),
                                'title' => $product->get_name(),
                                'url'   => get_permalink($product->get_id()),
                                'price' => $product->get_price(),
                                'image' => wp_get_attachment_url($product->get_image_id()),
                            ];
                        }
                    }
                } elseif ($key === 'category_data') {
                    $decoded = json_decode($value, true);
                    if (!empty($decoded['term_id'])) {
                        $term = get_term($decoded['term_id'], 'product_cat');
                        if ($term && !is_wp_error($term)) {
                            $data['category'] = [
                                'term_id' => $term->term_id,
                                'name'    => $term->name,
                                'slug'    => $term->slug,
                                'url'     => get_term_link($term),
                            ];
                        }
                    }
                } else {
                    $data[$key] = $value;
                }
            }

            $banner_data[] = $data;
        }

        if (!empty($banner_data)) {
            return buildecom_response_success('banners_found', 'Banners found', [
                'banners' => $banner_data
            ]);
        } else {
            return buildecom_response_error('banner_not_found', 'No banner found', [
                'banners' => []
            ]);
        }
    }


    /**
     * Get the payment gateways from woocommerce
     * @return WP_REST_Response
     */
    public function getPaymentGateways()
    {
        $gateways         = WC()->payment_gateways->get_available_payment_gateways();
        $enabled_gateways = [];

        if ($gateways) {
            $enabled_gateways = array_filter($gateways, fn($gateway) => 'yes' == $gateway->enabled);
        }

        if ($enabled_gateways) {
            return buildecom_response_success('gateways_found', 'Payment gateways found', [
                'gateways' => $enabled_gateways
            ]);
        } else {
            return buildecom_response_success('gateways_not_found', 'No payment gateways found', [
                'gateways' => []
            ]);
        }
    }

    public function fetchStatus()
    {
        $status = json_decode(get_option('buildecom_status_sorted_order'), true);

        $status_list = json_decode(get_option('buildecom_status_order'), true);

        global $wpdb;

        $table_name = $wpdb->prefix . 'buildecom_order_statuses';

        $custom_status = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE status = %d ORDER BY id DESC",
                1
            )
        );

        $custom_status_colors = array(
            'wc-pending'         => '#ffcc00',
            'wc-pending-payment' => '#ffcc00',
            'wc-processing'      => '#fadd85',
            'wc-completed'       => '#0000ff',
            'wc-on-hold'         => '#ff6600',
            'wc-cancelled'       => '#cc0000',
            'wc-refunded'        => '#009999',
            'wc-failed'          => '#990000',
            'wc-draft'           => '#000000',
        );

        $status_list  = [];
        $storedStatus = json_decode(get_option('buildecom_status_sorted_order'), true);
        foreach ($storedStatus as $key => $status) {
            if (!isset($custom_status_colors[$key])) {
                error_log(' the custom one is : ' . $key);
            }

            $status_list[$key] = [
                'status'       => $status,
                'status_color' => isset($custom_status_colors[$key])
                    ? $custom_status_colors[$key]
                    : (($found = array_find($custom_status, function ($arrayItem) use ($key) {
                        return 'wc-' . strtolower($arrayItem->status_label) === $key;
                    })) ? $found->status_color : null),
            ];
        }

        if ($status_list) {
            return buildecom_response_success('status_found', 'Status found', [
                'tracking_status' => $status_list
            ]);
        } else {
            return buildecom_response_error('status_not_found', 'Status found', [
                'tracking_status' => []
            ]);
        }
    }

    /**
     * Update user profile
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function updateUserProfile(WP_REST_Request $request)
    {
        $user = $this->getUser();

        $user_id = $user ? $user->ID : 0;

        if (!$user) {
            return buildecom_response_error('authentication_error', 'User is not authenticated', [], 401);
        }

        $params     = $request->get_body_params();

        $first_name = sanitize_text_field($params['first_name'] ?? '');
        $last_name  = sanitize_text_field($params['last_name'] ?? '');
        $phone      = sanitize_text_field($params['phone'] ?? '');
        $address    = sanitize_text_field($params['address'] ?? '');
        $city       = sanitize_text_field($params['city'] ?? '');
        $state      = sanitize_text_field($params['state'] ?? '');
        $postcode   = sanitize_text_field($params['postcode'] ?? '');
        $country    = sanitize_text_field($params['country'] ?? '');

        if (empty($first_name) || empty($last_name)) {
            return buildecom_response_error('invalid_request', 'First name and last name is required', [], 400);
        }

        $user = get_user_by('id', $user_id);

        if (!$user) {
            return buildecom_response_error('user_not_found', 'User not found', [], 404);
        }

        $userdata = [
            'ID'         => $user->ID,
            'first_name' => $first_name,
            'last_name'  => $last_name,
        ];

        wp_update_user($userdata);

        update_user_meta($user->ID, 'buildecom_phone', $phone);
        update_user_meta($user->ID, 'buildecom_address', $address);
        update_user_meta($user->ID, 'buildecom_city', $city);
        update_user_meta($user->ID, 'buildecom_state', $state);
        update_user_meta($user->ID, 'buildecom_postcode', $postcode);
        update_user_meta($user->ID, 'buildecom_country', $country);

        if (!function_exists('wp_handle_upload')) {
            require_once(ABSPATH . 'wp-admin/includes/file.php');
        }

        if (isset($_FILES['profile_picture']) && !empty($_FILES['profile_picture'])) {
            $file          = $_FILES['profile_picture'];
            $ext           = strtolower(pathinfo(sanitize_file_name($file['name']), PATHINFO_EXTENSION));
            $allowed_types = ['jpeg', 'jpg', 'png'];

            if (in_array($ext, $allowed_types)) {

                $upload = wp_handle_upload($file, ['test_form' => false]);

                if (isset($upload['error']) && $upload['error'] !== '') {
                    return buildecom_response_error('upload_error', sanitize_text_field($upload['error']), [], 400);
                } else {
                    $attachment_id = wp_insert_attachment([
                        'guid'           => esc_url_raw($upload['url']),
                        'post_mime_type' => sanitize_mime_type($upload['type']),
                        'post_title'     => sanitize_file_name(basename($upload['file'])),
                        'post_content'   => '',
                        'post_status'    => 'inherit',
                    ], $upload['file']);

                    if (is_wp_error($attachment_id)) {
                        return buildecom_response_error('attachment_error', 'Failed to insert attachment into media library', [], 500);
                    }

                    require_once(ABSPATH . 'wp-admin/includes/image.php');

                    $attach_data = wp_generate_attachment_metadata($attachment_id, $upload['file']);
                    wp_update_attachment_metadata($attachment_id, $attach_data);

                    update_user_meta($user->ID, 'profile_picture', $attachment_id);
                }
            } else {
                return buildecom_response_error('invalid_file_type', 'Only JPG, JPEG, and PNG files are allowed', [], 400);
            }
        }  

        return buildecom_response_success('profile_updated', 'Profile updated successfully');
    }

    private function authenticateUser($auth)
    {
        $payload = $auth->getUserInfo(false);
        $user_id = $payload?->data?->user?->id;

        if ($user_id) {
            $user = get_user_by('id', $user_id);
            if ($user) {
                wp_set_current_user($user_id);
                wp_set_auth_cookie($user_id);
                return $user;
            }
        }

        return null;
    }

    /**
     * Get the user from the bearer token verification
     * @return bool|WP_User|null
     */
    private function getUser()
    {
        if (!class_exists('Buildecom_Api_Auth')) {
            require_once __DIR__ . '/class-api-auth.php';
        }

        $auth = new Buildecom_Api_Auth();
        $user = $this->authenticateUser($auth);

        return $user;
    }
}
