<?php

if (!defined('ABSPATH')) exit;

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class Buildecom_Api_Auth
{

    /**
     * Store errors to display if the JWT is wrong
     */
    private $jwt_error = null;

    /**
     * Collection of translate-able messages.
     */
    private $messages = array();

    public function __construct()
    {
        $this->messages = array(
            'buildecom_auth_no_auth_header'  => __('Authorization header not found.', 'buildecom'),
            'buildecom_auth_bad_auth_header' => __('Authorization header malformed.', 'buildecom'),
        );
    }

    /**
     * Add CORs suppot to the request.
     */
    public function add_cors_support()
    {
        global $wp_version;

        $enable_cors = defined('BUILDECOM_CORS_ENABLE') ? BUILDECOM_CORS_ENABLE : false;

        if (! $enable_cors) {
            return;
        }

        // Hook exists since 5.5.0
        if (version_compare($wp_version, '5.5.0', '>=')) {
            add_filter('rest_allowed_cors_headers', function (array $headers) {

                $filters = 'X-Requested-With, Content-Type, Accept, Origin, Authorization, Cookie';

                $split = preg_split("/[\s,]+/", $filters);

                return array_unique(array_merge($headers, $split));
            });
        } else if (! headers_sent()) {
            $headers = 'X-Requested-With, Content-Type, Accept, Origin, Authorization, Cookie';

            header(sprintf('Access-Control-Allow-Headers: %s', $headers));
        }
    }

    public function register_user(WP_REST_Request $request)
    {
        $params    = $request->get_body_params();
        $username  = isset($params['username']) ? sanitize_text_field($params['username']) : '';
        $password  = isset($params['password']) ? sanitize_text_field($params['password']) : '';
        $email     = isset($params['email']) ? sanitize_email($params['email']) : '';
        $firstName = isset($params['firstname']) ? sanitize_text_field($params['firstname']) : '';
        $lastName  = isset($params['lastname']) ? sanitize_text_field($params['lastname']) : '';

        if (empty($username)) {
            $errors[] = 'Error: The username field is empty.';
        }

        if (empty($password)) {
            $errors[] = 'Error: The password field is empty.';
        }

        if (empty($email)) {
            $errors[] = 'Error: The email field is empty.';
        }

        if (empty($firstName)) {
            $errors[] = 'Error: The first name field is empty.';
        }

        if (empty($lastName)) {
            $errors[] = 'Error: The last name field is empty.';
        }

        if (!empty($errors)) {
            return buildecom_response_error('missing_fields',  $errors, [], 400);
        }

        if (username_exists($username)) {
            return buildecom_response_error('user_exists', 'Username already exists', [], 400);
        }

        if (email_exists($email)) {
            return buildecom_response_error('user_exists', 'Email already exists', [], 400);
        }

        $user_id = wp_create_user($username, $password, $email);

        if (is_wp_error($user_id)) {
            return buildecom_response_error('user_creation_failed', 'Failed to create user', [], 500);
        }

        wp_update_user(array(
            'ID'         => $user_id,
            'first_name' => $firstName,
            'last_name'  => $lastName,
        ));

        $user = get_user_by('id', $user_id);
        $data = $this->generate_token($user, false);

        // this option is excluded from admin setting
        $isEnableEmailVerification = false;

        if (!$isEnableEmailVerification) {
            update_user_meta($user_id, 'email_confirmed', 1);
        } else {

            $code = buildecom_generate_password_reset_token();
            update_user_meta($user->ID, 'buildecom_email_verification_code', $code);
        }

        $response = array(
            'status'  => 'success',
            'remark'  => 'user_created',
            'message' => [
                'success' => ['Registration successful.']
            ],
            'data'    => $data,
        );

        return $response;
    }

    public function sendEmail(WP_REST_Request $request)
    {
        $payload = $this->getUserInfo(false);
        $user    = get_user_by('id', @$payload->data->user->id);
        if (!$user) {
            return buildecom_response_error('user_not_found', 'User not found', [], 400);
        }
    
        $request_id = $request->get_header('X-Request-Id') ?: 'send-email-'.$user->ID.'-'.gmdate('YmdHi');
        if ( get_transient($request_id) ) {
            return buildecom_response_success('email_already_sent', 'Email already sent just now.', []);
        }
        set_transient($request_id, 1, 60); 
    
        $code = buildecom_generate_password_reset_token();
        update_user_meta($user->ID, 'buildecom_email_verification_code', $code);
        $this->send_confirmation_email($user, $code);
    
        return buildecom_response_success('email_sent', 'Email sent successfully.', []);
    }

    public function send_confirmation_email($user, $code)
    {
        $to      = $user->user_email;
        $subject = 'Email Confirmation';
        $message = 'Please confirm your email by using the given token: ' . $code;

        $siteName  = get_bloginfo('name');
        $mailFrom  = get_option('admin_email');
        $headers   = "From: $siteName <$mailFrom>\r\n";
        $headers  .= "Reply-To: $siteName <$mailFrom> \r\n";
        $headers  .= "MIME-Version: 1.0\r\n";
        $headers  .= "Content-Type: text/html; charset=utf-8\r\n";

        wp_mail($to, $subject, $message, $headers);
    }

    public function confirmEmail(WP_REST_Request $request)
    {
        $params  = $request->get_body_params();
        $code    = $params['code'];
        $payload = $this->getUserInfo(false);
        $user_id = @$payload->data->user->id;
        $user    = get_user_by('id', $user_id);

        if (!$user) {
            return buildecom_response_error('user_not_found', 'User not found', [], 400);
        }

        if (empty($code)) {
            return buildecom_response_error('empty_code', 'Please provide verification code', [], 400);
        }

        if (get_user_meta($user_id, 'buildecom_email_verification_code', true) !== $code) {
            return buildecom_response_error('invalid_code', 'Invalid code', [], 400);
        }

        update_user_meta($user_id, 'email_confirmed', true);

        delete_user_meta($user_id, 'email_verification_code');

        return buildecom_response_success('email_confirmed', 'Email verified successfully');
    }

    public function get_token( WP_REST_Request $request ) {
		$secret_key = defined( 'BUILDECOM_SECRET_KEY' ) ? BUILDECOM_SECRET_KEY : false;

		$body_params = $request->get_body_params();
		$username    = isset( $body_params['username'] ) ? $body_params['username'] : '';
		$password    = isset( $body_params['password'] ) ? $body_params['password'] : '';

		if ( ! $secret_key ) {
			return new WP_REST_Response(
				array(
					'status'  => 'error',
					'remark'  => 'buildecom_auth_bad_config',
					'message' => ['error' =>[ 'Buildecom Auth Secret Key is not configured properly.']],
				),
				500
			);
		}

		if (is_email($username)) {
			$user_data = get_user_by('email', $username);
			if ($user_data) {
				$username = $user_data->user_login;
			}
		}

		$user = wp_authenticate_username_password(null, $username, $password );;
		
		if ( is_wp_error( $user ) ) {
			$error_code = $user->get_error_code();
			
			return new WP_REST_Response(
				array(
					'status'  => 'error',
					'remark'  => $error_code,
					'message' => ['error' => [wp_strip_all_tags( $user->get_error_message( $error_code ) )]],
				),
				401
			);
		}

		$data = $this->generate_token( $user, false );

		$response = array(
			'status'  => 'success',
			'remark'  => 'login_success',
			'message' => ['success' => ['Login successful.']],
			'data'    => $data,
		);

		return $response;
	}

    public function generate_token($user, $return_raw = true)
    {
        $secret_key    = defined('BUILDECOM_SECRET_KEY') ? BUILDECOM_SECRET_KEY : false;
        $issued_at     = time();
        $not_before    = $issued_at;
        $not_before    = apply_filters('buildecom_auth_not_before', $not_before, $issued_at);
        $expire        = $issued_at + MONTH_IN_SECONDS;
        $expire        = apply_filters('buildecom_auth_expire', $expire, $issued_at);
        $emailConfimed = get_user_meta($user->ID, 'email_confirmed', true) ? 1 : 0;

        $payload = array(
            'iss'  => $this->get_iss(),
            'iat'  => $issued_at,
            'nbf'  => $not_before,
            'exp'  => $expire,
            'data' => array(
                'user' => array(
                    'id' => $user->ID,
                ),
            ),
        );

        $alg = $this->get_alg();

        $token = JWT::encode($payload, $secret_key, $alg);

        if ($return_raw) {
            return $token;
        }

        $phone           = get_user_meta($user->ID, 'buildecom_phone', true);
        $address         = get_user_meta($user->ID, 'buildecom_address', true);
        $city            = get_user_meta($user->ID, 'buildecom_city', true);
        $state           = get_user_meta($user->ID, 'buildecom_state', true);
        $postcode        = get_user_meta($user->ID, 'buildecom_postcode', true);
        $country         = get_user_meta($user->ID, 'buildecom_country', true);
        $profile_picture = get_user_meta($user->ID, 'profile_picture', true);
        $image           = wp_get_attachment_image_url($profile_picture, 'medium');

        $response = array(
            'user' => [
                'id'              => $user->ID,
                'email'           => $user->user_email,
                'nicename'        => $user->user_nicename,
                'userName'        => $user->user_login,
                'firstName'       => $user->first_name,
                'lastName'        => $user->last_name,
                'displayName'     => $user->display_name,
                'phone'           => $phone,
                'address'         => $address,
                'city'            => $city,
                'state'           => $state,
                'postcode'        => $postcode,
                'country'         => $country,
                'ev'              => $emailConfimed,
                'profile_picture' => $image,
            ],
            'token'      => $token,
            'expires'    => $expire,
            "token_type" => "Bearer",
        );

        return $response;
    }


    public function send_refresh_token(\WP_REST_Request $request)
    {
        $user          = wp_get_current_user();
        $refresh_token = bin2hex(random_bytes(32));
        $created       = time();
        $expires       = $created + DAY_IN_SECONDS * 30;
        $expires       = apply_filters('buildecom_auth_refresh_expire', $expires, $created);

        setcookie('refresh_token', $user->ID . '.' . $refresh_token, $expires, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);

        $user_refresh_tokens = get_user_meta($user->ID, 'buildecom_auth_refresh_tokens', true);
        if (! is_array($user_refresh_tokens)) {
            $user_refresh_tokens = array();
        }
        $device = $request->get_param('device') ?: '';
        $user_refresh_tokens[$device] = array(
            'token'   => $refresh_token,
            'expires' => $expires,
        );
        update_user_meta($user->ID, 'buildecom_auth_refresh_tokens', $user_refresh_tokens);

        $expires_next = $expires;
        foreach ($user_refresh_tokens as $device) {
            if ($device['expires'] < $expires_next) {
                $expires_next = $device['expires'];
            }
        }
        update_user_meta($user->ID, 'buildecom_auth_refresh_tokens_expires_next', $expires_next);
    }

    public function get_iss()
    {
        return apply_filters('buildecom_auth_iss', get_bloginfo('url'));
    }

    public function get_alg()
    {
        return apply_filters('buildecom_auth_alg', 'HS256');
    }

    public function is_error_response($response)
    {
        if (! empty($response) && property_exists($response, 'data') && is_array($response->data)) {
            if (! isset($response->data['success']) || ! $response->data['success']) {
                return true;
            }
        }

        return false;
    }

    public function getUserInfo($return_response = true)
    {
        $headerkey = apply_filters('buildecom_auth_authorization_header', 'HTTP_AUTHORIZATION');
        $auth      = isset($_SERVER[$headerkey]) ? sanitize_text_field(wp_unslash($_SERVER[$headerkey])) : false;

        if (! $auth) {
            $auth = isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION']) ? sanitize_text_field(wp_unslash($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) : false;
        }

        if (! $auth) {
            return buildecom_response_error('no_auth_header', $this->messages['buildecom_auth_no_auth_header'], [], 401);
        }

        list($token) = sscanf($auth, 'Bearer %s');

        if (! $token) {
            return buildecom_response_error('bad_auth_header', $this->messages['buildecom_auth_bad_auth_header'], [], 401);
        }

        $secret_key = defined('BUILDECOM_SECRET_KEY') ? BUILDECOM_SECRET_KEY : false;

        if (! $secret_key) {
            return buildecom_response_error('auth_bad_config', 'Buildecom Auth secret key is not configured properly', [], 401);
        }

        // Try to decode the token.
        try {
            $alg     = $this->get_alg();
            $payload = JWT::decode($token, new Key($secret_key, $alg));
            if ($payload->iss !== $this->get_iss()) {
                return buildecom_response_error('auth_bad_iss', 'The iss do not match with this server', [], 401);
            }

            if (! isset($payload->data->user->id)) {
                return buildecom_response_error('buildecom_auth_bad_request', 'User ID not found in the token', [], 401);
            }

            $user = get_user_by('id', $payload->data->user->id);

            if (! $user) {
                return buildecom_response_error('auth_user_not_found', 'User doesn\'t exist', [], 401);
            }

            $failed_msg = apply_filters('buildecom_auth_extra_token_check', '', $user, $token, $payload);

            if (! empty($failed_msg)) {
                return buildecom_response_error('auth_obsolete_token', 'Token is obsolete', [], 401);
            }

            if (! $return_response) {
                return $payload;
            }

            $phone           = get_user_meta($user->ID, 'buildecom_phone', true);
            $address         = get_user_meta($user->ID, 'buildecom_address', true);
            $city            = get_user_meta($user->ID, 'buildecom_city', true);
            $state           = get_user_meta($user->ID, 'buildecom_state', true);
            $postcode        = get_user_meta($user->ID, 'buildecom_postcode', true);
            $country         = get_user_meta($user->ID, 'buildecom_country', true);
            $profile_picture = get_user_meta($user->ID, 'profile_picture', true);
            $image           = wp_get_attachment_image_url($profile_picture, 'medium');
            $emailConfimed   = get_user_meta($user->ID, 'email_confirmed', true) ? 1 : 0;

            $response = [
                'status'  => 'success',
                'remark'  => 'user_information',
                'message' => ['success' => ['User Information']],
                'data'    => [
                    'user' => [
                        'id'              => $user->ID,
                        'email'           => $user->user_email,
                        'userName'        => $user->user_login,
                        'nicename'        => $user->user_nicename,
                        'firstName'       => $user->first_name,
                        'lastName'        => $user->last_name,
                        'displayName'     => $user->display_name,
                        'phone'           => $phone,
                        'address'         => $address,
                        'city'            => $city,
                        'state'           => $state,
                        'postcode'        => $postcode,
                        'country'         => $country,
                        'ev'              => $emailConfimed,
                        'profile_picture' => $image,
                    ],
                    'billing_address' => [
                        'first_name'  => get_user_meta($user->ID, 'billing_first_name', true),
                        'last_name'   => get_user_meta($user->ID, 'billing_last_name', true),
                        'company'     => get_user_meta($user->ID, 'billing_company', true),
                        'address_1'   => get_user_meta($user->ID, 'billing_address_1', true),
                        'address_2'   => get_user_meta($user->ID, 'billing_address_2', true),
                        'city'        => get_user_meta($user->ID, 'billing_city', true),
                        'state'       => get_user_meta($user->ID, 'billing_state', true),
                        'postcode'    => get_user_meta($user->ID, 'billing_postcode', true),
                        'country'     => get_user_meta($user->ID, 'billing_country', true),
                        'email'       => $user->user_email,
                        'phone'       => get_user_meta($user->ID, 'billing_phone', true),
                    ],
                    'shipping_address' => [
                        'first_name'  => get_user_meta($user->ID, 'shipping_first_name', true),
                        'last_name'   => get_user_meta($user->ID, 'shipping_last_name', true),
                        'company'     => get_user_meta($user->ID, 'shipping_company', true),
                        'address_1'   => get_user_meta($user->ID, 'shipping_address_1', true),
                        'address_2'   => get_user_meta($user->ID, 'shipping_address_2', true),
                        'city'        => get_user_meta($user->ID, 'shipping_city', true),
                        'state'       => get_user_meta($user->ID, 'shipping_state', true),
                        'postcode'    => get_user_meta($user->ID, 'shipping_postcode', true),
                        'country'     => get_user_meta($user->ID, 'shipping_country', true),
                        'phone'       => get_user_meta($user->ID, 'shipping_phone', true),
                    ]
                ],
            ];

            $response = apply_filters('buildecom_auth_valid_token_response', $response, $user, $token, $payload);

            return new WP_REST_Response($response);
        } catch (\Exception $e) {
            return buildecom_response_error('auth_invalid_token', $e->getMessage(), [], 401);
        }
    }

    public function rest_pre_dispatch($result, WP_REST_Server $server, WP_REST_Request $request)
    {
        if ($this->is_error_response($this->jwt_error)) {
            return $this->jwt_error;
        }

        if (empty($result)) {
            return $result;
        }

        return $result;
    }

    public function passwordReset($data)
    {
        if (empty($data['email']) || $data['email'] === '') {
            return buildecom_response_error('no_email', 'You must provide an email address');
        }

        $exists = email_exists($data['email']);

        if (! $exists) {
            return buildecom_response_error('bad_email', 'No user found with this email address', [], 500);
        }

        try {
            $user = buildecom_get_user($exists);
            $user->send_reset_code();
        } catch (Exception $e) {
            return buildecom_response_error('bad_request', $e->getMessage(), [], 500);
        }

        return buildecom_response_success('password_reset_email_sent', 'A password reset email has been sent to your email address');
    }


    public function passwordChangeRequest()
    {
        $code  = isset($_POST['code']) ? wp_unslash($_POST['code']) : false;
        $email = isset($_POST['email']) ? wp_unslash($_POST['email']) : false;

        $errors = [];

        if (empty($code)) {
            $errors['code'] = 'Reset code is required.';
        }

        if (empty($email)) {
            $errors['email'] = 'Email address is required.';
        } elseif (!is_email($email)) {
            $errors['email'] = 'Email address is not valid.';
        }

        if (!empty($errors)) {
            return buildecom_response_error('validation_error', $errors, [], 400);
        }

        $code   = intval($code);
        $user_id = email_exists($email);

        if (!$user_id) {
            return buildecom_response_error('user_not_found', 'No user found with the provided email address', [], 404);
        }

        if (!class_exists('BuildecomUser')) {
            require_once __DIR__ . '/class-buildecom-user.php';
        }

        try {
            $user = new BuildecomUser($user_id);
            $user->validate_code($code);

            update_user_meta($user->ID, 'buildecom_password_request_approved', 1);

            return buildecom_response_success('password_reset_token_verified', 'Password reset token verified');
        } catch (Exception $e) {
            return buildecom_response_error('bad_request', $e->getMessage(), [], 400);
        }
    }

    public function resetPasswordChangeConfirm()
    {
        $code                  = isset($_POST['code']) ? wp_unslash($_POST['code']) : false;
        $password              = isset($_POST['password']) ? wp_unslash($_POST['password']) : false;
        $password_confirmation = isset($_POST['password_confirmation']) ? wp_unslash($_POST['password_confirmation']) : false;
        $email                 = isset($_POST['email']) ? wp_unslash($_POST['email']) : false;

        $errors = [];

        if (empty($code)) {
            $errors['code'] = 'Reset code is required.';
        } elseif (!is_numeric($code)) {
            $errors['code'] = 'Reset code must be an integer.';
        }

        if (empty($password)) {
            $errors['password'] = 'Password is required.';
        }

        if ($password !== $password_confirmation) {
            $errors['password_confirmation'] = 'Password confirmation does not match.';
        }

        if (empty($email)) {
            $errors['email'] = 'Email is required.';
        } elseif (!is_email($email)) {
            $errors['email'] = 'Email address is not valid.';
        }

        if (!empty($errors)) {
            return buildecom_response_error('validation_error', $errors, [], 400);
        }

        $token = intval($code);
        $user_id = email_exists($email);

        if (!$user_id) {
            return buildecom_response_error('user_not_found', 'No user found with the provided email address', [], 404);
        }

        if (!class_exists('BuildecomUser')) {
            require_once __DIR__ . '/class-buildecom-user.php';
        }

        try {
            $user = new BuildecomUser($user_id);
            $user->set_new_password($token, $password);

            delete_user_meta($user->ID, 'buildecom_password_request_approved');

            return buildecom_response_success('password_reset_success', 'Password changed successfully');
        } catch (Exception $e) {
            return buildecom_response_error('bad_request', $e->getMessage(), [], 400);
        }
    }

        public function changePassword()
    {
        $current_password          = isset($_POST['current_password']) ? wp_unslash($_POST['current_password']) : false;
        $new_password              = isset($_POST['new_password']) ? wp_unslash($_POST['new_password']) : false;
        $new_password_confirmation = isset($_POST['new_password_confirmation']) ? wp_unslash($_POST['new_password_confirmation']) : false;

        $errors = [];

        if (empty($current_password)) {
            $errors[] = 'Current password is required.';
        }

        if (empty($new_password)) {
            $errors[] = 'New password is required.';
        }

        if ($new_password !== $new_password_confirmation) {
            $errors[] = 'Password confirmation does not match.';
        }

        if (!empty($errors)) {
            return buildecom_response_error('validation_error', $errors, [], 400);
        }

        $payload = $this->getUserInfo(false);
        $user_id = isset($payload->data->user->id) ? intval($payload->data->user->id) : 0;

        $user = get_user_by('id', $user_id);

        if (!$user) {
            return buildecom_response_error('user_not_found', 'User not found', [], 404);
        }

        if (!wp_check_password($current_password, $user->user_pass, $user->ID)) {
            return buildecom_response_error('incorrect_password', 'Old password is incorrect', [], 400);
        }

        wp_set_password($new_password, $user->ID);

        return buildecom_response_success('password_changed', 'Password changed successfully');
    }
}
