<?php

if (!defined('ABSPATH'))
    exit;

/*
 * Plugin Name: BuildEcom
 * Plugin URI: https://buildecom.app/wp-plugin/
 * Description: The official WordPress plugin for BuildEcom. Build eCommerce mobile application through woocommerce and wordpress
 * Version: 1.1.2
 * Author: BuildEcom
 * Author URI: https://buildecom.app
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: buildecom
 * Requires: WooCommerce
 */

/**
 * Define some constants for global uses
 */
defined('BUILDECOM_DIR_PATH') or define('BUILDECOM_DIR_PATH', plugin_dir_path(__FILE__));
defined('BUILDECOM_ASSET_PATH') or define('BUILDECOM_ASSET_PATH', plugin_dir_url(__FILE__) . 'assets/');
defined('BUILDECOM_VERSION') or define('BUILDECOM_VERSION', "1.0");
defined('BUILDECOM_API_V1') or define('BUILDECOM_API_V1', "buildecom/v1");
defined('BUILDECOM_SECRET_KEY') or define('BUILDECOM_SECRET_KEY', 'asdfasdfasdf20d203.com');
defined('BUILDECOM_CORS_ENABLE') or define('BUILDECOM_CORS_ENABLE', false);

final class Buildecom
{
    private static $_instance = null;

    /**
     * When the initialize, init the plugin, enqueu assets for the admin
     */
    public function __construct()
    {
        add_action('plugins_loaded', [$this, 'init_plugin']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }


    /**
     * Enqueu admin assets(css/js) 
     * @return void
     */
    public function enqueue_admin_assets()
    {
        if (!isset($_GET['page']) || strpos($_GET['page'], 'buildecom') !== 0) {
            return;
        }

        wp_enqueue_Script('buildecom-bootrap-js', plugin_dir_url(__FILE__) . 'assets/js/bootstrap.bundle.min.js', [], BUILDECOM_VERSION, true);
        wp_enqueue_style('buildecom-admin-style', plugin_dir_url(__FILE__) . 'assets/css/bootstrap.min.css', [], BUILDECOM_VERSION);
        wp_enqueue_style('buildecom-admin-lineawesome', plugin_dir_url(__FILE__) . 'assets/css/line-awesome.min.css', [], BUILDECOM_VERSION);
        wp_enqueue_style('buildecom-admin-custom-style', plugin_dir_url(__FILE__) . 'assets/css/style.css', [], BUILDECOM_VERSION);
        wp_enqueue_script('jquery-ui-core');
        wp_enqueue_script('jquery-ui-sortable');
    }

    /**
     * Runs when the plugin initialized
     * @return void
     */
    public function init_plugin()
    {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('woocommerce_order_status_changed', 'buildecom_wc_order_status_changed', 10, 2);
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu()
    {
        add_menu_page(
            __('BuildEcom Dashboard', 'buildecom'),
            __('BuildEcom', 'buildecom'),
            'manage_options',
            'buildecom-dashboard',
            [$this, 'render_buildecom_home_page'],
            plugin_dir_url(__FILE__) . 'assets/images/buildecom-icon.svg',
            26
        );

        add_submenu_page(
            'buildecom-dashboard',
            __('Dashboard', 'buildecom'),
            __('Dashboard', 'buildecom'),
            'manage_options',
            'buildecom-dashboard',
            [$this, 'render_buildecom_home_page']
        );

        add_submenu_page(
            'buildecom-dashboard',
            __('Banners', 'buildecom'),
            __('Banners', 'buildecom'),
            'manage_options',
            'buildecom-banners',
            [$this, 'render_buildecom_banners_page']
        );

        add_submenu_page(
            'buildecom-dashboard',
            __('Testimonials', 'buildecom'),
            __('Testimonials', 'buildecom'),
            'manage_options',
            'buildecom-testimonials',
            [$this, 'render_buildecom_testimonials_page']
        );

        add_submenu_page(
            'buildecom-dashboard',
            __('Order Status', 'buildecom'),
            __('Order Status', 'buildecom'),
            'manage_options',
            'buildecom-order-status',
            [$this, 'render_buildecom_order_status_page']
        );

        add_submenu_page(
            'buildecom-dashboard',
            __('Notification', 'buildecom'),
            __('Notification', 'buildecom'),
            'manage_options',
            'buildecom-notification',
            [$this, 'render_buildecom_notification_page']
        );

        add_submenu_page(
            'buildecom-dashboard',
            __('Setting', 'buildecom'),
            __('Settings', 'buildecom'),
            'manage_options',
            'buildecom-setting',
            [$this, 'render_buildecom_settings_page']
        );
    }

    /**
     * Render all pages of banners 
     * @return void
     */
    public function render_buildecom_banners_page()
    {
        $page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : false;
        $current_action = isset($_GET['action']) ? sanitize_key(wp_unslash($_GET['action'])) : false;

        if ('buildecom-banners' !== $page || !$page) {
            wp_die(esc_html__('You cannot see the page', 'buildecom'));
        }

        wp_enqueue_media();

        wp_enqueue_script('buildecom-banners-script', BUILDECOM_ASSET_PATH . '/js/banners.js', ['jquery'], BUILDECOM_VERSION);

        wp_localize_script('buildecom-banners-script', 'buildecom_banner', [
            'choose_banner_image' => __('Choose Banner Image', 'buildecom'),
            'choose_image' => __('Choose Image', 'buildecom')
        ]);

        // Load list.php even if there's an action like delete/toggle
        if (
            !$current_action ||
            in_array($current_action, ['delete', 'toggle_status'], true)
        ) {
            // This ensures the action variable is available in list.php
            require_once BUILDECOM_DIR_PATH . 'inc/banners/list.php';
        } elseif ('add' === $current_action) {
            require_once BUILDECOM_DIR_PATH . 'inc/banners/add.php';
        } elseif ('edit' === $current_action && isset($_GET['id'])) {
            require_once BUILDECOM_DIR_PATH . 'inc/banners/edit.php';
        } elseif ('categories-add' === $current_action) {
            require_once BUILDECOM_DIR_PATH . 'inc/banner-categories/add.php';
        } elseif ('category-edit' === $current_action && isset($_GET['id'])) {
            require_once BUILDECOM_DIR_PATH . 'inc/banner-categories/edit.php';
        } else {
            require_once BUILDECOM_DIR_PATH . 'inc/banner-categories/list.php';
        }
    }


    /**
     * Render all pages for testimonials
     * @return void
     */
    public function render_buildecom_testimonials_page()
    {
        $page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : false;
        $action = isset($_GET['action']) ? sanitize_key(wp_unslash($_GET['action'])) : false;

        if ('buildecom-testimonials' !== $page || !$page) {
            wp_die(esc_html__('You cannot see the page', 'buildecom'));
        }

        if (in_array($action, ['testimonial_toggle_status', 'delete'], true)) {
            if (
                !isset($_GET['_buildecom_nonce']) ||
                !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_buildecom_nonce'])), 'buildecom_testimonial_action')
            ) {
                wp_die(esc_html__('Security check failed', 'buildecom'));
            }
        }

        require_once plugin_dir_path(__FILE__) . 'inc/testimonials/index.php';

        if (!$action || 'testimonial_toggle_status' === $action || 'delete' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/testimonials/list.php';
        } else {
            buildecom_enqueue_testimonial_assets();
            wp_enqueue_media();

            if ('add-testimonial' === $action) {
                require_once plugin_dir_path(__FILE__) . 'inc/testimonials/add.php';
            } else if ('edit' === $action) {
                require_once plugin_dir_path(__FILE__) . 'inc/testimonials/edit.php';
            }
        }
    }

    /**
     * Render all pages for settings of buildecom
     * @return void
     */
    public function render_buildecom_settings_page()
    {
        $page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : false;
        $action = isset($_GET['action']) ? sanitize_key(wp_unslash($_GET['action'])) : 'privacy-policy';

        if ('buildecom-setting' !== $page || !$page) {
            wp_die(esc_html__('You cannot see the page', 'buildecom'));
        }

        require_once plugin_dir_path(__FILE__) . 'inc/setting/index.php';

        buildecom_enqueue_settings_assets();

        // the general settting page
        if (!$action) {
            require_once plugin_dir_path(__FILE__) . 'inc/setting/general.php';
        } else if ('privacy-policy' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/setting/privacy-policy.php';
            buildecom_enqueue_nicedit_assets();
        } else if ('terms-and-condition' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/setting/terms-and-condition.php';
            buildecom_enqueue_nicedit_assets();
        } else if ('about-us' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/setting/about-us.php';
            buildecom_enqueue_nicedit_assets();
        }
    }

    /**
     * Render all the page related to notification configuration
     * @return void
     */
    public function render_buildecom_notification_page()
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'buildecom'));
        }

        $page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : false;
        $action = isset($_GET['action']) ? sanitize_key(wp_unslash($_GET['action'])) : false;

        if (
            $action &&
            (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'buildecom_notification_action'))
        ) {
            wp_die(esc_html__('Security check failed. Please refresh the page and try again.', 'buildecom'));
        }

        if ('buildecom-notification' !== $page || !$page) {
            wp_die(esc_html__('You cannot see the page', 'buildecom'));
        }

        require_once plugin_dir_path(__FILE__) . 'inc/notification/index.php';

        buildecom_enqueue_notification_assets();

        if (!$action) {
            require_once plugin_dir_path(__FILE__) . 'inc/notification/templates.php';
        } else if ('edit-template' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/notification/edit-template.php';
            buildecom_enqueue_notification_assets();
        } else if ('notification-config' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/notification/notification-config.php';
            buildecom_enqueue_notification_config_assets();
        }
    }

    /**
     * Render all pages for order statuses
     * @return void
     */
    public function render_buildecom_order_status_page()
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'buildecom'));
        }

        if (!isset($_GET['page']) || 'buildecom-order-status' !== sanitize_key(wp_unslash($_GET['page']))) {
            wp_die(esc_html__('You are not allowed to see the page', 'buildecom'));
        }

        $action = false;

        if (isset($_GET['action'])) {
            $action_raw = sanitize_key(wp_unslash($_GET['action']));

            $actions_require_nonce = ['order-status_toggle_status', 'delete-order-status', 'edit-order-status', 'add-order-status', 'render-tracking-page'];

            if (in_array($action_raw, $actions_require_nonce, true)) {
                $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
                if (!wp_verify_nonce($nonce, 'buildecom_order_status_action')) {
                    wp_die(esc_html__('Security check failed. Please try again.', 'buildecom'));
                }
            }

            $action = $action_raw;
        }

        require_once plugin_dir_path(__FILE__) . 'inc/order-status/index.php';

        if (!$action || 'order-status_toggle_status' === $action || 'delete-order-status' === $action) {
            require_once plugin_dir_path(__FILE__) . 'inc/order-status/list.php';
        } else if ('add-order-status' === $action) {
            buildecom_enqueue_order_status_assets();
            require_once plugin_dir_path(__FILE__) . 'inc/order-status/add.php';
        } else if ('edit-order-status' === $action) {
            buildecom_enqueue_order_status_assets();
            require_once plugin_dir_path(__FILE__) . 'inc/order-status/edit.php';
        } else if ('render-tracking-page' === $action) {
            buildecom_enqueue_track_order_status_assets();
            require_once plugin_dir_path(__FILE__) . 'inc/track-order-status/index.php';
        }
    }

    /**
     * Render plugin admin home page
     * @return void
     */
    public function render_buildecom_home_page()
    {
        require_once plugin_dir_path(__FILE__) . 'buildecom-home.php';

        buildecom_render_home();
    }


    /**
     * Activate the plugin also create the needed mysql tables
     * @return void
     */
    public static function activate()
    {
        self::create_tables();
    }

    /**
     * Cleanup things and delete mysql tables created by the plugin
     * @return void
     */
    public static function deactivate()
    {
        self::drop_tables();
    }

    /**
     * Insert tables to the database
     * @return void
     */
    private static function create_tables()
    {
        $sql_file = plugin_dir_path(__FILE__) . 'database.sql';

        if (!file_exists($sql_file)) {
            return;
        }

        $sql = file_get_contents($sql_file);

        global $wpdb;

        $sql = str_replace("{{prefix}}", $wpdb->prefix, $sql);

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Drop all the tables customized for this plugin
     * @return void
     */
    private static function drop_tables()
    {
        global $wpdb;

        $tables = [
            $wpdb->prefix . 'buildecom_extensions',
            $wpdb->prefix . 'buildecom_forms',
            $wpdb->prefix . 'buildecom_device_tokens',
            $wpdb->prefix . 'buildecom_banners',
            $wpdb->prefix . 'buildecom_categories',
            $wpdb->prefix . 'buildecom_testimonials',
            $wpdb->prefix . 'buildecom_order_statuses',
            $wpdb->prefix . 'buildecom_notification_templates',
        ];

        foreach ($tables as $table) {
            $query = $wpdb->prepare("DROP TABLE IF EXISTS %s", $table);
            $wpdb->query($query);
        }
    }

    /**
     * Instansiate the class if not and get it 
     * @return buildecom
     */
    public static function get_instance()
    {
        if (!self::$_instance) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }
}

require_once BUILDECOM_DIR_PATH . 'inc/helpers.php';
require_once BUILDECOM_DIR_PATH . 'inc/actions.php';

register_activation_hook(__FILE__, [Buildecom::class, 'activate']);
register_deactivation_hook(__FILE__, [Buildecom::class, 'deactivate']);


require_once BUILDECOM_DIR_PATH . 'vendor/autoload.php';
require_once BUILDECOM_DIR_PATH . 'api.php';

add_action('rest_api_init', 'buildecom_api_routes');

add_action('template_redirect', function () {
    if (!is_page() || !isset($_GET['token'])) {
        return;
    }

    $request_uri = $_SERVER['REQUEST_URI'] ?? '';
    if (strpos($request_uri, '/checkout/order-pay/') === false) {
        return;
    }

    $jwt = sanitize_text_field($_GET['token'] ?? '');
    if (!$jwt)
        return;

    if (!defined('BUILDECOM_SECRET_KEY')) {
        wp_die('JWT secret not configured.');
    }

    if (!class_exists(\Firebase\JWT\JWT::class)) {
        wp_die('JWT library missing.');
    }

    try {
        $payload = \Firebase\JWT\JWT::decode($jwt, new \Firebase\JWT\Key(BUILDECOM_SECRET_KEY, 'HS256'));
    } catch (Throwable $e) {
        wp_die('Invalid token.');
    }

    $iss_ok = isset($payload->iss) && $payload->iss === rtrim(get_site_url(), '/');
    $now = time();
    $nbf_ok = !isset($payload->nbf) || $payload->nbf <= $now;
    $exp_ok = !isset($payload->exp) || $now < $payload->exp;

    if (!$iss_ok || !$nbf_ok || !$exp_ok) {
        wp_die('Token not acceptable.');
    }

    $user_id = (int) ($payload->data->user->id ?? 0);
    if ($user_id <= 0 || !get_user_by('ID', $user_id)) {
        wp_die('User not found.');
    }

    if (!empty($payload->jti)) {
        $jti_key = 'myapp_jti_' . sanitize_key($payload->jti);
        if (get_transient($jti_key)) {
            wp_die('This link has already been used.');
        }
        $ttl = max(60, ($payload->exp ?? ($now + 600)) - $now);
        set_transient($jti_key, 1, $ttl);
    }

    wp_set_current_user($user_id);
    wp_set_auth_cookie($user_id, true, is_ssl());

    if (!function_exists('WC'))
        return;
    if (!WC()->session) {
        WC()->initialize_session();
    } elseif (!WC()->session->has_session()) {
        WC()->session->set_customer_session_cookie(true);
    }
    if (!WC()->customer) {
        WC()->customer = new \WC_Customer($user_id);
    }

    $order_id = 0;
    if (function_exists('get_query_var')) {
        $order_id = absint(get_query_var('order-pay'));
    }
    if (!$order_id) {
        if (preg_match('#/order-pay/(\d+)#', $request_uri, $m)) {
            $order_id = absint($m[1]);
        }
    }

    $provided_key = sanitize_text_field($_GET['key'] ?? '');

    $order = wc_get_order($order_id);
    if (!$order)
        wp_die('Order not found.');

    if ((int) $order->get_user_id() !== $user_id) {
        wp_die('Order does not belong to this user.');
    }

    if (!hash_equals($order->get_order_key(), $provided_key)) {
        wp_die('Invalid order key.');
    }

    if (!in_array($order->get_status(), ['pending', 'failed'], true)) {
        wp_safe_redirect($order->get_view_order_url());
        exit;
    }

    $pay_url = $order->get_checkout_payment_url();
    wp_safe_redirect($pay_url);
    exit;
});

add_action('init', function () {
    global $wpdb;

    $rows = $wpdb->get_results("
        SELECT status_slug, status_label
        FROM {$wpdb->prefix}buildecom_order_statuses
        WHERE status = 1
    ");

    if (!$rows) return;

    foreach ($rows as $r) {
        // Ensure wc- prefix and a clean key
        $key   = trim($r->status_slug);
        $key   = strpos($key, 'wc-') === 0 ? $key : 'wc-' . sanitize_title($key);
        $label = $r->status_label ?: ucfirst(str_replace('wc-', '', $key));

        register_post_status($key, array(
            'label'                     => $label,
            'public'                    => true,
            'exclude_from_search'       => false,
            'show_in_admin_all_list'    => true,
            'show_in_admin_status_list' => true,
            'label_count'               => _n_noop("$label (%s)", "$label (%s)", 'buildecom'),
        ));
    }
}, 5);

add_filter('wc_order_statuses', function ($order_statuses) {
    global $wpdb;

    $custom_statuses = $wpdb->get_results("
        SELECT status_slug, status_label
        FROM {$wpdb->prefix}buildecom_order_statuses
        WHERE status = 1
    ");

    if ($custom_statuses) {
        foreach ($custom_statuses as $s) { 
            $key_raw = trim((string) $s->status_slug);
            $key     = 'wc-' . sanitize_title(str_replace('wc-', '', $key_raw)); // <— normalize
            $label   = $s->status_label ?: ucfirst(str_replace('wc-', '', $key));

            $order_statuses[$key] = $label;
        }
    }

    return $order_statuses;
}, 20);


Buildecom::get_instance();
