<?php

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

if (!defined('ABSPATH')) exit;

require_once BUILDECOM_DIR_PATH . 'inc/api/class-api-controller.php';
require_once BUILDECOM_DIR_PATH . 'inc/api/class-api-auth.php';

/**
 * Check whether the user has permission to access the resource
 * @return bool
 */
function buildecom_check_auth(): bool
{
    $headers = getallheaders();

    if (!isset($headers['Authorization'])) {
        return false;
    }

    $auth_header = $headers['Authorization'];
    if (!preg_match('/Bearer\s(\S+)/', $auth_header, $matches)) {
        return false;
    }

    $token = $matches[1];

    $secret_key = defined('BUILDECOM_SECRET_KEY') ? BUILDECOM_SECRET_KEY : false;
    $alg        = 'HS256';

    try {
        $decoded = JWT::decode($token, new Key($secret_key, $alg));

        if ($decoded->exp < time()) {
            return false;
        }

        $user_id = $decoded->data->user->id;
        wp_set_current_user($user_id);

        return true;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Register api routes for buildecom
 * @return void
 */
function buildecom_api_routes()
{
    add_filter('woocommerce_rest_prepare_product_object', 'buildecom_custom_change_product_response', 20, 3);
    add_filter('woocommerce_rest_prepare_product_cat', 'buildecom_custom_product_category', 20, 3);
    add_filter('woocommerce_rest_product_cat_collection_params', 'buildecom_customize_rest_api_params', 10, 1);
    add_filter('woocommerce_rest_prepare_shop_order_object', 'custom_add_order_status_color', 10, 3);
    
    /*** ROUTES WHERE AUTH CHECK IS NOT MANDATORY */
    register_rest_route(BUILDECOM_API_V1, 'get-settings', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'fetchSettings'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'login', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'get_token'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'get-testimonials', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'fetchTestimonials'],
        'permission_callback' => '__return_true'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'get-banner-categories', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'fetchBannerCategories'],
        'permission_callback' => '__return_true'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'get-banners', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'fetchBanners'],
        'permission_callback' => '__return_true'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'login/refresh', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'send_refresh_token'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'registration', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'register_user'],
        'permission_callback' => '__return_true'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'password-reset', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'passwordReset'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'password-change-request', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'passwordChangeRequest'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'password-change-confirm', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'resetPasswordChangeConfirm'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'change-password', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'changePassword'],
        'permission_callback' => '__return_true',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'payment-gateways', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'getPaymentGateways'],
        'permission_callback' => '__return_true'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'get-status', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'fetchStatus'],
        'permission_callback' => '__return_true'
    ]);

    /*** ROUTES THOSE ONLY FOR AUTHENTICATED USERS */
    register_rest_route(BUILDECOM_API_V1, 'save-device-token', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'tokenStore'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'login/user-info', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Auth(), 'getUserInfo'],
        'permission_callback' => 'buildecom_check_auth'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'send-email', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'sendEmail'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'confirm-email', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Auth(), 'confirmEmail'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'profile-update', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'updateUserProfile'],
        'permission_callback' => 'buildecom_check_auth'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'eligible-shipping-methods', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'get_eligible_shipping_methods'],
        'permission_callback' => 'buildecom_check_auth'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'delete-account', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'deleteAccount'],
        'permission_callback' => 'buildecom_check_auth'
    ]);

    register_rest_route(BUILDECOM_API_V1, 'product-review', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'handleWoocommerceReview'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'cart/add-item', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'addItemToCart'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, 'checkout', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'checkoutHandler'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, '/cart/coupons', [
        'methods'             => 'POST',
        'callback'            => [new Buildecom_Api_Controller(), 'addcartCoupon'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, '/cart/coupons/(?P<code>[a-zA-Z0-9-_]+)', [
        'methods'             => 'DELETE',
        'callback'            => [new Buildecom_Api_Controller(), 'deleteCartCoupon'],
        'permission_callback' => 'buildecom_check_auth',
    ]);

    register_rest_route(BUILDECOM_API_V1, '/cart/coupons', [
        'methods'             => 'GET',
        'callback'            => [new Buildecom_Api_Controller(), 'deleteAllCartCoupons'],
        'permission_callback' => 'buildecom_check_auth',
    ]);
}



function buildecom_custom_change_product_response($response, $object, $request)
{
    // Will load the product variations if this request is for a specific product
    // product by ID
    $is_detail_api = isset($request->get_params()['id']);

    // The `is_all_data` can be String, Boolean or null. So it can be wrong if
    // check 'false' == true
    if (filter_var($request['is_all_data'], FILTER_VALIDATE_BOOLEAN)) {
        $is_detail_api = true;
    }

    $product = wc_get_product($response->data['id']);

    /* Update price for product variant */
    if ($product && $product->is_type('variable')) {
        $prices = $product->get_variation_prices();
        if (!empty($prices['price'])) {
            $response->data['price'] = current($prices['price']);
            $response->data['regular_price'] = current($prices['regular_price']);
            $response->data['sale_price'] = current($prices['sale_price']);
            $response->data['min_price'] = wc_get_price_to_display(  $product, array( 'price' => $product->get_variation_price() ) );
            $response->data['max_price'] = wc_get_price_to_display(  $product, array( 'price' => $product->get_variation_price('max') ) );
            
            if(!$response->data['min_price']){
                $response->data['min_price'] = '0';
            }
            if(!$response->data['max_price']){
                $response->data['max_price'] = '0';
            }
            //convert to string
            $response->data['min_price'] = strval($response->data['min_price']);
            $response->data['max_price'] = strval($response->data['max_price']);

            if($is_detail_api){
                $variations = $response->data['variations'];
                $controller = new \WC_REST_Product_Variations_V2_Controller();
                $variation_arr = array();
                foreach($variations as $variation_id){
                    $variation = new \WC_Product_Variation($variation_id);
                    $variation_data = $controller->prepare_object_for_response($variation, $request)->get_data();
                    $variation_arr[] = $variation_data;
                }
                $response->data['variation_lists'] = $variation_arr;
            }
            
        }
    }

    if ($product->is_type('grouped')) {
        $children_ids = $product->get_children();
        $min_price    = null;
        $max_price    = null;

        foreach ($children_ids as $child_id) {
            $child = wc_get_product($child_id);
            if (!$child) {
                continue;
            }

            if ($child->is_type('variable')) {
                $child_min = $child->get_variation_price();         // min
                $child_max = $child->get_variation_price('max');    // max
            } else {
                $child_min = $child->get_price();
                $child_max = $child_min;
            }

            if ($child_min === '' && $child_max === '') {
                continue;
            }

            if ($child_min !== '') {
                $child_min = wc_get_price_to_display($child, ['price' => (float) $child_min]);
            }
            if ($child_max !== '') {
                $child_max = wc_get_price_to_display($child, ['price' => (float) $child_max]);
            }

            if ($child_min !== '' && ($min_price === null || $child_min < $min_price)) {
                $min_price = $child_min;
            }
            if ($child_max !== '' && ($max_price === null || $child_max > $max_price)) {
                $max_price = $child_max;
            }
        }

        $response->data['min_price'] = ($min_price !== null) ? strval($min_price) : '0';
        $response->data['max_price'] = ($max_price !== null) ? strval($max_price) : '0'; 
        $response->data['price']     = $response->data['min_price'];
    }

    if($product) {
        $attributes = $product->get_attributes();
        $attributesData = [];
        foreach ($attributes as $key => $attr) {
            if(!is_string($attr)){
                $check = $attr->is_taxonomy();
                if ($check) {
                    $taxonomy = $attr->get_taxonomy_object();
                    $label = $taxonomy->attribute_label;
                } else {
                    $label = $attr->get_name();
                }
                $attrOptions = wc_get_product_terms($response->data['id'], $attr["name"]);
                $attrOptions = empty($attrOptions) ? array_map(function ($v){
                    return ['name'=>$v, 'slug' => $v];
                },$attr["options"]) : $attrOptions;
                $attributesData[] = array_merge($attr->get_data(), ["label" => $label, "name" => urldecode($key)], ['options' =>$attrOptions]);
            }
        }
        $response->data['attributes_lists'] = $attributesData;
    }

    return $response;

}

function buildecom_custom_product_category($response, $object, $request) {
    $id = $response->data['id'];
    $children = get_term_children($id, 'product_cat');

    if (empty($children)) {
        $response->data['has_children_category'] = false;
        $response->data['child_categories'] = array();
    } else {
        $response->data['has_children_category'] = true;
        $response->data['child_categories'] = get_child_categories_recursive($children);
    }

    return $response;
}

function get_child_categories_recursive($children) {
    $child_categories = array();

    foreach ($children as $child_id) {
        $child_term = get_term($child_id, 'product_cat');
        if (!is_wp_error($child_term)) {
            $child_term_data = array(
                'id' => $child_term->term_id,
                'name' => $child_term->name,
                'slug' => $child_term->slug,
                'description' => $child_term->description,
                'count' => $child_term->count,
                'parent' => $child_term->parent,
                'child_categories' => array()
            );

            $grandchildren = get_term_children($child_id, 'product_cat');
            if (!empty($grandchildren)) {
                $child_term_data['child_categories'] = get_child_categories_recursive($grandchildren);
            }

            $child_categories[] = $child_term_data;
        }
    }

    return $child_categories;
}

function buildecom_customize_rest_api_params($params) {
    $params['per_page']['maximum'] = 500; // Increase the maximum limit to 500
    return $params;
}

function custom_add_order_status_color($response, $object, $request) {
    global $wpdb;

    $order_id = $object->get_id();
    $order    = wc_get_order($order_id);

    // Fetch rows directly from your custom table
    $statuses = $wpdb->get_results(
        "SELECT * FROM {$wpdb->prefix}buildecom_order_statuses WHERE status = 1 ORDER BY id DESC"
    );

    $custom_status_colors = array(
        'pending'         => '#ffcc00',
        'pending-payment' => '#ffcc00',
        'processing'      => '#fadd85',
        'completed'       => '#0000ff',
        'on-hold'         => '#ff6600',
        'cancelled'       => '#cc0000',
        'refunded'        => '#009999',
        'failed'          => '#990000',
        'draft'           => '#000000',
    );

    foreach ($statuses as $status) {
        $slug = buildecom_make_slug($status->status_label);
        $custom_status_colors[$slug] = $status->status_color;
    }

    $status = $order->get_status();

    if (isset($custom_status_colors[$status])) {
        $status_color = $custom_status_colors[$status];
    }

    $response->data['status_color'] = $status_color;

    return $response;
}
