<?php
/**
 * OpenAI API integration for the chatbot plugin
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Bubblibot_OpenAI {
    
    private $api_key;
    private $model = 'gpt-4o-mini-2024-07-18';
    
    public function __construct() {
        $this->api_key = get_option('bubblibot_openai_api_key');
        $saved_model = get_option('bubblibot_openai_model');
        if (!empty($saved_model)) {
            $this->model = $saved_model;
        }
    }

    /**
     * Determine which parameter name to use for limiting completion tokens.
     * Some newer models (e.g., GPT-5 family) require 'max_completion_tokens'.
     */
    private function get_max_tokens_param_key() {
        // GPT-5 models use 'max_completion_tokens' instead of 'max_tokens'
        if ($this->is_gpt5_model()) {
            return 'max_completion_tokens';
        }
        return 'max_tokens';
    }

    /**
     * Whether the current model supports the 'temperature' parameter.
     * GPT-5 models only support the default temperature and reject custom values.
     */
    private function model_supports_temperature() {
        // GPT-5 models don't support custom temperature values
        if ($this->is_gpt5_model()) {
            return false;
        }
        return true;
    }

    /**
     * Check if the current model is a GPT-5 model
     */
    public function is_gpt5_model() {
        $is_gpt5 = strpos($this->model, 'gpt-5') === 0;
        write_debug_log('Bubblibot OpenAI: Checking if model "' . $this->model . '" is GPT-5: ' . ($is_gpt5 ? 'yes' : 'no'));
        return $is_gpt5;
    }
    
    /**
     * Get GPT-5 specific parameters for the API call
     */
    private function get_gpt5_parameters() {
        $params = array();
        
        // GPT-5 models support reasoning_effort parameter
        // Note: verbosity parameter seems to cause issues with GPT-5 nano, so omitting it
        if ($this->is_gpt5_model()) {
            // Set reasoning effort to low for GPT-5 nano (ultra-fast model)
            // This should provide quick responses without empty content
            $params['reasoning_effort'] = 'low';
            
            write_debug_log('Bubblibot OpenAI: GPT-5 parameters added - reasoning_effort: low');
        }
        
        return $params;
    }
    
    /**
     * Generate response using OpenAI
     */

    public function generate_response($user_message, $context_data = array(), $detected_language = 'en') {
        write_debug_log('Bubblibot OpenAI: Starting response generation');
        write_debug_log('Bubblibot OpenAI: User message: ' . $user_message);
        write_debug_log('Bubblibot OpenAI: Context data count: ' . count($context_data));
        write_debug_log('Bubblibot OpenAI: Detected language: ' . $detected_language);
        
        if (empty($this->api_key)) {
            write_debug_log('Bubblibot OpenAI: Error - API key not configured');
            return array(
                'success' => false,
                'error' => 'OpenAI API key is not configured'
            );
        }
        
        $messages = array();
        
        // System message to set the context
        $system_prompt = $this->get_system_prompt($context_data, $detected_language);
        write_debug_log('Bubblibot OpenAI: System prompt: ' . $system_prompt);
        
        $messages[] = array(
            'role' => 'system',
            'content' => $system_prompt
        );
        
        // Add context information directly in the system message (with safe size budget)
        if (!empty($context_data)) {
            $context_text = "\n\nHere is the relevant content from the website:\n\n";
            $budget = $this->get_context_char_budget();
            $used = 0;
            foreach ($context_data as $context) {
                $title = isset($context['title']) ? $context['title'] : '';
                $url = isset($context['url']) ? $context['url'] : '';
                $content = isset($context['content']) ? $context['content'] : '';
                write_debug_log('Bubblibot OpenAI: Adding context from: ' . $title);
                $header = "Article: \"{$title}\"\nURL: {$url}\nContent: ";
                $remaining = max(0, $budget - $used - strlen($header));
                if ($remaining <= 0) {
                    break;
                }
                // Truncate content to remaining budget per total, and also limit per-item chunk
                $perItemLimit = min($remaining, 2000);
                if (function_exists('mb_substr')) {
                    $snippet = mb_substr($content, 0, $perItemLimit);
                } else {
                    $snippet = substr($content, 0, $perItemLimit);
                }
                $block = $header . $snippet . "\n\n";
                $context_text .= $block;
                $used += strlen($block);
                if ($used >= $budget) {
                    break;
                }
            }
            // Add context information to system message
            $messages[0]['content'] .= $context_text;
            write_debug_log('Bubblibot OpenAI: Added context information to system message');
        }
        
        // Add the user's message
        $messages[] = array(
            'role' => 'user',
            'content' => $user_message
        );
        
        try {
            write_debug_log('Bubblibot OpenAI: Preparing API request');
            write_debug_log('Bubblibot OpenAI: Using model: ' . $this->model);
            write_debug_log('Bubblibot OpenAI: Is GPT-5 model: ' . ($this->is_gpt5_model() ? 'yes' : 'no'));
            
            $payload = array(
                'model' => $this->model,
                'messages' => $messages,
            );
            
            // Handle temperature parameter
            if ($this->model_supports_temperature()) {
                $payload['temperature'] = 0.7;
                write_debug_log('Bubblibot OpenAI: Added temperature: 0.7');
            } else {
                write_debug_log('Bubblibot OpenAI: Temperature not supported for this model');
            }
            
            // Handle max tokens parameter
            $max_tokens_key = $this->get_max_tokens_param_key();
            // GPT-5 models may need more tokens to generate proper responses
            $max_tokens = $this->is_gpt5_model() ? 1000 : 500;
            $payload[$max_tokens_key] = $max_tokens;
            write_debug_log('Bubblibot OpenAI: Using max tokens key: ' . $max_tokens_key . ' = ' . $max_tokens);
            
            // Add GPT-5 specific parameters if applicable
            if ($this->is_gpt5_model()) {
                $gpt5_params = $this->get_gpt5_parameters();
                $payload = array_merge($payload, $gpt5_params);
                write_debug_log('Bubblibot OpenAI: Added GPT-5 specific parameters');
            }
            
            // Log the complete payload for debugging
            write_debug_log('Bubblibot OpenAI: Complete payload: ' . json_encode($payload, JSON_PRETTY_PRINT));
            write_debug_log('Bubblibot OpenAI: Sending request to OpenAI API');

            $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode($payload),
                'timeout' => 30
            ));
            
            if (is_wp_error($response)) {
                write_debug_log('Bubblibot OpenAI: WP Error - ' . $response->get_error_message());
                return array(
                    'success' => false,
                    'error' => $response->get_error_message()
                );
            }
            
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['error'])) {
                write_debug_log('Bubblibot OpenAI: API Error - ' . $body['error']['message']);
                write_debug_log('Bubblibot OpenAI: Error details: ' . json_encode($body['error'], JSON_PRETTY_PRINT));
                write_debug_log('Bubblibot OpenAI: Request model: ' . $this->model);
                write_debug_log('Bubblibot OpenAI: Request payload that caused error: ' . json_encode($payload, JSON_PRETTY_PRINT));
                return array(
                    'success' => false,
                    'error' => $body['error']['message']
                );
            }
            
            $ai_response = $this->extract_response_text($body);
            
            // Special handling for GPT-5 models that might return empty content
            if (empty($ai_response) && $this->is_gpt5_model() && isset($body['choices'][0]['finish_reason'])) {
                write_debug_log('Bubblibot OpenAI: GPT-5 returned empty response, finish_reason: ' . $body['choices'][0]['finish_reason']);
                write_debug_log('Bubblibot OpenAI: Full response body: ' . json_encode($body, JSON_PRETTY_PRINT));
                
                // Try to extract any partial content if available
                if (isset($body['choices'][0]['message']) && is_array($body['choices'][0]['message'])) {
                    write_debug_log('Bubblibot OpenAI: Message structure: ' . json_encode($body['choices'][0]['message'], JSON_PRETTY_PRINT));
                }
                
                // For now, return an error instead of a fallback to help diagnose the issue
                return array(
                    'success' => false,
                    'error' => 'GPT-5 nano returned empty response. This may be a model-specific issue. Please try using a different model.'
                );
            }
            
            if (!empty($ai_response)) {
                
                // Post-process the response to fix link formatting
                $processed_response = $this->fix_link_formatting($ai_response, $context_data);
                
                write_debug_log('Bubblibot OpenAI: Successfully generated response');
                write_debug_log('Bubblibot OpenAI: Original response: ' . $ai_response);
                write_debug_log('Bubblibot OpenAI: Processed response: ' . $processed_response);
                
                return array(
                    'success' => true,
                    'message' => $processed_response
                );
            }
            
            // Log the full body structure for diagnostics (without huge truncation)
            write_debug_log('Bubblibot OpenAI: Invalid response format from API. Body keys: ' . (is_array($body) ? implode(',', array_keys($body)) : 'not-array'));
            if (is_array($body) && !empty($body['choices'][0])) {
                $choice = $body['choices'][0];
                $messageKeys = isset($choice['message']) && is_array($choice['message']) ? implode(',', array_keys($choice['message'])) : 'none';
                $contentType = isset($choice['message']['content']) ? gettype($choice['message']['content']) : 'unset';
                write_debug_log('Bubblibot OpenAI: Choice[0] finish_reason: ' . ($choice['finish_reason'] ?? 'n/a') . ', message keys: ' . $messageKeys . ', content type: ' . $contentType);
                if (!empty($choice['message'])) {
                    $snippet = wp_json_encode($choice['message']);
                    if (is_string($snippet) && strlen($snippet) > 1500) {
                        $snippet = substr($snippet, 0, 1500) . '...';
                    }
                    write_debug_log('Bubblibot OpenAI: message snippet: ' . $snippet);
                }
            }
            return array(
                'success' => false,
                'error' => 'Invalid response from OpenAI API'
            );
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot OpenAI: Exception - ' . $e->getMessage());
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }
    
    /**
     * Fix link formatting in AI response
     */
    private function fix_link_formatting($response, $context_data) {
        if (empty($context_data)) {
            return $response;
        }
        
        // Create a mapping of titles to URLs from context data
        $title_to_url = array();
        foreach ($context_data as $context) {
            $title_to_url[$context['title']] = $context['url'];
        }
        
        // Fix pattern: [Title] (URL) -> [Title](URL)
        $response = preg_replace('/\[([^\]]+?)\]\s*\(([^)]+?)\)/', '[$1]($2)', $response);
        
        // Fix pattern where article title is mentioned but not linked properly
        foreach ($title_to_url as $title => $url) {
            // Skip if the title is already properly linked with this URL
            if (strpos($response, '[' . $title . '](' . $url . ')') !== false) {
                continue;
            }
            
            // Pattern: "Title" (URL) -> [Title](URL)
            $escaped_title = preg_quote($title, '/');
            $response = preg_replace('/\[?' . $escaped_title . '\]?\s*\(' . preg_quote($url, '/') . '\)/', '[' . $title . '](' . $url . ')', $response);
            
            // Pattern: mention of title without link -> [Title](URL)
            // But be careful not to replace if it's already part of any Markdown link
            if (strpos($response, '[' . $title . ']') === false) {
                $escaped_title = preg_quote($title, '/');
                // Replace title with link, handling punctuation that might follow
                $response = preg_replace('/\b(' . $escaped_title . ')(\s*[\.\!\?]?)(?![^\[]*\])/m', '[$1](' . $url . ')$2', $response);
            }
        }
        
        return $response;
    }
    
    /**
     * Build system prompt based on context and mode
    */
    private function get_system_prompt($context_data, $detected_language = 'en') {
        $mode = get_option('bubblibot_response_mode', 'hybrid_smart');
        $base_prompt = "You are a helpful AI assistant for this website. Your responses should be clear, accurate, and based on the available information. ";
        
        if (!empty($context_data)) {
            $base_prompt .= "I will provide you with relevant content from the website including article titles and their URLs. ";
            
            if ($mode === 'company_only') {
                $base_prompt .= "You must ONLY use the provided website content to answer questions. If you cannot find the answer in the provided content, politely inform the user that you don't have enough information to answer their question accurately. ";
            } else if ($mode === 'hybrid_smart') {
                $base_prompt .= "PRIORITIZE using the provided website content when answering questions. The content provided is directly relevant to the user's question and comes from this website. If the website content fully answers the question, base your response primarily on that content. If the website content doesn't fully answer the question but you have relevant general knowledge, you can combine both to provide a complete answer. Always make it clear which parts of your answer come from the website and which are general knowledge. ";
            } else { // hybrid_always
                $base_prompt .= "Use both the provided website content and your general knowledge to provide comprehensive answers. Try to reference the website content when relevant, but feel free to expand on topics with your general knowledge. ";
            }
            
            $base_prompt .= "\n\nCRITICAL FORMATTING RULE: When you mention any article or page from the website content, you MUST format it as a clickable link using this exact format: [Article Title](URL). For example: [About Our Services](https://example.com/services). This is essential for proper link display in the chat interface. ";
            
            // Add specific instruction for product content
            $base_prompt .= "\n\nIMPORTANT: If the provided content includes product information (prices, stock status, product details), treat this as authoritative information about products available on this website. When discussing products, include relevant details like pricing and availability that are provided in the content. ";
            
            // Add instruction for focusing on relevant matches
            $base_prompt .= "\n\nPrioritize information that best matches the user's query, focusing on the most relevant content terms rather than just partial matches. ";
            
        } else {
            if ($mode === 'company_only') {
                $base_prompt .= "Since no relevant website content was found for this query, inform the user that you cannot provide specific information from the website. Suggest they rephrase their question or ask about something else. ";
            } else {
                $base_prompt .= "When no specific website content is available, you can provide general information while making it clear that you're not referencing website-specific content. ";
            }
        }
        
        // Language-specific instructions
        $language_names = array(
            'en' => 'English',
            'es' => 'Spanish',
            'fr' => 'French',
            'de' => 'German',
            'fi' => 'Finnish',
            'it' => 'Italian',
            'pt' => 'Portuguese',
            'nl' => 'Dutch',
            'sv' => 'Swedish',
            'no' => 'Norwegian',
            'da' => 'Danish',
            'pl' => 'Polish',
            'ru' => 'Russian',
            'ja' => 'Japanese',
            'ko' => 'Korean',
            'zh' => 'Chinese'
        );
        
        // Add language instructions - more explicit for GPT-5 models
        if ($this->is_gpt5_model()) {
            $language_name = isset($language_names[$detected_language]) ? $language_names[$detected_language] : 'English';
            $base_prompt .= "\n\nCRITICAL LANGUAGE REQUIREMENT: You MUST respond ONLY in " . strtoupper($language_name) . " (" . strtoupper($detected_language) . "). The user is communicating in " . $language_name . ". Never use Finnish, Swedish, German, or any other language except " . $language_name . ". This is absolutely critical - if you respond in the wrong language, it will cause serious problems.";
        } else {
            $base_prompt .= "\n\nIMPORTANT: Detect the language of the user's message and respond in the SAME language. If the user writes in Finnish, respond in Finnish. If they write in English, respond in English. Always match the user's language choice.";
        }
        
        // Add context about being a website assistant
        $base_prompt .= "\n\nRemember: You are representing this specific website and should focus on helping users with information and services available on this site. Be helpful, professional, and informative.";
        
        // Add general instruction about responding as business representative
        $base_prompt .= "\n\nCRITICAL: Always respond as a representative of this business/website. When users ask about 'your business hours', 'your prices', or 'your services', respond with 'our business hours', 'our prices', 'our services' etc. You work FOR this company, not as an external helper.";
        
        // Add emoji usage instruction
        $base_prompt .= "\n\nEMOJI USAGE: You may use emojis occasionally to make your responses more friendly and engaging, but use them sparingly and appropriately. A single emoji at the beginning of key points or to convey emotion is fine (e.g., '✅ Yes, we offer...', '📞 You can contact us...', '🎉 Great news!'), but avoid overusing them. Keep your responses professional while being approachable.";
        
        return $base_prompt;
    }

    /**
     * Character budget for appended context to system prompt.
     * Conservative limits for smaller GPT-5 models to avoid hitting context window.
     */
    private function get_context_char_budget() {
        // GPT-5 nano has optimized context handling
        if ($this->model === 'gpt-5-nano-2025-08-07') {
            return 16000; // Slightly higher budget for GPT-5 nano
        }
        return 12000; // default budget for models in use
    }
    
    /**
     * Get model-specific configuration information
     */
    public function get_model_info() {
        $models = array(
            'gpt-4o-mini-2024-07-18' => array(
                'name' => 'GPT-4o Mini',
                'pricing' => '$0.15/1M input, $0.60/1M output tokens',
                'description' => 'Cost-efficient model for simple tasks'
            ),
            'gpt-4.1-mini-2025-04-14' => array(
                'name' => 'GPT-4.1 Mini',
                'pricing' => '$0.30/1M input, $1.20/1M output tokens',
                'description' => 'Enhanced mini model with better reasoning'
            ),
            'gpt-5-nano-2025-08-07' => array(
                'name' => 'GPT-5 Nano',
                'pricing' => '$0.05/1M input, $0.40/1M output tokens',
                'description' => 'Ultra-low latency GPT-5 model with 45% fewer errors, optimized for instant responses'
            )
        );
        
        return isset($models[$this->model]) ? $models[$this->model] : null;
    }
    
    /**
     * Test API key validity
     */
    public function test_api_key() {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error' => 'API key not provided'
            );
        }
        
        // Simple test with minimal tokens to verify API key
        $messages = array(
            array(
                'role' => 'user',
                'content' => 'Test'
            )
        );
        
        try {
            $payload = array(
                'model' => $this->model,
                'messages' => $messages,
            );
            if ($this->model_supports_temperature()) {
                $payload['temperature'] = 0.1;
            }
            $payload[$this->get_max_tokens_param_key()] = 5; // Minimal tokens for testing

            $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode($payload),
                'timeout' => 15
            ));
            
            if (is_wp_error($response)) {
                return array(
                    'success' => false,
                    'error' => 'Network error: ' . $response->get_error_message()
                );
            }
            
            $http_code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            if ($http_code === 401) {
                return array(
                    'success' => false,
                    'error' => 'Invalid API key: Authentication failed'
                );
            }
            
            if ($http_code === 429) {
                return array(
                    'success' => false,
                    'error' => 'Rate limit exceeded: Please try again later'
                );
            }
            
            if ($http_code === 403) {
                return array(
                    'success' => false,
                    'error' => 'Access denied: Check your API key permissions'
                );
            }
            
            if (isset($body['error'])) {
                $error_message = $body['error']['message'] ?? 'Unknown API error';
                $error_type = $body['error']['type'] ?? '';
                
                if ($error_type === 'insufficient_quota') {
                    return array(
                        'success' => false,
                        'error' => 'Insufficient quota: Your OpenAI account has no available credits'
                    );
                }
                
                return array(
                    'success' => false,
                    'error' => $error_message
                );
            }
            
            // If we reached here without an error and with a 200 code, the key is valid
            // Newer models may return empty or structured content; we don't rely on it for validation
            $maybeText = $this->extract_response_text($body);
            if (is_string($maybeText)) {
                $maybeText = trim($maybeText);
            }
            return array(
                'success' => true,
                'message' => 'API key is valid and working correctly'
            );
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'error' => 'Exception: ' . $e->getMessage()
            );
        }
    }

    /**
     * Extract response text across different API shapes (Chat Completions vs Responses API)
     */
    private function extract_response_text($body) {
        // Chat Completions shape
        if (is_array($body)) {
            if (isset($body['choices'][0]['message']['content'])) {
                $content = $body['choices'][0]['message']['content'];
                // If it's already a string, return directly (even if empty for GPT-5)
                if (is_string($content)) {
                    // Log if content is empty
                    if (empty($content)) {
                        write_debug_log('Bubblibot OpenAI: Response content is empty string');
                    }
                    return $content;
                }
                // If the content is an array of parts, concatenate any 'text' fields
                if (is_array($content)) {
                    $collected = array();
                    foreach ($content as $part) {
                        if (is_string($part)) {
                            $collected[] = $part;
                        } elseif (is_array($part)) {
                            if (!empty($part['text']) && is_string($part['text'])) {
                                $collected[] = $part['text'];
                            } elseif (!empty($part['content']) && is_string($part['content'])) {
                                $collected[] = $part['content'];
                            } elseif (!empty($part['value']) && is_string($part['value'])) {
                                $collected[] = $part['value'];
                            }
                        }
                    }
                    if (!empty($collected)) {
                        return implode("\n", $collected);
                    }
                    // Fallback common nested location
                    if (!empty($content[0]['text']) && is_string($content[0]['text'])) {
                        return $content[0]['text'];
                    }
                }
            }
            // Try a broader recursive extraction under message if present
            if (!empty($body['choices'][0]['message'])) {
                $maybeText = $this->extract_text_recursive($body['choices'][0]['message']);
                if (!empty($maybeText)) {
                    return $maybeText;
                }
            }
            // Responses API shape (text aggregated under output_text)
            if (!empty($body['output_text'])) {
                return is_array($body['output_text']) ? implode("\n", $body['output_text']) : $body['output_text'];
            }
            // Some Responses API variants: choices[0].message? or output[0].content[0].text
            if (!empty($body['output'][0]['content'][0]['text'])) {
                return $body['output'][0]['content'][0]['text'];
            }
        }
        return '';
    }

    /**
     * Recursively collect any string values under keys that typically hold text.
     */
    private function extract_text_recursive($node, $parentKey = '') {
        $textKeys = array('text', 'content', 'output_text', 'value');
        $collected = array();
        if (is_string($node)) {
            // Only accept bare strings when they are under a text-bearing key
            return in_array($parentKey, $textKeys, true) ? $node : '';
        }
        if (is_array($node)) {
            foreach ($node as $key => $value) {
                if (in_array($key, $textKeys, true) && is_string($value)) {
                    $collected[] = $value;
                } else {
                    $sub = $this->extract_text_recursive($value, is_string($key) ? $key : '');
                    if (is_string($sub) && $sub !== '') {
                        $collected[] = $sub;
                    }
                }
            }
        }
        return !empty($collected) ? implode("\n", $collected) : '';
    }

    /**
     * Detect language using AI
     */
    public function detect_language($message) {
        write_debug_log('Bubblibot OpenAI: Starting AI language detection for message: ' . $message);
        
        if (empty($this->api_key)) {
            write_debug_log('Bubblibot OpenAI: Error - API key not configured for language detection');
            return null;
        }
        
        // Simple, general prompt that focuses on grammatical structure
        $messages = array(
            array(
                'role' => 'system',
                'content' => 'Detect the language of the user message by analyzing its grammatical structure and function words. Ignore proper nouns (names of people, places, brands) as they can appear in any language. Focus on the sentence structure, articles, prepositions, and verb patterns to determine the actual language being used. Respond with only the ISO 639-1 language code (en, fi, sv, de, etc.).'
            ),
            array(
                'role' => 'user', 
                'content' => $message
            )
        );
        
        try {
            write_debug_log('Bubblibot OpenAI: Sending language detection request to OpenAI API');
            
            $payload = array(
                'model' => $this->model,
                'messages' => $messages,
            );
            if ($this->model_supports_temperature()) {
                $payload['temperature'] = 0.1; // Low temperature for consistent results
            }
            $payload[$this->get_max_tokens_param_key()] = 10; // Very short response needed

            $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode($payload),
                'timeout' => 15 // Shorter timeout for quick detection
            ));
            
            if (is_wp_error($response)) {
                write_debug_log('Bubblibot OpenAI: Language detection WP Error - ' . $response->get_error_message());
                return null;
            }
            
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            if (isset($body['error'])) {
                write_debug_log('Bubblibot OpenAI: Language detection API Error - ' . $body['error']['message']);
                return null;
            }
            
            if (!empty($body['choices'][0]['message']['content'])) {
                $detected_language = trim(strtolower($body['choices'][0]['message']['content']));
                
                // Validate the response is a proper language code (2-3 characters, letters only)
                if (preg_match('/^[a-z]{2,3}$/', $detected_language)) {
                    write_debug_log('Bubblibot OpenAI: AI detected language: ' . $detected_language);
                    return $detected_language;
                } else {
                    write_debug_log('Bubblibot OpenAI: Invalid language code format received: ' . $detected_language);
                    return null;
                }
            }
            
            write_debug_log('Bubblibot OpenAI: No language detection response from API');
            return null;
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot OpenAI: Language detection exception - ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Check if a language needs word normalization for better search results
     */
    public function language_needs_normalization($language_code) {
        // Free version uses basic approach without complex language normalization
        return false;
    }
    
    /**
     * Normalize query words to their base forms for better search matching
     * Free version uses basic approach without complex language normalization
     */
    public function normalize_query_words($query, $detected_language = 'en') {
        write_debug_log('Bubblibot OpenAI: Free version - skipping word normalization');
        return $query;
    }
}