<?php
/**
 * Simplified language detection for free version
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Bubblibot_Language_Detector {
    
    private $supported_languages = array(
        'en' => 'English',
        'es' => 'Spanish',
        'fr' => 'French', 
        'de' => 'German',
        'it' => 'Italian',
        'pt' => 'Portuguese',
        'nl' => 'Dutch',
        'sv' => 'Swedish',
        'no' => 'Norwegian',
        'da' => 'Danish',
        'fi' => 'Finnish',
        'pl' => 'Polish',
        'ru' => 'Russian',
        'ja' => 'Japanese',
        'ko' => 'Korean',
        'zh' => 'Chinese'
    );
    
    /**
     * Detect user language from message - simplified for free version
     * 
     * @param string $message The user's message
     * @return string Language code (e.g., 'en', 'es', 'fi')
     */
    public function detect_user_language($message) {
        // Free version uses basic language detection
        $message_lower = strtolower($message);
        
        // Simple pattern matching for common words
        $language_patterns = array(
            'fi' => array('hei', 'kiitos', 'kyllä', 'ei', 'moi', 'terve', 'mitä', 'mikä', 'kuinka', 'voiko', 'voiko', 'tänään', 'huomenna', 'auki', 'kiinni'),
            'es' => array('hola', 'gracias', 'por favor', 'sí', 'no', 'buenos días', 'buenas tardes', 'cómo', 'qué', 'cuándo', 'dónde'),
            'fr' => array('bonjour', 'merci', 's\'il vous plaît', 'oui', 'non', 'comment', 'quoi', 'quand', 'où', 'pourquoi'),
            'de' => array('hallo', 'danke', 'bitte', 'ja', 'nein', 'wie', 'was', 'wann', 'wo', 'warum'),
            'it' => array('ciao', 'grazie', 'prego', 'sì', 'no', 'come', 'cosa', 'quando', 'dove', 'perché'),
            'pt' => array('olá', 'obrigado', 'por favor', 'sim', 'não', 'como', 'o que', 'quando', 'onde', 'por que')
        );
        
        $detected_scores = array();
        
        foreach ($language_patterns as $lang => $patterns) {
            $score = 0;
            foreach ($patterns as $pattern) {
                if (strpos($message_lower, $pattern) !== false) {
                    $score++;
                }
            }
            if ($score > 0) {
                $detected_scores[$lang] = $score;
            }
        }
        
        if (!empty($detected_scores)) {
            arsort($detected_scores);
            return array_keys($detected_scores)[0];
        }
        
        // Fallback to WordPress locale or default
        $wp_locale = get_locale();
        $wp_lang_code = substr($wp_locale, 0, 2);
        
        if (isset($this->supported_languages[$wp_lang_code])) {
            return $wp_lang_code;
        }
        
        return 'en'; // Default fallback
    }
    
    /**
     * Get language name from code
     */
    public function get_language_name($language_code) {
        return isset($this->supported_languages[$language_code]) 
            ? $this->supported_languages[$language_code] 
            : 'English';
    }
    
    /**
     * Get all supported languages
     */
    public function get_supported_languages() {
        return $this->supported_languages;
    }
    
    /**
     * Basic content translation (free version returns original content)
     */
    public function translate_content($content, $target_language) {
        // Free version doesn't include translation - return original
        return $content;
    }
    
    /**
     * Debug language detection for troubleshooting
     */
    public function debug_detection($message) {
        if (!get_option('bubblibot_debug_enabled', '0')) {
            return;
        }
        
        $detected = $this->detect_user_language($message);
        if (function_exists('write_debug_log')) {
            write_debug_log("Language Detection: '{$message}' detected as '{$detected}'");
        }
    }
    
    /**
     * Get UI strings for different languages (basic set for free version)
     */
    public function get_ui_strings($language_code) {
        $strings = array(
            'welcome_message'   => 'Hello! How can I help you today?',
            'placeholder_text'  => 'Type your message...',
            'header_text'       => 'Chat Support',
            'error_message'     => 'Sorry, something went wrong.',
            'typing_indicator'  => 'Typing...',
            'send_button'       => 'Send',
            'thinking'          => 'Thinking...'
        );
        
        // Basic translations for common languages
        switch ($language_code) {
            case 'fi':
                $strings = array_merge($strings, array(
                    'welcome_message'  => 'Hei! Kuinka voin auttaa?',
                    'placeholder_text' => 'Kirjoita viestisi...',
                    'header_text'      => 'Chat-tuki',
                    'error_message'    => 'Jotain meni vikaan.',
                    'typing_indicator' => 'Kirjoittaa...',
                    'send_button'      => 'Lähetä',
                    'thinking'         => 'Ajattelen...'
                ));
                break;
            case 'es':
                $strings = array_merge($strings, array(
                    'welcome_message'  => '¡Hola! ¿Cómo puedo ayudarte?',
                    'placeholder_text' => 'Escribe tu mensaje...',
                    'header_text'      => 'Soporte en línea',
                    'error_message'    => 'Algo salió mal.',
                    'typing_indicator' => 'Escribiendo...',
                    'send_button'      => 'Enviar',
                    'thinking'         => 'Pensando...'
                ));
                break;
            case 'fr':
                $strings = array_merge($strings, array(
                    'welcome_message'  => 'Bonjour ! Comment puis-je vous aider ?',
                    'placeholder_text' => 'Tapez votre message...',
                    'header_text'      => 'Assistance',
                    'error_message'    => 'Une erreur est survenue.',
                    'typing_indicator' => 'Saisie...',
                    'send_button'      => 'Envoyer',
                    'thinking'         => 'Réflexion...'
                ));
                break;
            case 'de':
                $strings = array_merge($strings, array(
                    'welcome_message'  => 'Hallo! Wie kann ich Ihnen helfen?',
                    'placeholder_text' => 'Nachricht eingeben...',
                    'header_text'      => 'Chat Support',
                    'error_message'    => 'Etwas ist schief gelaufen.',
                    'typing_indicator' => 'Schreibt...',
                    'send_button'      => 'Senden',
                    'thinking'         => 'Denken...'
                ));
                break;
        }
        
        return $strings;
    }
} 