<?php
/**
 * Content indexer for the chatbot plugin
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Bubblibot_Indexer {
    
    private $database;
    public function __construct($database) {
        $this->database = $database;
    }
    
    /**
     * Index all existing posts, pages, and products (Free version: max 50 items)
     */
    public function index_all_content() {
        try {
            write_debug_log('Bubblibot Indexer: Starting content indexing');
            
            // Get posts and pages
            $posts = get_posts(array(
                'post_type' => array('post', 'page'),
                'posts_per_page' => -1,
                'post_status' => 'publish'
            ));
            
            // Get WooCommerce products if enabled
            $products = array();
            $woocommerce_enabled = get_option('bubblibot_woocommerce_enabled', '0');
            if ($woocommerce_enabled === '1' && class_exists('WooCommerce')) {
                $products = get_posts(array(
                    'post_type' => 'product',
                    'posts_per_page' => -1,
                    'post_status' => 'publish'
                ));
            }
            
            $total_posts = count($posts);
            $total_products = count($products);
            $total = $total_posts + $total_products;
            
            // Free version: content limit removed - unlimited indexing allowed
            // Other free version limitations still apply (no embeddings, no memory, etc.)
            
            write_debug_log("Bubblibot Indexer: Found {$total_posts} posts/pages and {$total_products} products to index (total: {$total})");
            
            if ($total === 0) {
                throw new Exception('No published posts, pages, or products found to index.');
            }
            
            // Store initial progress
            update_option('bubblibot_indexing_progress', array(
                'total' => $total,
                'current' => 0,
                'status' => 'running',
                'last_error' => '',
                'start_time' => current_time('mysql')
            ));
            
            $count = 0;
            
            // Index posts and pages
            foreach ($posts as $post) {
                try {
                    // Free version: content limit removed - unlimited indexing allowed
                    
                    if ($this->index_single_post($post->ID)) {
                        $count++;
                    } else {
                        write_debug_log("Bubblibot Indexer: Failed to index post {$post->ID} - {$post->post_title}");
                    }
                    
                    // Update progress after each item
                    $this->update_progress($total, $count);
                    
                } catch (Exception $e) {
                    // Log error but continue with next post
                    write_debug_log('Bubblibot Indexer ERROR: Failed to index post ' . $post->ID . ' - ' . $e->getMessage());
                }
            }
            
            // Index WooCommerce products
            if (!empty($products)) {
                foreach ($products as $product) {
                    try {
                        // Free version: content limit removed - unlimited indexing allowed
                        
                        if ($this->index_single_product($product->ID)) {
                            $count++;
                            write_debug_log("Bubblibot Indexer: Successfully indexed product {$product->ID} - {$product->post_title}");
                        } else {
                            write_debug_log("Bubblibot Indexer: Failed to index product {$product->ID} - {$product->post_title}");
                        }
                        
                        // Update progress after each item
                        $this->update_progress($total, $count);
                        
                    } catch (Exception $e) {
                        // Log error but continue with next product
                        write_debug_log('Bubblibot Indexer ERROR: Failed to index product ' . $product->ID . ' - ' . $e->getMessage());
                    }
                }
            } else {
                write_debug_log('Bubblibot Indexer: Skipping WooCommerce product indexing - no products found or WooCommerce support disabled');
            }
            
            if ($count === 0) {
                throw new Exception('Failed to index any content. Check error log for details.');
            }
            
            // Update final status
            update_option('bubblibot_last_full_index', current_time('mysql'));
            $final_message = $count;
            
            update_option('bubblibot_indexing_progress', array(
                'total' => $total,
                'current' => $count,
                'status' => 'completed',
                'last_error' => '',
                'start_time' => get_option('bubblibot_indexing_progress')['start_time'],
                'final_message' => $final_message
            ));
            
            write_debug_log("Bubblibot Indexer: Completed indexing with {$count} successful items out of {$total} total items");
            
            return $count;
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot Indexer ERROR: ' . $e->getMessage());
            write_debug_log('Bubblibot Indexer ERROR: Stack trace - ' . $e->getTraceAsString());
            
            // Update progress with error
            update_option('bubblibot_indexing_progress', array(
                'total' => isset($total) ? $total : 0,
                'current' => isset($count) ? $count : 0,
                'status' => 'error',
                'last_error' => $e->getMessage(),
                'start_time' => isset($start_time) ? $start_time : current_time('mysql')
            ));
            
            throw $e;
        }
    }
    
    /**
     * Update indexing progress
     */
    private function update_progress($total, $current) {
        update_option('bubblibot_indexing_progress', array(
            'total' => $total,
            'current' => $current,
            'status' => 'running',
            'last_error' => '',
            'start_time' => get_option('bubblibot_indexing_progress')['start_time']
        ));
    }
    
    /**
     * Check if a post should be excluded from indexing
     */
    private function should_exclude_post($post, $content) {
        // Skip if post is empty or invalid
        if (!$post || empty($post->post_title)) {
            return true;
        }
        
        $title = strtolower(trim($post->post_title));
        $content_lower = strtolower(trim($content));
        $combined = $title . ' ' . $content_lower;
        
        // Font-related exclusions (WooCommerce, themes, etc.)
        $font_patterns = [
            // Font family names
            '/^(inter|bodoni moda|overpass|albert sans|lora|montserrat|arvo|rubik|newsreader|cormorant|work sans|raleway|roboto|open sans|helvetica|arial|times|georgia|verdana|calibri)$/i',
            // Font CSS/technical patterns  
            '/^[a-z\s]+;(normal|italic|bold);[0-9]+;[0-9]+%;u\+[0-9a-f\-]+$/i',
            // Font file patterns
            '/\.(woff|woff2|ttf|otf|eot)$/i',
            // CSS font declarations
            '/font-family|font-weight|font-style|@font-face/i'
        ];
        
        foreach ($font_patterns as $pattern) {
            if (preg_match($pattern, $title) || preg_match($pattern, $combined)) {
                write_debug_log("Bubblibot Indexer: Excluding font-related post {$post->ID} - {$post->post_title}");
                return true;
            }
        }
        
        // WooCommerce system pages exclusions
        $woocommerce_patterns = [
            // English WooCommerce pages
            '/^(shop|cart|checkout|my account|orders|order)$/i',
            // WooCommerce shortcode content patterns
            '/\[woocommerce_my_account\]/',
            '/\[woocommerce_cart\]/',
            '/\[woocommerce_checkout\]/',
            '/\[shop_messages\]/',
            // Common WooCommerce page content patterns
            '/your cart is empty/i',
            '/you may also like/i',
            '/new in store/i'
        ];
        
        foreach ($woocommerce_patterns as $pattern) {
            if (preg_match($pattern, $title) || preg_match($pattern, $combined)) {
                write_debug_log("Bubblibot Indexer: Excluding WooCommerce system page {$post->ID} - {$post->post_title}");
                return true;
            }
        }

        // Technical/system posts exclusions
        $technical_patterns = [
            // WooCommerce technical posts
            '/^(custom-css|custom-logo|site-icon|background-image)$/i',
            // Plugin configuration posts
            '/^(elementor|gutenberg|block-editor|wp-config|settings|configuration)$/i',
            // JSON/technical data
            '/^\{.*\}$/',
            '/^[a-f0-9]{32,}$/i', // Hash-like content
            // Short technical strings (but exclude common abbreviations like FAQ, API, etc.)
            '/^[a-z0-9\-_]{1,3}$/i' // Only 1-3 characters (not 4-5 like FAQ, API)
        ];
        
        foreach ($technical_patterns as $pattern) {
            if (preg_match($pattern, $title) || preg_match($pattern, $combined)) {
                write_debug_log("Bubblibot Indexer: Excluding technical post {$post->ID} - {$post->post_title}");
                return true;
            }
        }
        
        // Exclude posts with very short titles and technical content
        if (strlen($title) < 3 && strlen($content_lower) < 50) {
            write_debug_log("Bubblibot Indexer: Excluding short technical post {$post->ID} - {$post->post_title}");
            return true;
        }
        
        // Exclude posts with minimal meaningful content (mostly shortcodes, placeholders, etc.)
        $meaningful_content = preg_replace('/\[.*?\]/', '', $content_lower); // Remove shortcodes
        $meaningful_content = preg_replace('/\{.*?\}/', '', $meaningful_content); // Remove placeholders like {business_idea_generator}
        $meaningful_content = preg_replace('/[^a-zA-Z0-9\s]/', ' ', $meaningful_content); // Remove special chars
        $meaningful_content = preg_replace('/\s+/', ' ', trim($meaningful_content)); // Clean whitespace
        
        // If after cleaning there's very little meaningful text, exclude it
        if (strlen($meaningful_content) < 20) {
            write_debug_log("Bubblibot Indexer: Excluding post with minimal content {$post->ID} - {$post->post_title} (meaningful content: '{$meaningful_content}')");
            return true;
        }
        
        return false;
    }
    
    /**
     * Index a single post (Free version: check content limits)
     */
    public function index_single_post($post_id) {
        write_debug_log("Bubblibot Indexer: Attempting to index post {$post_id}");
        
        // Free version: content limit removed - unlimited indexing allowed
        // Other free version limitations still apply (no embeddings, no memory, etc.)
        
        $post = get_post($post_id);
        
        if (!$post || $post->post_status !== 'publish') {
            write_debug_log("Bubblibot Indexer: Skipping post {$post_id} - invalid or not published (status: " . ($post ? $post->post_status : 'null') . ")");
            return false;
        }
        
        write_debug_log("Bubblibot Indexer: Post {$post_id} is valid and published - Title: {$post->post_title}");
        
        // Get post content first for proper exclusion checking
        $content = $post->post_title . "\n\n" . $post->post_content;
        $content = wp_strip_all_tags($content);
        $content = preg_replace('/\s+/', ' ', $content);
        $content = trim($content);
        
        // Check if post should be excluded (now with processed content)
        if ($this->should_exclude_post($post, $content)) {
            write_debug_log("Bubblibot Indexer: Excluding post {$post_id} - {$post->post_title} (matches exclusion criteria)");
            return false;
        }
        
        write_debug_log("Bubblibot Indexer: Post {$post_id} passed exclusion check");
        
        if (empty($content)) {
            write_debug_log("Bubblibot Indexer: Skipping post {$post_id} - content is empty after processing");
            return false;
        }
        
        write_debug_log("Bubblibot Indexer: Indexing post {$post_id} - {$post->post_title} (content length: " . strlen($content) . ")");
        
        // Use the actual post type for content_type instead of hardcoded 'post'
        $content_type = $post->post_type === 'page' ? 'page' : 'post';
        
        $this->database->update_content($post_id, $content, '', $content_type, '', $post->post_title);
        
        write_debug_log("Bubblibot Indexer: Successfully indexed {$content_type} {$post_id}");
        
        return true;
    }
    
    /**
     * Index a single WooCommerce product (Free version: check content limits)
     */
    public function index_single_product($product_id) {
        if (!class_exists('WooCommerce')) {
            write_debug_log('Bubblibot Indexer: Cannot index product - WooCommerce not available');
            return false;
        }
        
        // Free version: content limit removed - unlimited indexing allowed
        // Other free version limitations still apply (no embeddings, no memory, etc.)
        
        $product = wc_get_product($product_id);
        
        if (!$product) {
            write_debug_log("Bubblibot Indexer: Product {$product_id} not found");
            return false;
        }
        
        // Index simple and variable products, skip grouped, external, etc.
        $indexable_types = array('simple', 'variable');
        if (!in_array($product->get_type(), $indexable_types)) {
            write_debug_log("Bubblibot Indexer: Skipping product {$product_id} (type: {$product->get_type()})");
            return false;
        }
        
        // Get product data
        $title = $product->get_name();
        $description = $product->get_description();
        $short_description = $product->get_short_description();
        $price = $product->get_price();
        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $stock_status = $product->get_stock_status(); // 'instock', 'outofstock', 'onbackorder'
        $stock_quantity = $product->get_stock_quantity();
        $sku = $product->get_sku();
        $categories = $product->get_category_ids();
        
        // Get category names
        $category_names = array();
        foreach ($categories as $category_id) {
            $term = get_term($category_id, 'product_cat');
            if ($term && !is_wp_error($term)) {
                $category_names[] = $term->name;
            }
        }
        
        // Build searchable content
        $content_parts = array();
        
        if (!empty($title)) {
            $content_parts[] = "Product: " . $title;
        }
        
        if (!empty($sku)) {
            $content_parts[] = "SKU: " . $sku;
        }
        
        if (!empty($short_description)) {
            $content_parts[] = "Summary: " . wp_strip_all_tags($short_description);
        }
        
        if (!empty($description)) {
            $content_parts[] = "Description: " . wp_strip_all_tags($description);
        }
        
        if (!empty($category_names)) {
            $content_parts[] = "Categories: " . implode(', ', $category_names);
        }
        
        // Add variation attributes for variable products
        if ($product->get_type() === 'variable') {
            $attributes = $product->get_variation_attributes();
            if (!empty($attributes)) {
                $attribute_info = array();
                foreach ($attributes as $attribute_name => $values) {
                    if (!empty($values)) {
                        $attribute_label = wc_attribute_label($attribute_name);
                        $attribute_info[] = $attribute_label . ": " . implode(', ', $values);
                    }
                }
                if (!empty($attribute_info)) {
                    $content_parts[] = "Available options: " . implode('; ', $attribute_info);
                }
            }
        }
        
        // Add price information for better search
        if (!empty($price)) {
            $currency = get_woocommerce_currency();
            $content_parts[] = "Price: " . $price . " " . $currency;
            
            if (!empty($sale_price) && $sale_price != $price) {
                $content_parts[] = "Sale Price: " . $sale_price . " " . $currency;
                $content_parts[] = "Regular Price: " . $regular_price . " " . $currency;
                $content_parts[] = "On Sale: Yes";
            } else {
                $content_parts[] = "Regular Price: " . $price . " " . $currency;
            }
            
            // Convert to different currencies for search (basic conversions)
            if ($currency === 'EUR') {
                $usd_approx = round($price * 1.1, 2); // Rough conversion
                $content_parts[] = "Price USD: $" . $usd_approx;
            }
        }
        
        // Add stock information for better search
        if (!empty($stock_status)) {
            $stock_text = ucfirst(str_replace('_', ' ', $stock_status));
            $content_parts[] = "Stock Status: " . $stock_text;
            
            if ($stock_quantity !== null && $stock_quantity > 0) {
                $content_parts[] = "Available Quantity: " . $stock_quantity;
                $content_parts[] = "In Stock: " . $stock_quantity . " units";
            }
        }
        
        $content = implode("\n", $content_parts);
        $content = preg_replace('/\s+/', ' ', $content);
        $content = trim($content);
        
        if (empty($content)) {
            write_debug_log("Bubblibot Indexer: Product {$product_id} has no indexable content");
            return false;
        }
        
        write_debug_log("Bubblibot Indexer: Indexing product {$product_id} - {$title}");
        
        // Prepare structured product data
        $product_data = array(
            'price' => $price,
            'sale_price' => (!empty($sale_price) && $sale_price > 0) ? $sale_price : null,
            'regular_price' => $regular_price,
            'currency' => get_woocommerce_currency(),
            'stock_status' => $stock_status,
            'stock_quantity' => $stock_quantity,
            'sku' => $sku,
            'type' => $product->get_type()
        );
        
        $this->database->update_content($product_id, $content, '', 'product', '', $title, $product_data);
        return true;
    }

    /**
     * Validate and clean extracted content
     */
    private function validate_and_clean_content($content) {
        // Basic content cleaning
        $content = preg_replace('/\s+/', ' ', $content);
        $content = trim($content);

        // Check if content is meaningful (e.g., not just spaces or symbols)
        if (strlen(preg_replace('/[^a-zA-Z0-9]/', '', $content)) < 20) { // Require at least 20 alphanumeric characters
             write_debug_log("Bubblibot Indexer: Content rejected - too few alphanumeric characters: '" . substr($content, 0, 50) . "...'");
            return '';
        }

        // Further validation can be added here (e.g., language detection, keyword filtering)

        return $content;
    }
    
    /**
     * Clean title string
     */
    private function clean_title($title) {
        if (empty($title)) return '';
        $title = wp_strip_all_tags(strval($title));
        $title = preg_replace('/\s+/', ' ', $title);
        $title = trim($title);
        // Truncate if too long
        if (mb_strlen($title) > 250) {
            $title = mb_substr($title, 0, 250) . '...';
        }
        return $title;
    }
    
    /**
     * Clean metadata field string
     */
    private function clean_metadata_field($field) {
        if (empty($field) || !is_string($field)) return '';
        $field = wp_strip_all_tags($field);
        $field = preg_replace('/\s+/', ' ', $field);
        $field = trim($field);
        // Truncate if too long
        if (mb_strlen($field) > 200) {
            $field = mb_substr($field, 0, 200) . '...';
        }
        return $field;
    }
    
    /**
     * Prepare content for database insertion (e.g., truncate)
     */
    private function prepare_content_for_database($content) {
        // Max content length in DB (adjust as needed)
        $max_length = 60000; // TEXT type in MySQL is typically 65,535 bytes
        if (mb_strlen($content) > $max_length) {
            $content = mb_substr($content, 0, $max_length);
            write_debug_log("Bubblibot Indexer: Content truncated to {$max_length} characters for database storage.");
        }
        return $content;
    }
    
    /**
     * Remove post from index
     */
    public function remove_post_from_index($post_id) {
        $this->database->delete_content($post_id);
        write_debug_log("Bubblibot Indexer: Removed post {$post_id} from index");
    }
    
    /**
     * Handle attachment upload (for potential future use)
     */
    public function handle_attachment_upload($attachment_id) {
        // Currently no file types are indexed, but this method is kept for potential future use
        write_debug_log("Bubblibot Indexer: Attachment {$attachment_id} uploaded (no indexing performed)");
    }

    /**
     * Handle attachment deletion (for potential future use)
     */
    public function handle_attachment_delete($attachment_id) {
        write_debug_log("Bubblibot Indexer: handle_attachment_delete triggered for ID: {$attachment_id}");
        
        // Ensure $attachment_id is an integer
        $attachment_id = intval($attachment_id);
        if ($attachment_id <= 0) {
            write_debug_log("Bubblibot Indexer: Invalid attachment_id ({$attachment_id}) in handle_attachment_delete.");
            return; // Don't proceed with an invalid ID
        }

        // Currently no file types are indexed, but this method is kept for potential future use
        write_debug_log("Bubblibot Indexer: Attachment {$attachment_id} deleted (no index removal needed)");
    }
    
    /**
     * Remove product from index
     */
    public function remove_product_from_index($product_id) {
        $this->database->delete_content($product_id, 'product');
        write_debug_log("Bubblibot Indexer: Removed product {$product_id} from index");
    }
    
    /**
     * Handle WooCommerce product save/update
     */
    public function handle_product_save($product_id) {
        $woocommerce_enabled = get_option('bubblibot_woocommerce_enabled', '0');
        
        if ($woocommerce_enabled === '1' && class_exists('WooCommerce')) {
            write_debug_log("Bubblibot Indexer: Product {$product_id} saved, indexing");
            $this->index_single_product($product_id);
        }
    }
    
    /**
     * Handle WooCommerce product deletion
     */
    public function handle_product_delete($product_id) {
        write_debug_log("Bubblibot Indexer: Product {$product_id} deleted, removing from index");
        $this->remove_product_from_index($product_id);
    }

    /**
     * Get indexing statistics
     */
    public function get_stats() {
        return $this->database->get_stats();
    }
}