<?php
/**
 * Frontend chat interface for Bubblibot
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Bubblibot_Frontend {
    
    private $database;
    private $openai;
    private $language_detector;
    private $analytics;
    private $display_controller;
    
    public function __construct($database, $openai, $language_detector, $analytics = null, $display_controller = null) {
        $this->database = $database;
        $this->openai = $openai;
        $this->language_detector = $language_detector;
        $this->analytics = $analytics;
        $this->display_controller = $display_controller;
    }
    
    public function init() {
        // Initialize display controller first  
        if (!$this->display_controller) {
            return;
        }
        
        // Initialize AJAX handlers first
        $this->init_ajax_handlers();
        
        // Only initialize UI elements if chatbot should be displayed
        if (!$this->display_controller || !$this->display_controller->should_display_chatbot()) {
            return;
        }
        
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('wp_footer', array($this, 'render_chat_widget'));
        add_action('wp_head', array($this, 'add_mobile_viewport_meta'));
        
        // CSS injection for page builders that don't load stylesheets properly
        add_filter('the_content', array($this, 'inject_css_fallback'), 999);
        
        // Dynamic CSS rendering - Load late to override theme styles
        add_action('wp_head', array($this, 'render_dynamic_css'), 9999);
        add_action('wp_footer', array($this, 'render_dynamic_css_fallback'), 1);
    }
    
    /**
     * Initialize AJAX handlers
     */
    private function init_ajax_handlers() {
        // Core chat functionality
        add_action('wp_ajax_bubblibot_message', array($this, 'handle_chat_message'));
        add_action('wp_ajax_nopriv_bubblibot_message', array($this, 'handle_chat_message'));
    }
    
    /**
     * Add mobile viewport meta tag for safe area support
     */
    public function add_mobile_viewport_meta() {
        // Enhanced mobile viewport support for all devices
        
        // Ensure proper viewport meta tag exists
        add_action('wp_head', function() {
            // Check if viewport meta tag already exists
            $existing_viewport = false;
            if (function_exists('wp_get_custom_css')) {
                // Simple check - most themes will have viewport meta
                $existing_viewport = true;
            }
            
            // Add our enhanced viewport meta tag if needed
            if (!$existing_viewport || wp_is_mobile()) {
                echo '<meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">' . "\n";
            }
        }, 1);
        
        // Add comprehensive safe area CSS variables
        wp_add_inline_style( 'bubblibot-frontend', '
            :root { 
                --safe-area-inset-top: 0px;
                --safe-area-inset-right: 0px;
                --safe-area-inset-bottom: 0px;
                --safe-area-inset-left: 0px;
            }
        ');

        // Add enhanced inline script for safe area support
        wp_add_inline_script(
            'bubblibot-frontend',
            '(function() {
                // Enhanced safe area support detection and setup
                if (window.CSS && CSS.supports("padding: env(safe-area-inset-bottom)")) {
                    const root = document.documentElement;
                    root.style.setProperty("--safe-area-inset-top", "env(safe-area-inset-top, 0px)");
                    root.style.setProperty("--safe-area-inset-right", "env(safe-area-inset-right, 0px)");
                    root.style.setProperty("--safe-area-inset-bottom", "env(safe-area-inset-bottom, 0px)");
                    root.style.setProperty("--safe-area-inset-left", "env(safe-area-inset-left, 0px)");
                }
                
                // Mobile viewport height fix for better full-screen support
                function updateVHProperty() {
                    const vh = window.innerHeight * 0.01;
                    document.documentElement.style.setProperty("--vh", vh + "px");
                }
                
                updateVHProperty();
                window.addEventListener("resize", updateVHProperty);
                window.addEventListener("orientationchange", function() {
                    setTimeout(updateVHProperty, 500);
                });
            })();',
            'after'
        );
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_frontend_scripts() {
        wp_enqueue_style(
            'bubblibot-frontend',
            BUBBLIBOT_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            BUBBLIBOT_VERSION
        );
        
        wp_enqueue_script(
            'bubblibot-frontend',
            BUBBLIBOT_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            BUBBLIBOT_VERSION,
            true
        );
        
        // Get WordPress locale for initial UI language
        $wp_locale = get_locale();
        $wp_lang_code = substr($wp_locale, 0, 2);
        $ui_strings = $this->language_detector->get_ui_strings($wp_lang_code);
        
        // Prepare localization data
        $localize_data = array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bubblibot_nonce'),
            'welcome_message' => get_option('bubblibot_welcome_message', $ui_strings['welcome_message']),
            'placeholder_text' => get_option('bubblibot_placeholder_text', $ui_strings['placeholder_text']),
            'header_text' => get_option('bubblibot_header_text', $ui_strings['header_text']),
            'error_message' => $ui_strings['error_message'],
            'typing_indicator' => $ui_strings['typing_indicator'],
            'bot_avatar_url' => get_option('bubblibot_bot_avatar_url', ''),
            // Free version specific data
            'is_free_version' => defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION,
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        );
        
        // Free version: Add conversation limit information
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $localize_data['conversation_limit'] = 50;
            $localize_data['upgrade_messages'] = array(
                'limit_reached' => __('Conversation limit reached. Upgrade to Pro for unlimited access.', 'bubblibot'),
                'upgrade_banner' => __('Upgrade to Pro for unlimited conversations, lead capture, and human handoff!', 'bubblibot')
            );
        }
        
        wp_localize_script('bubblibot-frontend', 'BubbliBot', $localize_data);
        
        // Enqueue dynamic CSS to ensure it's loaded with the frontend styles
        $this->enqueue_dynamic_css();
        
        // Add comprehensive JavaScript fallback for font size and colors (universal theme compatibility)
        $font_size        = get_option('bubblibot_font_size', '14');
        $header_font_size = max(14, intval($font_size) + 2);
        $small_font_size  = max(10, intval($font_size) - 2);
        // Fetch color settings for JS fallback
        $primary_color    = get_option('bubblibot_primary_color', '#0073aa');
        $secondary_color  = get_option('bubblibot_secondary_color', '#006291');
        $text_color       = get_option('bubblibot_text_color', '#ffffff');
        $bot_message_bg   = get_option('bubblibot_bot_message_bg', '#f0f0f0');
        $bot_message_text = get_option('bubblibot_bot_message_text', '#333333');
        $chat_window_bg  = get_option('bubblibot_chat_window_bg', '#ffffff');
        
        wp_add_inline_script('bubblibot-frontend', 
            'jQuery(document).ready(function($) {
                // Comprehensive font size application
                function applyBubbliFont() {
                    var widget = $("#bubblibot-widget");
                    if (!widget.length) return;
                    
                    // Set CSS custom properties
                    widget.css({
                        "--bubblibot-font-size": "' . esc_js($font_size) . 'px",
                        "--bubblibot-header-font-size": "' . esc_js($header_font_size) . 'px",
                        "--bubblibot-small-font-size": "' . esc_js($small_font_size) . 'px",
                        "--bubblibot-primary-color": "' . esc_js($primary_color) . '",
                        "--bubblibot-secondary-color": "' . esc_js($secondary_color) . '",
                        "--bubblibot-text-color": "' . esc_js($text_color) . '",
                        "--bubblibot-bot-bg": "' . esc_js($bot_message_bg) . '",
                        "--bubblibot-bot-text": "' . esc_js($bot_message_text) . '"
                    });
                    
                    // Force font sizes on all elements
                    var mainElements = widget.find(".bubblibot-message-content, .bubblibot-message-content *, .bubblibot-input, input, textarea, button, .bubblibot-send");
                    var headerElements = widget.find(".bubblibot-header h3, .bubblibot-header-text");
                    var smallElements = widget.find(".bubblibot-typing, .bubblibot-powered-by, .bubblibot-powered-by span, .bubblibot-timestamp");
                    
                    mainElements.css({
                        "font-size": "' . esc_js($font_size) . 'px !important",
                        "line-height": "1.4 !important"
                    });
                    
                    headerElements.css({
                        "font-size": "' . esc_js($header_font_size) . 'px !important",
                        "line-height": "1.2 !important"
                    });
                    
                    smallElements.css({
                        "font-size": "' . esc_js($small_font_size) . 'px !important"
                    });
                    
                    // Apply color customizations
                    widget.find(".bubblibot-toggle").css({
                        "background": "' . esc_js($primary_color) . '",
                        "color": "' . esc_js($text_color) . '"
                    });
                    widget.find(".bubblibot-header").css({
                        "background": "' . esc_js($primary_color) . '",
                        "color": "' . esc_js($text_color) . '"
                    });
                    widget.find(".bubblibot-header h3, .bubblibot-close").css({
                        "color": "' . esc_js($text_color) . '"
                    });
                    widget.find(".bubblibot-user .bubblibot-message-content").css({
                        "background": "' . esc_js($primary_color) . '",
                        "color": "' . esc_js($text_color) . '"
                    });
                    widget.find(".bubblibot-bot .bubblibot-message-content, .bubblibot-agent .bubblibot-message-content").css({
                        "background": "' . esc_js($bot_message_bg) . '",
                        "color": "' . esc_js($bot_message_text) . '"
                    });
                    widget.find(".bubblibot-send").css({
                        "background": "' . esc_js($primary_color) . '",
                        "color": "' . esc_js($text_color) . '"
                    });
                    widget.find(".bubblibot-window, .bubblibot-messages").css({
                        "background": "' . esc_js($chat_window_bg) . '"
                    });
                    // Hover state adjustments using secondary color
                    widget.find(".bubblibot-send").off("mouseenter.bubbli hover mouseleave.bubbli").on("mouseenter.bubbli", function(){
                        $(this).css("background", "' . esc_js($secondary_color) . '");
                    }).on("mouseleave.bubbli", function(){
                        $(this).css("background", "' . esc_js($primary_color) . '");
                    });
                    widget.find("#bubblibot-toggle").off("mouseenter.bubbli mouseleave.bubbli").on("mouseenter.bubbli", function(){
                        $(this).css("background", "' . esc_js($secondary_color) . '");
                    }).on("mouseleave.bubbli", function(){
                        $(this).css("background", "' . esc_js($primary_color) . '");
                    });
                }
                
                // Apply immediately
                applyBubbliFont();
                
                // Apply when widget is opened
                $(document).on("click", ".bubblibot-toggle", function() {
                    setTimeout(applyBubbliFont, 50);
                    setTimeout(applyBubbliFont, 200);
                });
                
                // Apply when new messages are added
                var observer = new MutationObserver(function(mutations) {
                    var shouldApply = false;
                    mutations.forEach(function(mutation) {
                        if (mutation.addedNodes.length > 0) {
                            mutation.addedNodes.forEach(function(node) {
                                if (node.nodeType === 1 && (node.classList.contains("bubblibot-message") || node.querySelector(".bubblibot-message"))) {
                                    shouldApply = true;
                                }
                            });
                        }
                    });
                    if (shouldApply) {
                        setTimeout(applyBubbliFont, 10);
                    }
                });
                
                // Start observing when widget exists
                function startObserving() {
                    var widget = document.getElementById("bubblibot-widget");
                    if (widget) {
                        observer.observe(widget, { childList: true, subtree: true });
                    } else {
                        setTimeout(startObserving, 500);
                    }
                }
                startObserving();
            });', 
            'after'
        );
    }
    
    /**
     * Enqueue dynamic CSS as inline styles (alternative method for page builders)
     */
    public function enqueue_dynamic_css() {
        // Prevent duplicate CSS if already rendered
        if (did_action('bubblibot_dynamic_css_rendered')) {
            return;
        }
        
        // Escape all color values and font size late when building the CSS
        $primary_color = esc_attr(get_option('bubblibot_primary_color', '#0073aa'));
        $secondary_color = esc_attr(get_option('bubblibot_secondary_color', '#006291'));
        $text_color = esc_attr(get_option('bubblibot_text_color', '#ffffff'));
        $bot_message_bg = esc_attr(get_option('bubblibot_bot_message_bg', '#f0f0f0'));
        $bot_message_text = esc_attr(get_option('bubblibot_bot_message_text', '#333333'));
        $font_size = esc_attr(get_option('bubblibot_font_size', '14'));
        
        // Calculate related font sizes for consistency
        $header_font_size = max(14, intval($font_size) + 2);
        $small_font_size = max(10, intval($font_size) - 2);
        
        // Build CSS using sprintf for better readability and escaping
        $css = sprintf('
            #bubblibot-widget .bubblibot-toggle {
                background: %1$s !important;
            }
            #bubblibot-widget .bubblibot-toggle:hover {
                background: %2$s !important;
            }
            #bubblibot-widget .bubblibot-toggle svg {
                color: %3$s !important;
            }
            #bubblibot-widget .bubblibot-header {
                background: %1$s !important;
                color: %3$s !important;
            }
            #bubblibot-widget .bubblibot-header h3 {
                color: %3$s !important;
                margin: 0 !important;
            }
            #bubblibot-widget .bubblibot-close {
                color: %3$s !important;
            }
            #bubblibot-widget .bubblibot-user .bubblibot-message-content {
                background: %1$s !important;
                color: %3$s !important;
            }
            #bubblibot-widget .bubblibot-user .bubblibot-message-content a {
                color: %3$s !important;
            }
            #bubblibot-widget .bubblibot-bot .bubblibot-message-content {
                background: %4$s !important;
                color: %5$s !important;
            }
            #bubblibot-widget .bubblibot-agent .bubblibot-message-content {
                background: %4$s !important;
                color: %5$s !important;
            }
            #bubblibot-widget .bubblibot-bot .bubblibot-message-content a {
                color: %1$s !important;
            }
            #bubblibot-widget .bubblibot-agent .bubblibot-message-content a {
                color: %1$s !important;
            }
            #bubblibot-widget .bubblibot-bot .bubblibot-message-content a:hover {
                background-color: %1$s1a !important;
            }
            #bubblibot-widget .bubblibot-agent .bubblibot-message-content a:hover {
                background-color: %1$s1a !important;
            }
            #bubblibot-widget .bubblibot-agent-joined .bubblibot-message-content {
                background: %4$s !important;
                color: %5$s !important;
            }
            #bubblibot-widget .bubblibot-agent-joined .bubblibot-agent-name {
                color: %5$s !important;
            }
            #bubblibot-widget .bubblibot-agent-joined .bubblibot-agent-message {
                color: %5$s !important;
            }
            #bubblibot-widget .bubblibot-handoff .bubblibot-message-content {
                background: %4$s !important;
                color: %5$s !important;
            }
            #bubblibot-widget .bubblibot-input:focus {
                border-color: %1$s !important;
            }
            #bubblibot-widget .bubblibot-send {
                background: %1$s !important;
                color: %3$s !important;
            }
            #bubblibot-widget .bubblibot-send:hover {
                background: %2$s !important;
            }
            #bubblibot-widget .bubblibot-typing span {
                background: %1$s !important;
            }
            #bubblibot-widget .bubblibot-message-content a {
                color: %1$s !important;
            }
            
            /* Lead Form Specific Styling */
            #bubblibot-widget .bubblibot-lead-form .bubblibot-message-content {
                border-color: %1$s !important;
            }
            
            #bubblibot-widget .bubblibot-form-field input:focus,
            #bubblibot-widget .bubblibot-form-field textarea:focus {
                border-color: %1$s !important;
                box-shadow: 0 0 0 2px %1$s1a !important;
            }
            
            #bubblibot-widget .bubblibot-form-field input:required,
            #bubblibot-widget .bubblibot-form-field textarea:required {
                border-left-color: %1$s !important;
            }
            
            #bubblibot-widget .bubblibot-form-submit {
                background: %1$s !important;
                color: %3$s !important;
                border-color: %1$s !important;
            }
            
            #bubblibot-widget .bubblibot-form-submit:hover {
                background: %2$s !important;
                border-color: %2$s !important;
            }
            
            #bubblibot-widget .bubblibot-agent-form .bubblibot-form-header {
                background: %1$s1a !important;
                border-color: %1$s !important;
            }
            
            #bubblibot-widget .bubblibot-agent-form .bubblibot-form-header h4,
            #bubblibot-widget .bubblibot-agent-form .bubblibot-form-header p {
                color: %1$s !important;
            }
            
            #bubblibot-widget .bubblibot-agent-form .bubblibot-form-content {
                border-color: %1$s !important;
            }
            
            /* COMPREHENSIVE FONT SIZE OVERRIDES - Universal theme compatibility */
            
            /* CSS Custom Properties for consistent font sizing */
            #bubblibot-widget {
                --bubblibot-font-size: %6$spx;
                --bubblibot-header-font-size: %7$spx;
                --bubblibot-small-font-size: %8$spx;
                --bubblibot-primary-color: %1$s;
                --bubblibot-secondary-color: %2$s;
                --bubblibot-text-color: %3$s;
                --bubblibot-bot-bg: %4$s;
                --bubblibot-bot-text: %5$s;
            }
            
            /* Message Content - Override ALL possible selectors */
            html body #bubblibot-widget .bubblibot-message-content,
            html body #bubblibot-widget .bubblibot-message-content *,
            html body #bubblibot-widget .bubblibot-message-content p,
            html body #bubblibot-widget .bubblibot-message-content div,
            html body #bubblibot-widget .bubblibot-message-content span,
            html body #bubblibot-widget .bubblibot-message-content a,
            html body #bubblibot-widget .bubblibot-message-content strong,
            html body #bubblibot-widget .bubblibot-message-content em,
            html body #bubblibot-widget .bubblibot-message-content li,
            html body #bubblibot-widget .bubblibot-message-content ul,
            html body #bubblibot-widget .bubblibot-message-content ol,
            html body #bubblibot-widget .bubblibot-bot .bubblibot-message-content,
            html body #bubblibot-widget .bubblibot-user .bubblibot-message-content,
            html body #bubblibot-widget .bubblibot-agent .bubblibot-message-content {
                font-size: var(--bubblibot-font-size) !important;
                line-height: 1.4 !important;
            }
            
            /* Input Fields */
            html body #bubblibot-widget .bubblibot-input,
            html body #bubblibot-widget input,
            html body #bubblibot-widget input[type="text"],
            html body #bubblibot-widget input[type="email"],
            html body #bubblibot-widget input[type="tel"],
            html body #bubblibot-widget textarea,
            html body #bubblibot-widget select,
            html body #bubblibot-widget .bubblibot-form-field input,
            html body #bubblibot-widget .bubblibot-form-field textarea,
            html body #bubblibot-widget .bubblibot-form-field select {
                font-size: var(--bubblibot-font-size) !important;
                line-height: 1.4 !important;
            }
            
            /* Header - Proportional sizing */
            html body #bubblibot-widget .bubblibot-header h3,
            html body #bubblibot-widget .bubblibot-header .bubblibot-header-text {
                font-size: var(--bubblibot-header-font-size) !important;
                line-height: 1.2 !important;
            }
            
            /* Buttons and Send Button */
            html body #bubblibot-widget .bubblibot-send,
            html body #bubblibot-widget button,
            html body #bubblibot-widget .bubblibot-form-submit {
                font-size: var(--bubblibot-font-size) !important;
            }
            
            /* Typing Indicator */
            html body #bubblibot-widget .bubblibot-typing,
            html body #bubblibot-widget .bubblibot-typing-text {
                font-size: var(--bubblibot-small-font-size) !important;
            }
            
            /* Small Text Elements */
            html body #bubblibot-widget .bubblibot-powered-by,
            html body #bubblibot-widget .bubblibot-powered-by span,
            html body #bubblibot-widget .bubblibot-timestamp {
                font-size: var(--bubblibot-small-font-size) !important;
            }
            
            /* Override specific theme classes that might interfere */
            html body.kadence-theme #bubblibot-widget,
            html body.astra-theme #bubblibot-widget,
            html body.generatepress #bubblibot-widget,
            html body.oceanwp-theme #bubblibot-widget,
            html body.hello-elementor #bubblibot-widget {
                font-size: initial !important;
            }
            
            html body.kadence-theme #bubblibot-widget *,
            html body.astra-theme #bubblibot-widget *,
            html body.generatepress #bubblibot-widget *,
            html body.oceanwp-theme #bubblibot-widget *,
            html body.hello-elementor #bubblibot-widget * {
                font-size: var(--bubblibot-font-size) !important;
            }',
            $primary_color,
            $secondary_color,
            $text_color,
            $bot_message_bg,
            $bot_message_text,
            $font_size,
            $header_font_size,
            $small_font_size
        );
        
        wp_add_inline_style('bubblibot-frontend', $css);
        
        // Mark as rendered to prevent duplicates
        do_action('bubblibot_dynamic_css_rendered');
    }
    
    /**
     * Render chat widget HTML
     */
    public function render_chat_widget() {
        // Get WordPress locale for UI language
        $wp_locale = get_locale();
        $wp_lang_code = substr($wp_locale, 0, 2);
        $ui_strings = $this->language_detector->get_ui_strings($wp_lang_code);
        
        $header_text = get_option('bubblibot_header_text', $ui_strings['header_text']);
        $placeholder_text = get_option('bubblibot_placeholder_text', $ui_strings['placeholder_text']);
        
        // Get position classes and styles
        $position_classes = $this->display_controller ? $this->display_controller->get_position_classes() : 'bubblibot-widget position-bottom-right';
        $position_styles = $this->display_controller ? $this->display_controller->generate_position_css() : 'bottom: 20px; right: 20px;';
        
        // Get custom colors for inline CSS fallback
        $primary_color = get_option('bubblibot_primary_color', '#0073aa');
        $secondary_color = get_option('bubblibot_secondary_color', '#006291');
        $text_color = get_option('bubblibot_text_color', '#ffffff');
        $chat_window_bg = get_option('bubblibot_chat_window_bg', '#ffffff');
        $bot_message_bg = get_option('bubblibot_bot_message_bg', '#f0f0f0');
        $bot_message_text = get_option('bubblibot_bot_message_text', '#333333');
        
        // Add inline CSS via wp_add_inline_style
        $inline_css = sprintf(
            '#bubblibot-widget .bubblibot-toggle { background: %1$s !important; }
            #bubblibot-widget .bubblibot-toggle:hover { background: %2$s !important; }
            #bubblibot-widget .bubblibot-toggle svg { color: %3$s !important; }
            #bubblibot-widget .bubblibot-header { background: %1$s !important; color: %3$s !important; }
            #bubblibot-widget .bubblibot-header h3 { color: %3$s !important; margin: 0 !important; }
            #bubblibot-widget .bubblibot-close { color: %3$s !important; }
            #bubblibot-widget .bubblibot-user .bubblibot-message-content { background: %1$s !important; color: %3$s !important; }
            #bubblibot-widget .bubblibot-user .bubblibot-message-content a { color: %3$s !important; }
            #bubblibot-widget .bubblibot-bot .bubblibot-message-content { background: %4$s !important; color: %5$s !important; }
            #bubblibot-widget .bubblibot-agent .bubblibot-message-content { background: %4$s !important; color: %5$s !important; }
            #bubblibot-widget .bubblibot-bot .bubblibot-message-content a { color: %1$s !important; }
            #bubblibot-widget .bubblibot-agent .bubblibot-message-content a { color: %1$s !important; }
            #bubblibot-widget .bubblibot-bot .bubblibot-message-content a:hover { background-color: %1$s1a !important; }
            #bubblibot-widget .bubblibot-agent .bubblibot-message-content a:hover { background-color: %1$s1a !important; }
            #bubblibot-widget .bubblibot-agent-joined .bubblibot-message-content { background: %4$s !important; color: %5$s !important; }
            #bubblibot-widget .bubblibot-agent-joined .bubblibot-agent-name { color: %5$s !important; }
            #bubblibot-widget .bubblibot-agent-joined .bubblibot-agent-message { color: %5$s !important; }
            #bubblibot-widget .bubblibot-handoff .bubblibot-message-content { background: %4$s !important; color: %5$s !important; }
            #bubblibot-widget .bubblibot-input:focus { border-color: %1$s !important; }
            #bubblibot-widget .bubblibot-send { background: %1$s !important; color: %3$s !important; }
            #bubblibot-widget .bubblibot-send:hover { background: %2$s !important; }
            #bubblibot-widget .bubblibot-typing span { background: %1$s !important; }
            #bubblibot-widget .bubblibot-window { background: %6$s !important; }
            #bubblibot-widget .bubblibot-messages { background: %6$s !important; }
            #bubblibot-widget .bubblibot-powered-by a:hover { color: %1$s !important; }',
            esc_attr($primary_color),
            esc_attr($secondary_color),
            esc_attr($text_color),
            esc_attr($bot_message_bg),
            esc_attr($bot_message_text),
            esc_attr($chat_window_bg)
        );
        wp_add_inline_style('bubblibot-frontend', $inline_css);
        
        ?>
        <div id="bubblibot-widget" class="<?php echo esc_attr($position_classes); ?>" style="<?php echo esc_attr($position_styles); ?>">
            <!-- Chat toggle button -->
            <div id="bubblibot-toggle" class="bubblibot-toggle" style="background: <?php echo esc_attr( $primary_color ); ?>; color: <?php echo esc_attr( $text_color ); ?>;">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M20 2H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h4l4 4 4-4h4c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-2 12H6v-2h12v2zm0-3H6V9h12v2zm0-3H6V6h12v2z"/>
                </svg>
            </div>
            
            <!-- Chat window -->
            <div id="bubblibot-window" class="bubblibot-window" style="display: none;">
                <!-- Header -->
                <div class="bubblibot-header">
                    <h3><?php echo esc_html($header_text); ?></h3>
                    <button id="bubblibot-close" class="bubblibot-close">&times;</button>
                </div>
                

                
                <!-- Messages container -->
                <div id="bubblibot-messages" class="bubblibot-messages"></div>
                
                <!-- Typing indicator -->
                <div id="bubblibot-typing" class="bubblibot-typing" style="display: none;">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
                
                <!-- Input container -->
                <div class="bubblibot-input-container">
                    <input type="text" id="bubblibot-input" class="bubblibot-input" placeholder="<?php echo esc_attr($placeholder_text); ?>">
                    <button id="bubblibot-send" class="bubblibot-send">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="18" height="18">
                            <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                        </svg>
                    </button>
                </div>
                

            </div>
        </div>
        <?php
    }
    
    /**
     * Handle chat message AJAX request
     */
    public function handle_chat_message() {
        check_ajax_referer('bubblibot_nonce', 'nonce');
        
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $raw_message = isset( $_POST['message'] ) ? wp_unslash( $_POST['message'] ) : '';
        $user_message = sanitize_text_field( $raw_message );
        $session_id = isset( $_POST['session_id'] ) && ! empty( $_POST['session_id'] ) ? sanitize_text_field( wp_unslash( $_POST['session_id'] ) ) : $this->generate_session_id();
        
        if (empty($user_message)) {
            wp_send_json_error('Message cannot be empty');
        }
        
        $start_time = microtime(true);
        
        // Detect user language
        $detected_language = $this->get_or_detect_session_language($session_id, $user_message);
        if (function_exists('write_debug_log')) {
            write_debug_log('Frontend: Detected language for message: ' . $detected_language);
        }
        
        // Search for relevant content (basic search only in free version)
        $context_data = $this->database->search_content($user_message, 2);
        
        // Generate response using OpenAI with language information
        $response = $this->openai->generate_response($user_message, $context_data, $detected_language);
        
        // Calculate response time
        $response_time = round((microtime(true) - $start_time) * 1000);
        
        // Track analytics if available (basic analytics only)
        if ($this->analytics) {
            $context_found = !empty($context_data) && count($context_data) > 0;
            $this->analytics->track_query(
                $user_message,
                $context_found,
                $detected_language,
                $response_time
            );
        }
        
        // Prepare response data (simplified for free version - no pro features)
        $response_data = array(
            'message' => $response['message'],
            'context_found' => !empty($context_data),
            'detected_language' => $detected_language,
            'language_name' => $this->language_detector->get_language_name($detected_language),
            'response_time' => $response_time,
            'session_id' => $session_id,
            // Free version: Pro features disabled
            'handoff_triggered' => false,
            'lead_capture_triggered' => false,
            'is_with_human' => false,
            'handoff_available' => false,
            'lead_capture_available' => false
        );
        
        wp_send_json_success($response_data);
    }
    
    /**
     * Generate a session ID for tracking conversations
     */
    private function generate_session_id() {
        return 'session_' . uniqid() . '_' . time();
    }
    
    /**
     * Get conversation history for a session (simplified for now)
     */
    private function get_session_history($session_id) {
        // For now, return empty array. In a full implementation, 
        // this would retrieve the conversation history from the database
        return array();
    }
    
    /**
     * Render dynamic CSS for color customization
     */
    public function render_dynamic_css() {
        // Use enqueue_dynamic_css() instead of outputting <style> tags
        $this->enqueue_dynamic_css();
    }
    
    /**
     * Fallback CSS rendering in footer for theme compatibility
     */
    public function render_dynamic_css_fallback() {
        // Only render if not already done in head
        if (!did_action('bubblibot_dynamic_css_rendered')) {
            $this->enqueue_dynamic_css();
        }
    }
    
    /**
     * Inject CSS as fallback for page builders (content filter)
     */
    public function inject_css_fallback($content) {
        // Trigger enqueue_dynamic_css if not already done
        if (!did_action('bubblibot_dynamic_css_rendered')) {
            $this->enqueue_dynamic_css();
        }
        
        // Just return content without modifying it
            return $content;
    }

    
    /**
     * Get or detect language for session with AI-powered detection and caching
     */
    private function get_or_detect_session_language($session_id, $user_message) {
        // Get cached language but don't rely on it exclusively (allow language switching)
        $cached_language = null;
        try {
            $conversation = $this->database->get_conversation_by_session($session_id);
            if ($conversation && !empty($conversation->detected_language)) {
                $cached_language = $conversation->detected_language;
                if (function_exists('write_debug_log')) {
                    write_debug_log("Frontend: Found cached language: {$cached_language}");
                }
            }
        } catch (Exception $e) {
            if (function_exists('write_debug_log')) {
                write_debug_log("Frontend: Error getting conversation for language check: " . $e->getMessage());
            }
        }
        
        // Check if language detection is enabled
        $language_detection_enabled = get_option('bubblibot_language_detection_enabled', '1') === '1';
        $default_language = get_option('bubblibot_default_language', 'en');
        
        if (!$language_detection_enabled) {
            // Language detection disabled, use default language
            if (function_exists('write_debug_log')) {
                write_debug_log("Frontend: Language detection disabled, using default language: {$default_language}");
            }
            return $default_language;
        }
        
        // No cached language, use AI to detect
        if (function_exists('write_debug_log')) {
            write_debug_log("Frontend: No cached language found, using AI detection for message: {$user_message}");
        }
        
        $ai_detected_language = $this->openai->detect_language($user_message);
        
        // If GPT-5 detection failed, try with a fallback model (GPT-4o mini)
        if (!$ai_detected_language && $this->openai->is_gpt5_model()) {
            if (function_exists('write_debug_log')) {
                write_debug_log("Frontend: GPT-5 language detection failed, trying fallback model");
            }
            
            // Create temporary OpenAI instance with GPT-4o mini for fallback detection
            $fallback_openai = new Bubblibot_OpenAI();
            $original_model = get_option('bubblibot_openai_model');
            update_option('bubblibot_openai_model', 'gpt-4o-mini-2024-07-18');
            $fallback_openai = new Bubblibot_OpenAI(); // Reinitialize with new model
            
            $ai_detected_language = $fallback_openai->detect_language($user_message);
            
            // Restore original model
            update_option('bubblibot_openai_model', $original_model);
            
            if ($ai_detected_language && function_exists('write_debug_log')) {
                write_debug_log("Frontend: Fallback model detected language: {$ai_detected_language}");
            }
        }
        
        if ($ai_detected_language) {
            if (function_exists('write_debug_log')) {
                write_debug_log("Frontend: AI detected language: {$ai_detected_language}");
            }
            $detected_language = $ai_detected_language;
        } else {
            // AI detection failed, use fallback chain: cached -> WordPress locale -> default
            if ($cached_language) {
                $detected_language = $cached_language;
                if (function_exists('write_debug_log')) {
                    write_debug_log("Frontend: AI detection failed, using cached language: {$detected_language}");
                }
            } else {
                $wp_locale = get_locale();
                $wp_lang_code = substr($wp_locale, 0, 2);
                
                // Check if WordPress locale is supported
                $supported_languages = array('en', 'fi', 'sv', 'no', 'da', 'de', 'fr', 'es', 'it', 'pt', 'nl', 'pl', 'ru', 'ja', 'ko', 'zh');
                if (in_array($wp_lang_code, $supported_languages)) {
                    $detected_language = $wp_lang_code;
                    if (function_exists('write_debug_log')) {
                        write_debug_log("Frontend: AI detection failed, using WordPress locale: {$detected_language}");
                    }
                } else {
                    $detected_language = $default_language;
                    if (function_exists('write_debug_log')) {
                        write_debug_log("Frontend: AI detection failed, using default language: {$detected_language}");
                    }
                }
            }
        }
        
        // Cache the detected language by updating/creating conversation
        try {
            $this->database->get_or_create_conversation($session_id, null, null, $detected_language);
            if (function_exists('write_debug_log')) {
                write_debug_log("Frontend: Cached detected language {$detected_language} for session {$session_id}");
            }
        } catch (Exception $e) {
            if (function_exists('write_debug_log')) {
                write_debug_log("Frontend: Error caching language: " . $e->getMessage());
            }
        }
        
        return $detected_language;
    }
}