<?php
/**
 * Display Controller for Bubblibot
 * Handles visibility logic and positioning
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Bubblibot_Display_Controller {
    
    public function __construct() {
        // Constructor
    }
    
    /**
     * Check if chatbot should be displayed on current page
     */
    public function should_display_chatbot() {
        // Check if chatbot is enabled
        if (get_option('bubblibot_enabled', '1') !== '1') {
            return false;
        }
        
        // Check mobile setting
        if (wp_is_mobile() && get_option('bubblibot_mobile_enabled', '1') !== '1') {
            return false;
        }
        
        // Check if current page is excluded
        if ($this->is_page_excluded()) {
            return false;
        }
        
        // Check display pages setting
        $display_pages = get_option('bubblibot_display_pages', array('all'));
        if (!is_array($display_pages)) {
            $display_pages = array('all');
        }
        
        // If "all" is selected, show everywhere (except excluded)
        if (in_array('all', $display_pages)) {
            return true;
        }
        
        // Check specific page types
        // Enhanced front page detection (fallback for page builders like Thrive Architect)
        if ($this->is_front_page_enhanced()) {
            return in_array('home', $display_pages);
        }
        
        // Enhanced single posts/articles detection (fallback for page builders)
        if ($this->is_single_post_enhanced()) {
            return in_array('posts', $display_pages);
        }
        
        // Enhanced regular pages detection (fallback for page builders)
        if ($this->is_regular_page_enhanced()) {
            return in_array('pages', $display_pages);
        }
        
        // WooCommerce checks
        if (class_exists('WooCommerce')) {
            if (is_shop()) {
                return in_array('shop', $display_pages);
            }
            
            if (is_product()) {
                return in_array('products', $display_pages);
            }
        }
        
        // Archive pages (categories, tags, etc.)
        if (is_category() || is_tag() || is_archive()) {
            return in_array('archive', $display_pages);
        }
        
        // Search results
        if (is_search()) {
            return in_array('search', $display_pages);
        }
        
        return false;
    }
    
    /**
     * Enhanced front page detection for page builders
     * 
     * This is a fallback method for page builders like Thrive Architect
     * that don't trigger WordPress's standard is_front_page() function.
     * For regular WordPress sites, is_front_page() works fine.
     */
    private function is_front_page_enhanced() {
        // Standard WordPress front page detection (works for most sites)
        if (is_front_page()) {
            return true;
        }
        
        // Fallback: Check if we're on the root URL (for page builders)
        $clean_uri = $this->get_clean_request_uri();
        if ($clean_uri === '/' || $clean_uri === '') {
            return true;
        }
        
        // Fallback: Check if current page ID matches the front page setting
        $front_page_id = get_option('page_on_front');
        if ($front_page_id && is_page($front_page_id)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Enhanced regular page detection for page builders
     * 
     * This is a fallback method for page builders like Thrive Architect
     * that don't trigger WordPress's standard is_page() function.
     * For regular WordPress sites, is_page() works fine.
     */
    private function is_regular_page_enhanced() {
        // Standard WordPress page detection (works for most sites)
        if (is_page() && !is_front_page()) {
            return true;
        }
        
        // Fallback: Check wp_query object for page builders
        global $wp_query;
        if (isset($wp_query->queried_object) && 
            isset($wp_query->queried_object->post_type) && 
            $wp_query->queried_object->post_type === 'page' &&
            !$this->is_front_page_enhanced()) {
            return true;
        }
        
        // Fallback: Check database for actual post type (for page builders)
        $post_type = $this->get_current_post_type();
        if ($post_type === 'page' && !$this->is_front_page_enhanced()) {
            return true;
        }
        
        // Final fallback: URL-based detection (only if post type unknown)
        if (!$this->is_front_page_enhanced() && 
            !is_single() && 
            !is_category() && 
            !is_tag() && 
            !is_archive() && 
            !is_search() && 
            !is_404() &&
            $post_type === null) {
            
            $clean_uri = $this->get_clean_request_uri();
            
            // Check if URL looks like a page
            if ($clean_uri !== '/' && 
                !preg_match('/\/(page|category|tag|author|date|search)\//', $clean_uri) &&
                !preg_match('/\/\d{4}\/\d{2}\//', $clean_uri) && 
                !preg_match('/\.(php|html|htm|asp|aspx)$/', $clean_uri)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Enhanced single post detection for page builders
     * 
     * This is a fallback method for page builders like Thrive Architect
     * that don't trigger WordPress's standard is_single() function.
     * For regular WordPress sites, is_single() works fine.
     */
    private function is_single_post_enhanced() {
        // Standard WordPress single post detection (works for most sites)
        if (is_single() && !is_attachment()) {
            return true;
        }
        
        // Fallback: Check database for actual post type (for page builders)
        $post_type = $this->get_current_post_type();
        if ($post_type === 'post') {
            return true;
        }
        
        return false;
    }
    
    /**
     * Get the actual post type from database based on current URL
     * 
     * This is used as a fallback when WordPress conditional functions
     * don't work properly with page builders.
     */
    private function get_current_post_type() {
        global $wpdb, $wp_query;
        
        // First try to get from wp_query if available
        if (isset($wp_query->queried_object) && isset($wp_query->queried_object->post_type)) {
            return $wp_query->queried_object->post_type;
        }
        
        // Get the current URL path
        $clean_uri = $this->get_clean_request_uri();
        
        // Remove leading and trailing slashes
        $slug = trim($clean_uri, '/');
        
        if (empty($slug)) {
            return null; // This is the home page
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $post = $wpdb->get_row( $wpdb->prepare(
            "SELECT post_type FROM {$wpdb->posts} WHERE post_name = %s AND post_status = 'publish' LIMIT 1",
            $slug
        ) );
        
        return $post ? $post->post_type : null;
    }
    
    /**
     * Get clean request URI without query parameters
     */
    private function get_clean_request_uri() {
        // Sanitize the incoming request URI immediately to satisfy PHPCS
        $uri = sanitize_text_field( isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '/' );
        
        // Remove query parameters
        $uri = strtok($uri, '?');
        
        // Remove common cache-busting parameters that might be in the path
        $uri = preg_replace('/\/\?[^\/]*$/', '/', $uri);
        
        return $uri;
    }
    
    /**
     * Check if current page is in exclusion list
     */
    private function is_page_excluded() {
        $exclude_pages = get_option('bubblibot_exclude_pages', '');
        if (empty($exclude_pages)) {
            return false;
        }
        
        $excluded_paths = array_filter(array_map('trim', explode("\n", $exclude_pages)));
        $current_url = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        
        foreach ($excluded_paths as $path) {
            // Handle page/post ID format (page:123, post:456)
            if (preg_match('/^(page|post):(\d+)$/', $path, $matches)) {
                $type = $matches[1];
                $id = (int) $matches[2];
                
                if ($type === 'page' && is_page($id)) {
                    return true;
                }
                
                if ($type === 'post' && is_single($id)) {
                    return true;
                }
            }
            // Handle URL path format
            else {
                // Ensure path starts with /
                if (substr($path, 0, 1) !== '/') {
                    $path = '/' . $path;
                }
                
                // Check if current URL matches excluded path
                if (substr($current_url, 0, strlen($path)) === $path) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get chatbot position CSS classes
     */
    public function get_position_classes() {
        $position = get_option('bubblibot_position', 'bottom-right');
        
        $classes = array('bubblibot-widget');
        
        switch ($position) {
            case 'bottom-left':
                $classes[] = 'position-bottom-left';
                break;
            case 'bottom-center':
                $classes[] = 'position-bottom-center';
                break;
            case 'right-center':
                $classes[] = 'position-right-center';
                break;
            case 'left-center':
                $classes[] = 'position-left-center';
                break;
            case 'bottom-right':
            default:
                $classes[] = 'position-bottom-right';
                break;
        }
        
        return implode(' ', $classes);
    }
    
    /**
     * Get position-specific CSS styles
     */
    public function get_position_styles() {
        $position = get_option('bubblibot_position', 'bottom-right');
        
        $styles = array();
        
        switch ($position) {
            case 'bottom-left':
                $styles = array(
                    'bottom' => '20px',
                    'left' => '20px',
                    'right' => 'auto'
                );
                break;
            case 'bottom-center':
                $styles = array(
                    'bottom' => '20px',
                    'left' => '50%',
                    'transform' => 'translateX(-50%)',
                    'right' => 'auto'
                );
                break;
            case 'right-center':
                $styles = array(
                    'top' => '50%',
                    'right' => '20px',
                    'transform' => 'translateY(-50%)',
                    'bottom' => 'auto'
                );
                break;
            case 'left-center':
                $styles = array(
                    'top' => '50%',
                    'left' => '20px',
                    'transform' => 'translateY(-50%)',
                    'bottom' => 'auto',
                    'right' => 'auto'
                );
                break;
            case 'bottom-right':
            default:
                $styles = array(
                    'bottom' => '20px',
                    'right' => '20px'
                );
                break;
        }
        
        return $styles;
    }
    
    /**
     * Generate inline CSS for positioning
     */
    public function generate_position_css() {
        $styles = $this->get_position_styles();
        $css_rules = array();
        
        foreach ($styles as $property => $value) {
            $css_rules[] = $property . ': ' . $value;
        }
        
        return implode('; ', $css_rules);
    }
} 