<?php
/**
 * Database operations for the chatbot plugin
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

class Bubblibot_Database {
    
    private $table_name;
    private $analytics_table_name;
    private $conversations_table_name;
    private $messages_table_name;
    private $handoff_queue_table_name;
    private $leads_table_name;
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'bubblibot_content';
        $this->analytics_table_name = $wpdb->prefix . 'bubblibot_analytics';
        $this->conversations_table_name = $wpdb->prefix . 'bubblibot_conversations';
        $this->messages_table_name = $wpdb->prefix . 'bubblibot_messages';
        $this->handoff_queue_table_name = $wpdb->prefix . 'bubblibot_handoff_queue';
        $this->leads_table_name = $wpdb->prefix . 'bubblibot_leads';
    }
    
    /**
     * Create the chatbot content table
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            content longtext NOT NULL,
            content_type varchar(50) DEFAULT 'post',
            file_path varchar(500),
            source_title varchar(500),
            embedding_id varchar(255),
            product_price decimal(10,2),
            product_sale_price decimal(10,2),
            product_regular_price decimal(10,2),
            product_currency varchar(10),
            product_stock_status varchar(20),
            product_stock_quantity int,
            product_sku varchar(100),
            product_type varchar(50),
            last_updated datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY post_id (post_id),
            KEY content_type (content_type),
            KEY product_price (product_price),
            KEY product_stock_status (product_stock_status),
            FULLTEXT KEY content (content),
            FULLTEXT KEY content_title (content, source_title)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Create analytics table
        $analytics_sql = "CREATE TABLE IF NOT EXISTS {$this->analytics_table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_session varchar(32),
            question_text longtext NOT NULL,
            detected_language varchar(10),
            context_found tinyint(1) DEFAULT 0,
            response_generated tinyint(1) DEFAULT 0,
            user_satisfied tinyint(1) DEFAULT NULL,
            response_time int DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            user_ip varchar(45),
            user_agent varchar(500),
            PRIMARY KEY (id),
            KEY user_session (user_session),
            KEY created_at (created_at),
            KEY context_found (context_found),
            KEY detected_language (detected_language),
            FULLTEXT KEY question_text (question_text)
        ) $charset_collate;";
        
        dbDelta($analytics_sql);
        
        // Create conversations table for handoff functionality
        $conversations_sql = "CREATE TABLE IF NOT EXISTS {$this->conversations_table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            session_id varchar(64),
            user_email varchar(100),
            user_name varchar(100),
            detected_language varchar(10) DEFAULT 'en',
            status enum('active', 'waiting_human', 'with_human', 'resolved', 'abandoned') DEFAULT 'active',
            assigned_to bigint(20),
            priority enum('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            resolved_at datetime,
            PRIMARY KEY (id),
            UNIQUE KEY session_id (session_id),
            KEY status (status),
            KEY assigned_to (assigned_to),
            KEY priority (priority),
            KEY detected_language (detected_language),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        // Create messages table for handoff functionality
        $messages_sql = "CREATE TABLE IF NOT EXISTS {$this->messages_table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            conversation_id bigint(20),
            sender_type enum('user', 'bot', 'human', 'system') NOT NULL,
            sender_id bigint(20),
            message longtext NOT NULL,
            metadata text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY conversation_id (conversation_id),
            KEY sender_type (sender_type),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        // Create handoff queue table
        $handoff_queue_sql = "CREATE TABLE IF NOT EXISTS {$this->handoff_queue_table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            conversation_id bigint(20),
            reason varchar(255),
            urgency int DEFAULT 5,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            claimed_at datetime,
            claimed_by bigint(20),
            PRIMARY KEY (id),
            KEY conversation_id (conversation_id),
            KEY urgency (urgency),
            KEY created_at (created_at),
            KEY claimed_by (claimed_by)
        ) $charset_collate;";
        
        // Create leads table for lead capture functionality
        $leads_sql = "CREATE TABLE IF NOT EXISTS {$this->leads_table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            conversation_id bigint(20),
            session_id varchar(64),
            email varchar(100) NOT NULL,
            name varchar(100),
            phone varchar(20),
            company varchar(100),
            intent_type varchar(50),
            intent_message text,
            source_page varchar(500),
            lead_score int DEFAULT 0,
            status enum('new', 'contacted', 'converted', 'lost') DEFAULT 'new',
            notes text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            contacted_at datetime,
            PRIMARY KEY (id),
            KEY conversation_id (conversation_id),
            KEY session_id (session_id),
            KEY email (email),
            KEY intent_type (intent_type),
            KEY status (status),
            KEY lead_score (lead_score),
            KEY created_at (created_at)
        ) $charset_collate;";

        if (function_exists('write_debug_log')) {
            write_debug_log("Database: Creating conversations table with SQL: " . $conversations_sql);
        }
        
        $result1 = dbDelta($conversations_sql);
        $result2 = dbDelta($messages_sql);
        $result3 = dbDelta($handoff_queue_sql);
        $result4 = dbDelta($leads_sql);
        
        if (function_exists('write_debug_log')) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
            write_debug_log( 'Database: dbDelta results - Conversations: ' . wp_json_encode( $result1 ) );
            if ($wpdb->last_error) {
                write_debug_log("Database: MySQL Error: " . $wpdb->last_error);
            }
        }
        
        // Only log errors, not successful operations
        if (function_exists('write_debug_log')) {
            // Check if tables actually exist and log only if there are issues
            global $wpdb;
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $conversations_exists = $wpdb->get_var( "SHOW TABLES LIKE '{$this->conversations_table_name}'" );
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $leads_exists        = $wpdb->get_var( "SHOW TABLES LIKE '{$this->leads_table_name}'" );
            
            if (!$conversations_exists) {
                write_debug_log('BubbliBot Database ERROR: Conversations table creation failed');
            }
            if (!$leads_exists) {
                write_debug_log('BubbliBot Database ERROR: Leads table creation failed');
            }
        }
    }
    
    /**
     * Migrate conversations table to add language support
     */
    public function migrate_conversations_language_support() {
        global $wpdb;
        
        // Check if language column already exists
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$this->conversations_table_name} LIKE 'detected_language'");
        
        if (empty($column_exists)) {
            write_debug_log('Database: Adding detected_language column to conversations table');
            
            $sql = "ALTER TABLE {$this->conversations_table_name} 
                    ADD COLUMN detected_language varchar(10) DEFAULT 'en' 
                    AFTER user_name";
            
            $result = $wpdb->query($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            
            if ($result === false) {
                write_debug_log('Database: Error adding detected_language column: ' . $wpdb->last_error);
            } else {
                write_debug_log('Database: Successfully added detected_language column');
                
                // Add index for the new column
                $index_sql = "ALTER TABLE {$this->conversations_table_name} 
                              ADD INDEX detected_language (detected_language)";
                $wpdb->query($index_sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            }
        } else {
            write_debug_log('Database: detected_language column already exists in conversations table');
        }
    }
    
    /**
     * Insert or update content in the index
     */
    public function update_content($post_id, $content, $embedding_id = '', $content_type = 'post', $file_path = '', $source_title = '', $product_data = array()) {
        global $wpdb;
        
        $existing = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id FROM {$this->table_name} WHERE post_id = %d AND content_type = %s",
                $post_id,
                $content_type
            )
        );
        
        // Free version: content limit removed - unlimited indexing allowed
        // Other free version limitations still apply (no embeddings, no memory, etc.)
        
        // Prepare update/insert data
        $data = array(
            'content' => $content,
            'content_type' => $content_type,
            'file_path' => $file_path,
            'source_title' => $source_title,
            'embedding_id' => $embedding_id,
            'last_updated' => current_time('mysql')
        );
        
        $formats = array('%s', '%s', '%s', '%s', '%s', '%s');
        
        // Add product data if provided
        if (!empty($product_data)) {
            if (isset($product_data['price'])) {
                $data['product_price'] = $product_data['price'];
                $formats[] = '%f';
            }
            if (isset($product_data['sale_price'])) {
                $data['product_sale_price'] = $product_data['sale_price'];
                $formats[] = '%f';
            }
            if (isset($product_data['regular_price'])) {
                $data['product_regular_price'] = $product_data['regular_price'];
                $formats[] = '%f';
            }
            if (isset($product_data['currency'])) {
                $data['product_currency'] = $product_data['currency'];
                $formats[] = '%s';
            }
            if (isset($product_data['stock_status'])) {
                $data['product_stock_status'] = $product_data['stock_status'];
                $formats[] = '%s';
            }
            if (isset($product_data['stock_quantity'])) {
                $data['product_stock_quantity'] = $product_data['stock_quantity'];
                $formats[] = '%d';
            }
            if (isset($product_data['sku'])) {
                $data['product_sku'] = $product_data['sku'];
                $formats[] = '%s';
            }
            if (isset($product_data['type'])) {
                $data['product_type'] = $product_data['type'];
                $formats[] = '%s';
            }
        }

        if ($existing) {
            $result = $wpdb->update(
                $this->table_name,
                $data,
                array('post_id' => $post_id, 'content_type' => $content_type),
                $formats,
                array('%d', '%s')
            );
            if (function_exists('write_debug_log') && $result === false) {
                write_debug_log("BubbliBot Database: UPDATE error for {$content_type} ID {$post_id}: " . $wpdb->last_error);
            } elseif (function_exists('write_debug_log') && $result !== false) {
                write_debug_log("BubbliBot Database: Successfully UPDATED {$content_type} ID {$post_id} - {$source_title}");
            }
        } else {
            // Add post_id to the beginning of the data array
            $data = array_merge(array('post_id' => $post_id), $data);
            // Add corresponding format to the beginning
            $formats = array_merge(array('%d'), $formats);
            
            $result = $wpdb->insert(
                $this->table_name,
                $data,
                $formats
            );
            if (function_exists('write_debug_log') && $result === false) {
                write_debug_log("BubbliBot Database: INSERT error for {$content_type} ID {$post_id}: " . $wpdb->last_error);
            } elseif (function_exists('write_debug_log') && $result !== false) {
                write_debug_log("BubbliBot Database: Successfully INSERTED {$content_type} ID {$post_id} - {$source_title}");
            }
        }
    }
    
    /**
     * Remove content from index
     */
    public function delete_content($post_id, $content_type = null) {
        global $wpdb;
        
        if ($content_type) {
            $wpdb->delete(
                $this->table_name, 
                array('post_id' => $post_id, 'content_type' => $content_type), 
                array('%d', '%s')
            );
        } else {
            // Delete all content types for this post_id
            $wpdb->delete($this->table_name, array('post_id' => $post_id), array('%d'));
        }
        
        if (function_exists('write_debug_log')) {
            write_debug_log("BubbliBot Database: Deleted content for ID {$post_id}" . ($content_type ? " (type: {$content_type})" : ""));
        }
    }
    
    /**
     * Search content for relevant information
     */
    public function search_content($query, $limit = 2) {
        global $wpdb;
        
        write_debug_log('Bubblibot Database: Starting content search for query: ' . $query);
        
        // Try to normalize words to their base forms using AI (only for complex languages)
        $normalized_query = $query;
        if (class_exists('Bubblibot_OpenAI') && class_exists('Bubblibot_Language_Detector')) {
            $openai = new Bubblibot_OpenAI();
            $language_detector = new Bubblibot_Language_Detector();
            
            // Use the sophisticated language detector instead of simple heuristics
            $detected_language = $language_detector->detect_user_language($query);
            write_debug_log('BubbliBot Database: Detected language for normalization: ' . $detected_language);
            
            $normalized_query = $openai->normalize_query_words($query, $detected_language);
            
            if ($normalized_query !== $query) {
                write_debug_log('BubbliBot Database: Using normalized query: ' . $normalized_query);
            }
        }
        
        // Clean and prepare search terms intelligently
        $search_data = $this->analyze_search_query($normalized_query);
        
        write_debug_log('BubbliBot Database: Query analysis - Original terms: ' . implode(', ', $search_data['original_terms']) . ' | Key terms: ' . implode(', ', $search_data['key_terms']));
        
        if (empty($search_data['key_terms'])) {
            write_debug_log('BubbliBot Database: No valid search terms found');
            return array();
        }
        
        // Filter out stop words to get meaningful search terms
        $meaningful_terms = array_filter($search_data['key_terms'], function($term) {
            return !$this->is_stop_word($term);
        });
        
        if (empty($meaningful_terms)) {
            $meaningful_terms = array_slice($search_data['key_terms'], 0, 3);
        }
        
        write_debug_log('BubbliBot Database: Meaningful terms for search: ' . implode(', ', $meaningful_terms));
        
        // PRIORITY 1: WooCommerce products (if available)
        if (class_exists('WooCommerce')) {
            $product_results = $this->simple_woocommerce_search($meaningful_terms, 5);
            if (!empty($product_results)) {
                write_debug_log('BubbliBot Database: Found WooCommerce products: ' . count($product_results));
                return $product_results;
            }
        }
        
        // FALLBACK: Other content search strategies
        $title_results = $this->search_by_title($meaningful_terms, $limit, $normalized_query);
        if (!empty($title_results)) {
            write_debug_log('BubbliBot Database: Found title-based matches: ' . count($title_results));
            return $title_results;
        }
        
        $enhanced_product_results = $this->search_product_category($meaningful_terms, $normalized_query, $limit);
        if (!empty($enhanced_product_results)) {
            write_debug_log('BubbliBot Database: Found product category matches: ' . count($enhanced_product_results));
            return $enhanced_product_results;
        }
        
        $product_insights = $this->analyze_product_query($normalized_query, $search_data);
        if (!empty($product_insights['product_names'])) {
            write_debug_log('BubbliBot Database: Detected potential product names: ' . implode(', ', $product_insights['product_names']));
            
            foreach ($product_insights['product_names'] as $product_name) {
                $product_results = $this->search_product_name($product_name, $limit);
                if (!empty($product_results)) {
                    write_debug_log('BubbliBot Database: Found product name matches for "' . $product_name . '": ' . count($product_results));
                    return $product_results;
                }
            }
        }
        
        $exact_phrase_results = $this->search_exact_phrase($normalized_query, $limit);
        if (!empty($exact_phrase_results)) {
            write_debug_log('BubbliBot Database: Found exact phrase matches: ' . count($exact_phrase_results));
            return $exact_phrase_results;
        }
        
        $all_terms_results = $this->search_all_terms_required($meaningful_terms, $limit);
        if (!empty($all_terms_results)) {
            write_debug_log('BubbliBot Database: Found all-terms-required matches: ' . count($all_terms_results));
            return $all_terms_results;
        }
        
        $multi_term_results = $this->search_multiple_terms_flexible($meaningful_terms, $limit);
        if (!empty($multi_term_results)) {
            write_debug_log('BubbliBot Database: Found multi-term matches: ' . count($multi_term_results));
            return $multi_term_results;
        }
        
        $similarity_results = $this->search_with_similarity($meaningful_terms, $normalized_query, $limit);
        if (!empty($similarity_results)) {
            write_debug_log('BubbliBot Database: Found similarity matches: ' . count($similarity_results));
            return $similarity_results;
        }
        
        $individual_results = $this->search_individual_terms_flexible($meaningful_terms, $normalized_query, $limit);
        if (!empty($individual_results)) {
            write_debug_log('BubbliBot Database: Found individual term matches: ' . count($individual_results));
            return $individual_results;
        }
        
        // If no content-based matches found, fallback to title-based results
        if (!empty($title_results)) {
            write_debug_log('BubbliBot Database: Fallback to title-based matches');
            return $title_results;
        }
        
        write_debug_log('BubbliBot Database: No relevant results found');
        return array();
    }
    
    /**
     * Simple WooCommerce product search - no complex logic, just find matching products
     */
    private function simple_woocommerce_search($terms, $limit = 5) {
        global $wpdb;
        
        if (empty($terms)) return array();
        
        write_debug_log('BubbliBot Database: Simple WooCommerce search for: ' . implode(', ', $terms));
        
        // Create simple search terms including variations
        $all_search_terms = array();
        foreach ($terms as $term) {
            $variations = $this->create_word_variations($term);
            $all_search_terms = array_merge($all_search_terms, $variations);
        }
        
        // Build search conditions with proper placeholders
        $search_conditions = array();
        $search_values = array();
        foreach ($all_search_terms as $search_term) {
            $search_conditions[] = "(c.source_title LIKE %s OR c.content LIKE %s)";
            $like_term = '%' . $wpdb->esc_like($search_term) . '%';
            $search_values[] = $like_term;
            $search_values[] = $like_term;
        }
        
        if (empty($search_conditions)) {
            return array();
        }
        
        // Build the base query with placeholders
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        $sql = "SELECT c.*, 
                (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish') as title,
                c.post_id,
                c.content_type
            FROM {$this->table_name} c
            WHERE c.content_type = 'product' 
            AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
            AND (" . implode(' OR ', $search_conditions) . ")
            ORDER BY c.last_updated DESC
            LIMIT %d";
        
        // Add limit to values array
        $search_values[] = $limit;
        
        // Prepare the complete query
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Complex query assembled with placeholders above
        $prepared_sql = $wpdb->prepare($sql, $search_values);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query was prepared above
        $results = $wpdb->get_results($prepared_sql);
        
        if (empty($results)) {
            write_debug_log('BubbliBot Database: No WooCommerce products found');
            return array();
        }
        
        write_debug_log('BubbliBot Database: Found ' . count($results) . ' WooCommerce products');
        
        return array_map(function($row) use ($terms) {
            return $this->format_search_result($row, implode(' ', $terms));
        }, $results);
    }
    
    /**
     * Search specifically by title content (highest priority)
     */
    private function search_by_title($terms, $limit, $original_query = '') {
        global $wpdb;
        
        if (empty($terms)) return array();
        
        write_debug_log('BubbliBot Database: Searching titles for terms: ' . implode(', ', $terms));
        
        // Build conditions for title search with word variations
        $title_conditions = array();
        $all_variations = array();
        
        foreach ($terms as $term) {
            $variations = $this->create_word_variations($term);
            $all_variations = array_merge($all_variations, $variations);
            
            $term_conditions = array();
            foreach ($variations as $variation) {
                $term_conditions[] = $wpdb->prepare("c.source_title LIKE %s", '%' . $wpdb->esc_like($variation) . '%');
            }
            
            // Group variations for this term with OR
            if (!empty($term_conditions)) {
                $title_conditions[] = '(' . implode(' OR ', $term_conditions) . ')';
            }
        }
        
        write_debug_log('BubbliBot Database: Expanded search to variations: ' . implode(', ', array_unique($all_variations)));
        
        // For special product queries, increase the limit to get more results
        $actual_limit = $limit;
        // Use original query if provided, otherwise fallback to terms
        $query_for_pricing = !empty($original_query) ? $original_query : implode(' ', $terms);
        if ($this->detect_pricing_intent($query_for_pricing)) {
            $actual_limit = min($limit * 4, 15); // Get more results for pricing queries - same as product category search
        }
        
        // Build score parts with placeholders
        $score_parts = array();
        $score_values = array();
        foreach ($all_variations as $variation) {
            $score_parts[] = "CASE WHEN c.source_title LIKE %s THEN 1 ELSE 0 END";
            $score_values[] = '%' . $wpdb->esc_like($variation) . '%';
        }
        
        // Build the complete SQL with placeholders
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        $sql = "SELECT c.*, 
                CASE 
                    WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                    WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                    WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                    ELSE c.source_title
                END as title,
                c.post_id,
                c.content_type,
                (" . implode(' + ', $score_parts) . ") as title_match_score
            FROM {$this->table_name} c
            WHERE (
                (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                OR (c.content_type NOT IN ('product', 'post', 'page'))
            )
            AND (" . implode(' OR ', $title_conditions) . ")
            ORDER BY title_match_score DESC, c.last_updated DESC
            LIMIT %d";
        
        // Combine all values and add limit
        $all_values = $score_values;
        $all_values[] = $actual_limit;
        
        // Prepare and execute
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query was prepared above
        $prepared_sql = $wpdb->prepare($sql, $all_values);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query was prepared above
        $results = $wpdb->get_results($prepared_sql);
        
        if (empty($results)) {
            write_debug_log('BubbliBot Database: No title matches found');
            return array();
        }
        
        // Apply smarter scoring based on term importance
        foreach ($results as &$row) {
            $title_lower = strtolower($row->title);
            $important_terms_found = 0;
            $total_important_terms = 0;
            
            // Identify important terms (proper nouns and content words, skip common words)
            $skip_words = array('tell', 'me', 'about', 'the', 'a', 'an', 'is', 'are', 'was', 'were', 'what', 'how', 'why', 'when', 'where');
            
            foreach ($terms as $term) {
                $term_lower = strtolower($term);
                
                // Skip common/question words
                if (in_array($term_lower, $skip_words)) {
                    continue;
                }
                
                // This is an important term
                $total_important_terms++;
                
                // Check if it's in the title
                if (strpos($title_lower, $term_lower) !== false) {
                    $important_terms_found++;
                }
            }
            
            // Calculate match percentage for important terms only
            if ($total_important_terms > 0) {
                $match_ratio = $important_terms_found / $total_important_terms;
                
                // Boost scoring based on match ratio
                if ($match_ratio == 1.0) {
                    // All important terms found - big boost
                    $row->title_match_score = $row->title_match_score * 2;
                } elseif ($match_ratio >= 0.75) {
                    // Most important terms found - small boost
                    $row->title_match_score = $row->title_match_score * 1.5;
                } elseif ($match_ratio < 0.5) {
                    // Less than half important terms - small penalty
                    $row->title_match_score = $row->title_match_score * 0.8;
                }
                // 50-75% match keeps original score
            }
        }
        
        // Re-sort by adjusted scores
        usort($results, function($a, $b) {
            return $b->title_match_score <=> $a->title_match_score;
        });
        
        write_debug_log('BubbliBot Database: Found ' . count($results) . ' title matches');
        
        return array_map(function($row) use ($terms) {
            return $this->format_search_result($row, implode(' ', $terms));
        }, $results);
    }
    
    /**
     * Analyze query for product names and pricing intent
     */
    private function analyze_product_query($query, $search_data) {
        $insights = array(
            'product_names' => array(),
            'is_pricing_query' => false,
            'pricing_terms' => array()
        );
        
        // Check for pricing intent
        $pricing_patterns = array(
            '/\b(cost|costs|price|pricing|how much|what.*(cost|price)|expensive|cheap|fee|fees)\b/i'
        );
        
        foreach ($pricing_patterns as $pattern) {
            if (preg_match($pattern, $query)) {
                $insights['is_pricing_query'] = true;
                break;
            }
        }
        
        // Extract potential product names (capitalized sequences or meaningful term combinations)
        $original_query = $query;
        $terms = $search_data['original_terms'];
        
        // Look for meaningful term combinations that could be product names
        $potential_products = array();
        
        // Strategy 1: Look for sequences of 2-4 meaningful terms
        for ($i = 0; $i < count($terms) - 1; $i++) {
            $term1 = $terms[$i];
            $term2 = $terms[$i + 1];
            
            // Skip very common words for product names
            if ($this->is_stop_word($term1) || $this->is_stop_word($term2)) {
                continue;
            }
            
            // Try 2-word combinations
            if (strlen($term1) > 3 && strlen($term2) > 3) {
                $potential_products[] = $term1 . ' ' . $term2;
                
                // Try 3-word combinations
                if (isset($terms[$i + 2]) && strlen($terms[$i + 2]) > 3 && !$this->is_stop_word($terms[$i + 2])) {
                    $potential_products[] = $term1 . ' ' . $term2 . ' ' . $terms[$i + 2];
                    
                    // Try 4-word combinations for longer product names
                    if (isset($terms[$i + 3]) && strlen($terms[$i + 3]) > 3 && !$this->is_stop_word($terms[$i + 3])) {
                        $potential_products[] = $term1 . ' ' . $term2 . ' ' . $terms[$i + 2] . ' ' . $terms[$i + 3];
                    }
                }
            }
        }
        
        // Strategy 2: Look for patterns that suggest product names
        // Common product name patterns like "X Vault", "X System", "X Course", etc.
        $product_suffixes = array('vault', 'system', 'course', 'program', 'guide', 'toolkit', 'framework', 'method', 'blueprint', 'training', 'academy', 'masterclass');
        
        foreach ($terms as $i => $term) {
            if (in_array(strtolower($term), $product_suffixes)) {
                // Look for 1-3 words before this suffix
                for ($j = max(0, $i - 3); $j < $i; $j++) {
                    if (!$this->is_stop_word($terms[$j]) && strlen($terms[$j]) > 2) {
                        $product_parts = array();
                        for ($k = $j; $k <= $i; $k++) {
                            $product_parts[] = $terms[$k];
                        }
                        $potential_products[] = implode(' ', $product_parts);
                    }
                }
            }
        }
        
        // Remove duplicates and filter
        $potential_products = array_unique($potential_products);
        foreach ($potential_products as $product) {
            if (strlen($product) > 6) { // Must be reasonably long
                $insights['product_names'][] = $product;
            }
        }
        
        return $insights;
    }
    
    /**
     * Check if a word is a common stop word for product names
     */
    private function is_stop_word($word) {
        // Basic English stop words only in free version
        $stop_words = array(
            'the', 'is', 'at', 'which', 'on', 'a', 'an', 'and', 'or', 'but', 'in', 'with', 'to', 'for', 'of', 'as', 'by',
            'that', 'this', 'it', 'from', 'they', 'we', 'you', 'have', 'had', 'has', 'do', 'does', 'did', 'will', 'would',
            'could', 'should', 'may', 'might', 'must', 'can', 'be', 'been', 'being', 'am', 'are', 'was', 'were',
            'what', 'when', 'where', 'why', 'how', 'who', 'whom', 'whose', 'which', 'that',
            'about', 'cost', 'costs', 'price', 'pricing', 'expensive', 'cheap'
        );
        
        return in_array(strtolower($word), $stop_words);
    }
    
    /**
     * Create word variations for better matching (basic English variations only)
     */
    private function create_word_variations($word) {
        $variations = array($word);
        $word_lower = strtolower($word);
        
        // Basic English plural/singular variations
        if (substr($word_lower, -1) === 's') {
            // Remove 's' for potential singular
            $base = substr($word, 0, -1);
            $variations[] = $base;
        } else {
            // Add 's' for potential plural
            $variations[] = $word . 's';
        }
        
        return array_unique($variations);
    }
    
    /**
     * Enhanced product search by category/type with better matching
     */
    private function search_product_category($terms, $query, $limit) {
        global $wpdb;
        
        if (empty($terms)) return array();
        
        // Look for product-related terms and expand them
        $product_terms = array();
        $expanded_terms = array();
        
        foreach ($terms as $term) {
            // Skip stop words (which include common pricing terms)
            if ($this->is_stop_word($term)) {
                continue;
            }
            
            $variations = $this->create_word_variations($term);
            $product_terms[] = $term;
            $expanded_terms = array_merge($expanded_terms, $variations);
        }
        
        if (empty($product_terms)) {
            return array();
        }
        
        write_debug_log('BubbliBot Database: Product category search for terms: ' . implode(', ', $product_terms));
        write_debug_log('BubbliBot Database: Expanded to variations: ' . implode(', ', array_unique($expanded_terms)));
        
        // Build comprehensive search conditions
        $search_conditions = array();
        foreach ($expanded_terms as $term) {
            $like_term = '%' . $wpdb->esc_like($term) . '%';
            $search_conditions[] = $wpdb->prepare("(c.source_title LIKE %s OR c.content LIKE %s)", $like_term, $like_term);
        }
        
        if (empty($search_conditions)) {
            return array();
        }
        
        // Increase limit for pricing queries to find all matching products
        $actual_limit = $this->detect_pricing_intent($query) ? min($limit * 4, 15) : $limit;
        
        // Build title match conditions for relevance scoring
        $title_match_conditions = array();
        foreach ($expanded_terms as $term) {
            $title_match_conditions[] = $wpdb->prepare("c.source_title LIKE %s", '%' . $wpdb->esc_like($term) . '%');
        }
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        $sql = "SELECT c.*, 
                CASE 
                    WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                    WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                    WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                    ELSE c.source_title
                END as title,
                c.post_id,
                c.content_type,
                -- Prioritize products and title matches
                CASE 
                    WHEN c.content_type = 'product' THEN 10
                    ELSE 1
                END +
                CASE 
                    WHEN " . implode(' OR ', $title_match_conditions) . " THEN 5
                    ELSE 1
                END as relevance_score
            FROM {$this->table_name} c
            WHERE (
                (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                OR (c.content_type NOT IN ('product', 'post', 'page'))
            )
            AND (" . implode(' OR ', $search_conditions) . ")
            ORDER BY relevance_score DESC, c.content_type = 'product' DESC, c.last_updated DESC
            LIMIT " . intval($actual_limit);
        
        $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        
        if (empty($results)) {
            return array();
        }
        
        write_debug_log('BubbliBot Database: Found ' . count($results) . ' product category matches');
        
        return array_map(function($row) use ($query) {
            return $this->format_search_result($row, $query);
        }, $results);
    }
    
    /**
     * Search specifically for product names
     */
    private function search_product_name($product_name, $limit) {
        global $wpdb;
        
        write_debug_log('BubbliBot Database: Searching for product name: ' . $product_name);
        
        // Search for the product name as a phrase with high priority for titles
        $sql = $wpdb->prepare(
            "SELECT c.*, 
                CASE 
                    WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                    WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                    WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                    ELSE c.source_title
                END as title,
                c.post_id,
                c.content_type,
                CASE 
                    WHEN c.source_title LIKE %s THEN 100
                    WHEN c.content LIKE %s THEN 50
                    ELSE 0
                END as relevance_score
            FROM {$this->table_name} c
            WHERE (
                (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                OR (c.content_type NOT IN ('product', 'post', 'page'))
            )
            AND (c.source_title LIKE %s OR c.content LIKE %s)
            ORDER BY relevance_score DESC, c.last_updated DESC
            LIMIT %d",
            '%' . $product_name . '%', // For relevance score calculation
            '%' . $product_name . '%', // For relevance score calculation
            '%' . $product_name . '%', // For WHERE clause
            '%' . $product_name . '%', // For WHERE clause
            intval($limit)
        );
        
        $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        
        if (empty($results)) {
            return array();
        }
        
        write_debug_log('BubbliBot Database: Found ' . count($results) . ' results for product name "' . $product_name . '"');
        
        return array_map(function($row) use ($product_name) {
            return $this->format_search_result($row, $product_name);
        }, $results);
    }
    
    /**
     * Search for exact phrase matches
     */
    private function search_exact_phrase($query, $limit) {
        global $wpdb;
        
        // Clean the query but keep meaningful phrases
        $clean_query = trim(preg_replace('/[^\p{L}\p{N}\s]/u', ' ', $query));
        $clean_query = preg_replace('/\s+/', ' ', $clean_query);
        
        // Skip very short queries
        if (strlen($clean_query) < 4) {
            return array();
        }
        
        $sql = $wpdb->prepare(
            "SELECT c.*, 
                CASE 
                    WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                    WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                    WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                    ELSE c.source_title
                END as title,
                c.post_id,
                c.content_type
            FROM {$this->table_name} c
            WHERE (
                (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                OR (c.content_type NOT IN ('product', 'post', 'page'))
            )
            AND (c.content LIKE %s OR c.source_title LIKE %s)
            ORDER BY 
                CASE WHEN c.source_title LIKE %s THEN 1 ELSE 2 END,
                c.last_updated DESC
            LIMIT %d",
            '%' . $clean_query . '%',
            '%' . $clean_query . '%',
            '%' . $clean_query . '%',
            intval($limit)
        );
        
        $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        
        if (empty($results)) {
            return array();
        }
        
        return array_map(function($row) use ($query) {
            return $this->format_search_result($row, $query);
        }, $results);
    }
    
    /**
     * Search for multiple terms with flexible matching
     */
    private function search_multiple_terms_flexible($terms, $limit) {
        global $wpdb;
        
        if (count($terms) < 2) return array();
        
        // Use the most important terms (longest ones first)
        $important_terms = array_slice($terms, 0, 3);
        
        // Build conditions that allow for partial matching
        $title_conditions = array();
        $content_conditions = array();
        $title_placeholders = array();
        $content_placeholders = array();
        $title_values = array();
        $content_values = array();
        
        foreach ($important_terms as $term) {
            if (strlen($term) > 3) { // Only use meaningful terms
                $title_placeholders[] = "c.source_title LIKE %s";
                $content_placeholders[] = "c.content LIKE %s";
                $like_term = '%' . $wpdb->esc_like($term) . '%';
                $title_values[] = $like_term;
                $content_values[] = $like_term;
            }
        }
        
        if (empty($title_placeholders)) return array();
        
        // Build conditions for at least 2 terms matching
        $title_cond_str = count($title_placeholders) >= 2 ? 
            '(' . implode(' AND ', array_slice($title_placeholders, 0, 2)) . ')' : 
            '(' . implode(' AND ', $title_placeholders) . ')';
        $content_cond_str = count($content_placeholders) >= 2 ? 
            '(' . implode(' AND ', array_slice($content_placeholders, 0, 2)) . ')' : 
            '(' . implode(' AND ', $content_placeholders) . ')';
        
        $multi_term_condition = '(' . $title_cond_str . ' OR ' . $content_cond_str . ')';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        $sql = "SELECT c.*, 
            CASE 
                WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                ELSE c.source_title
            END as title,
            c.post_id,
            c.content_type
        FROM {$this->table_name} c
        WHERE (
            (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
            OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
            OR (c.content_type NOT IN ('product', 'post', 'page'))
        )
        AND {$multi_term_condition}
        ORDER BY c.last_updated DESC
        LIMIT %d";
        
        // Prepare values: title values for first condition, content values for second, then limit
        $prepare_values = array_merge(
            array_slice($title_values, 0, 2),
            array_slice($content_values, 0, 2),
            array($limit)
        );
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query was prepared above
        $prepared_sql = $wpdb->prepare($sql, $prepare_values);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query was prepared above
        $results = $wpdb->get_results($prepared_sql);
        
        if (empty($results)) {
            return array();
        }
        
        return array_map(function($row) use ($terms) {
            return $this->format_search_result($row, implode(' ', $terms));
        }, $results);
    }
    
    /**
     * Search with intelligent similarity matching for word inflections and partial matches
     */
    private function search_with_similarity($terms, $original_query, $limit) {
        global $wpdb;
        
        if (empty($terms)) return array();
        
        $all_results = array();
        $processed_terms = array();
        
        // Process each term to create similarity patterns
        foreach ($terms as $term) {
            if (strlen($term) < 4) continue; // Skip very short terms
            
            $similarity_patterns = $this->generate_similarity_patterns($term);
            $processed_terms[] = array(
                'original' => $term,
                'patterns' => $similarity_patterns
            );
        }
        
        if (empty($processed_terms)) return array();
        
        write_debug_log('BubbliBot Database: Similarity search for terms: ' . implode(', ', array_column($processed_terms, 'original')));
        
        // Search for each pattern
        foreach ($processed_terms as $term_data) {
            $term = $term_data['original'];
            $patterns = $term_data['patterns'];
            
            foreach ($patterns as $pattern_type => $pattern) {
                $sql = $wpdb->prepare(
                    "SELECT c.*, 
                        CASE 
                            WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                            WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                            WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                            ELSE c.source_title
                        END as title,
                        c.post_id,
                        c.content_type,
                        %s as matched_pattern,
                        %s as pattern_type
                    FROM {$this->table_name} c
                    WHERE (
                        (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                        OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                        OR (c.content_type NOT IN ('product', 'post', 'page'))
                    )
                    AND (c.source_title LIKE %s OR c.content LIKE %s)
                    ORDER BY 
                        CASE 
                            WHEN c.source_title LIKE %s THEN 1 
                            WHEN c.source_title LIKE %s THEN 2
                            ELSE 3 
                        END,
                        c.last_updated DESC
                    LIMIT %d",
                    $pattern,
                    $pattern_type,
                    '%' . $pattern . '%',
                    '%' . $pattern . '%',
                    '%' . $term . '%',  // Exact term in title gets highest priority
                    '%' . $pattern . '%', // Pattern match in title gets second priority
                    intval($limit * 2)
                );
                
                $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                
                if (!empty($results)) {
                    write_debug_log("BubbliBot Database: Found " . count($results) . " results for {$pattern_type} pattern '{$pattern}' from term '{$term}'");
                    
                    foreach ($results as $result) {
                        // Calculate similarity score
                        $result->similarity_score = $this->calculate_similarity_score($result, $term, $pattern, $pattern_type);
                        $all_results[] = $this->format_search_result($result, $original_query);
                    }
                }
            }
        }
        
        if (empty($all_results)) {
            return array();
        }
        
        // Remove duplicates and sort by relevance
        $unique_results = array();
        $seen_ids = array();
        
        foreach ($all_results as $result) {
            $key = $result['post_id'] . '_' . $result['content_type'];
            if (!in_array($key, $seen_ids)) {
                $seen_ids[] = $key;
                $unique_results[] = $result;
            }
        }
        
        return array_slice($unique_results, 0, $limit);
    }
    
    /**
     * Generate similarity patterns for intelligent word matching
     */
    private function generate_similarity_patterns($term) {
        $patterns = array();
        $term_len = mb_strlen($term);
        
        // Root/stem pattern - remove common endings (language-agnostic approach)
        if ($term_len >= 6) {
            // Try removing last 1-3 characters to find word stems
            $patterns['stem_1'] = mb_substr($term, 0, -1);
            $patterns['stem_2'] = mb_substr($term, 0, -2);
            if ($term_len >= 8) {
                $patterns['stem_3'] = mb_substr($term, 0, -3);
            }
        }
        
        // Prefix pattern - first part of the word
        if ($term_len >= 5) {
            $patterns['prefix'] = mb_substr($term, 0, intval($term_len * 0.7)); // 70% of the word
        }
        
        // Core pattern - middle part of longer words
        if ($term_len >= 7) {
            $start = intval($term_len * 0.1);
            $length = intval($term_len * 0.8);
            $patterns['core'] = mb_substr($term, $start, $length);
        }
        
        return $patterns;
    }
    
    /**
     * Calculate similarity score for ranking results
     */
    private function calculate_similarity_score($result, $original_term, $matched_pattern, $pattern_type) {
        $score = 0;
        $title = strtolower($result->title);
        $content = strtolower($result->content);
        $original_lower = strtolower($original_term);
        $pattern_lower = strtolower($matched_pattern);
        
        // Base score by pattern type
        switch ($pattern_type) {
            case 'stem_1':
                $score += 80; // High score for close stems
                break;
            case 'stem_2':
                $score += 70;
                break;
            case 'stem_3':
                $score += 60;
                break;
            case 'prefix':
                $score += 50;
                break;
            case 'core':
                $score += 40;
                break;
        }
        
        // Bonus for title matches
        if (strpos($title, $pattern_lower) !== false) {
            $score += 30;
        }
        
        // Bonus for exact term in title (highest priority)
        if (strpos($title, $original_lower) !== false) {
            $score += 50;
        }
        
        // Bonus for product content type (often more specific)
        if ($result->content_type === 'product') {
            $score += 20;
        }
        
        return $score;
    }
    
    /**
     * Analyze search query to extract meaningful terms and context
     */
    private function analyze_search_query($query) {
        // Normalize the query
        $normalized = preg_replace('/[^\p{L}\p{N}\s]/u', ' ', strtolower($query));
        $normalized = preg_replace('/\s+/', ' ', trim($normalized));
        
        // Split into words
        $words = explode(' ', $normalized);
        $words = array_filter($words, function($word) {
            return mb_strlen($word) > 1; // Keep words longer than 1 character
        });
        
        // Calculate term importance dynamically
        $term_scores = array();
        foreach ($words as $word) {
            $score = $this->calculate_term_importance($word, $words);
            if ($score > 0) {
                $term_scores[$word] = $score;
            }
        }
        
        // Sort by importance
        arsort($term_scores);
        
        return array(
            'original_query' => $query,
            'original_terms' => $words,
            'key_terms' => array_keys($term_scores),
            'term_scores' => $term_scores
        );
    }
    
    /**
     * Calculate the importance of a term dynamically
     */
    private function calculate_term_importance($term, $all_terms) {
        $length = mb_strlen($term);
        
        // Base score on length (longer terms are usually more specific)
        $score = $length * 10;
        
        // Reduce score for very common short words (but don't eliminate them)
        if ($length <= 3) {
            $score *= 0.3;
        }
        
        // Boost score for terms that appear to be meaningful
        // Check if it starts with uppercase (proper nouns are important)
        if (ctype_upper(substr($term, 0, 1))) {
            $score *= 1.5;
        }
        
        // Boost score for less common words (frequency inverse weighting)
        $frequency = array_count_values($all_terms)[$term];
        if ($frequency == 1) {
            $score *= 1.2; // Unique terms are more important
        }
        
        return $score;
    }
    
    /**
     * Search individual terms but filter for relevance
     */
    private function search_individual_terms_flexible($terms, $original_query, $limit) {
        global $wpdb;
        
        if (empty($terms)) return array();
        
        // Only use terms that are longer than 4 characters (more specific)
        $meaningful_terms = array_filter($terms, function($term) {
            return strlen($term) > 4;
        });
        
        if (empty($meaningful_terms)) {
            // Fallback to all terms if no long terms
            $meaningful_terms = array_slice($terms, 0, 2);
        }
        
        $all_results = array();
        
        // Search for each meaningful term
        foreach ($meaningful_terms as $term) {
            $sql = $wpdb->prepare(
                "SELECT c.*, 
                    CASE 
                        WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                        WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                        WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                        ELSE c.source_title
                    END as title,
                    c.post_id,
                    c.content_type
                FROM {$this->table_name} c
                WHERE (
                    (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                    OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                    OR (c.content_type NOT IN ('product', 'post', 'page'))
                )
                AND (c.source_title LIKE %s OR c.content LIKE %s)
                ORDER BY 
                    CASE WHEN c.source_title LIKE %s THEN 1 ELSE 2 END,
                    c.last_updated DESC
                LIMIT %d",
                '%' . $term . '%',
                '%' . $term . '%',
                '%' . $term . '%',
                intval($limit * 2)
            );
            
            $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            
            foreach ($results as $result) {
                $all_results[] = $this->format_search_result($result, $original_query);
            }
        }
        
        // Remove duplicates and return top results
        $unique_results = array();
        $seen_ids = array();
        
        foreach ($all_results as $result) {
            $key = $result['post_id'] . '_' . $result['content_type'];
            if (!in_array($key, $seen_ids)) {
                $seen_ids[] = $key;
                $unique_results[] = $result;
            }
        }
        
        return array_slice($unique_results, 0, $limit);
    }
    
    /**
     * Format raw database results
     */
    private function format_raw_results($raw_results) {
        if (empty($raw_results)) return array();
        
        $formatted = array();
        foreach ($raw_results as $row) {
            $formatted[] = $this->format_search_result($row, '');
        }
        
        return $formatted;
    }
    
    /**
     * Extract relevant snippet from content - improved version for multiple terms
     */
    private function extract_relevant_snippet($content, $query, $context_length = 1000) {
        if (empty($query)) {
            return substr($content, 0, $context_length) . (strlen($content) > $context_length ? '...' : '');
        }
        
        // Extract meaningful terms (longer than 3 chars, not common words)
        $terms = explode(' ', strtolower(trim(preg_replace('/[^\p{L}\p{N}\s]/u', ' ', $query))));
        $meaningful_terms = array_filter($terms, function($term) {
            return strlen($term) > 2 && !$this->is_stop_word($term);
        });
        
        if (empty($meaningful_terms)) {
            return substr($content, 0, $context_length) . '...';
        }
        
        $content_lower = strtolower($content);
        $best_position = 0;
        $best_score = 0;
        $found_terms = array();
        
        // Look for sections where multiple terms appear close together
        foreach ($meaningful_terms as $primary_term) {
            $positions = array();
            $offset = 0;
            
            // Find all positions of the primary term
            while (($pos = strpos($content_lower, $primary_term, $offset)) !== false) {
                $positions[] = $pos;
                $offset = $pos + 1;
            }
            
            // For each position of the primary term, check how many other terms are nearby
            foreach ($positions as $pos) {
                $score = 1; // Base score for finding the primary term
                $terms_found = array($primary_term);
                
                // Check for other terms within a 500-character window around this position
                $window_start = max(0, $pos - 250);
                $window_end = min(strlen($content_lower), $pos + 250);
                $window = substr($content_lower, $window_start, $window_end - $window_start);
                
                foreach ($meaningful_terms as $term) {
                    if ($term !== $primary_term && strpos($window, $term) !== false) {
                        $score += 2; // Bonus for each additional term in the vicinity
                        $terms_found[] = $term;
                    }
                }
                
                // Extra bonus for exact phrase matches like "kymppi xxl"
                if (count($meaningful_terms) >= 2 && strpos($window, implode(' ', array_slice($meaningful_terms, 0, 2))) !== false) {
                    $score += 5;
                }
                
                // If this is the best section found so far, use it
                if ($score > $best_score) {
                    $best_score = $score;
                    $best_position = max(0, $pos - 200);
                    $found_terms = $terms_found;
                }
            }
        }
        
        write_debug_log('BubbliBot Database: Enhanced snippet extraction - found terms: ' . implode(', ', $found_terms) . ' at position ' . $best_position . ' (score: ' . $best_score . ')');
        
        return $this->extract_snippet_at_position($content, $best_position, $context_length);
    }
    
    /**
     * Detect if query has pricing intent - simplified version
     */
    private function detect_pricing_intent($query) {
        $pricing_words = array(
            // English pricing words
            'cost', 'costs', 'price', 'pricing', 'expensive', 'cheap', 'much', 'money', 'fee', 'charge'
        );
        $query_lower = strtolower($query);
        
        foreach ($pricing_words as $word) {
            if (strpos($query_lower, $word) !== false) {
                return true;
            }
        }
        
        return false;
    }
    

    
    /**
     * Extract pricing snippet - simplified version
     */
    private function extract_pricing_snippet($content, $query_terms, $context_length) {
        $content_lower = strtolower($content);
        
        // Look for currency symbols and price patterns
        $pricing_patterns = array('$', '€', '£', 'cost', 'price', 'fee');
        
        foreach ($pricing_patterns as $pattern) {
            $pos = strpos($content_lower, $pattern);
            if ($pos !== false) {
                $start_pos = max(0, $pos - 300);
                return $this->extract_snippet_at_position($content, $start_pos, $context_length);
            }
        }
        
        // Fallback to regular snippet extraction
        return $this->extract_relevant_snippet($content, implode(' ', $query_terms), $context_length);
    }
    
    /**
     * Extract snippet by terms - simplified version  
     */
    private function extract_snippet_by_terms($content, $terms, $context_length) {
        return $this->extract_relevant_snippet($content, implode(' ', $terms), $context_length);
    }
    
    /**
     * Remove complex pattern generation - keep it simple
     */
    private function create_term_patterns($term) {
        // Just return the term itself and simple plural
        $patterns = array($term);
        
        // Add simple plural/singular variations
        if (substr($term, -1) === 's') {
            $patterns[] = substr($term, 0, -1); // Remove 's' for singular
        } else {
            $patterns[] = $term . 's'; // Add 's' for plural
        }
        
        return $patterns;
    }
    
    /**
     * Extract snippet by density - simplified version
     */
    private function extract_snippet_by_density($content, $terms, $context_length) {
        return $this->extract_relevant_snippet($content, implode(' ', $terms), $context_length);
    }
    
    /**
     * Extract snippet at specific position with word boundaries
     */
    private function extract_snippet_at_position($content, $start, $context_length) {
        // Adjust start to word boundary
        if ($start > 0) {
            $word_start = strrpos(substr($content, 0, $start), ' ');
            if ($word_start !== false) {
                $start = $word_start + 1;
            }
        }
        
        // Get the snippet
        $snippet = substr($content, $start, $context_length);
        
        // Adjust end to word boundary
        if (strlen($content) > ($start + $context_length)) {
            $lastSpace = strrpos($snippet, ' ');
            if ($lastSpace !== false) {
                $snippet = substr($snippet, 0, $lastSpace);
            }
        }
        
        // Add ellipsis
        if ($start > 0) $snippet = '...' . $snippet;
        if (strlen($content) > ($start + strlen($snippet))) $snippet .= '...';
        
        return $snippet;
    }
    
    /**
     * Choose the best snippet from multiple candidates
     */
    private function choose_best_snippet($snippets, $query_terms) {
        if (empty($snippets)) {
            return '';
        }
        
        $best_snippet = $snippets[0];
        $best_score = 0;
        
        foreach ($snippets as $snippet) {
            $score = 0;
            $snippet_lower = strtolower($snippet);
            
            // Score based on how many query terms appear in this snippet
            foreach ($query_terms as $term) {
                $score += substr_count($snippet_lower, $term) * strlen($term);
            }
            
            if ($score > $best_score) {
                $best_score = $score;
                $best_snippet = $snippet;
            }
        }
        
        return $best_snippet;
    }
    
    /**
     * Get total indexed content count
     */
    public function get_stats() {
        global $wpdb;
        
        // Count both 'post' and 'page' content types for posts/pages
        $post_count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE content_type IN ('post', 'page')");
        $product_count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE content_type = 'product'");
        $total_indexed = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        
        $stats = array(
            'total_posts' => $this->count_indexable_posts(),
            'total_products' => $this->count_woocommerce_products(),
            'indexed_posts' => $post_count,
            'indexed_products' => $product_count,
            'indexed_count' => $total_indexed,
            'last_full_index' => get_option('bubblibot_last_full_index', 'Never')
        );
        
        return $stats;
    }
    

    
    /**
     * Count WooCommerce products
     */
    private function count_woocommerce_products() {
        global $wpdb;
        
        if (!class_exists('WooCommerce')) {
            return 0;
        }
        
        return $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
             WHERE post_type = 'product' 
             AND post_status = 'publish'"
        );
    }
    
    /**
     * Count indexable posts/pages (excluding WooCommerce system pages and other excluded content)
     */
    private function count_indexable_posts() {
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ));
        
        $indexable_count = 0;
        
        foreach ($posts as $post) {
            // Replicate Indexer preprocessing (include title, strip tags, normalize spaces)
            $raw_content = $post->post_title . "\n\n" . $post->post_content;
            $clean_content = wp_strip_all_tags($raw_content);
            $clean_content = preg_replace('/\s+/', ' ', $clean_content);
            $clean_content = trim($clean_content);

            if (!$this->should_exclude_post($post, $clean_content)) {
                $indexable_count++;
            }
        }
        
        return $indexable_count;
    }
    
    /**
     * Check if a post should be excluded from indexing (same logic as indexer)
     */
    private function should_exclude_post($post, $content = null) {
        // Skip if post is empty or invalid
        if (!$post || empty($post->post_title)) {
            return true;
        }
        
        $title = strtolower(trim($post->post_title));
        
        // If content is provided, use it; otherwise fall back to post_content
        if ($content !== null) {
            $content_lower = strtolower(trim($content));
        } else {
            $content_lower = strtolower(trim($post->post_content));
        }
        
        $combined = $title . ' ' . $content_lower;
        
        // Font-related exclusions (WooCommerce, themes, etc.)
        $font_patterns = [
            // Font family names
            '/^(inter|bodoni moda|overpass|albert sans|lora|montserrat|arvo|rubik|newsreader|cormorant|work sans|raleway|roboto|open sans|helvetica|arial|times|georgia|verdana|calibri)$/i',
            // Font CSS/technical patterns  
            '/^[a-z\s]+;(normal|italic|bold);[0-9]+;[0-9]+%;u\+[0-9a-f\-]+$/i',
            // Font file patterns
            '/\.(woff|woff2|ttf|otf|eot)$/i',
            // CSS font declarations
            '/font-family|font-weight|font-style|@font-face/i'
        ];
        
        foreach ($font_patterns as $pattern) {
            if (preg_match($pattern, $title) || preg_match($pattern, $combined)) {
                return true;
            }
        }
        
        // WooCommerce system pages exclusions
        $woocommerce_patterns = [
            // English WooCommerce pages
            '/^(shop|cart|checkout|my account|orders|order)$/i',
            // WooCommerce shortcode content patterns
            '/\[woocommerce_my_account\]/',
            '/\[woocommerce_cart\]/',
            '/\[woocommerce_checkout\]/',
            '/\[shop_messages\]/',
            // Common WooCommerce page content patterns
            '/your cart is empty/i',
            '/you may also like/i',
            '/new in store/i'
        ];
        
        foreach ($woocommerce_patterns as $pattern) {
            if (preg_match($pattern, $title) || preg_match($pattern, $combined)) {
                return true;
            }
        }

        // Technical/system posts exclusions
        $technical_patterns = [
            // WooCommerce technical posts
            '/^(custom-css|custom-logo|site-icon|background-image)$/i',
            // Plugin configuration posts
            '/^(elementor|gutenberg|block-editor|wp-config|settings|configuration)$/i',
            // JSON/technical data
            '/^\{.*\}$/',
            '/^[a-f0-9]{32,}$/i', // Hash-like content
            // Short technical strings (but exclude common abbreviations like FAQ, API, etc.)
            '/^[a-z0-9\-_]{1,3}$/i' // Only 1-3 characters (not 4-5 like FAQ, API)
        ];
        
        foreach ($technical_patterns as $pattern) {
            if (preg_match($pattern, $title) || preg_match($pattern, $combined)) {
                return true;
            }
        }
        
        // Exclude posts with very short titles and technical content
        if (strlen($title) < 3 && strlen($content_lower) < 50) {
            return true;
        }
        
        // Exclude posts with minimal meaningful content (mostly shortcodes, placeholders, etc.)
        $meaningful_content = preg_replace('/\[.*?\]/', '', $content_lower); // Remove shortcodes
        $meaningful_content = preg_replace('/\{.*?\}/', '', $meaningful_content); // Remove placeholders like {business_idea_generator}
        $meaningful_content = preg_replace('/[^a-zA-Z0-9\s]/', ' ', $meaningful_content); // Remove special chars
        $meaningful_content = preg_replace('/\s+/', ' ', trim($meaningful_content)); // Clean whitespace
        
        // If after cleaning there's very little meaningful text, exclude it
        if (strlen($meaningful_content) < 20) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Clear all indexed content
     */
    public function clear_all_content() {
        global $wpdb;
        
        // Try TRUNCATE first (faster and resets auto-increment)
        $result = $wpdb->query("TRUNCATE TABLE {$this->table_name}");
        
        if ($result === false) {
            // If TRUNCATE fails, try DELETE (slower but more reliable)
            if (function_exists('write_debug_log')) {
                write_debug_log("BubbliBot Database: TRUNCATE failed, trying DELETE - " . $wpdb->last_error);
            }
            $result = $wpdb->query("DELETE FROM {$this->table_name}");
        }
        
        // Verify the table is actually empty
        $count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        
        if (function_exists('write_debug_log')) {
            if ($count == 0) {
                write_debug_log("BubbliBot Database: Successfully cleared all content");
            } else {
                write_debug_log("BubbliBot Database: WARNING - Table still contains {$count} rows after clear attempt");
            }
        }
        
        return $result !== false && $count == 0;
    }
    
    /**
     * Get table name
     */
    public function get_table_name() {
        return $this->table_name;
    }
    
    private function format_search_result($row, $query) {
        // Use the title from the SQL query (already handles COALESCE)
        $title = isset($row->title) ? $row->title : 'Unknown';
        
        // Generate URL based on content type
        if ($row->content_type === 'product') {
            // For WooCommerce products
            $url = get_permalink($row->post_id);
        } else {
            // For regular posts/pages
            $url = get_permalink($row->post_id);
        }
        
        // Use specialized pricing snippet when user asks about prices/fees
        if ($this->detect_pricing_intent($query)) {
            $snippet_content = $this->extract_pricing_snippet($row->content, explode(' ', $query), 1000);
        } else {
            $snippet_content = $this->extract_relevant_snippet($row->content, $query);
        }
        
        $result = array(
            'title' => $title,
            'content' => $snippet_content,
            'url' => $url ? $url : '#',
            'content_type' => $row->content_type,
            'file_path' => isset($row->file_path) ? $row->file_path : ''
        );
        
        // Add structured product data for WooCommerce products
        if ($row->content_type === 'product') {
            $result['product_data'] = array(
                'price' => isset($row->product_price) ? $row->product_price : null,
                'sale_price' => isset($row->product_sale_price) ? $row->product_sale_price : null,
                'regular_price' => isset($row->product_regular_price) ? $row->product_regular_price : null,
                'currency' => isset($row->product_currency) ? $row->product_currency : 'EUR',
                'stock_status' => isset($row->product_stock_status) ? $row->product_stock_status : null,
                'stock_quantity' => isset($row->product_stock_quantity) ? $row->product_stock_quantity : null,
                'sku' => isset($row->product_sku) ? $row->product_sku : null,
                'type' => isset($row->product_type) ? $row->product_type : null
            );
            
            // Enhance content with structured pricing info for better AI understanding
            if ($result['product_data']['price']) {
                $price_info = "\n\nPRICE INFORMATION:\n";
                $currency = $result['product_data']['currency'];
                
                if (!empty($result['product_data']['sale_price']) && $result['product_data']['sale_price'] > 0 && $result['product_data']['sale_price'] < $result['product_data']['regular_price']) {
                    $price_info .= "Current Price: {$currency} {$result['product_data']['sale_price']} (ON SALE)\n";
                    $price_info .= "Regular Price: {$currency} {$result['product_data']['regular_price']}\n";
                } else {
                    $price_info .= "Price: {$currency} {$result['product_data']['price']}\n";
                }
                
                if ($result['product_data']['stock_status']) {
                    $stock_text = ucfirst(str_replace('_', ' ', $result['product_data']['stock_status']));
                    if ($result['product_data']['stock_quantity']) {
                        $stock_text .= " ({$result['product_data']['stock_quantity']} available)";
                    }
                    $price_info .= "Stock Status: {$stock_text}\n";
                }
                
                $result['content'] .= $price_info;
            }
        }
        
        return $result;
    }
    
    /**
     * Get a single indexed item by post_id and content_type
     */
    public function get_indexed_item_by_post_id_and_type($post_id, $content_type) {
        global $wpdb;
        return $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE post_id = %d AND content_type = %s",
                $post_id,
                $content_type
            ),
            ARRAY_A // Return as an associative array
        );
    }
    
    /**
     * Get analytics table name
     */
    public function get_analytics_table_name() {
        return $this->analytics_table_name;
    }
    
    public function get_conversations_table_name() {
        return $this->conversations_table_name;
    }
    
    public function get_messages_table_name() {
        return $this->messages_table_name;
    }
    
    public function get_handoff_queue_table_name() {
        return $this->handoff_queue_table_name;
    }
    
    public function get_leads_table_name() {
        return $this->leads_table_name;
    }
    
    /**
     * Track a user query for analytics
     */
    public function track_query($question, $detected_language, $context_found, $response_generated, $response_time = null) {
        global $wpdb;
        
        // Generate session ID (simple hash of IP + User Agent + today's date)
        $remote_ip  = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        $session_data = $remote_ip . $user_agent . gmdate( 'Y-m-d' );
        $user_session = substr(md5($session_data), 0, 32);
        
        // Anonymize IP address for privacy (remove last octet)
        $user_ip = $this->anonymize_ip( $remote_ip );
        
        $result = $wpdb->insert(
            $this->analytics_table_name,
            array(
                'user_session' => $user_session,
                'question_text' => $question,
                'detected_language' => $detected_language,
                'context_found' => $context_found ? 1 : 0,
                'response_generated' => $response_generated ? 1 : 0,
                'response_time' => $response_time,
                'user_ip' => $user_ip,
                'user_agent' => substr( $user_agent, 0, 500 ),
                'created_at' => gmdate('Y-m-d')
            ),
            array('%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s', '%s')
        );
        
        if (function_exists('write_debug_log')) {
            write_debug_log('Analytics: Tracked query - ' . ($result ? 'Success' : 'Failed'));
        }
        
        return $result;
    }
    
    /**
     * Anonymize IP address for privacy compliance
     */
    private function anonymize_ip($ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            // IPv4: remove last octet
            return preg_replace('/\.\d+$/', '.0', $ip);
        } elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            // IPv6: remove last 80 bits (5 groups)
            $parts = explode(':', $ip);
            return implode(':', array_slice($parts, 0, 3)) . '::';
        }
        return 'unknown';
    }
    
    /**
     * Update user satisfaction for a query
     */
    public function update_user_satisfaction($session_id, $satisfied) {
        global $wpdb;
        
        return $wpdb->update(
            $this->analytics_table_name,
            array('user_satisfied' => $satisfied ? 1 : 0),
            array(
                'user_session' => $session_id,
                'created_at' => gmdate('Y-m-d')
            ),
            array('%d'),
            array('%s', '%s')
        );
    }
    
    /**
     * Clear old analytics data
     */
    public function clear_old_analytics($days = 90) {
        global $wpdb;
        
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->analytics_table_name} WHERE created_at < %s",
            $cutoff_date
        ));
        
        if (function_exists('write_debug_log')) {
            write_debug_log("Analytics: Cleared {$deleted} old analytics records older than {$days} days");
        }
        
        return $deleted;
    }
    
    // ========== FREE VERSION CONTENT LIMITS ==========
    
    /**
     * Get total number of indexed items
     * Free version: used for enforcing content limits
     */
    public function get_total_indexed_content_count() {
        global $wpdb;
        
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name}"
        );
    }
    
    /**
     * Check if content indexing limit has been reached
     * Free version: limit to 50 indexed items
     */
    public function has_exceeded_content_limit() {
        // Content limit removed for free version - unlimited indexing allowed
        return false;
    }
    
    /**
     * Get remaining content slots
     * Free version: show how many more items can be indexed
     */
    public function get_remaining_content_slots() {
        // Content limit removed for free version - unlimited indexing allowed
        return -1; // -1 indicates unlimited
    }
    
    /**
     * Check if new content can be indexed
     * Free version: prevent indexing when limit is reached
     */
    public function can_index_new_content($post_id, $content_type = 'post') {
        // Content limit removed for free version - unlimited indexing allowed
        return true;
    }
    
    // ========== HANDOFF FUNCTIONALITY ==========
    
    /**
     * Create or get conversation by session ID
     */
    public function get_or_create_conversation($session_id, $user_email = null, $user_name = null, $detected_language = null) {
        global $wpdb;
        
        // Make sure migration has run
        $this->migrate_conversations_language_support();
        
        // Debug: Check if table exists
        if (function_exists('write_debug_log')) {
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->conversations_table_name}'");
            write_debug_log("Database: Conversations table '{$this->conversations_table_name}' exists: " . ($table_exists ? 'YES' : 'NO'));
            
            if (!$table_exists) {
                write_debug_log("Database: Creating missing tables...");
                $this->create_tables();
                
                // Check again
                $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->conversations_table_name}'");
                write_debug_log("Database: After creation, table exists: " . ($table_exists ? 'YES' : 'NO'));
            }
        }
        
        // Try to get existing conversation
        $conversation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->conversations_table_name} WHERE session_id = %s",
            $session_id
        ));
        
        if ($conversation) {
            if (function_exists('write_debug_log')) {
                write_debug_log("Database: Found existing conversation ID: {$conversation->id}");
            }
            
            // Update language if provided and not set
            if ($detected_language && (!isset($conversation->detected_language) || empty($conversation->detected_language) || $conversation->detected_language === 'en')) {
                $this->update_conversation_language($conversation->id, $detected_language);
                $conversation->detected_language = $detected_language;
            }
            
            return $conversation;
        }
        
        // Create new conversation
        if (function_exists('write_debug_log')) {
            write_debug_log("Database: Creating new conversation for session: {$session_id}");
        }
        
        $data = array(
            'session_id' => $session_id,
            'user_email' => $user_email,
            'user_name' => $user_name,
            'status' => 'active',
            'created_at' => gmdate('Y-m-d')
        );
        
        $formats = array('%s', '%s', '%s', '%s', '%s');
        
        // Add language if detected
        if ($detected_language) {
            $data['detected_language'] = $detected_language;
            $formats[] = '%s';
        }
        
        $result = $wpdb->insert($this->conversations_table_name, $data, $formats);
        
        if ($result) {
            $conversation_id = $wpdb->insert_id;
            if (function_exists('write_debug_log')) {
                write_debug_log("Database: Successfully created conversation ID: {$conversation_id}");
            }
            
            return $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->conversations_table_name} WHERE id = %d",
                $conversation_id
            ));
        } else {
            if (function_exists('write_debug_log')) {
                write_debug_log("Database ERROR: Failed to create conversation. Error: " . $wpdb->last_error);
                write_debug_log("Database ERROR: Last query: " . $wpdb->last_query);
            }
        }
        
        return false;
    }
    
    /**
     * Update conversation language
     */
    public function update_conversation_language($conversation_id, $detected_language) {
        global $wpdb;
        
        write_debug_log("Database: Updating conversation {$conversation_id} language to: {$detected_language}");
        
        return $wpdb->update(
            $this->conversations_table_name,
            array('detected_language' => $detected_language),
            array('id' => $conversation_id),
            array('%s'),
            array('%d')
        );
    }
    
    /**
     * Save a message to the conversation
     */
    public function save_message($conversation_id, $sender_type, $message, $sender_id = null, $metadata = null) {
        global $wpdb;
        
        $data = array(
            'conversation_id' => $conversation_id,
            'sender_type' => $sender_type,
            'message' => $message,
            'created_at' => gmdate('Y-m-d')
        );
        
        $formats = array('%d', '%s', '%s', '%s');
        
        if ($sender_id) {
            $data['sender_id'] = $sender_id;
            $formats[] = '%d';
        }
        
        if ($metadata) {
            $data['metadata'] = json_encode($metadata);
            $formats[] = '%s';
        }
        
        $result = $wpdb->insert($this->messages_table_name, $data, $formats);
        
        if ($result) {
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    /**
     * Get conversation messages
     */
    public function get_conversation_messages($conversation_id, $limit = 50) {
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT m.id, m.conversation_id, m.sender_type, m.sender_id, m.message, m.metadata, m.created_at, u.display_name as sender_name 
             FROM {$this->messages_table_name} m
             LEFT JOIN {$wpdb->users} u ON m.sender_id = u.ID AND m.sender_type = 'human'
             WHERE m.conversation_id = %d
             ORDER BY m.created_at ASC
             LIMIT %d",
            $conversation_id,
            $limit
        ));
    }
    
    /**
     * Update conversation status
     */
    public function update_conversation_status($conversation_id, $status, $assigned_to = 'no_change') {
        global $wpdb;
        
        $data = array(
            'status' => $status,
            'updated_at' => gmdate('Y-m-d')
        );
        
        $formats = array('%s', '%s');
        
        if ($assigned_to !== 'no_change') {
            if ($assigned_to === null) {
                // Use raw SQL to set NULL
                $wpdb->query($wpdb->prepare(
                    "UPDATE {$this->conversations_table_name} SET assigned_to = NULL WHERE id = %d",
                    $conversation_id
                ));
            } else {
                $data['assigned_to'] = $assigned_to;
                $formats[] = '%d';
            }
        }
        
        if ($status === 'resolved') {
            $data['resolved_at'] = gmdate('Y-m-d');
            $formats[] = '%s';
        }
        
        return $wpdb->update(
            $this->conversations_table_name,
            $data,
            array('id' => $conversation_id),
            $formats,
            array('%d')
        );
    }
    
    /**
     * Add conversation to handoff queue
     * Free version: disabled - returns false and logs upgrade message
     */
    public function add_to_handoff_queue($conversation_id, $reason, $urgency = 5) {
        // Free version: handoff is a Pro feature
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            if (function_exists('write_debug_log')) {
                write_debug_log('Free Version: Handoff queue add attempted but disabled - Pro feature');
            }
            return false;
        }
        
        global $wpdb;
        
        // Check if already in queue
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->handoff_queue_table_name} WHERE conversation_id = %d AND claimed_at IS NULL",
            $conversation_id
        ));
        
        if ($existing) {
            return $existing; // Already in queue
        }
        
        $result = $wpdb->insert(
            $this->handoff_queue_table_name,
            array(
                'conversation_id' => $conversation_id,
                'reason' => $reason,
                'urgency' => $urgency,
                'created_at' => gmdate('Y-m-d')
            ),
            array('%d', '%s', '%d', '%s')
        );
        
        return $result ? $wpdb->insert_id : false;
    }
    
    /**
     * Get handoff queue
     * Free version: returns empty array - Pro feature only
     */
    public function get_handoff_queue($limit = 20) {
        // Free version: handoff is a Pro feature
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            return array();
        }
        
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT hq.id, hq.conversation_id, hq.reason, hq.urgency, hq.created_at as queue_created_at, 
                    c.session_id, c.user_email, c.user_name, c.created_at as conversation_started,
                    (SELECT m.message FROM {$this->messages_table_name} m 
                     WHERE m.conversation_id = hq.conversation_id 
                     AND m.sender_type = 'user' 
                     AND m.created_at <= hq.created_at 
                     ORDER BY m.created_at DESC LIMIT 1) as trigger_message
             FROM {$this->handoff_queue_table_name} hq
             JOIN {$this->conversations_table_name} c ON hq.conversation_id = c.id
             WHERE hq.claimed_at IS NULL
             ORDER BY hq.urgency DESC, hq.created_at DESC
             LIMIT %d",
            $limit
        ));
    }
    
    /**
     * Claim conversation from queue
     */
    public function claim_conversation($queue_id, $agent_id) {
        global $wpdb;
        
        // Update queue entry
        $queue_result = $wpdb->update(
            $this->handoff_queue_table_name,
            array(
                'claimed_at' => gmdate('Y-m-d'),
                'claimed_by' => $agent_id
            ),
            array('id' => $queue_id),
            array('%s', '%d'),
            array('%d')
        );
        
        if ($queue_result) {
            // Get conversation ID
            $conversation_id = $wpdb->get_var($wpdb->prepare(
                "SELECT conversation_id FROM {$this->handoff_queue_table_name} WHERE id = %d",
                $queue_id
            ));
            
            // Update conversation status
            $this->update_conversation_status($conversation_id, 'with_human', $agent_id);
            
            return $conversation_id;
        }
        
        return false;
    }
    
    /**
     * Get agent's active conversations
     */
    public function get_agent_conversations($agent_id) {
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT c.*, 
                    (SELECT COUNT(*) FROM {$this->messages_table_name} WHERE conversation_id = c.id) as message_count,
                    (SELECT message FROM {$this->messages_table_name} WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message
             FROM {$this->conversations_table_name} c
             WHERE c.assigned_to = %d AND c.status = 'with_human'
             ORDER BY c.updated_at DESC",
            $agent_id
        ));
    }
    
    /**
     * Get conversation by session ID
     */
    public function get_conversation_by_session($session_id) {
        global $wpdb;
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->conversations_table_name} WHERE session_id = %s",
            $session_id
        ));
    }
    
    /**
     * Get conversation by ID
     */
    public function get_conversation_by_id($conversation_id) {
        global $wpdb;
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->conversations_table_name} WHERE id = %d",
            $conversation_id
        ));
    }
    
    /**
     * Check if user has pending handoff
     */
    public function has_pending_handoff($session_id) {
        global $wpdb;
        
        $conversation = $this->get_conversation_by_session($session_id);
        if (!$conversation) {
            return false;
        }
        
        return in_array($conversation->status, ['waiting_human', 'with_human']);
    }
    
    /**
     * Get handoff statistics
     */
    public function get_handoff_stats($days = 30) {
        global $wpdb;
        
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT 
                COUNT(*) as total_handoffs,
                COUNT(CASE WHEN c.status = 'resolved' THEN 1 END) as resolved_handoffs,
                COUNT(CASE WHEN c.status = 'with_human' THEN 1 END) as active_handoffs,
                COUNT(CASE WHEN c.status = 'waiting_human' THEN 1 END) as waiting_handoffs,
                AVG(TIMESTAMPDIFF(MINUTE, hq.created_at, hq.claimed_at)) as avg_wait_time,
                AVG(TIMESTAMPDIFF(MINUTE, c.created_at, c.resolved_at)) as avg_resolution_time
             FROM {$this->handoff_queue_table_name} hq
             JOIN {$this->conversations_table_name} c ON hq.conversation_id = c.id
             WHERE hq.created_at >= %s",
            $cutoff_date
        ), ARRAY_A);
    }
    
    /**
     * Update conversation contact information
     */
    public function update_conversation_contact_info($conversation_id, $email, $name = '') {
        global $wpdb;
        
        $data = array(
            'user_email' => $email,
            'updated_at' => gmdate('Y-m-d')
        );
        
        $formats = array('%s', '%s');
        
        if (!empty($name)) {
            $data['user_name'] = $name;
            $formats[] = '%s';
        }
        
        return $wpdb->update(
            $this->conversations_table_name,
            $data,
            array('id' => $conversation_id),
            $formats,
            array('%d')
        );
    }
    
    /**
     * Remove conversation from handoff queue
     */
    public function remove_from_handoff_queue($queue_id) {
        global $wpdb;
        
        // Get conversation ID before deleting
        $conversation_id = $wpdb->get_var($wpdb->prepare(
            "SELECT conversation_id FROM {$this->handoff_queue_table_name} WHERE id = %d",
            $queue_id
        ));
        
        if ($conversation_id) {
            // Delete from queue
            $result = $wpdb->delete(
                $this->handoff_queue_table_name,
                array('id' => $queue_id),
                array('%d')
            );
            
            if ($result) {
                // Update conversation status to resolved and clear assignment
                $this->update_conversation_status($conversation_id, 'resolved', null);
                return $conversation_id;
            }
        }
        
        return false;
    }
    
    /**
     * Search requiring ALL meaningful terms to be present (boolean AND search)
     */
    private function search_all_terms_required($terms, $limit) {
        global $wpdb;
        
        if (empty($terms)) return array();
        
        // Filter out only stop words, keep all other terms regardless of length
        $meaningful_terms = array_filter($terms, function($term) {
            return !$this->is_stop_word($term);
        });
        
        if (empty($meaningful_terms)) {
            // Fallback to original terms if no meaningful terms found
            $meaningful_terms = array_slice($terms, 0, 3); // Take first 3 terms
        }
        
        write_debug_log('BubbliBot Database: ALL-terms search for: ' . implode(', ', $meaningful_terms));
        
        // Try strict ALL-terms search first
        $strict_results = $this->search_with_all_terms_strict($meaningful_terms, $limit);
        if (!empty($strict_results)) {
            return $strict_results;
        }
        
        // If strict search fails, try flexible search (most terms, not all)
        write_debug_log('BubbliBot Database: Strict ALL-terms failed, trying flexible search');
        return $this->search_with_most_terms($meaningful_terms, $limit);
    }
    
    /**
     * Strict search requiring ALL terms
     */
    private function search_with_all_terms_strict($terms, $limit) {
        global $wpdb;
        
        // Build WHERE conditions that require ALL terms to be present
        $title_conditions = array();
        $content_conditions = array();
        
        foreach ($terms as $term) {
            $like_term = '%' . $wpdb->esc_like($term) . '%';
            $title_conditions[] = $wpdb->prepare("c.source_title LIKE %s", $like_term);
            $content_conditions[] = $wpdb->prepare("c.content LIKE %s", $like_term);
        }
        
        // Require ALL terms in title OR ALL terms in content
        $all_terms_condition = '((' . implode(' AND ', $title_conditions) . ') OR (' . implode(' AND ', $content_conditions) . '))';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        $sql = "SELECT c.*, 
                CASE 
                    WHEN c.content_type = 'product' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish')
                    WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish')
                    WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                    ELSE c.source_title
                END as title,
                c.post_id,
                c.content_type
            FROM {$this->table_name} c
            WHERE (
                (c.content_type = 'product' AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'product' AND post_status = 'publish'))
                OR (c.content_type IN ('post', 'page') AND EXISTS (SELECT 1 FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type IN ('post', 'page') AND post_status = 'publish'))
                OR (c.content_type NOT IN ('product', 'post', 'page'))
            )
            AND {$all_terms_condition}
            ORDER BY c.last_updated DESC
            LIMIT " . intval($limit);
        
        $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        
        if (!empty($results)) {
            write_debug_log('BubbliBot Database: Found ' . count($results) . ' results requiring ALL terms: ' . implode(', ', $terms));
            return array_map(function($row) use ($terms) {
                return $this->format_search_result($row, implode(' ', $terms));
            }, $results);
        }
        
        return array();
    }
    
    /**
     * Flexible search requiring most terms (not all) - enhanced scoring for free version (posts/pages only)
     */
    private function search_with_most_terms($terms, $limit) {
        global $wpdb;
        
        if (count($terms) < 2) return array();
        
        // Build OR conditions and scoring parts for each term
        $condition_parts = array();
        $match_score_parts = array(); // simple presence count
        $relevance_score_parts = array(); // frequency * length weighting
        
        foreach ($terms as $term) {
            $like_term = '%' . $wpdb->esc_like($term) . '%';
            // OR condition (title OR content contains term)
            $condition_parts[] = $wpdb->prepare("(c.source_title LIKE %s OR c.content LIKE %s)", $like_term, $like_term);
            // +1 if term present → match_score
            $match_score_parts[] = $wpdb->prepare("CASE WHEN (c.source_title LIKE %s OR c.content LIKE %s) THEN 1 ELSE 0 END", $like_term, $like_term);
            // Frequency-weighted relevance (count occurrences / char_length(term))
            $relevance_score_parts[] = $wpdb->prepare("((CHAR_LENGTH(c.source_title) - CHAR_LENGTH(REPLACE(LOWER(c.source_title), LOWER(%s), ''))) / CHAR_LENGTH(%s) * 5 + (CHAR_LENGTH(c.content) - CHAR_LENGTH(REPLACE(LOWER(c.content), LOWER(%s), ''))) / CHAR_LENGTH(%s))", $term, $term, $term, $term);
        }
        
        // Require at least 60 % of terms to match
        $min_matches = max(1, floor(count($terms) * 0.6));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- complex dynamic query assembled safely
        $sql = "SELECT c.*, 
                (" . implode(' + ', $match_score_parts) . ") AS match_score,
                (" . implode(' + ', $relevance_score_parts) . ") AS relevance_score,
                CASE 
                    WHEN c.content_type = 'post' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'post' AND post_status = 'publish')
                    WHEN c.content_type = 'page' THEN (SELECT post_title FROM {$wpdb->posts} WHERE ID = c.post_id AND post_type = 'page' AND post_status = 'publish')
                    ELSE c.source_title
                END AS title,
                c.post_id,
                c.content_type
            FROM {$this->table_name} c
            WHERE (
                c.content_type IN ('post','page')
                AND EXISTS (
                    SELECT 1 FROM {$wpdb->posts} p 
                    WHERE p.ID = c.post_id 
                    AND p.post_type = c.content_type 
                    AND p.post_status = 'publish'
                )
            )
            AND (" . implode(' OR ', $condition_parts) . ")
            ORDER BY relevance_score DESC, c.last_updated DESC
            LIMIT " . intval($limit);
        
        $results = $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        
        // Keep only rows that match the minimum required term count
        $filtered = array_filter($results, function($row) use ($min_matches) {
            return isset($row->match_score) && $row->match_score >= $min_matches;
        });
        
        if (empty($filtered)) {
            write_debug_log('BubbliBot Database: No results found requiring most terms (enhanced search)');
            return array();
        }
        
        write_debug_log('BubbliBot Database: Found ' . count($filtered) . ' results with most terms (enhanced search)');
        
        return array_map(function($row) use ($terms) {
            return $this->format_search_result($row, implode(' ', $terms));
        }, $filtered);
    }
    
    /**
     * Clean up orphaned content that references deleted posts/pages/attachments
     */
    public function cleanup_orphaned_content() {
        global $wpdb;
        
        $orphaned_count = 0;
        
        // Clean up orphaned posts/pages
        $orphaned_posts = $wpdb->get_results(
            "SELECT c.id, c.post_id, c.content_type 
             FROM {$this->table_name} c 
             WHERE c.content_type IN ('post', 'page') 
             AND NOT EXISTS (
                 SELECT 1 FROM {$wpdb->posts} p 
                 WHERE p.ID = c.post_id 
                 AND p.post_type IN ('post', 'page') 
                 AND p.post_status = 'publish'
             )"
        );
        
        foreach ($orphaned_posts as $orphaned) {
            $wpdb->delete(
                $this->table_name,
                array('id' => $orphaned->id),
                array('%d')
            );
            $orphaned_count++;
            
            if (function_exists('write_debug_log')) {
                write_debug_log("BubbliBot Database: Removed orphaned {$orphaned->content_type} ID {$orphaned->post_id} from index");
            }
        }
        

        
        // Clean up orphaned WooCommerce products
        if (class_exists('WooCommerce')) {
            $orphaned_products = $wpdb->get_results(
                "SELECT c.id, c.post_id 
                 FROM {$this->table_name} c 
                 WHERE c.content_type = 'product' 
                 AND NOT EXISTS (
                     SELECT 1 FROM {$wpdb->posts} p 
                     WHERE p.ID = c.post_id 
                     AND p.post_type = 'product' 
                     AND p.post_status = 'publish'
                 )"
            );
            
            foreach ($orphaned_products as $orphaned) {
                $wpdb->delete(
                    $this->table_name,
                    array('id' => $orphaned->id),
                    array('%d')
                );
                $orphaned_count++;
                
                if (function_exists('write_debug_log')) {
                    write_debug_log("BubbliBot Database: Removed orphaned product ID {$orphaned->post_id} from index");
                }
            }
        }
        
        if (function_exists('write_debug_log')) {
            write_debug_log("BubbliBot Database: Cleanup completed - removed {$orphaned_count} orphaned entries");
        }
        
        return $orphaned_count;
    }
}