<?php
/**
 * Analytics functionality for the chatbot plugin
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
class Bubblibot_Analytics {
    
    private $database;
    
    public function __construct($database) {
        $this->database = $database;
    }
    
    /**
     * Track a user query
     */
    public function track_query($question, $context_found, $detected_language = 'en', $response_time = null, $user_satisfied = null) {
        // Check if analytics is enabled
        if (get_option('bubblibot_analytics_enabled', '1') !== '1') {
            return false;
        }
        
        // $context_found is now passed directly as a boolean
        $response_generated = true; // If we're tracking, we guenerated a response
        
        return $this->database->track_query(
            $question,
            $detected_language,
            $context_found,
            $response_generated,
            $response_time
        );
    }
    
    /**
     * Get comprehensive analytics insights (Free version: max 30 days)
     */
    public function get_insights($days = 30) {
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        return array(
            'summary' => $this->get_summary_stats($days),
            'top_questions' => $this->get_top_questions($days),
            'unanswered_questions' => $this->get_failed_queries($days),
            'popular_topics' => $this->analyze_query_topics($days),
            'peak_hours' => $this->get_usage_patterns($days),
            'language_distribution' => $this->get_language_distribution($days),
            'satisfaction_rate' => $this->get_satisfaction_rate($days),
            'is_free_version' => defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION
        );
    }
    
    /**
     * Get summary statistics (Free version: max 30 days)
     */
    public function get_summary_stats($days = 30) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        $stats = $wpdb->get_row($wpdb->prepare("
            SELECT 
                COUNT(*) as total_queries,
                COUNT(DISTINCT user_session) as unique_users,
                SUM(CASE WHEN context_found = 1 THEN 1 ELSE 0 END) as answered_queries,
                SUM(CASE WHEN context_found = 0 THEN 1 ELSE 0 END) as unanswered_queries,
                AVG(response_time) as avg_response_time,
                COUNT(CASE WHEN user_satisfied = 1 THEN 1 END) as satisfied_responses,
                COUNT(CASE WHEN user_satisfied IS NOT NULL THEN 1 END) as total_feedback
            FROM {$table_name}
            WHERE created_at >= %s
        ", $cutoff_date), ARRAY_A);
        
        if ($stats) {
            $stats['answer_rate'] = $stats['total_queries'] > 0 ? 
                round(($stats['answered_queries'] / $stats['total_queries']) * 100, 1) : 0;
            $stats['satisfaction_rate'] = $stats['total_feedback'] > 0 ? 
                round(($stats['satisfied_responses'] / $stats['total_feedback']) * 100, 1) : 0;
            $stats['avg_response_time'] = $stats['avg_response_time'] ? round($stats['avg_response_time'] / 1000, 1) : 0;
        }
        
        return $stats;
    }
    
    /**
     * Get top questions by frequency (Free version: max 30 days)
     */
    public function get_top_questions($days = 30, $limit = 10) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        return $wpdb->get_results($wpdb->prepare("
            SELECT 
                question_text,
                COUNT(*) as frequency,
                AVG(CASE WHEN context_found = 1 THEN 1 ELSE 0 END) as answer_rate,
                detected_language,
                MAX(created_at) as last_asked
            FROM {$table_name}
            WHERE created_at >= %s
            GROUP BY LOWER(TRIM(question_text))
            ORDER BY frequency DESC
            LIMIT %d
        ", $cutoff_date, $limit), ARRAY_A);
    }
    
    /**
     * Get queries that failed to find answers (Free version: max 30 days)
     */
    public function get_failed_queries($days = 30, $limit = 20) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        return $wpdb->get_results($wpdb->prepare("
            SELECT 
                question_text,
                COUNT(*) as frequency,
                detected_language,
                MAX(created_at) as last_asked
            FROM {$table_name}
            WHERE created_at >= %s
            AND context_found = 0
            GROUP BY LOWER(TRIM(question_text))
            ORDER BY frequency DESC
            LIMIT %d
        ", $cutoff_date, $limit), ARRAY_A);
    }
    
    /**
     * Analyze query topics using keyword extraction (Free version: max 30 days)
     */
    public function analyze_query_topics($days = 30, $limit = 10) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        // Get all questions
        $questions = $wpdb->get_col($wpdb->prepare("
            SELECT question_text
            FROM {$table_name}
            WHERE created_at >= %s
        ", $cutoff_date));
        
        // Simple keyword extraction
        $keywords = array();
        $stopwords = array('the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'can', 'may', 'might', 'i', 'you', 'he', 'she', 'it', 'we', 'they', 'this', 'that', 'these', 'those', 'what', 'where', 'when', 'why', 'how', 'who');
        
        foreach ($questions as $question) {
            $words = str_word_count(strtolower($question), 1);
            foreach ($words as $word) {
                if (strlen($word) > 3 && !in_array($word, $stopwords)) {
                    $keywords[$word] = ($keywords[$word] ?? 0) + 1;
                }
            }
        }
        
        arsort($keywords);
        return array_slice($keywords, 0, $limit, true);
    }
    
    /**
     * Get usage patterns by hour (Free version: max 30 days)
     */
    public function get_usage_patterns($days = 30) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        return $wpdb->get_results($wpdb->prepare("
            SELECT 
                HOUR(created_at) as hour,
                COUNT(*) as query_count,
                COUNT(DISTINCT user_session) as unique_users
            FROM {$table_name}
            WHERE created_at >= %s
            GROUP BY HOUR(created_at)
            ORDER BY hour
        ", $cutoff_date), ARRAY_A);
    }
    
    /**
     * Get language distribution (Free version: max 30 days)
     */
    public function get_language_distribution($days = 30) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        return $wpdb->get_results($wpdb->prepare("
            SELECT 
                detected_language,
                COUNT(*) as query_count,
                COUNT(DISTINCT user_session) as unique_users,
                AVG(CASE WHEN context_found = 1 THEN 1 ELSE 0 END) as answer_rate
            FROM {$table_name}
            WHERE created_at >= %s
            GROUP BY detected_language
            ORDER BY query_count DESC
        ", $cutoff_date), ARRAY_A);
    }
    
    /**
     * Get satisfaction rate (Free version: max 30 days)
     */
    public function get_satisfaction_rate($days = 30) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        return $wpdb->get_row($wpdb->prepare("
            SELECT 
                COUNT(CASE WHEN user_satisfied = 1 THEN 1 END) as satisfied,
                COUNT(CASE WHEN user_satisfied = 0 THEN 1 END) as dissatisfied,
                COUNT(CASE WHEN user_satisfied IS NOT NULL THEN 1 END) as total_feedback
            FROM {$table_name}
            WHERE created_at >= %s
        ", $cutoff_date), ARRAY_A);
    }
    
    /**
     * Get daily query trends (Free version: max 30 days)
     */
    public function get_daily_trends($days = 30) {
        global $wpdb;
        
        // Free version: enforce 30-day maximum  
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $days = min($days, 30);
        }
        
        $table_name = $this->database->get_analytics_table_name();
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
        
        return $wpdb->get_results($wpdb->prepare("
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as total_queries,
                COUNT(DISTINCT user_session) as unique_users,
                SUM(CASE WHEN context_found = 1 THEN 1 ELSE 0 END) as answered_queries
            FROM {$table_name}
            WHERE created_at >= %s
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        ", $cutoff_date), ARRAY_A);
    }
    
    /**
     * Clean up old analytics data (Free version: force 30-day retention)
     */
    public function cleanup_old_data() {
        // Free version: force 30-day retention regardless of setting
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            $retention_days = 30;
        } else {
            $retention_days = get_option('bubblibot_analytics_retention', 90);
        }
        
        return $this->database->clear_old_analytics($retention_days);
    }
} 