<?php
/**
 * Admin panel for the chatbot plugin
 */
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

require_once BUBBLIBOT_PLUGIN_PATH . 'includes/class-database.php';
require_once BUBBLIBOT_PLUGIN_PATH . 'includes/class-indexer.php';
require_once BUBBLIBOT_PLUGIN_PATH . 'includes/class-openai.php';

class Bubblibot_Admin {
    
    public function init() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_bubblibot_reindex', array($this, 'handle_reindex_ajax'));
        add_action('wp_ajax_bubblibot_check_progress', array($this, 'handle_check_progress_ajax'));
        add_action('wp_ajax_bubblibot_refresh_debug_log', array($this, 'handle_refresh_debug_log_ajax'));
        
        // Add admin notice for missing API key
        add_action('admin_notices', array($this, 'show_api_key_notice'));
        
        // Handle admin actions
        add_action('admin_action_bubblibot_migrate_db', array($this, 'handle_admin_actions'));
        
        // Handle lead capture AJAX
        add_action('wp_ajax_bubblibot_send_lead_form', array($this, 'handle_send_lead_form_ajax'));
        add_action('wp_ajax_nopriv_bubblibot_send_lead_form', array($this, 'handle_send_lead_form_ajax'));
        
        // Handoff AJAX handlers  
        add_action('wp_ajax_bubblibot_send_agent_message', array($this, 'handle_send_agent_message_ajax'));
        add_action('wp_ajax_bubblibot_get_conversation_messages', array($this, 'handle_get_conversation_messages_ajax'));
        add_action('wp_ajax_bubblibot_update_conversation_status', array($this, 'handle_update_conversation_status_ajax'));
        add_action('wp_ajax_bubblibot_check_agent_status', array($this, 'handle_check_agent_status_ajax'));
        add_action('wp_ajax_bubblibot_get_new_messages', array($this, 'handle_get_new_messages_ajax'));
        add_action('wp_ajax_bubblibot_update_lead_status', array($this, 'handle_update_lead_status_ajax'));
        add_action('wp_ajax_bubblibot_verify_api_key', array($this, 'handle_verify_api_key_ajax'));
        add_action('wp_ajax_bubblibot_clear_api_verification', array($this, 'handle_clear_api_verification_ajax'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Bubblibot Settings',
            'Bubblibot',
            'manage_options',
            'bubblibot-settings',
            array($this, 'admin_page'),
            'dashicons-format-chat',
            30
        );
        
        // Free version: Only show basic analytics
        add_submenu_page(
            'bubblibot-settings',
            'Analytics Dashboard',
            'Analytics',
            'manage_options',
            'bubblibot-analytics',
            array($this, 'analytics_page')
        );
        
        // Show upgrade page for pro features
        if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION) {
            add_submenu_page(
                'bubblibot-settings',
                'Upgrade to Pro',
                '⭐ Upgrade to Pro',
                'manage_options',
                'bubblibot-upgrade',
                array($this, 'upgrade_page')
            );
        }
        
        // Debug log (keep for troubleshooting)
        add_submenu_page(
            'bubblibot-settings',
            'Debug Log',
            'Debug Log',
            'manage_options',
            'bubblibot-debug',
            array($this, 'debug_log_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        // Register all plugin options with proper sanitization callbacks so that PluginCheck sniff passes

        // 1. Simple text options.
        $text_settings = array(
            'bubblibot_openai_api_key',
            'bubblibot_openai_model',
            'bubblibot_response_mode',
            'bubblibot_welcome_message',
            'bubblibot_placeholder_text',
            'bubblibot_header_text',
            'bubblibot_send_button_text',
            'bubblibot_position',
            'bubblibot_default_language',
            'bubblibot_bot_avatar_url',
        );
        foreach ( $text_settings as $option ) {
            register_setting( 'bubblibot_settings', $option, array(
                'sanitize_callback' => 'sanitize_text_field',
            ) );
        }

        // 2. Hex colour pickers.
        $color_settings = array(
            'bubblibot_primary_color',
            'bubblibot_secondary_color',
            'bubblibot_text_color',
            'bubblibot_chat_window_bg',
            'bubblibot_bot_message_bg',
            'bubblibot_bot_message_text',
        );
        foreach ( $color_settings as $option ) {
            register_setting( 'bubblibot_settings', $option, array(
                'sanitize_callback' => 'sanitize_hex_color',
            ) );
        }

        // 3. Integer (numeric) values.
        register_setting( 'bubblibot_settings', 'bubblibot_font_size', array(
            'sanitize_callback' => 'absint',
        ) );

        // 4. Boolean / checkbox settings.
        $bool_settings = array(
            'bubblibot_enabled',
            'bubblibot_language_detection_enabled',
            'bubblibot_analytics_enabled',
            'bubblibot_debug_enabled',
            'bubblibot_mobile_enabled',
        );
        foreach ( $bool_settings as $option ) {
            register_setting( 'bubblibot_settings', $option, array(
                'sanitize_callback' => array( $this, 'sanitize_checkbox' ),
            ) );
        }

        // 5. Array of page types.
        register_setting( 'bubblibot_settings', 'bubblibot_display_pages', array(
            'sanitize_callback' => array( $this, 'sanitize_text_array' ),
        ) );

        // 6. Multiline textarea (page exclusions).
        register_setting( 'bubblibot_settings', 'bubblibot_exclude_pages', array(
            'sanitize_callback' => 'sanitize_textarea_field',
        ) );
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if (!in_array($hook, ['toplevel_page_bubblibot-settings', 'bubblibot_page_bubblibot-analytics', 'bubblibot_page_bubblibot-debug', 'bubblibot_page_bubblibot-handoff', 'bubblibot_page_bubblibot-agent-chat'])) {
            return;
        }
        
        // Enqueue jQuery first
        wp_enqueue_script('jquery');
        
        // Enqueue admin CSS
        wp_enqueue_style(
            'bubblibot-admin',
            BUBBLIBOT_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            BUBBLIBOT_VERSION
        );
        
        // Enqueue our admin script with proper version and dependencies
        wp_enqueue_script(
            'bubblibot-admin',
            BUBBLIBOT_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            BUBBLIBOT_VERSION,
            true
        );
        
        // Add some inline CSS for better styling
        wp_add_inline_style('bubblibot-admin', '
            .bubblibot-color-preview {
                display: inline-block;
                width: 30px;
                height: 30px;
                border: 1px solid #ddd;
                border-radius: 4px;
                vertical-align: middle;
                margin-left: 10px;
            }
            .form-table th {
                width: 200px;
            }
            .form-table input[type="color"] {
                width: 60px;
                height: 30px;
                border: none;
                border-radius: 4px;
                cursor: pointer;
            }
            .bubblibot-settings-section {
                margin-top: 30px;
            }
            /* Floating Save Button */
            .bubblibot-floating-save {
                position: fixed;
                top: 60px;
                right: 20px;
                z-index: 9999;
                background: #0073aa;
                color: white;
                border: none;
                padding: 12px 24px;
                border-radius: 6px;
                font-size: 14px;
                font-weight: 600;
                cursor: pointer;
                box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
                transition: all 0.2s ease;
                min-width: 120px;
                text-align: center;
                line-height: 1.2;
            }
            .bubblibot-floating-save:hover {
                background: #005a87;
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
            }
            .bubblibot-floating-save:active {
                transform: translateY(0);
            }
            .bubblibot-floating-save:disabled {
                background: #ccc;
                cursor: not-allowed;
                transform: none;
                box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            }
            .bubblibot-floating-save:focus {
                outline: 2px solid #005a87;
                outline-offset: 2px;
            }
            /* Responsive adjustments */
            @media screen and (max-width: 782px) {
                .bubblibot-floating-save {
                    top: 70px;
                    right: 10px;
                    padding: 10px 16px;
                    font-size: 13px;
                    min-width: 100px;
                }
            }
            @media screen and (max-width: 600px) {
                .bubblibot-floating-save {
                    position: relative;
                    top: auto;
                    right: auto;
                    display: block;
                    margin: 10px auto 20px;
                    width: 200px;
                }
            }
            /* Hide original submit button */
            .bubblibot-settings-form .submit {
                display: none;
            }
            /* Add some top padding to account for floating button */
            .wrap {
                padding-top: 20px;
            }
            /* Business Hours Table Styling */
            .bubblibot-business-hours-table th {
                padding-left: 15px !important;
                background-color: #f9f9f9;
                font-weight: 600;
            }
            .bubblibot-business-hours-table input[type="time"] {
                width: 120px !important;
                font-family: monospace;
                font-size: 14px;
            }
            /* Analytics Dashboard Styling */
            .bubblibot-analytics-summary .postbox-header h3 {
                margin-left: 15px !important;
                padding-left: 0 !important;
            }
            .postbox-header h3 {
                margin-left: 15px;
            }
            /* Handoff Dashboard Styling */
            .bubblibot-handoff .postbox-header h3 {
                margin-left: 15px !important;
                padding-left: 15px !important;
            }
            /* General postbox header styling for all admin pages */
            .postbox .postbox-header h3 {
                padding-left: 15px !important;
            }
            /* Lead status styling */
            .lead-status-new { color: #f56e28; font-weight: bold; }
            .lead-status-contacted { color: #0073aa; }
            .lead-status-converted { color: #00a32a; font-weight: bold; }
            .lead-status-lost { color: #dc3232; }
            /* Lead intent styling */
            .lead-intent-quote { color: #dc3232; font-weight: bold; }
            .lead-intent-demo { color: #f56e28; font-weight: bold; }
            .lead-intent-purchase { color: #00a32a; font-weight: bold; }
            .lead-intent-consultation { color: #0073aa; }
            .lead-intent-contact { color: #666; }
            /* Free version styling */
            .bubblibot-free-notice {
                background: #f0f6fc;
                border: 1px solid #0969da;
                border-radius: 6px;
                padding: 12px;
                margin: 10px 0;
            }
            /* API key styling */
            .bubblibot-api-status {
                display: inline-block;
                font-weight: 500;
                font-size: 13px;
                padding: 4px 8px;
                border-radius: 4px;
                background: #f0f0f0;
            }
            .bubblibot-api-status.valid {
                color: #00a32a;
                background: #e6f7e6;
            }
            .bubblibot-api-status.invalid {
                color: #dc3232;
                background: #fce6e6;
            }
            .bubblibot-api-status.empty {
                color: #ff6900;
                background: #fff3e0;
            }
            .bubblibot-api-status.unverified {
                color: #0073aa;
                background: #e6f3ff;
            }
            .bubblibot-api-status.verifying {
                color: #666;
                background: #f0f0f0;
            }
        ');
        
        // Add inline JS and CSS specifically for the Debug Log page
        if ( 'bubblibot_page_bubblibot-debug' === $hook ) {
            $debug_nonce = wp_create_nonce( 'bubblibot_debug_nonce' );
            $inline_js = "jQuery(document).ready(function($){\n                function refreshLog(){\n                    $.ajax({\n                        url: ajaxurl,\n                        type: 'POST',\n                        data: { action: 'bubblibot_refresh_debug_log', nonce: '{$debug_nonce}' },\n                        success: function(response){\n                            if(response.success){\n                                $('#log-content pre').html(response.data);\n                                $('#log-content pre').scrollTop($('#log-content pre')[0].scrollHeight);\n                            }\n                        }\n                    });\n                }\n                $('#refresh-log').on('click', function(){ refreshLog(); });\n                $('#copy-log').on('click', function(){\n                    var logContent = $('#log-content pre').text();\n                    var \$temp = $('<textarea>');\n                    \$temp.val(logContent).css({position:'absolute',left:'-9999px'});\n                    $('body').append(\$temp);\n                    \$temp.select();\n                    try{ document.execCommand('copy'); $('#copy-success').show().delay(2000).fadeOut(); }catch(err){ alert('Copy failed.'); }\n                    \$temp.remove();\n                });\n                var refreshInterval;\n                $('#auto-refresh').on('change', function(){\n                    if($(this).is(':checked')){\n                        refreshInterval = setInterval(refreshLog, 5000);\n                    }else{\n                        clearInterval(refreshInterval);\n                    }\n                });\n            });";
            wp_add_inline_script( 'bubblibot-admin', $inline_js );
            wp_add_inline_style( 'bubblibot-admin', '#copy-success { position: fixed; top: 50px; right: 20px; z-index: 9999; }' );
        }
        
        // Localize the script with new data
        wp_localize_script(
            'bubblibot-admin',
            'bubblibot_admin',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('bubblibot_admin_nonce'),
                'i18n' => array(
                    'starting' => __('Starting indexing...', 'bubblibot'),
                    'processing' => __('Processing...', 'bubblibot'),
                    'error' => __('Error:', 'bubblibot'),
                    'success' => __('Success:', 'bubblibot'),
                    'saving' => __('Saving...', 'bubblibot'),
                    'save_changes' => __('Save Changes', 'bubblibot'),
                    'saved' => __('Saved!', 'bubblibot')
                )
            )
        );
    }
    
    /**
     * Admin page HTML
     */
    public function admin_page() {
        try {
            // Handle form submission
            if (isset($_POST['submit']) && check_admin_referer('bubblibot_settings_nonce')) {
                $this->save_settings();
                echo '<div class="notice notice-success"><p>' . esc_html__('Settings saved successfully!', 'bubblibot') . '</p></div>';
            }
            
            // Get current settings
            $api_key = get_option('bubblibot_openai_api_key', '');
            $response_mode = get_option('bubblibot_response_mode', 'hybrid_smart');
            $enabled = get_option('bubblibot_enabled', '1');
            $welcome_message = get_option('bubblibot_welcome_message', 'Hello! How can I help you today?');
            $placeholder_text = get_option('bubblibot_placeholder_text', 'Type your message...');
            
            // Get appearance settings
            $header_text = get_option('bubblibot_header_text', 'Chat Support');

            $primary_color = get_option('bubblibot_primary_color', '#0073aa');
            $secondary_color = get_option('bubblibot_secondary_color', '#006291');
            $text_color = get_option('bubblibot_text_color', '#ffffff');
            $chat_window_bg = get_option('bubblibot_chat_window_bg', '#ffffff');
            $bot_message_bg = get_option('bubblibot_bot_message_bg', '#f0f0f0');
            $bot_message_text = get_option('bubblibot_bot_message_text', '#333333');
            $font_size = get_option('bubblibot_font_size', '14');
            
            // Get indexing stats
            try {
                $database = new Bubblibot_Database();
                $indexer = new Bubblibot_Indexer($database);
                $stats = $indexer->get_stats();
            } catch (Exception $e) {
                if (function_exists('write_debug_log')) {
                    write_debug_log('Bubblibot Admin ERROR: Failed to load indexing stats - ' . $e->getMessage());
                }
                $stats = array(
                    'total_posts' => 0,
                    'indexed_count' => 0,
                    'last_full_index' => 'Never'
                );
            }
            
            ?>
            <div class="wrap">
                <h1><?php echo esc_html__('Bubblibot Settings', 'bubblibot'); ?></h1>
                
                <!-- Floating Save Button -->
                <button type="submit" form="bubblibot-settings-form" class="bubblibot-floating-save" id="floating-save-btn" name="submit" value="Save Changes">
                    <?php esc_html_e('Save Changes', 'bubblibot'); ?>
                </button>
                
                <form method="post" action="" id="bubblibot-settings-form" class="bubblibot-settings-form">
                    <?php wp_nonce_field('bubblibot_settings_nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable Chatbot', 'bubblibot'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="bubblibot_enabled" value="1" <?php checked($enabled, '1'); ?>>
                                    <?php esc_html_e('Enable the chatbot on your website', 'bubblibot'); ?>
                                </label>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('OpenAI API Key', 'bubblibot'); ?></th>
                            <td>
                                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 8px;">
                                    <input type="password" id="bubblibot_openai_api_key" name="bubblibot_openai_api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text" style="width: 400px; max-width: 100%;">
                                    <button type="button" id="verify-api-key" class="button button-secondary" style="white-space: nowrap;">
                                        <?php esc_html_e('Verify Key', 'bubblibot'); ?>
                                    </button>
                                </div>
                                <div id="api-key-status" style="margin-bottom: 8px;">
                                    <?php $this->display_api_key_status(); ?>
                                </div>
                                <?php /* translators: %s: HTML link to the OpenAI Platform. */ ?>
                                <p class="description"><?php printf( wp_kses_post( __( 'Get your API key from %s. Required for chatbot functionality.', 'bubblibot' ) ), '<a href="https://platform.openai.com/api-keys" target="_blank">' . esc_html__( 'OpenAI Platform', 'bubblibot' ) . '</a>' ); ?></p>
                            </td>
                        </tr>
                        
                        <?php if (defined('BUBBLIBOT_FREE_VERSION') && BUBBLIBOT_FREE_VERSION): ?>
                        <tr>
                            <th scope="row">⭐ <?php esc_html_e('Upgrade to Pro', 'bubblibot'); ?></th>
                            <td>
                                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; border-left: 4px solid #0073aa;">
                                    <p style="margin: 0 0 10px 0; font-weight: 500;"><?php esc_html_e('You are using the free version with these limitations:', 'bubblibot'); ?></p>
                                    <ul style="margin: 5px 0 10px 20px;">
                                        <li><?php esc_html_e('Standard answer accuracy (no embeddings)', 'bubblibot'); ?></li>
                                        <li><?php esc_html_e('No chat conversation memory', 'bubblibot'); ?></li>
                                        <li><?php esc_html_e('Basic customization options', 'bubblibot'); ?></li>
                                        <li><?php esc_html_e('No lead capture or human handoff', 'bubblibot'); ?></li>
                                        <li><?php esc_html_e('Limited analytics (30-day retention)', 'bubblibot'); ?></li>
                                    </ul>
                                    <a href="<?php echo esc_url( admin_url('admin.php?page=bubblibot-upgrade') ); ?>" class="button button-primary"><?php esc_html_e('View Pro Features', 'bubblibot'); ?></a>
                                </div>
                            </td>
                        </tr>
                        <?php endif; ?>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('OpenAI Model', 'bubblibot'); ?></th>
                            <td>
                                <?php $openai_model = get_option('bubblibot_openai_model', 'gpt-4o-mini-2024-07-18'); ?>
                                <select name="bubblibot_openai_model">
                                    <option value="gpt-4o-mini-2024-07-18" <?php selected($openai_model, 'gpt-4o-mini-2024-07-18'); ?>>gpt-4o-mini ($0.15/1M input)</option>
                                    <option value="gpt-4.1-mini-2025-04-14" <?php selected($openai_model, 'gpt-4.1-mini-2025-04-14'); ?>>gpt-4.1-mini ($0.30/1M input)</option>
                                    <option value="gpt-5-nano-2025-08-07" <?php selected($openai_model, 'gpt-5-nano-2025-08-07'); ?>>gpt-5-nano ($0.05/1M input)</option>
                                </select>
                                <p class="description"><?php esc_html_e('Choose the OpenAI model to use for generating responses. GPT-4o-mini is recommended in most cases', 'bubblibot'); ?></p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('Response Mode', 'bubblibot'); ?></th>
                            <td>
                                <select name="bubblibot_response_mode">
                                    <option value="company_only" <?php selected($response_mode, 'company_only'); ?>><?php esc_html_e('Company Data Only', 'bubblibot'); ?></option>
                                    <option value="hybrid_smart" <?php selected($response_mode, 'hybrid_smart'); ?>><?php esc_html_e('Smart Hybrid', 'bubblibot'); ?></option>
                                    <option value="hybrid_always" <?php selected($response_mode, 'hybrid_always'); ?>><?php esc_html_e('Always Hybrid', 'bubblibot'); ?></option>
                                </select>
                                <p class="description">
                                    <strong><?php esc_html_e('Company Data Only:', 'bubblibot'); ?></strong> <?php esc_html_e('Only use content from your website', 'bubblibot'); ?><br>
                                    <strong><?php esc_html_e('Smart Hybrid:', 'bubblibot'); ?></strong> <?php esc_html_e('Use company data when available, fall back to general AI knowledge', 'bubblibot'); ?><br>
                                    <strong><?php esc_html_e('Always Hybrid:', 'bubblibot'); ?></strong> <?php esc_html_e('Combine company and general knowledge when relevant', 'bubblibot'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('Welcome Message', 'bubblibot'); ?></th>
                            <td>
                                <input type="text" name="bubblibot_welcome_message" value="<?php echo esc_attr($welcome_message); ?>" class="regular-text">
                                <p class="description"><?php esc_html_e('The first message users see when opening the chat', 'bubblibot'); ?></p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('Placeholder Text', 'bubblibot'); ?></th>
                            <td>
                                <input type="text" name="bubblibot_placeholder_text" value="<?php echo esc_attr($placeholder_text); ?>" class="regular-text">
                                <p class="description"><?php esc_html_e('Placeholder text in the message input field', 'bubblibot'); ?></p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('Header Text', 'bubblibot'); ?></th>
                            <td>
                                <input type="text" name="bubblibot_header_text" value="<?php echo esc_attr($header_text); ?>" class="regular-text">
                                <p class="description"><?php esc_html_e('Text displayed in the chat window header', 'bubblibot'); ?></p>
                            </td>
                        </tr>
                        

                    </table>
                    
                    <div class="bubblibot-settings-section">
                        <h2><?php esc_html_e('Appearance Settings', 'bubblibot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Primary Color', 'bubblibot'); ?></th>
                                <td>
                                    <input type="color" name="bubblibot_primary_color" value="<?php echo esc_attr($primary_color); ?>">
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Header background, chat toggle button, user message bubbles, and send button background', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Secondary Color', 'bubblibot'); ?></th>
                                <td>
                                    <input type="color" name="bubblibot_secondary_color" value="<?php echo esc_attr($secondary_color); ?>">
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Hover effects on buttons (darker shade of primary color)', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Text Color', 'bubblibot'); ?></th>
                                <td>
                                    <input type="color" name="bubblibot_text_color" value="<?php echo esc_attr($text_color); ?>">
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Header text, button text, and user message text color', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Chat Window Background', 'bubblibot'); ?></th>
                                <td>
                                    <input type="color" name="bubblibot_chat_window_bg" value="<?php echo esc_attr($chat_window_bg); ?>">
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Background color for the entire chat window', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Bot Message Background', 'bubblibot'); ?></th>
                                <td>
                                    <input type="color" name="bubblibot_bot_message_bg" value="<?php echo esc_attr($bot_message_bg); ?>">
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Background color for bot response messages', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Bot Message Text', 'bubblibot'); ?></th>
                                <td>
                                    <input type="color" name="bubblibot_bot_message_text" value="<?php echo esc_attr($bot_message_text); ?>">
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Text color for bot response messages', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Font Size', 'bubblibot'); ?></th>
                                <td>
                                    <input type="number" name="bubblibot_font_size" value="<?php echo esc_attr($font_size); ?>" min="10" max="24" step="1" class="small-text"> <?php esc_html_e('px', 'bubblibot'); ?>
                                    <p class="description"><strong><?php esc_html_e('Controls:', 'bubblibot'); ?></strong> <?php esc_html_e('Font size for chat messages (10-24 pixels)', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <div class="bubblibot-settings-section">
                        <h2><?php esc_html_e('Display & Position Settings', 'bubblibot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Show chatbot on pages', 'bubblibot'); ?></th>
                                <td>
                                    <?php 
                                    $display_pages = get_option('bubblibot_display_pages', array('all'));
                                    if (!is_array($display_pages)) {
                                        $display_pages = array('all');
                                    }
                                    ?>
                                    <fieldset>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="all" <?php checked(in_array('all', $display_pages)); ?>>
                                            <?php esc_html_e('All pages (default)', 'bubblibot'); ?>
                                        </label><br>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="home" <?php checked(in_array('home', $display_pages)); ?>>
                                            <?php esc_html_e('Front page', 'bubblibot'); ?>
                                        </label><br>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="posts" <?php checked(in_array('posts', $display_pages)); ?>>
                                            <?php esc_html_e('Articles', 'bubblibot'); ?>
                                        </label><br>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="pages" <?php checked(in_array('pages', $display_pages)); ?>>
                                            <?php esc_html_e('Pages', 'bubblibot'); ?>
                                        </label><br>
                                        <?php if (class_exists('WooCommerce')): ?>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="shop" <?php checked(in_array('shop', $display_pages)); ?>>
                                            <?php esc_html_e('WooCommerce shop', 'bubblibot'); ?>
                                        </label><br>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="products" <?php checked(in_array('products', $display_pages)); ?>>
                                            <?php esc_html_e('Product pages', 'bubblibot'); ?>
                                        </label><br>
                                        <?php endif; ?>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="archive" <?php checked(in_array('archive', $display_pages)); ?>>
                                            <?php esc_html_e('Archive pages (categories, tags)', 'bubblibot'); ?>
                                        </label><br>
                                        <label>
                                            <input type="checkbox" name="bubblibot_display_pages[]" value="search" <?php checked(in_array('search', $display_pages)); ?>>
                                            <?php esc_html_e('Search results', 'bubblibot'); ?>
                                        </label>
                                    </fieldset>
                                    <p class="description"><?php esc_html_e('Select which page types should display the chatbot. If "All pages" is selected, other options are ignored.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Chatbot position', 'bubblibot'); ?></th>
                                <td>
                                    <?php $position = get_option('bubblibot_position', 'bottom-right'); ?>
                                    <select name="bubblibot_position">
                                        <option value="bottom-right" <?php selected($position, 'bottom-right'); ?>><?php esc_html_e('Bottom right (default)', 'bubblibot'); ?></option>
                                        <option value="bottom-left" <?php selected($position, 'bottom-left'); ?>><?php esc_html_e('Bottom left', 'bubblibot'); ?></option>
                                        <option value="bottom-center" <?php selected($position, 'bottom-center'); ?>><?php esc_html_e('Bottom center', 'bubblibot'); ?></option>
                                        <option value="right-center" <?php selected($position, 'right-center'); ?>><?php esc_html_e('Right center', 'bubblibot'); ?></option>
                                        <option value="left-center" <?php selected($position, 'left-center'); ?>><?php esc_html_e('Left center', 'bubblibot'); ?></option>
                                    </select>
                                    <p class="description"><?php esc_html_e('Choose where the chatbot button appears on the page.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Hide on specific pages', 'bubblibot'); ?></th>
                                <td>
                                    <?php $exclude_pages = get_option('bubblibot_exclude_pages', ''); ?>
                                    <textarea name="bubblibot_exclude_pages" rows="4" cols="50" placeholder="<?php esc_attr_e('Example:', 'bubblibot'); ?>&#10;/contact/&#10;/thank-you/&#10;/checkout/"><?php echo esc_textarea($exclude_pages); ?></textarea>
                                    <p class="description">
                                        <?php esc_html_e('Enter page paths (one per line) where the chatbot should be hidden. Example:', 'bubblibot'); ?> <code>/contact/</code>, <code>/checkout/</code><br>
                                        <?php esc_html_e('You can also use page IDs in format:', 'bubblibot'); ?> <code>page:123</code> <?php esc_html_e('or', 'bubblibot'); ?> <code>post:456</code>
                                    </p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Mobile devices', 'bubblibot'); ?></th>
                                <td>
                                    <?php $mobile_enabled = get_option('bubblibot_mobile_enabled', '1'); ?>
                                    <label>
                                        <input type="checkbox" name="bubblibot_mobile_enabled" value="1" <?php checked($mobile_enabled, '1'); ?>>
                                        <?php esc_html_e('Show chatbot on mobile devices', 'bubblibot'); ?>
                                    </label>
                                    <p class="description"><?php esc_html_e('Uncheck to hide the chatbot on mobile devices.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <div class="bubblibot-settings-section">
                        <h2><?php esc_html_e('Language Settings', 'bubblibot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Language Detection', 'bubblibot'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="bubblibot_language_detection_enabled" value="1" <?php checked(get_option('bubblibot_language_detection_enabled', '1'), '1'); ?>>
                                        <?php esc_html_e('Enable automatic language detection from user messages', 'bubblibot'); ?>
                                    </label>
                                    <p class="description"><?php esc_html_e('When enabled, the chatbot will detect the language of user messages and respond in the same language.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Default Language', 'bubblibot'); ?></th>
                                <td>
                                    <?php 
                                    $default_language = get_option('bubblibot_default_language', 'en');
                                    $supported_languages = array(
                                        'en' => 'English',
                                        'es' => 'Spanish',
                                        'fr' => 'French',
                                        'de' => 'German',
                                        'fi' => 'Finnish',
                                        'it' => 'Italian',
                                        'pt' => 'Portuguese',
                                        'nl' => 'Dutch',
                                        'sv' => 'Swedish',
                                        'no' => 'Norwegian',
                                        'da' => 'Danish',
                                        'pl' => 'Polish',
                                        'ru' => 'Russian',
                                        'ja' => 'Japanese',
                                        'ko' => 'Korean',
                                        'zh' => 'Chinese'
                                    );
                                    ?>
                                    <select name="bubblibot_default_language">
                                        <?php foreach ($supported_languages as $code => $name): ?>
                                            <option value="<?php echo esc_attr($code); ?>" <?php selected($default_language, $code); ?>>
                                                <?php echo esc_html($name); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="description"><?php esc_html_e('Fallback language when automatic detection fails or is disabled.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Language Status', 'bubblibot'); ?></th>
                                <td>
                                    <?php 
                                    $wp_locale = get_locale();
                                    $wp_lang_code = substr($wp_locale, 0, 2);
                                    ?>
                                    <p><strong><?php esc_html_e('WordPress Locale:', 'bubblibot'); ?></strong> <?php echo esc_html($wp_locale); ?></p>
                                    <p><strong><?php esc_html_e('Detected Language Code:', 'bubblibot'); ?></strong> <?php echo esc_html($wp_lang_code); ?></p>
                                    <p><strong><?php esc_html_e('Supported by Plugin:', 'bubblibot'); ?></strong> 
                                        <?php if (isset($supported_languages[$wp_lang_code])): ?>
                                            <?php /* translators: %s: Detected language name. */ ?>
                                            <span style="color: #46b450;">✓ <?php printf( esc_html__( 'Yes (%s)', 'bubblibot' ), esc_html( $supported_languages[ $wp_lang_code ] ) ); ?></span>
                                        <?php else: ?>
                                            <span style="color: #dc3232;">✗ <?php esc_html_e( 'No (will use default language)', 'bubblibot' ); ?></span>
                                        <?php endif; ?>
                                    </p>
                                    <p class="description">
                                        <?php esc_html_e('The plugin automatically detects your WordPress language and uses appropriate UI strings.', 'bubblibot'); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    

                    
                    <!-- Free Version Limitations Notice -->
                    <div class="bubblibot-settings-section" style="background: #f8f9fa; border: 1px solid #e2e4e7; border-left: 4px solid #0073aa; padding: 20px; margin: 20px 0;">
                        <h2 style="margin-top: 0; color: #0073aa;">🚀 <?php esc_html_e('Upgrade to Bubblibot Pro', 'bubblibot'); ?></h2>
                        <p><?php esc_html_e('Unlock advanced features to take your chatbot to the next level:', 'bubblibot'); ?></p>
                        
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; margin: 15px 0;">
                            <div style="background: white; padding: 15px; border-radius: 5px; border: 1px solid #ddd;">
                                <h4 style="margin-top: 0; color: #0073aa;">🧠 Advanced AI Technology</h4>
                                <ul style="margin: 0; padding-left: 15px;">
                                    <li>OpenAI Embeddings support</li>
                                    <li>Semantic search capabilities</li>
                                    <li>Ultra-accurate context understanding</li>
                                    <li>Superior answer relevance</li>
                                </ul>
                            </div>
                            
                            <div style="background: white; padding: 15px; border-radius: 5px; border: 1px solid #ddd;">
                                <h4 style="margin-top: 0; color: #0073aa;">💭 Chat Memory & Context</h4>
                                <ul style="margin: 0; padding-left: 15px;">
                                    <li>Conversation memory retention</li>
                                    <li>Context-aware responses</li>
                                    <li>Follow-up question handling</li>
                                    <li>Configurable memory length</li>
                                    <li>Smart conversation continuity</li>
                                </ul>
                            </div>
                            
                            <div style="background: white; padding: 15px; border-radius: 5px; border: 1px solid #ddd;">
                                <h4 style="margin-top: 0; color: #0073aa;">🤝 AI-to-Human Handoff</h4>
                                <ul style="margin: 0; padding-left: 15px;">
                                    <li>Seamless agent transfers</li>
                                    <li>Queue management</li>
                                    <li>Business hours support</li>
                                    <li>Agent avatars & profiles</li>
                                </ul>
                            </div>
                            
                            <div style="background: white; padding: 15px; border-radius: 5px; border: 1px solid #ddd;">
                                <h4 style="margin-top: 0; color: #0073aa;">🎯 Smart Lead Capture</h4>
                                <ul style="margin: 0; padding-left: 15px;">
                                    <li>Automatic intent detection</li>
                                    <li>Contact form generation</li>
                                    <li>Lead management dashboard</li>
                                    <li>Email notifications</li>
                                </ul>
                            </div>
                            
                            <div style="background: white; padding: 15px; border-radius: 5px; border: 1px solid #ddd;">
                                <h4 style="margin-top: 0; color: #0073aa;">🛍️ WooCommerce Integration</h4>
                                <ul style="margin: 0; padding-left: 15px;">
                                    <li>Product search & recommendations</li>
                                    <li>Inventory information</li>
                                    <li>Price comparisons</li>
                                    <li>Shopping assistance</li>
                                </ul>
                            </div>
                            
                            <div style="background: white; padding: 15px; border-radius: 5px; border: 1px solid #ddd;">
                                <h4 style="margin-top: 0; color: #0073aa;">📊 Advanced Analytics</h4>
                                <ul style="margin: 0; padding-left: 15px;">
                                    <li>Extended data retention</li>
                                    <li>Conversation insights</li>
                                    <li>Performance metrics</li>
                                    <li>Custom reports</li>
                                </ul>
                            </div>
                            

                        </div>
                        
                        <div style="text-align: center; margin-top: 20px;">
                            <a href="https://www.plugnify.com/bubblibot" target="_blank" class="button button-primary" style="padding: 10px 30px; font-size: 16px;">
                                <?php esc_html_e('Upgrade to Pro →', 'bubblibot'); ?>
                            </a>
                        </div>
                    </div>
                    
                    <div class="bubblibot-settings-section">
                        <h2><?php esc_html_e('Analytics Settings', 'bubblibot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Enable Analytics', 'bubblibot'); ?></th>
                                <td>
                                    <?php $analytics_enabled = get_option('bubblibot_analytics_enabled', '1'); ?>
                                    <label>
                                        <input type="checkbox" name="bubblibot_analytics_enabled" value="1" <?php checked($analytics_enabled, '1'); ?>>
                                        <?php esc_html_e('Track user queries and generate analytics insights', 'bubblibot'); ?>
                                    </label>
                                    <p class="description"><?php esc_html_e('Analytics data helps you understand what users are asking and improve your content. All data is anonymized for privacy.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Data Retention', 'bubblibot'); ?></th>
                                <td>
                                    <select name="bubblibot_analytics_retention" disabled>
                                        <option value="30" selected><?php esc_html_e('30 days (Free Version)', 'bubblibot'); ?></option>
                                    </select>
                                    <p class="description">
                                        <?php esc_html_e('Free version includes 30 days of analytics data retention.', 'bubblibot'); ?>
                                        <a href="https://www.plugnify.com/bubblibot" target="_blank"><?php esc_html_e('Upgrade to Pro for extended retention (90 days, 180 days, 1 year)', 'bubblibot'); ?></a>
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <div class="bubblibot-settings-section">
                        <h2><?php esc_html_e('Debug Settings', 'bubblibot'); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Enable Debug Logging', 'bubblibot'); ?></th>
                                <td>
                                    <?php $debug_enabled = get_option('bubblibot_debug_enabled', '0'); ?>
                                    <label>
                                        <input type="checkbox" name="bubblibot_debug_enabled" value="1" <?php checked($debug_enabled, '1'); ?>>
                                        <?php esc_html_e('Enable detailed debug logging for troubleshooting', 'bubblibot'); ?>
                                    </label>
                                    <p class="description"><?php esc_html_e('When enabled, the plugin will log detailed information about its operations. This can help with troubleshooting but may impact performance.', 'bubblibot'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <?php submit_button(); ?>
                </form>
                
                <hr>
                
                <h2><?php esc_html_e('Content Indexing', 'bubblibot'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Indexing Status', 'bubblibot'); ?></th>
                        <td>
                            <p><strong><?php esc_html_e('Posts/Pages:', 'bubblibot'); ?></strong> <?php echo esc_html( $stats['indexed_posts'] ); ?> <?php esc_html_e('of', 'bubblibot'); ?> <?php echo esc_html( $stats['total_posts'] ); ?> <?php esc_html_e('indexed', 'bubblibot'); ?></p>
    
                            <?php if (class_exists('WooCommerce')): ?>
                                <p><strong><?php esc_html_e('WooCommerce Products:', 'bubblibot'); ?></strong> <?php echo esc_html( $stats['indexed_products'] ); ?> <?php esc_html_e('of', 'bubblibot'); ?> <?php echo esc_html( $stats['total_products'] ); ?> <?php esc_html_e('indexed', 'bubblibot'); ?></p>
                            <?php endif; ?>
                            <p><strong><?php esc_html_e('Total Indexed Content:', 'bubblibot'); ?></strong> <?php echo esc_html( $stats['indexed_count'] ); ?></p>
                            <p><strong><?php esc_html_e('Last Full Index:', 'bubblibot'); ?></strong> <?php echo esc_html( $stats['last_full_index'] ); ?></p>
                            
                            <p>
                                <button type="button" id="reindex-content" class="button button-secondary">
                                    <?php esc_html_e('Re-index All Content', 'bubblibot'); ?>
                                </button>
                                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?action=bubblibot_migrate_db'), 'bubblibot_migrate_db')); ?>" class="button button-secondary" style="margin-left: 10px;">
                                    <?php esc_html_e('Fix Database Issues', 'bubblibot'); ?>
                                </a>
                                <button type="button" onclick="location.reload();" class="button button-secondary" style="margin-left: 10px;">
                                    <?php esc_html_e('Refresh Stats', 'bubblibot'); ?>
                                </button>
                                <span id="reindex-status" style="margin-left: 10px;"></span>
                            </p>
                            
                            <p class="description">
                                <?php esc_html_e('Content is automatically indexed when you publish or update posts/pages or save WooCommerce products.', 'bubblibot'); ?>
                                <?php esc_html_e('Use the re-index button if you want to rebuild the entire index.', 'bubblibot'); ?>
                                <?php esc_html_e('Use the fix database issues button to clean up orphaned content from deleted posts/pages/files.', 'bubblibot'); ?>
                            </p>
                        </td>
                    </tr>
                </table>
            </div>
            <?php
        } catch (Exception $e) {
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin ERROR: Failed to load admin page - ' . $e->getMessage());
            }
            /* translators: %s: Detailed error message returned when loading the admin page. */
            echo '<div class="notice notice-error"><p>' . sprintf( esc_html__( 'Error loading admin page: %s', 'bubblibot' ), esc_html( $e->getMessage() ) ) . '</p></div>';
        }
    }
    
    /**
     * Analytics dashboard page
     */
    public function analytics_page() {
        try {
            // Initialize analytics
            $database = new Bubblibot_Database();
            $analytics = new Bubblibot_Analytics($database);
            
            // Get time period from request (default 30 days)
            // Sanitize the days query arg (no nonce needed, read-only)
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading a query var only.
            $days = isset( $_GET['days'] ) ? absint( wp_unslash( $_GET['days'] ) ) : 30;
            $days = max(1, min(365, $days)); // Limit between 1 and 365 days
            
            // Get analytics data
            $insights = $analytics->get_insights($days);
            
            ?>
            <div class="wrap">
                <h1><?php esc_html_e('Analytics Dashboard', 'bubblibot'); ?></h1>
                
                <!-- Free Version Limitations Notice -->
                <div style="background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px; padding: 15px; margin: 20px 0;">
                    <h3 style="margin-top: 0; color: #856404;">📊 Free Version Analytics</h3>
                    <p style="margin: 10px 0;">You're viewing the free version analytics with <strong>30-day data retention</strong>.</p>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 10px; margin: 10px 0;">
                        <div>✅ Basic metrics & insights</div>
                        <div>✅ User behavior tracking</div>
                        <div>✅ Popular questions analysis</div>
                        <div>❌ Extended retention (90d+)</div>
                        <div>❌ Advanced conversation insights</div>
                        <div>❌ Custom report exports</div>
                    </div>
                    <div style="text-align: center; margin-top: 15px;">
                        <a href="https://www.plugnify.com/bubblibot" target="_blank" class="button button-primary">
                            <?php esc_html_e('Upgrade to Pro for Advanced Analytics', 'bubblibot'); ?>
                        </a>
                    </div>
                </div>
                
                <div style="margin: 20px 0;">
                    <strong><?php esc_html_e('Time Period:', 'bubblibot'); ?></strong>
                    <a href="?page=bubblibot-analytics&days=7" class="<?php echo esc_attr( $days === 7 ? 'button-primary' : 'button' ); ?>">
                        <?php esc_html_e('7 days', 'bubblibot'); ?>
                    </a>
                    <a href="?page=bubblibot-analytics&days=30" class="<?php echo esc_attr( $days === 30 ? 'button-primary' : 'button' ); ?>">
                        <?php esc_html_e('30 days', 'bubblibot'); ?>
                    </a>
                    <span class="button" style="opacity: 0.5; cursor: not-allowed;" title="<?php esc_html_e('Upgrade to Pro for extended analytics', 'bubblibot'); ?>">
                        <?php esc_html_e('90 days (Pro)', 'bubblibot'); ?>
                    </span>
                </div>
                
                <!-- Summary Stats -->
                <div class="bubblibot-analytics-summary" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 20px 0;">
                    <?php if (!empty($insights['summary'])): ?>
                        <div class="postbox">
                            <div class="postbox-header"><h3><?php esc_html_e('Total Queries', 'bubblibot'); ?></h3></div>
                            <div class="inside" style="text-align: center; font-size: 24px; font-weight: bold; color: #0073aa;">
                                <?php echo esc_html( number_format_i18n( $insights['summary']['total_queries'] ) ); ?>
                            </div>
                        </div>
                        
                        <div class="postbox">
                            <div class="postbox-header"><h3><?php esc_html_e('Unique Users', 'bubblibot'); ?></h3></div>
                            <div class="inside" style="text-align: center; font-size: 24px; font-weight: bold; color: #46b450;">
                                <?php echo esc_html( number_format_i18n( $insights['summary']['unique_users'] ) ); ?>
                            </div>
                        </div>
                        
                        <div class="postbox">
                            <div class="postbox-header"><h3><?php esc_html_e('Answer Rate', 'bubblibot'); ?></h3></div>
                            <div class="inside" style="text-align: center; font-size: 24px; font-weight: bold; color: <?php echo esc_attr( ( $insights['summary']['answer_rate'] >= 70 ) ? '#46b450' : ( ( $insights['summary']['answer_rate'] >= 50 ) ? '#f56e28' : '#dc3232' ) ); ?>;">
                                <?php echo esc_html( $insights['summary']['answer_rate'] ); ?>%
                            </div>
                        </div>
                        
                        <div class="postbox">
                            <div class="postbox-header"><h3><?php esc_html_e('Avg Response Time', 'bubblibot'); ?></h3></div>
                            <div class="inside" style="text-align: center; font-size: 24px; font-weight: bold; color: #666;">
                                <?php echo esc_html( $insights['summary']['avg_response_time'] ); ?>s
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 20px 0;">
                    <!-- Top Questions -->
                    <div class="postbox">
                        <div class="postbox-header">
                            <h3><?php esc_html_e('Most Frequent Questions', 'bubblibot'); ?></h3>
                        </div>
                        <div class="inside">
                            <?php if (!empty($insights['top_questions'])): ?>
                                <table class="wp-list-table widefat fixed striped">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('Question', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Frequency', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Answer Rate', 'bubblibot'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach (array_slice($insights['top_questions'], 0, 10) as $question): ?>
                                            <tr>
                                                <td style="max-width: 300px; word-wrap: break-word;">
                                                    <?php echo esc_html(mb_substr($question['question_text'], 0, 100)); ?>
                                                    <?php if (strlen($question['question_text']) > 100) echo '...'; ?>
                                                </td>
                                                <td><?php echo esc_html( $question['frequency'] ); ?></td>
                                                <td>
                                                    <span style="color: <?php echo esc_attr( $question['answer_rate'] >= 0.7 ? '#46b450' : ( $question['answer_rate'] >= 0.5 ? '#f56e28' : '#dc3232' ) ); ?>;">
                                                        <?php echo esc_html( round( $question['answer_rate'] * 100 ) ); ?>%
                                                    </span>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            <?php else: ?>
                                <p><?php esc_html_e('No questions recorded yet.', 'bubblibot'); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Failed Queries -->
                    <div class="postbox">
                        <div class="postbox-header">
                            <h3><?php esc_html_e('Questions Without Answers', 'bubblibot'); ?></h3>
                        </div>
                        <div class="inside">
                            <?php if (!empty($insights['unanswered_questions'])): ?>
                                <table class="wp-list-table widefat fixed striped">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('Question', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Frequency', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Language', 'bubblibot'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach (array_slice($insights['unanswered_questions'], 0, 10) as $question): ?>
                                            <tr>
                                                <td style="max-width: 300px; word-wrap: break-word;">
                                                    <?php echo esc_html(mb_substr($question['question_text'], 0, 100)); ?>
                                                    <?php if (strlen($question['question_text']) > 100) echo '...'; ?>
                                                </td>
                                                <td><?php echo esc_html( $question['frequency'] ); ?></td>
                                                <td><?php echo esc_html( $question['detected_language'] ); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                                <p class="description">
                                    <?php esc_html_e('These questions indicate content gaps. Consider adding relevant content to improve answer rates.', 'bubblibot'); ?>
                                </p>
                            <?php else: ?>
                                <p style="color: #46b450;"><?php esc_html_e('Great! All questions are being answered successfully.', 'bubblibot'); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 20px 0;">
                    <!-- Popular Topics -->
                    <div class="postbox">
                        <div class="postbox-header">
                            <h3><?php esc_html_e('Popular Topics', 'bubblibot'); ?></h3>
                        </div>
                        <div class="inside">
                            <?php if (!empty($insights['popular_topics'])): ?>
                                <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                                    <?php foreach ($insights['popular_topics'] as $topic => $count): ?>
                                        <span style="background: #f0f0f0; padding: 5px 10px; border-radius: 15px; font-size: 14px;">
                                            <?php echo esc_html( $topic ); ?> (<?php echo esc_html( $count ); ?>)
                                        </span>
                                    <?php endforeach; ?>
                                </div>
                            <?php else: ?>
                                <p><?php esc_html_e('No topic data available yet.', 'bubblibot'); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Language Distribution -->
                    <div class="postbox">
                        <div class="postbox-header">
                            <h3><?php esc_html_e('Language Distribution', 'bubblibot'); ?></h3>
                        </div>
                        <div class="inside">
                            <?php if (!empty($insights['language_distribution'])): ?>
                                <table class="wp-list-table widefat fixed striped">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('Language', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Queries', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Users', 'bubblibot'); ?></th>
                                            <th><?php esc_html_e('Answer Rate', 'bubblibot'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($insights['language_distribution'] as $lang): ?>
                                            <tr>
                                                <td><?php echo esc_html(strtoupper($lang['detected_language'])); ?></td>
                                                <td><?php echo esc_html( $lang['query_count'] ); ?></td>
                                                <td><?php echo esc_html( $lang['unique_users'] ); ?></td>
                                                <td>
                                                    <span style="color: <?php echo esc_attr( $lang['answer_rate'] >= 0.7 ? '#46b450' : ( $lang['answer_rate'] >= 0.5 ? '#f56e28' : '#dc3232' ) ); ?>;">
                                                        <?php echo esc_html( round( $lang['answer_rate'] * 100 ) ); ?>%
                                                    </span>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            <?php else: ?>
                                <p><?php esc_html_e('No language data available yet.', 'bubblibot'); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Usage Patterns -->
                <?php if (!empty($insights['peak_hours'])): ?>
                    <div class="postbox">
                        <div class="postbox-header">
                            <h3><?php esc_html_e('Usage Patterns by Hour', 'bubblibot'); ?></h3>
                        </div>
                        <div class="inside">
                            <div style="display: grid; grid-template-columns: repeat(12, 1fr); gap: 5px; margin: 20px 0;">
                                <?php 
                                $max_queries = max(array_column($insights['peak_hours'], 'query_count')) ?: 1;
                                for ($hour = 0; $hour < 24; $hour++): 
                                    $hour_data = array_filter($insights['peak_hours'], function($h) use ($hour) { 
                                        return $h['hour'] == $hour; 
                                    });
                                    $hour_data = reset($hour_data);
                                    $queries = $hour_data ? $hour_data['query_count'] : 0;
                                    $height = ($queries / $max_queries) * 100;
                                ?>
                                    <div style="text-align: center;">
                                        <div style="height: 100px; display: flex; align-items: end;">
                                            <div style="background: #0073aa; width: 100%; height: <?php echo esc_attr( $height ); ?>%; margin: 0 2px;" title="<?php echo esc_attr( $hour ); ?>:00 - <?php echo esc_attr( $queries ); ?> queries"></div>
                                        </div>
                                        <small><?php echo esc_html( $hour ); ?>h</small>
                                    </div>
                                <?php endfor; ?>
                            </div>
                            <p class="description"><?php echo esc_html__('Hover over bars to see exact numbers. Peak hours help you understand when users are most active.', 'bubblibot'); ?></p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
            <?php
            
        } catch (Exception $e) {
            /* translators: %s: Detailed error message returned when loading the analytics dashboard. */
            echo '<div class="notice notice-error"><p>' . sprintf( esc_html__( 'Error loading analytics: %s', 'bubblibot' ), esc_html( $e->getMessage() ) ) . '</p></div>';
        }
    }
    
    /**
     * Save settings
     */
    private function save_settings() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Verified earlier in admin_page()
        $post = wp_unslash( $_POST );

        update_option( 'bubblibot_openai_api_key', isset( $post['bubblibot_openai_api_key'] ) ? sanitize_text_field( $post['bubblibot_openai_api_key'] ) : '' );
        update_option( 'bubblibot_openai_model', isset( $post['bubblibot_openai_model'] ) ? sanitize_text_field( $post['bubblibot_openai_model'] ) : '' );
        update_option( 'bubblibot_response_mode', isset( $post['bubblibot_response_mode'] ) ? sanitize_text_field( $post['bubblibot_response_mode'] ) : '' );
        update_option( 'bubblibot_enabled', isset( $post['bubblibot_enabled'] ) ? '1' : '0' );
        update_option( 'bubblibot_welcome_message', isset( $post['bubblibot_welcome_message'] ) ? sanitize_text_field( $post['bubblibot_welcome_message'] ) : '' );
        update_option( 'bubblibot_placeholder_text', isset( $post['bubblibot_placeholder_text'] ) ? sanitize_text_field( $post['bubblibot_placeholder_text'] ) : '' );
        
        // Free version - no license key needed
        
        // Save appearance settings
        update_option( 'bubblibot_header_text', isset( $post['bubblibot_header_text'] ) ? sanitize_text_field( $post['bubblibot_header_text'] ) : '' );
        update_option( 'bubblibot_primary_color', isset( $post['bubblibot_primary_color'] ) ? sanitize_hex_color( $post['bubblibot_primary_color'] ) : '' );
        update_option( 'bubblibot_secondary_color', isset( $post['bubblibot_secondary_color'] ) ? sanitize_hex_color( $post['bubblibot_secondary_color'] ) : '' );
        update_option( 'bubblibot_text_color', isset( $post['bubblibot_text_color'] ) ? sanitize_hex_color( $post['bubblibot_text_color'] ) : '' );
        update_option( 'bubblibot_chat_window_bg', isset( $post['bubblibot_chat_window_bg'] ) ? sanitize_hex_color( $post['bubblibot_chat_window_bg'] ) : '' );
        update_option( 'bubblibot_bot_message_bg', isset( $post['bubblibot_bot_message_bg'] ) ? sanitize_hex_color( $post['bubblibot_bot_message_bg'] ) : '' );
        update_option( 'bubblibot_bot_message_text', isset( $post['bubblibot_bot_message_text'] ) ? sanitize_hex_color( $post['bubblibot_bot_message_text'] ) : '' );
        update_option( 'bubblibot_font_size', isset( $post['bubblibot_font_size'] ) ? absint( $post['bubblibot_font_size'] ) : 14 );
        
        // Save language settings
        update_option( 'bubblibot_language_detection_enabled', isset( $post['bubblibot_language_detection_enabled'] ) ? '1' : '0' );
        update_option( 'bubblibot_default_language', isset( $post['bubblibot_default_language'] ) ? sanitize_text_field( $post['bubblibot_default_language'] ) : 'en' );
        
        // Save Analytics settings (free version - limited retention)
        update_option( 'bubblibot_analytics_enabled', isset( $post['bubblibot_analytics_enabled'] ) ? '1' : '0' );
        
        // Save Debug settings
        update_option( 'bubblibot_debug_enabled', isset( $post['bubblibot_debug_enabled'] ) ? '1' : '0' );
        
        // Save Display settings
        update_option( 'bubblibot_display_pages', isset( $post['bubblibot_display_pages'] ) ? array_map( 'sanitize_text_field', (array) $post['bubblibot_display_pages'] ) : array( 'all' ) );
        update_option( 'bubblibot_position', isset( $post['bubblibot_position'] ) ? sanitize_text_field( $post['bubblibot_position'] ) : 'bottom-right' );
        update_option( 'bubblibot_exclude_pages', isset( $post['bubblibot_exclude_pages'] ) ? sanitize_textarea_field( $post['bubblibot_exclude_pages'] ) : '' );
        update_option( 'bubblibot_mobile_enabled', isset( $post['bubblibot_mobile_enabled'] ) ? '1' : '0' );
    }
    
    /**
     * Handle re-index AJAX request
     */
    public function handle_reindex_ajax() {
        try {
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin: Starting reindex process');
            }
            
            check_ajax_referer('bubblibot_admin_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                throw new Exception('Unauthorized access');
            }
            
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin: Authorization successful');
            }
            
            $database = new Bubblibot_Database();
            $indexer = new Bubblibot_Indexer($database);
            
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin: Database and Indexer initialized');
            }
            
            // Clear existing content
            $database->clear_all_content();
            
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin: Cleared existing content');
            }
            
            // Re-index all content
            $count = $indexer->index_all_content();
            
            if (function_exists('write_debug_log')) {
                write_debug_log("Bubblibot Admin: Successfully reindexed {$count} items");
            }
            
            wp_send_json_success(array(
                'message' => "Successfully indexed {$count} posts, pages, and products.",
                'count' => $count
            ));
        } catch (Exception $e) {
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin ERROR: Reindex failed - ' . $e->getMessage());
                write_debug_log('Bubblibot Admin ERROR: Stack trace - ' . $e->getTraceAsString());
            }
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Handle check progress AJAX request
     */
    public function handle_check_progress_ajax() {
        try {
            check_ajax_referer('bubblibot_admin_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                throw new Exception('Unauthorized access');
            }
            
            $progress = get_option('bubblibot_indexing_progress', array(
                'total' => 0,
                'current' => 0,
                'status' => 'error',
                'last_error' => 'No indexing progress found',
                'start_time' => current_time('mysql')
            ));
            
            wp_send_json_success($progress);
            
        } catch (Exception $e) {
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin ERROR: Progress check failed - ' . $e->getMessage());
            }
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Handle admin actions
     */
    public function handle_admin_actions() {
        // Handle database migration action
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Action read-only.
        $action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : '';
        if ( $action === 'bubblibot_migrate_db' ) {
            // Check if we're in admin area and user has permissions
            if (is_admin() && current_user_can('manage_options')) {
                $this->handle_database_migration();
            }
        }
    }
    
    /**
     * Handle database migration via admin interface
     */
    private function handle_database_migration() {
        try {
            // Verify nonce with proper sanitization
            if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'bubblibot_migrate_db' ) ) {
                wp_die('Security check failed');
            }
            
            $database = new Bubblibot_Database();
            
            // Clean up orphaned content
            $orphaned_count = $database->cleanup_orphaned_content();
            
            // Also run any other database migrations/fixes
            $database->migrate_conversations_language_support();
            
            // Add admin notice
            $message = sprintf(
                /* translators: %d: Number of orphaned entries removed from the index. */
                __('Database cleanup completed successfully. Removed %d orphaned entries from the index.', 'bubblibot'), 
                $orphaned_count
            );
            
            add_settings_error(
                'bubblibot_messages',
                'database_cleanup',
                $message,
                'success'
            );
            
            // Log the action
            if (function_exists('write_debug_log')) {
                write_debug_log("Bubblibot Admin: Database cleanup completed - removed {$orphaned_count} orphaned entries");
            }
            
            // Redirect back to admin page with success message
            wp_redirect(admin_url('admin.php?page=bubblibot-settings&settings-updated=true'));
            exit;
            
        } catch (Exception $e) {
            if (function_exists('write_debug_log')) {
                write_debug_log('Bubblibot Admin ERROR: Database migration failed - ' . $e->getMessage());
            }
            
            add_settings_error(
                'bubblibot_messages',
                'database_migration_error',
                /* translators: %s: Error message returned when database cleanup fails. */
                sprintf(__('Database cleanup failed: %s', 'bubblibot'), $e->getMessage()),
                'error'
            );
            
            // Redirect back to admin page with error message
            wp_redirect(admin_url('admin.php?page=bubblibot-settings&settings-updated=false'));
            exit;
        }
    }
    
    /**
     * Debug log page
     */
    public function debug_log_page() {
        $debug_enabled = get_option('bubblibot_debug_enabled', '0');
        $log_file = $this->get_debug_log_file();
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Debug Log Viewer', 'bubblibot'); ?></h1>
            
            <?php if ($debug_enabled !== '1'): ?>
                <div class="notice notice-warning">
                    <p>
                        <strong><?php esc_html_e('Debug logging is currently disabled.', 'bubblibot'); ?></strong>
                        <a href="<?php echo esc_url( admin_url('admin.php?page=bubblibot-settings') ); ?>"><?php esc_html_e('Enable it in settings', 'bubblibot'); ?></a>
                        <a href="<?php echo esc_url( admin_url('admin.php?page=bubblibot-settings') ); ?>"><?php esc_html_e('Enable it in settings', 'bubblibot'); ?></a>
                        <?php esc_html_e('to start logging debug information.', 'bubblibot'); ?>
                    </p>
                </div>
            <?php endif; ?>
            
            <?php
            // Handle clear log action
            if (isset($_POST['clear_log']) && check_admin_referer('bubblibot_clear_debug_log')) {
                if (file_exists($log_file)) {
                    file_put_contents($log_file, '');
                    echo '<div class="notice notice-success"><p>' . esc_html__('Log cleared successfully!', 'bubblibot') . '</p></div>';
                }
            }
            ?>
            
            <div style="margin: 20px 0;">
                <button id="refresh-log" class="button button-secondary">
                    <?php esc_html_e('Refresh Log', 'bubblibot'); ?>
                </button>
                
                <button id="copy-log" class="button button-primary" style="margin-left: 10px;">
                    <span class="dashicons dashicons-clipboard" style="vertical-align: text-top;"></span>
                    <?php esc_html_e('Copy to Clipboard', 'bubblibot'); ?>
                </button>
                
                <form method="post" style="display: inline; margin-left: 10px;">
                    <?php wp_nonce_field('bubblibot_clear_debug_log'); ?>
                    <input type="submit" name="clear_log" class="button button-secondary" value="<?php esc_attr_e('Clear Log', 'bubblibot'); ?>">
                </form>
                
                <div style="float: right;">
                    <label>
                        <input type="checkbox" id="auto-refresh"> 
                        <?php esc_html_e('Auto-refresh (5 seconds)', 'bubblibot'); ?>
                    </label>
                </div>
            </div>
            
            <div id="copy-success" class="notice notice-success" style="display: none;">
                <p><?php esc_html_e('Log content copied to clipboard!', 'bubblibot'); ?></p>
            </div>
            
            <div id="log-content" style="background: #fff; padding: 20px; border: 1px solid #ccc; margin-top: 20px;">
                <pre style="white-space: pre-wrap; word-wrap: break-word; max-height: 500px; overflow-y: auto;"><?php
                    if (file_exists($log_file)) {
                        echo esc_html(file_get_contents($log_file));
                    } else {
                        esc_html_e('No log file found. Debug logging may be disabled or no events have been logged yet.', 'bubblibot');
                    }
                ?></pre>
            </div>
        </div>
        

        <?php
    }
    
    /**
     * Handle refresh debug log AJAX request
     */
    public function handle_refresh_debug_log_ajax() {
        check_ajax_referer('bubblibot_debug_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized access', 'bubblibot'));
        }
        
        $log_file = $this->get_debug_log_file();
        
        if (file_exists($log_file)) {
            wp_send_json_success(esc_html(file_get_contents($log_file)));
        } else {
            wp_send_json_success(__('No log file found. Debug logging may be disabled or no events have been logged yet.', 'bubblibot'));
        }
    }
    
    /**
     * Get debug log file path
     */
    private function get_debug_log_file() {
        $upload_dir = wp_upload_dir();
        return $upload_dir['basedir'] . '/bubblibot-debug.log';
    }
    
    /**
     * Upgrade page for free version users
     */
    public function upgrade_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Upgrade to Bubblibot Pro', 'bubblibot'); ?></h1>
            
            <div class="bubblibot-upgrade-container" style="max-width: 800px;">
                <div class="postbox" style="margin-top: 20px;">
                    <div class="postbox-header" style="background: linear-gradient(135deg, #0073aa 0%, #005a87 100%); border-bottom: none;">
                        <h2 class="hndle" style="color: white; font-size: 24px; font-weight: 600; padding: 20px; margin: 0; display: flex; align-items: center; gap: 12px;">
                            <span style="font-size: 28px; filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));">⭐</span>
                            <?php esc_html_e('Unlock Advanced Features', 'bubblibot'); ?>
                        </h2>
                    </div>
                    <div class="inside" style="padding: 20px;">
                        <p style="font-size: 16px; margin-bottom: 30px;">
                            <?php esc_html_e('You are currently using the free version of Bubblibot. Upgrade to Pro to unlock powerful features for your business:', 'bubblibot'); ?>
                        </p>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin: 30px 0;">
                            <div>
                                <h3 style="color: #0073aa; margin-bottom: 15px;">🧠 <?php esc_html_e('Advanced AI Technology', 'bubblibot'); ?></h3>
                                <ul style="margin-left: 20px;">
                                    <li><?php esc_html_e('OpenAI Embeddings support', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Semantic search capabilities', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Ultra-accurate context understanding', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Superior answer relevance', 'bubblibot'); ?></li>
                                </ul>
                            </div>
                            
                            <div>
                                <h3 style="color: #0073aa; margin-bottom: 15px;">💭 <?php esc_html_e('Chat Memory & Context', 'bubblibot'); ?></h3>
                                <ul style="margin-left: 20px;">
                                    <li><?php esc_html_e('Conversation memory retention', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Context-aware responses', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Follow-up question handling', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Configurable memory length', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Smart conversation continuity', 'bubblibot'); ?></li>
                                </ul>
                            </div>
                            
                            <div>
                                <h3 style="color: #0073aa; margin-bottom: 15px;">🚀 <?php esc_html_e('Human Handoff System', 'bubblibot'); ?></h3>
                                <ul style="margin-left: 20px;">
                                    <li><?php esc_html_e('Live agent chat integration', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Intelligent escalation triggers', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Agent dashboard & queue management', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Business hours support', 'bubblibot'); ?></li>
                                </ul>
                            </div>
                            
                            <div>
                                <h3 style="color: #0073aa; margin-bottom: 15px;">📈 <?php esc_html_e('Smart Lead Capture', 'bubblibot'); ?></h3>
                                <ul style="margin-left: 20px;">
                                    <li><?php esc_html_e('Automatic intent detection', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Lead scoring & qualification', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('CRM integration ready', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Email notifications', 'bubblibot'); ?></li>
                                </ul>
                            </div>
                            
                            <div>
                                <h3 style="color: #0073aa; margin-bottom: 15px;">🛒 <?php esc_html_e('WooCommerce Integration', 'bubblibot'); ?></h3>
                                <ul style="margin-left: 20px;">
                                    <li><?php esc_html_e('Product catalog integration', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Inventory & pricing answers', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Order status inquiries', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Shopping assistance', 'bubblibot'); ?></li>
                                </ul>
                            </div>
                            
                            <div>
                                <h3 style="color: #0073aa; margin-bottom: 15px;">📊 <?php esc_html_e('Advanced Analytics', 'bubblibot'); ?></h3>
                                <ul style="margin-left: 20px;">
                                    <li><?php esc_html_e('Unlimited conversation history', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Advanced reporting & insights', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Performance metrics', 'bubblibot'); ?></li>
                                    <li><?php esc_html_e('Custom retention periods', 'bubblibot'); ?></li>
                                </ul>
                            </div>
                            

                        </div>
                        
                        <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 30px 0;">
                            <h3 style="margin-top: 0;">📝 <?php esc_html_e('Free Version Limitations', 'bubblibot'); ?></h3>
                            <ul style="margin-left: 20px;">
                                <li><?php esc_html_e('Standard answer accuracy (no embeddings support)', 'bubblibot'); ?></li>
                                <li><?php esc_html_e('No chat conversation memory', 'bubblibot'); ?></li>
                                <li><?php esc_html_e('Basic customization options', 'bubblibot'); ?></li>
                                <li><?php esc_html_e('No lead capture or human handoff', 'bubblibot'); ?></li>
                                <li><?php esc_html_e('Limited analytics (30-day retention)', 'bubblibot'); ?></li>
                            </ul>
                        </div>
                        
                        <div style="text-align: center; margin: 40px 0;">
                            <a href="https://www.plugnify.com/bubblibot" target="_blank" class="button button-primary button-large" style="background: #0073aa; border-color: #0073aa; padding: 15px 30px; font-size: 16px; text-decoration: none;">
                                <?php esc_html_e('View Pro Plans & Pricing →', 'bubblibot'); ?>
                            </a>
                            <p style="margin-top: 15px; color: #666;">
                                <?php esc_html_e('Priority support included', 'bubblibot'); ?>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Pro feature upgrade page (replaces handoff dashboard in free version)
     */
    public function handoff_dashboard_page() {
        ?>
        <div class="wrap">
            <h1>🤝 <?php esc_html_e('AI-to-Human Handoff', 'bubblibot'); ?></h1>
            
            <div style="background: #f8f9fa; border: 1px solid #e2e4e7; border-left: 4px solid #0073aa; padding: 30px; margin: 20px 0; text-align: center;">
                <h2 style="color: #0073aa; margin-top: 0;">🚀 <?php esc_html_e('Upgrade Required', 'bubblibot'); ?></h2>
                <p style="font-size: 16px; margin: 20px 0;"><?php esc_html_e('AI-to-Human Handoff is a Pro feature that allows seamless transfer of conversations from AI to human agents.', 'bubblibot'); ?></p>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 30px 0; text-align: left;">
                    <div style="background: white; padding: 20px; border-radius: 8px; border: 1px solid #ddd;">
                        <h3 style="color: #0073aa; margin-top: 0;">🎯 Smart Queue Management</h3>
                        <ul style="margin: 0; padding-left: 20px;">
                            <li>Automatic conversation prioritization</li>
                            <li>Wait time estimation</li>
                            <li>Agent availability detection</li>
                            <li>Queue overflow protection</li>
                        </ul>
                    </div>
                    
                    <div style="background: white; padding: 20px; border-radius: 8px; border: 1px solid #ddd;">
                        <h3 style="color: #0073aa; margin-top: 0;">💬 Live Agent Chat</h3>
                        <ul style="margin: 0; padding-left: 20px;">
                            <li>Real-time messaging interface</li>
                            <li>Conversation history access</li>
                            <li>Agent performance tracking</li>
                            <li>Custom agent avatars</li>
                        </ul>
                    </div>
                    
                    <div style="background: white; padding: 20px; border-radius: 8px; border: 1px solid #ddd;">
                        <h3 style="color: #0073aa; margin-top: 0;">⏰ Business Hours Support</h3>
                        <ul style="margin: 0; padding-left: 20px;">
                            <li>Configurable operating hours</li>
                            <li>Holiday management</li>
                            <li>Timezone-aware scheduling</li>
                            <li>Off-hours messaging</li>
                        </ul>
                    </div>
                    
                    <div style="background: white; padding: 20px; border-radius: 8px; border: 1px solid #ddd;">
                        <h3 style="color: #0073aa; margin-top: 0;">📊 Lead Management</h3>
                        <ul style="margin: 0; padding-left: 20px;">
                            <li>Automatic lead capture</li>
                            <li>Contact form generation</li>
                            <li>Lead scoring & prioritization</li>
                            <li>CRM integration ready</li>
                        </ul>
                    </div>
                </div>
                
                <div style="margin: 30px 0;">
                    <a href="https://www.plugnify.com/bubblibot" target="_blank" class="button button-primary" style="padding: 15px 40px; font-size: 18px; font-weight: 600;">
                        <?php esc_html_e('Upgrade to Pro →', 'bubblibot'); ?>
                    </a>
                </div>
                
                <p style="color: #666; font-style: italic;"><?php esc_html_e('Join thousands of businesses using Bubblibot Pro for superior customer support.', 'bubblibot'); ?></p>
            </div>
        </div>
        <?php
    }

    /**
     * Agent chat upgrade page (replaces agent chat in free version)
     */
    public function agent_chat_page() {
        ?>
        <div class="wrap">
            <h1>💬 <?php esc_html_e('Agent Chat', 'bubblibot'); ?></h1>
            
            <div style="background: #f8f9fa; border: 1px solid #e2e4e7; border-left: 4px solid #0073aa; padding: 30px; margin: 20px 0; text-align: center;">
                <h2 style="color: #0073aa; margin-top: 0;">🚀 <?php esc_html_e('Upgrade Required', 'bubblibot'); ?></h2>
                <p style="font-size: 16px; margin: 20px 0;"><?php esc_html_e('Real-time Agent Chat is a Pro feature that enables live conversations between your team and customers.', 'bubblibot'); ?></p>
                
                <div style="background: white; padding: 20px; border-radius: 8px; border: 1px solid #ddd; margin: 20px auto; max-width: 600px; text-align: left;">
                    <h3 style="color: #0073aa; margin-top: 0;">✨ What you get with Pro:</h3>
                    <ul style="margin: 10px 0; padding-left: 20px;">
                        <li>Real-time messaging interface</li>
                        <li>Conversation history and context</li>
                        <li>Agent availability management</li>
                        <li>Customer handoff from AI</li>
                        <li>Multi-agent support</li>
                        <li>Typing indicators</li>
                        <li>File sharing capabilities</li>
                        <li>Performance analytics</li>
                    </ul>
                </div>
                
                <div style="margin: 30px 0;">
                    <a href="https://www.plugnify.com/bubblibot" target="_blank" class="button button-primary" style="padding: 15px 40px; font-size: 18px; font-weight: 600;">
                        <?php esc_html_e('Upgrade to Pro →', 'bubblibot'); ?>
                    </a>
                </div>
                
                <p style="color: #666; font-style: italic;"><?php esc_html_e('Transform your customer support with professional agent tools.', 'bubblibot'); ?></p>
            </div>
        </div>
        <?php
    }

    /**
     * Remove all pro AJAX handlers - these return errors directing users to upgrade
     */
    public function handle_update_lead_status_ajax() {
        wp_send_json_error(array(
            'message' => __('Lead management is a Pro feature. Upgrade to access advanced lead handling capabilities.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }

    public function handle_send_agent_message_ajax() {
        wp_send_json_error(array(
            'message' => __('Agent messaging is a Pro feature.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }

    public function handle_get_conversation_messages_ajax() {
        wp_send_json_error(array(
            'message' => __('Conversation management is a Pro feature.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }

    public function handle_update_conversation_status_ajax() {
        wp_send_json_error(array(
            'message' => __('Conversation status management is a Pro feature.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }

    public function handle_check_agent_status_ajax() {
        wp_send_json_error(array(
            'message' => __('Agent status checking is a Pro feature.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }

    public function handle_get_new_messages_ajax() {
        wp_send_json_error(array(
            'message' => __('Real-time messaging is a Pro feature.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }

    public function handle_send_lead_form_ajax() {
        wp_send_json_error(array(
            'message' => __('Lead capture forms are a Pro feature.', 'bubblibot'),
            'upgrade_url' => 'https://www.plugnify.com/bubblibot'
        ));
    }
    
    /**
     * Display API key status indicator
     */
    private function display_api_key_status() {
        $api_key = get_option('bubblibot_openai_api_key', '');
        
        if (empty($api_key)) {
            echo '<span class="bubblibot-api-status empty">⚠️ ' . esc_html__( 'No API key entered', 'bubblibot' ) . '</span>';
            return;
        }
        
        // Check if the key format looks valid (starts with sk- and reasonable length)
        if (!preg_match('/^sk-[a-zA-Z0-9_-]{20,}$/', $api_key)) {
            echo '<span class="bubblibot-api-status invalid">❌ ' . esc_html__( 'Invalid API key format', 'bubblibot' ) . '</span>';
            return;
        }
        
        // Check if this API key has been verified before
        $verified_key_hash = get_option('bubblibot_api_key_verified_hash', '');
        $current_key_hash = hash('sha256', $api_key);
        
        if ($verified_key_hash === $current_key_hash) {
            echo '<span class="bubblibot-api-status valid">✅ ' . esc_html__( 'API key is valid and working correctly', 'bubblibot' ) . '</span>';
            return;
        }
        
        // Show that the key needs verification
        echo '<span class="bubblibot-api-status unverified">🔍 ' . esc_html__( 'Click "Verify Key" to test', 'bubblibot' ) . '</span>';
    }
    
    /**
     * Handle API key verification AJAX request
     */
    public function handle_verify_api_key_ajax() {
        check_ajax_referer('bubblibot_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'bubblibot'));
        }
        
        $api_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
        
        if (empty($api_key)) {
            wp_send_json_error(__('API key is required', 'bubblibot'));
        }
        
        try {
            // Temporarily set the API key for testing
            $original_key = get_option('bubblibot_openai_api_key', '');
            update_option('bubblibot_openai_api_key', $api_key);
            
            // Create OpenAI instance and test the key
            $openai = new Bubblibot_OpenAI();
            $result = $openai->test_api_key();
            
            // Restore original key
            update_option('bubblibot_openai_api_key', $original_key);
            
            if ($result['success']) {
                // Store the verification status for this API key
                $api_key_hash = hash('sha256', $api_key);
                update_option('bubblibot_api_key_verified_hash', $api_key_hash);
                
                wp_send_json_success(array(
                    'message' => __('API key is valid and working correctly', 'bubblibot'),
                    'status' => 'valid'
                ));
            } else {
                wp_send_json_error(array(
                    'message' => $result['error'],
                    'status' => 'invalid'
                ));
            }
            
        } catch (Exception $e) {
            // Restore original key in case of exception
            if (isset($original_key)) {
                update_option('bubblibot_openai_api_key', $original_key);
            }
            
            wp_send_json_error(array(
                /* translators: %s: Error message returned when testing the API key. */
                'message' => sprintf(__('Error testing API key: %s', 'bubblibot'), $e->getMessage()),
                'status' => 'error'
            ));
        }
    }
    
    /**
     * Handle clear API verification AJAX request
     */
    public function handle_clear_api_verification_ajax() {
        check_ajax_referer('bubblibot_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'bubblibot'));
        }
        
        // Clear the verification status
        delete_option('bubblibot_api_key_verified_hash');
        
        wp_send_json_success(array(
            'message' => __('API key verification status cleared', 'bubblibot')
        ));
    }
    
    /**
     * Show admin notice when API key is missing
     */
    public function show_api_key_notice() {
        // Only show to users who can manage options
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Don't show on Bubblibot settings pages (user is already there)
        $current_screen = get_current_screen();
        if ($current_screen && strpos($current_screen->id, 'bubblibot') !== false) {
            return;
        }
        
        // Check if chatbot is enabled but API key is missing
        $enabled = get_option('bubblibot_enabled', '1');
        $api_key = get_option('bubblibot_openai_api_key', '');
        
        if ($enabled === '1' && empty($api_key)) {
            $settings_url = admin_url('admin.php?page=bubblibot-settings');
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><strong>⚠️ Bubblibot:</strong> ' . 
                 sprintf(
                     /* translators: 1: opening anchor tag to the Bubblibot settings page, 2: closing anchor tag. */
                     esc_html__( 'OpenAI API key is required for chatbot functionality. Please %1$senter your API key%2$s to enable Bubblibot.', 'bubblibot' ),
                     '<a href="' . esc_url($settings_url) . '">',
                     '</a>'
                 ) . 
                 '</p>';
            echo '</div>';
        }
    }

    /**
     * Cast any checkbox value to '1' or '0'.
     *
     * @param mixed $value Raw checkbox value.
     * @return string Sanitized value.
     */
    public function sanitize_checkbox( $value ) {
        return ! empty( $value ) ? '1' : '0';
    }

    /**
     * Sanitize an array of text values (or a single string).
     *
     * @param mixed $value Raw value from request.
     * @return array|string Sanitized value.
     */
    public function sanitize_text_array( $value ) {
        if ( is_array( $value ) ) {
            return array_map( 'sanitize_text_field', $value );
        }
        return sanitize_text_field( $value );
    }

}