<?php
/**
 * Plugin Name: Bubblibot
 * Plugin URI: https://wordpress.org/plugins/bubblibot/
 * Description: A WordPress plugin that adds an AI-powered chatbot to your website. Indexes your content and answers visitor questions using OpenAI.
 * Version: 1.1.1
 * Author: Plugnify
 * Author URI: https://www.plugnify.com/bubblibot
 * Text Domain: bubblibot
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.8
 * Tested up to: 6.8.2
 * Requires PHP: 8.0
 * Copyright (c) 2025 Plugnify. All rights reserved.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BUBBLIBOT_VERSION', '1.1.1');
define('BUBBLIBOT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BUBBLIBOT_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('BUBBLIBOT_FREE_VERSION', true); // Free version identifier

// Helper function to safely write debug logs
function write_debug_log($message) {
    // Check if debug logging is enabled
    $debug_enabled = get_option('bubblibot_debug_enabled', '0');
    if ($debug_enabled !== '1') {
        return; // Debug logging is disabled
    }
    
    // Use integrated debug logging
    bubblibot_write_debug_log($message);
}

// Integrated debug logging function
function bubblibot_write_debug_log($message) {
    $upload_dir = wp_upload_dir();
    $log_file = $upload_dir['basedir'] . '/bubblibot-debug.log';
    
    // Create log file if it does not exist using WordPress filesystem APIs
    if (!file_exists($log_file)) {
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . '/wp-admin/includes/file.php';
            WP_Filesystem();
        }
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_touch
        $wp_filesystem->put_contents( $log_file, '', FS_CHMOD_FILE );
    }
    
    $timestamp = current_time('mysql');
    $formatted_message = "[{$timestamp}] {$message}\n";
    
    file_put_contents($log_file, $formatted_message, FILE_APPEND);
}

// Maintain backward compatibility with existing debug log plugin
if (!function_exists('write_debug_log')) {
    function write_debug_log($message) {
        write_debug_log($message);
    }
}

// Main plugin class
class Bubblibot {
    
    private $admin;
    private $indexer;
    private $openai;
    private $frontend;
    private $database;
    private $language_detector;
    private $analytics;
    private $display_controller;
    private $load_error = '';
    
    public function __construct() {
        try {
            $this->load_dependencies();
            $this->init_hooks();
        } catch (Exception $e) {
            $this->load_error = $e->getMessage();
            write_debug_log('Bubblibot ERROR: Constructor failed - ' . $e->getMessage());
            
            // Add admin notice for load errors
            add_action('admin_notices', array($this, 'show_load_error'));
        }
    }
    
    public function show_load_error() {
        if (!empty($this->load_error)) {
            echo '<div class="notice notice-error"><p><strong>Bubblibot Error:</strong> ' . esc_html($this->load_error) . '</p></div>';
        }
    }
    
    private function load_dependencies() {
        $required_files = [
            'includes/class-database.php',
            'includes/class-admin.php',
            'includes/class-indexer.php',
            'includes/class-openai.php',
            'includes/class-frontend.php',
            'includes/class-language-detector.php',
            'includes/class-analytics.php',
            'includes/class-display-controller.php'
        ];
        
        foreach ($required_files as $file) {
            $file_path = BUBBLIBOT_PLUGIN_PATH . $file;
            if (!file_exists($file_path)) {
                throw new Exception(sprintf('Required file missing: %s', esc_html($file)));
            }
            require_once $file_path;
        }
        
        // Initialize classes
        try {
            $this->database = new Bubblibot_Database();
            $this->admin = new Bubblibot_Admin();
            $this->indexer = new Bubblibot_Indexer($this->database);
            $this->openai = new Bubblibot_OpenAI();
            $this->language_detector = new Bubblibot_Language_Detector();
            $this->analytics = new Bubblibot_Analytics($this->database);
            $this->display_controller = new Bubblibot_Display_Controller();
            $this->frontend = new Bubblibot_Frontend($this->database, $this->openai, $this->language_detector, $this->analytics, $this->display_controller);
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot ERROR: Failed to initialize classes - ' . $e->getMessage());
            throw new Exception(sprintf('Failed to initialize plugin classes: %s', esc_html($e->getMessage())));
        }
    }
    
    private function init_hooks() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        add_action('plugins_loaded', array($this, 'init'));
    }
    
    public function init() {
        // Skip initialization if there was a load error
        if (!empty($this->load_error)) {
            write_debug_log('Bubblibot: Skipping init due to load error');
            return;
        }
        
        try {
            // Check database version and migrate if needed
            $this->check_database_version();
            
            // Initialize components
            if ($this->admin) {
                $this->admin->init();
            }
            
            if ($this->frontend) {
                $this->frontend->init();
            }
            
            // Auto-index content when posts are saved
            // Use wp_after_insert_post for WordPress 5.6+ (more reliable)
            // and save_post as fallback for older versions
            add_action('wp_after_insert_post', array($this, 'handle_post_save'), 10, 3);
            add_action('save_post', array($this, 'handle_post_save_fallback'), 10, 3);
            add_action('delete_post', array($this, 'handle_post_delete'));
            
            // Auto-index attachments (for potential future use)
            add_action('add_attachment', array($this, 'handle_attachment_add'));
            add_action('delete_attachment', array($this, 'handle_attachment_delete'));
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot ERROR: Initialization failed - ' . $e->getMessage());
        }
    }
    
    /**
     * Handle post save
     */
    public function handle_post_save($post_id, $post, $update) {
        write_debug_log("Bubblibot: handle_post_save triggered for post {$post_id} (update: " . ($update ? 'true' : 'false') . ")");
        
        if ($this->indexer) {
            write_debug_log("Bubblibot: Indexer available, calling index_single_post");
            $this->indexer->index_single_post($post_id);
        } else {
            write_debug_log("Bubblibot: ERROR - Indexer not available!");
        }
        
        // Mark that wp_after_insert_post hook was called
        update_post_meta($post_id, '_bubblibot_indexed_after_insert', time());
    }
    
    /**
     * Handle post save fallback (for older WordPress versions)
     */
    public function handle_post_save_fallback($post_id, $post, $update) {
        // Only run if wp_after_insert_post didn't already handle this
        $already_indexed = get_post_meta($post_id, '_bubblibot_indexed_after_insert', true);
        
        if ($already_indexed && (time() - $already_indexed) < 30) {
            write_debug_log("Bubblibot: Skipping save_post fallback for post {$post_id} - already handled by wp_after_insert_post");
            return;
        }
        
        write_debug_log("Bubblibot: handle_post_save_fallback triggered for post {$post_id} (update: " . ($update ? 'true' : 'false') . ")");
        
        if ($this->indexer) {
            write_debug_log("Bubblibot: Indexer available, calling index_single_post (fallback)");
            $this->indexer->index_single_post($post_id);
        } else {
            write_debug_log("Bubblibot: ERROR - Indexer not available! (fallback)");
        }
    }
    
    /**
     * Handle post delete
     */
    public function handle_post_delete($post_id) {
        if ($this->indexer) {
            $this->indexer->remove_post_from_index($post_id);
        }
    }
    

    
    /**
     * Handle attachment add
     */
    public function handle_attachment_add($attachment_id) {
        if ($this->indexer) {
            $this->indexer->handle_attachment_upload($attachment_id);
        }
    }
    
    /**
     * Handle attachment delete
     */
    public function handle_attachment_delete($attachment_id) {
        if ($this->indexer) {
            $this->indexer->handle_attachment_delete($attachment_id);
        }
    }
    
    public function activate() {
        try {
            // Force load dependencies if not already loaded
            if (empty($this->database)) {
                $this->force_load_dependencies();
            }
            
            // Create database tables with fallback
            $this->create_database_with_fallback();
            
            // Set database version
            update_option('bubblibot_db_version', BUBBLIBOT_VERSION);
            
            // Schedule initial indexing
            if (!wp_next_scheduled('bubblibot_initial_index')) {
                wp_schedule_single_event(time() + 10, 'bubblibot_initial_index');
            }
            
            // Add the action for initial indexing
            add_action('bubblibot_initial_index', array($this, 'handle_initial_index'));
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot ERROR: Activation failed - ' . $e->getMessage());
            
            // Add admin notice for activation failure
            add_option('bubblibot_activation_error', $e->getMessage());
            add_action('admin_notices', array($this, 'show_activation_error'));
        }
    }
    
    public function handle_initial_index() {
        try {
            if ($this->indexer) {
                $this->indexer->index_all_content();
            }
        } catch (Exception $e) {
            write_debug_log('Bubblibot ERROR: Initial indexing failed - ' . $e->getMessage());
        }
    }
    
    /**
     * Force load dependencies during activation
     */
    private function force_load_dependencies() {
        $required_files = [
            'includes/class-database.php',
            'includes/class-admin.php',
            'includes/class-indexer.php',
            'includes/class-openai.php',
            'includes/class-frontend.php',

            'includes/class-language-detector.php',
            'includes/class-analytics.php',
            'includes/class-handoff-manager.php'
        ];
        
        foreach ($required_files as $file) {
            $file_path = BUBBLIBOT_PLUGIN_PATH . $file;
            if (!file_exists($file_path)) {
                throw new Exception(sprintf('Required file missing during activation: %s', esc_html($file)));
            }
            require_once $file_path;
        }
        
        // Initialize only the database class for activation
        $this->database = new Bubblibot_Database();
    }
    
    /**
     * Create database tables with fallback mechanisms
     */
    private function create_database_with_fallback() {
        if (!$this->database) {
            throw new Exception('Database object not available for table creation');
        }
        
        try {
            // Primary method: use the database class
            $this->database->create_tables();
            write_debug_log('Bubblibot: Database tables created via primary method');
            
            // Verify tables were created
            if (!$this->verify_database_tables()) {
                throw new Exception('Table verification failed after creation');
            }
            
        } catch (Exception $e) {
            write_debug_log('Bubblibot: Primary database creation failed, trying fallback: ' . $e->getMessage());
            
            // Fallback method: direct SQL execution
            $this->create_tables_fallback();
            
            // Verify fallback worked
            if (!$this->verify_database_tables()) {
                throw new Exception('Database table creation failed completely');
            }
            
            write_debug_log('Bubblibot: Database tables created via fallback method');
        }
    }
    
    /**
     * Fallback database table creation
     */
    private function create_tables_fallback() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        $content_table = $wpdb->prefix . 'bubblibot_content';
        $analytics_table = $wpdb->prefix . 'bubblibot_analytics';
        
        // Create content table
        $sql1 = "CREATE TABLE IF NOT EXISTS {$content_table} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            content longtext NOT NULL,
            content_type varchar(50) DEFAULT 'post',
            file_path varchar(500),
            source_title varchar(500),
            embedding_id varchar(255),
            product_price decimal(10,2),
            product_sale_price decimal(10,2),
            product_regular_price decimal(10,2),
            product_currency varchar(10),
            product_stock_status varchar(20),
            product_stock_quantity int,
            product_sku varchar(100),
            product_type varchar(50),
            last_updated datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY post_id (post_id),
            KEY content_type (content_type),
            KEY product_price (product_price),
            KEY product_stock_status (product_stock_status),
            FULLTEXT KEY content (content),
            FULLTEXT KEY content_title (content, source_title)
        ) $charset_collate;";
        
        // Create analytics table
        $sql2 = "CREATE TABLE IF NOT EXISTS {$analytics_table} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_session varchar(32),
            question_text longtext NOT NULL,
            detected_language varchar(10),
            context_found tinyint(1) DEFAULT 0,
            response_generated tinyint(1) DEFAULT 0,
            user_satisfied tinyint(1) DEFAULT NULL,
            response_time int DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            user_ip varchar(45),
            user_agent varchar(500),
            PRIMARY KEY (id),
            KEY user_session (user_session),
            KEY created_at (created_at),
            KEY context_found (context_found),
            KEY detected_language (detected_language),
            FULLTEXT KEY question_text (question_text)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        $result1 = dbDelta($sql1);
        $result2 = dbDelta($sql2);
        
        if (empty($result1) && empty($result2)) {
            throw new Exception('dbDelta returned no results for table creation');
        }
    }
    
    /**
     * Verify database tables exist and have correct structure
     */
    private function verify_database_tables() {
        global $wpdb;
        
        $content_table = $wpdb->prefix . 'bubblibot_content';
        $analytics_table = $wpdb->prefix . 'bubblibot_analytics';
        
        // Check if tables exist
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $content_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $content_table ) ) === $content_table;
        $analytics_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $analytics_table ) ) === $analytics_table; // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        if (!$content_exists || !$analytics_exists) {
            write_debug_log('Bubblibot: Table verification failed - tables do not exist');
            return false;
        }
        
        // Check if content_type column exists (this was the main issue)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $content_type_exists = $wpdb->get_var( $wpdb->prepare( "SHOW COLUMNS FROM {$content_table} LIKE %s", 'content_type' ) );
        
        if (!$content_type_exists) {
            write_debug_log('Bubblibot: Table verification failed - content_type column missing');
            return false;
        }
        
        write_debug_log('Bubblibot: Database table verification successful');
        return true;
    }
    
    /**
     * Check database version and migrate if needed
     */
    private function check_database_version() {
        $current_version = get_option('bubblibot_db_version', '0.0.0');
        
        if (version_compare($current_version, BUBBLIBOT_VERSION, '<')) {
            try {
                // Ensure database object is available
                if (!$this->database) {
                    $this->database = new Bubblibot_Database();
                }
                
                // Run database migration
                $this->database->create_tables();
                
                // Run language support migration  
                $this->database->migrate_conversations_language_support();
                
                // Update version
                update_option('bubblibot_db_version', BUBBLIBOT_VERSION);
                
            } catch (Exception $e) {
                write_debug_log('Bubblibot ERROR: Database migration failed - ' . $e->getMessage());
            }
        }
    }
    

    
    /**
     * Show activation error notice
     */
    public function show_activation_error() {
        $error = get_option('bubblibot_activation_error');
        if ($error) {
            echo '<div class="notice notice-error"><p><strong>Bubblibot Activation Error:</strong> ' . esc_html($error) . '</p></div>';
            delete_option('bubblibot_activation_error');
        }
    }
    
    /* The load_plugin_textdomain() call is no longer needed for WordPress 4.6+ where
     * translations are loaded automatically from translate.wordpress.org. The previous
     * implementation has been removed to satisfy PluginCheck’s discouraged-function sniff.
     */
    
    public function deactivate() {
        write_debug_log('Bubblibot: Plugin deactivation started');
        
        // Clean up scheduled events
        wp_clear_scheduled_hook('bubblibot_initial_index');
        
        write_debug_log('Bubblibot: Plugin deactivated');
    }
}

// Initialize the plugin
new Bubblibot();