(function($) {
    'use strict';
    
    // CACHE BUSTED: Emergency Fix v2 - Removed all isMobileDevice references
    // Wait for DOM ready and ensure jQuery is available
    $(document).ready(function() {
    
    // Debug: Check if everything is loaded properly
    console.log('🚀 BubbliBot: FIXED VERSION LOADED - No more isMobileDevice function!');
    console.log('BubbliBot: DOM ready, jQuery loaded', typeof $ !== 'undefined');
    console.log('BubbliBot: jQuery version', $.fn.jquery);
    console.log('BubbliBot: BubbliBot object available:', typeof BubbliBot !== 'undefined');
    
    if (typeof BubbliBot !== 'undefined') {
        console.log('BubbliBot: AJAX URL', BubbliBot.ajax_url);
        console.log('BubbliBot: Nonce', BubbliBot.nonce);
        console.log('BubbliBot: Welcome message', BubbliBot.welcome_message);
    }
    
    // Check if BubbliBot object exists
    if (typeof BubbliBot === 'undefined') {
        console.error('BubbliBot: BubbliBot object is not defined! Script localization failed.');
        console.error('BubbliBot: Available global objects:', Object.keys(window).filter(key => key.toLowerCase().includes('bubble') || key.toLowerCase().includes('chat')));
        return;
    }
    
    // Test AJAX functionality on page load
    console.log('BubbliBot: Testing AJAX functionality...');
            $.ajax({
            url: BubbliBot.ajax_url,
            type: 'POST',
            data: {
                action: 'bubblibot_test',
                nonce: BubbliBot.nonce
            },
        success: function(response) {
            console.log('BubbliBot: Test AJAX successful', response);
        },
        error: function(xhr, status, error) {
            console.error('BubbliBot: Test AJAX failed', {
                status: status,
                error: error,
                responseText: xhr.responseText,
                statusCode: xhr.status,
                readyState: xhr.readyState
            });
            console.error('BubbliBot: Full XHR object', xhr);
        }
    });
    
    // Chat widget elements
    const widget = $('#bubblibot-widget');
    const toggle = $('#bubblibot-toggle');
    const chatWindow = $('#bubblibot-window');
    const messages = $('#bubblibot-messages');
    const input = $('#bubblibot-input');
    const send = $('#bubblibot-send');
    const close = $('#bubblibot-close');
    const typing = $('#bubblibot-typing');
    
    // Check if essential elements exist
    if (widget.length === 0) {
        console.error('BubbliBot: Widget container not found (#bubblibot-widget)');
        return;
    }
    if (toggle.length === 0) {
        console.error('BubbliBot: Toggle button not found (#bubblibot-toggle)');
        return;
    }
    if (chatWindow.length === 0) {
        console.error('BubbliBot: Chat window not found (#bubblibot-window)');
        return;
    }
    
    console.log('BubbliBot: All essential elements found, initializing...');
    
    // Add welcome message
    const welcomeBotAvatarHtml = createBotAvatarHtml();
    const welcomeHasAvatar = welcomeBotAvatarHtml !== '';
    messages.append(`
        <div class="bubblibot-message bubblibot-bot${welcomeHasAvatar ? ' has-avatar' : ''}">
            ${welcomeBotAvatarHtml}
            <div class="bubblibot-message-content">${BubbliBot.welcome_message}</div>
        </div>
    `);
    
    // Set input placeholder
    input.attr('placeholder', BubbliBot.placeholder_text);
    
    // Set header text
    $('.bubblibot-header h3').text(BubbliBot.header_text);
    
    // Toggle chat window - RESTORED DESKTOP FUNCTIONALITY + Mobile enhancements
    toggle.on('click', function() {
        const isMobile = window.innerWidth <= 480; // Simple, reliable mobile detection
        console.log('BubbliBot: Toggle clicked, isMobile:', isMobile, 'window width:', window.innerWidth);
        
        if (isMobile) {
            // Mobile-specific full-screen experience
            if (chatWindow.is(':visible')) {
                // Close chat with slide down effect
                chatWindow.addClass('closing');
                chatWindow.css({
                    'transform': 'translateY(100%)',
                    'transition': 'transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94)'
                });
                
                setTimeout(function() {
                    chatWindow.hide();
                    chatWindow.removeClass('closing');
                    chatWindow.css({
                        'transform': 'translateY(0)',
                        'transition': ''
                    });
                    widget.removeClass('chat-open');
                    $('body').removeClass('bubblibot-mobile-open');
                    
                    // Re-enable scrolling with delay to prevent flash
                    setTimeout(() => {
                        $('body').css('position', '');
                    }, 50);
                }, 300);
            } else {
                // Open chat full-screen with slide up effect
                widget.addClass('chat-open');
                $('body').addClass('bubblibot-mobile-open');
                
                // Set initial position before showing
                chatWindow.css({
                    'transform': 'translateY(100%)',
                    'transition': 'transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94)',
                    'display': 'flex'
                });
                
                // Force reflow and trigger animation
                chatWindow[0].offsetHeight;
                
                requestAnimationFrame(function() {
                    chatWindow.css('transform', 'translateY(0)');
                });
                
                // Don't auto-focus input on mobile to prevent unwanted keyboard
                if (document.activeElement) {
                    document.activeElement.blur();
                }
            }
        } else {
            // DESKTOP BEHAVIOR - FIXED TIMING ISSUE
            console.log('BubbliBot: Desktop mode - calling slideToggle');
            chatWindow.slideToggle(200, function() {
                console.log('BubbliBot: slideToggle complete, visible:', chatWindow.is(':visible'));
                // Focus input after animation completes (only when opening)
                if (chatWindow.is(':visible')) {
                    input.focus();
                    console.log('BubbliBot: Input focused');
                }
            });
        }
    });
    
    // Close chat window
    close.on('click', function() {
        const isMobile = window.innerWidth <= 480; // Simple mobile detection
        
        if (isMobile) {
            // Mobile full-screen close with slide down effect
            chatWindow.addClass('closing');
            chatWindow.css({
                'transform': 'translateY(100%)',
                'transition': 'transform 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94)'
            });
            
            setTimeout(function() {
                chatWindow.hide();
                chatWindow.removeClass('closing');
                chatWindow.css({
                    'transform': 'translateY(0)',
                    'transition': ''
                });
                widget.removeClass('chat-open');
                $('body').removeClass('bubblibot-mobile-open');
                
                // Re-enable scrolling with delay to prevent flash
                setTimeout(() => {
                    $('body').css('position', '');
                }, 50);
            }, 300);
        } else {
            // DESKTOP CLOSE - EXACTLY AS ORIGINAL
            chatWindow.slideUp(200);
        }
    });
    
    // Handle window resize to switch between mobile and desktop modes
    let resizeTimeout;
    $(window).on('resize', function() {
        // Debounce resize events to improve performance
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            const isMobile = window.innerWidth <= 480; // Simple mobile detection
            const isOpen = chatWindow.is(':visible');
            
            if (!isMobile && isOpen) {
                // Switching from mobile to desktop - reset styles and classes
                widget.removeClass('chat-open');
                $('body').removeClass('bubblibot-mobile-open');
                chatWindow.css({
                    'transform': '',
                    'transition': '',
                    'display': ''
                });
                chatWindow.removeClass('closing');
                
                // Re-enable body scrolling if it was disabled
                $('body').css('position', '');
            } else if (isMobile && isOpen) {
                // Ensure mobile full-screen mode is properly set
                widget.addClass('chat-open');
                $('body').addClass('bubblibot-mobile-open');
                chatWindow.css('display', 'flex');
            }
        }, 150); // 150ms debounce
    });
    
    // Send message on button click or enter
    function sendMessage() {
        const message = input.val().trim();
        if (message === '') return;
        
        // Add user message to chat
        const userMessageElement = $(`
            <div class="bubblibot-message bubblibot-user">
                <div class="bubblibot-message-content">${escapeHtml(message)}</div>
            </div>
        `);
        messages.append(userMessageElement);
        
        // Clear input
        input.val('');
        
        // Show typing indicator
        typing.show();
        
        // Scroll to bottom
        messages.scrollTop(messages[0].scrollHeight);
        
        // Send to server
        $.ajax({
            url: BubbliBot.ajax_url,
            type: 'POST',
            data: {
                action: 'bubblibot_message',
                message: message,
                session_id: window.chatbotSessionId || '',
                nonce: BubbliBot.nonce
            },
            success: function(response) {
                typing.hide();
                
                if (response.success) {
                    // Store session ID
                    if (response.data.session_id) {
                        window.chatbotSessionId = response.data.session_id;
                    }
                    
                    // Add message ID to user message if available
                    if (response.data.user_message_id) {
                        userMessageElement.attr('data-message-id', response.data.user_message_id);
                    }
                    
                    // Check for pro feature triggers and show upgrade messages
                    if (response.data.handoff_triggered) {
                        showProFeatureMessage('AI-to-Human Handoff', 'This conversation would be transferred to a human agent in the Pro version.');
                    } else if (response.data.lead_capture_triggered) {
                        showProFeatureMessage('Smart Lead Capture', 'A lead capture form would appear here in the Pro version.');
                    } else {
                        // Add bot response message
                        const botAvatarHtml = createBotAvatarHtml();
                        const hasAvatar = botAvatarHtml !== '';
                        const messageText = (response.data && typeof response.data.message === 'string') ? response.data.message : '';
                        const botMessageHtml = `
                            <div class="bubblibot-message bubblibot-bot${hasAvatar ? ' has-avatar' : ''}"${response.data.bot_message_id ? ` data-message-id="${response.data.bot_message_id}"` : ''}>
                                ${botAvatarHtml}
                                <div class="bubblibot-message-content">${messageText ? formatResponse(messageText) : 'Sorry, I could not generate a response right now.'}</div>
                            </div>
                        `;
                        messages.append(botMessageHtml);
                    }
                } else {
                    messages.append(`
                        <div class="bubblibot-message bubblibot-error">
                            <div class="bubblibot-message-content">Sorry, there was an error: ${escapeHtml(response.data)}</div>
                        </div>
                    `);
                }
                
                messages.scrollTop(messages[0].scrollHeight);
            },
            error: function(xhr, status, error) {
                typing.hide();
                
                // Debug information
                console.error('AJAX Error Details:', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText,
                    statusCode: xhr.status,
                    readyState: xhr.readyState
                });
                
                // Try to parse response if it exists
                if (xhr.responseText) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        console.log('Parsed response:', response);
                        
                        // If we got a valid response, handle it as success
                        if (response && typeof response === 'object') {
                            if (response.success && response.data && response.data.message) {
                                console.log('Response was actually successful, handling as success');
                                
                                const botAvatarHtml = createBotAvatarHtml();
                                const hasAvatar = botAvatarHtml !== '';
                                const safeMsg = (response.data && typeof response.data.message === 'string') ? response.data.message : '';
                                const botMessageHtml = `
                                    <div class="bubblibot-message bubblibot-bot${hasAvatar ? ' has-avatar' : ''}">
                                        ${botAvatarHtml}
                                        <div class="bubblibot-message-content">${safeMsg ? formatResponse(safeMsg) : 'Sorry, I could not generate a response right now.'}</div>
                                    </div>
                                `;
                                messages.append(botMessageHtml);
                                messages.scrollTop(messages[0].scrollHeight);
                                return;
                            } else if (!response.success) {
                                // Show the actual error from the response
                                messages.append(`
                                    <div class="bubblibot-message bubblibot-error">
                                        <div class="bubblibot-message-content">Error: ${escapeHtml(response.data || 'Unknown error')}</div>
                                    </div>
                                `);
                                messages.scrollTop(messages[0].scrollHeight);
                                return;
                            }
                        }
                    } catch (parseError) {
                        console.error('Failed to parse response:', parseError);
                        console.log('Raw response text:', xhr.responseText);
                    }
                }
                
                messages.append(`
                    <div class="bubblibot-message bubblibot-error">
                        <div class="bubblibot-message-content">Sorry, there was an error communicating with the server. (Status: ${xhr.status})</div>
                    </div>
                `);
                messages.scrollTop(messages[0].scrollHeight);
            }
        });
    }
    
    // Send button click
    send.on('click', sendMessage);
    
    // Enter key press with mobile keyboard handling
    input.on('keypress', function(e) {
        if (e.which === 13) {
            sendMessage();
            
            // On mobile, blur input after sending to hide keyboard if needed
            if (window.innerWidth <= 480) {
                // Small delay to let the message send animation complete
                setTimeout(() => {
                    input.blur();
                }, 100);
            }
        }
    });
    
    // Enhanced mobile input handling (only apply if mobile)
    if (window.innerWidth <= 480) {
        // Handle input focus on mobile
        input.on('focus', function() {
            // Scroll messages to bottom when keyboard appears
            setTimeout(() => {
                if (messages.length) {
                    messages.scrollTop(messages[0].scrollHeight);
                }
            }, 300); // Wait for keyboard animation
        });
        
        // Handle viewport changes on mobile (keyboard show/hide)
        if (window.visualViewport) {
            window.visualViewport.addEventListener('resize', function() {
                // Adjust scroll position when keyboard appears/disappears
                setTimeout(() => {
                    if (messages.length) {
                        messages.scrollTop(messages[0].scrollHeight);
                    }
                }, 100);
            });
        }
        
        // Prevent zoom on input focus for iOS
        input.on('focus blur', function(e) {
            if (e.type === 'focus') {
                // Temporarily disable viewport zoom
                $('meta[name=viewport]').attr('content', 'width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=0');
            } else {
                // Re-enable viewport zoom
                setTimeout(() => {
                    $('meta[name=viewport]').attr('content', 'width=device-width, initial-scale=1.0');
                }, 500);
            }
        });
    }
    
    // Helper functions
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function createBotAvatarHtml() {
        const botAvatarUrl = BubbliBot.bot_avatar_url;
        if (!botAvatarUrl) {
            return '';
        }
        
        return `<div class="bubblibot-bot-message-avatar">
            <img src="${botAvatarUrl}" alt="Bot Avatar" onerror="this.parentElement.innerHTML='AI'; this.parentElement.classList.add('default-avatar');">
        </div>`;
    }
    
    function showProFeatureMessage(featureName, description) {
        const botAvatarHtml = createBotAvatarHtml();
        const hasAvatar = botAvatarHtml !== '';
        const upgradeMessage = `
            <div class="bubblibot-message bubblibot-bot${hasAvatar ? ' has-avatar' : ''}">
                ${botAvatarHtml}
                <div class="bubblibot-message-content">
                    <strong>🚀 ${featureName} (Pro Feature)</strong><br>
                    ${description}<br><br>
                    <a href="https://www.plugnify.com/bubblibot" target="_blank" style="color: #0073aa; text-decoration: none;">
                        ✨ Upgrade to Pro →
                    </a>
                </div>
            </div>
        `;
        messages.append(upgradeMessage);
        messages.scrollTop(messages[0].scrollHeight);
    }
    
    function formatResponse(text) {
        if (text == null) {
            return '';
        }
        if (typeof text !== 'string') {
            try { text = String(text); } catch(e) { return ''; }
        }
        const links = [];
        let currentIndex = 0;
    
        // Pre-process text: unescape common HTML entities for brackets/parentheses
        let processedText = text.replace(/&lbrack;/g, '[').replace(/&#91;/g, '[');
        processedText = processedText.replace(/&rbrack;/g, ']').replace(/&#93;/g, ']');
        processedText = processedText.replace(/&lpar;/g, '(').replace(/&#40;/g, '(');
        processedText = processedText.replace(/&rpar;/g, ')').replace(/&#41;/g, ')');
    
        // 1. Extract proper Markdown links [Title](URL) - handle punctuation at end
        processedText = processedText.replace(/\[([^\]]+?)\]\(([^)]+?)\)([.,;!?]*)("?)/g, (match, title, url, punctuation, quote) => {
            // Clean URL by removing trailing punctuation that shouldn't be part of the URL
            let cleanUrl = url.trim();
            // Remove common punctuation that gets accidentally included
            cleanUrl = cleanUrl.replace(/[.,;!?]+$/, '');
            
            links.push({ title: title.trim(), url: cleanUrl });
            return `__CHATBOT_LINK_PLACEHOLDER_${currentIndex++}__${punctuation}${quote}`;
        });
    
        // 2. Handle the specific pattern: [Title] (URL) - note the space, also handle punctuation
        processedText = processedText.replace(/\[([^\]]+?)\]\s+\(([^)]+?)\)([.,;!?]*)("?)/g, (match, title, url, punctuation, quote) => {
            let cleanUrl = url.trim().replace(/[.,;!?]+$/, '');
            links.push({ title: title.trim(), url: cleanUrl });
            return `__CHATBOT_LINK_PLACEHOLDER_${currentIndex++}__${punctuation}${quote}`;
        });
    
        // 3. Handle pattern where article title is mentioned and then URL in parentheses
        // Pattern: "artikkelista: [Title] (URL)" or similar, handle punctuation
        processedText = processedText.replace(/([:\s])(\[([^\]]+?)\])\s*\(([^)]+?)\)([.,;!?]*)("?)/g, (match, prefix, fullTitle, title, url, punctuation, quote) => {
            let cleanUrl = url.trim().replace(/[.,;!?]+$/, '');
            links.push({ title: title.trim(), url: cleanUrl });
            return `${prefix}__CHATBOT_LINK_PLACEHOLDER_${currentIndex++}__${punctuation}${quote}`;
        });
    
        // 4. Extract standalone URLs in parentheses (https://example.com) - handle punctuation
        processedText = processedText.replace(/\((https?:\/\/[^\s)]+)\)([.,;!?]*)("?)/g, (match, url, punctuation, quote) => {
            let cleanUrl = url.trim().replace(/[.,;!?]+$/, '');
            // Try to extract a meaningful title from the URL
            let title = cleanUrl.replace(/^https?:\/\//, '').split('/')[0];
            // If there's a path, try to make a better title
            const pathParts = cleanUrl.split('/');
            if (pathParts.length > 3) {
                const lastPart = pathParts[pathParts.length - 1];
                if (lastPart && lastPart !== '') {
                    title = lastPart.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
                }
            }
            links.push({ title: title, url: cleanUrl });
            return `__CHATBOT_LINK_PLACEHOLDER_${currentIndex++}__${punctuation}${quote}`;
        });
    
        // 5. Extract standalone URLs without parentheses https://example.com
        processedText = processedText.replace(/(?<!\()\b(https?:\/\/[^\s<>"]+)/g, (match, url) => {
            // Use the domain as the title
            const domain = url.replace(/^https?:\/\//, '').split('/')[0];
            links.push({ title: domain, url: url.trim() });
            return `__CHATBOT_LINK_PLACEHOLDER_${currentIndex++}__`;
        });
    
        // 6. HTML-escape the entire text
        const div = document.createElement('div');
        div.textContent = processedText;
        processedText = div.innerHTML;
    
        // 7. Process markdown formatting AFTER HTML escaping but BEFORE link replacement
        // Handle headers (###, ##, #) - process from most specific to least specific
        processedText = processedText.replace(/^###\s+(.+)$/gm, '<h3 style="margin: 8px 0 4px 0; font-size: 15px; font-weight: 600; color: #333;">$1</h3>');
        processedText = processedText.replace(/^##\s+(.+)$/gm, '<h2 style="margin: 10px 0 6px 0; font-size: 17px; font-weight: 600; color: #333;">$1</h2>');
        processedText = processedText.replace(/^#\s+(.+)$/gm, '<h1 style="margin: 12px 0 8px 0; font-size: 18px; font-weight: 600; color: #333;">$1</h1>');
        
        // Handle **bold** text
        processedText = processedText.replace(/\*\*([^*]+?)\*\*/g, '<strong>$1</strong>');
        
        // Handle *italic* text (but not if it's part of **bold**)
        processedText = processedText.replace(/(?<!\*)\*([^*]+?)\*(?!\*)/g, '<em>$1</em>');
        
        // Handle numbered lists with better spacing
        processedText = processedText.replace(/^(\d+)\.\s+(.+)$/gm, '<div style="margin: 4px 0; line-height: 1.4;"><strong>$1.</strong> $2</div>');
    
        // 8. Replace placeholders with actual <a> HTML tags
        links.forEach((linkInfo, index) => {
            const placeholder = `__CHATBOT_LINK_PLACEHOLDER_${index}__`;
            // The placeholder might have been HTML-escaped in step 6
            const divEscapedPlaceholder = document.createElement('div');
            divEscapedPlaceholder.textContent = placeholder;
            const escapedPlaceholder = divEscapedPlaceholder.innerHTML;
            
            const anchor = document.createElement('a');
            anchor.href = linkInfo.url;
            anchor.textContent = linkInfo.title;
            anchor.target = '_blank';
            anchor.rel = 'noopener noreferrer';
            processedText = processedText.replace(escapedPlaceholder, anchor.outerHTML);
        });
    
        // 9. Convert newline characters to <br> tags for display, but handle spacing compactly
        // First, convert newlines to <br>
        processedText = processedText.replace(/\n/g, '<br>');
        
        // Collapse line break immediately before punctuation so . , etc. stay on previous line
        processedText = processedText.replace(/<br>\s*([\.,;!?])/g, '$1');
        
        // Add minimal spacing around headers (only if not already present)
        processedText = processedText.replace(/(<br>)*(<h[1-3][^>]*>)/g, '<br>$2');
        processedText = processedText.replace(/(<\/h[1-3]>)(<br>)*/g, '$1<br>');
        
        // Clean up excessive <br> tags (limit to just one <br> for paragraph breaks)
        processedText = processedText.replace(/(<br>\s*){2,}/g, '<br>');
        
        // Remove leading and trailing <br> tags
        processedText = processedText.replace(/^(<br>\s*)+/, '');
        processedText = processedText.replace(/(<br>\s*)+$/, '');
    
        return processedText;
    }
    
    }); // End document ready
})(jQuery); // End jQuery wrapper