(function($) {
    'use strict';
    
    $(document).ready(function() {
        // Floating Save Button functionality
        var floatingSaveBtn = $('#floating-save-btn');
        var settingsForm = $('#bubblibot-settings-form');
        var hasUnsavedChanges = false;
        
        // Track form changes
        settingsForm.on('change input', 'input, select, textarea', function() {
            if (!hasUnsavedChanges) {
                hasUnsavedChanges = true;
                floatingSaveBtn.text(bubblibot_admin.i18n.save_changes + ' *');
                floatingSaveBtn.css('background', '#d63638'); // Orange to indicate unsaved changes
            }
        });
        
        // Handle floating save button click
        floatingSaveBtn.on('click', function(e) {
            var button = $(this);
            
            // Show saving state
            button.text(bubblibot_admin.i18n.saving);
            
            // Let the form submit naturally - don't interfere
        });
        
        // Handle form submission - reset unsaved changes flag
        settingsForm.on('submit', function() {
            // Reset the unsaved changes flag since we're submitting
            hasUnsavedChanges = false;
        });
        
        // Warn user about unsaved changes when leaving page
        $(window).on('beforeunload', function() {
            if (hasUnsavedChanges) {
                return 'You have unsaved changes. Are you sure you want to leave?';
            }
        });
        
        // Handle re-index button
        $('#reindex-content').on('click', function() {
            var button = $(this);
            var status = $('#reindex-status');
            
            button.prop('disabled', true).text('Indexing...');
            status.html('<span style="color: #0073aa;">Starting indexing process...</span>');
            
            $.ajax({
                url: bubblibot_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'bubblibot_reindex',
                    nonce: bubblibot_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        status.html('<span style="color: #46b450;">✓ ' + response.data.message + '</span>');
                        // Refresh the page after 2 seconds to show updated stats
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        status.html('<span style="color: #dc3232;">✗ Error: ' + response.data + '</span>');
                    }
                },
                error: function(xhr, status, error) {
                    status.html('<span style="color: #dc3232;">✗ AJAX Error: ' + error + '</span>');
                },
                complete: function() {
                    button.prop('disabled', false).text('Re-index All Content');
                }
            });
        });
        
        // Handle API key verification
        $('#verify-api-key').on('click', function() {
            var button = $(this);
            var apiKey = $('#bubblibot_openai_api_key').val();
            var statusDiv = $('#api-key-status');
            
            if (!apiKey.trim()) {
                alert('Please enter an API key first.');
                return;
            }
            
            button.prop('disabled', true).text('Verifying...');
            statusDiv.html('<span class="bubblibot-api-status verifying">🔍 Verifying API key...</span>');
            
            $.ajax({
                url: bubblibot_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'bubblibot_verify_api_key',
                    api_key: apiKey,
                    nonce: bubblibot_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        statusDiv.html('<span class="bubblibot-api-status valid">✅ ' + response.data.message + '</span>');
                    } else {
                        statusDiv.html('<span class="bubblibot-api-status invalid">❌ ' + response.data.message + '</span>');
                    }
                },
                error: function(xhr, status, error) {
                    statusDiv.html('<span class="bubblibot-api-status invalid">❌ Error verifying API key</span>');
                },
                complete: function() {
                    button.prop('disabled', false).text('Verify Key');
                }
            });
        });
        
        // Clear API verification status when key changes
        $('#bubblibot_openai_api_key').on('input', function() {
            var statusDiv = $('#api-key-status');
            var currentKey = $(this).val();
            
            if (!currentKey.trim()) {
                statusDiv.html('<span class="bubblibot-api-status empty">⚠️ No API key entered</span>');
            } else if (!currentKey.match(/^sk-[a-zA-Z0-9_-]{20,}$/)) {
                statusDiv.html('<span class="bubblibot-api-status invalid">❌ Invalid API key format</span>');
            } else {
                statusDiv.html('<span class="bubblibot-api-status unverified">🔍 Click "Verify Key" to test</span>');
            }
        });
    });
    
})(jQuery);