<?php
/**
 * Plugin Name: BT AI Alt Generator
 * Plugin URI: https://bt-ai.eu/
 * Description: ALT text generator with AI via API call GDPR-compliant thanks to AI server location in Europe (Germany).
 * Version: 1.0.0
 * Author: BT Webdesign
 * Author URI: https://www.bt-webdesign.de/
 * Text Domain: bt-ai-alt-generator
 * Domain Path: /languages
 * License: GPL-2.0+
 *
 * @package btaialge
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin Konstanten
 */
define( 'BTAIALGE_VERSION', '1.0.0' );
define( 'BTAIALGE_SERVER_URL', 'https://bt-ai.eu/' );
define( 'BTAIALGE_SERVER_URL_GENERATE', BTAIALGE_SERVER_URL . 'wp-json/bt-ai/v1/btaialge_generate_alt' );
define( 'BTAIALGE_SERVER_URL_KEY_INFO', BTAIALGE_SERVER_URL . 'wp-json/bt-ai/v1/btaialge_get_key_info' );
define(
	'BTAIALGE_AI_MODELS',
	array(
		array(
			'name'   => 'Mistral Small 24B',
			'model'  => 'mistralai/Mistral-Small-24B-Instruct',
			'status' => 'active',
		),
	)
);

define( 'BTAIALGE_IMG_MAX_WIDTH', 800 );

/**
 * Add Settings link to plugins page
 *
 * @since 1.0.0
 *
 * @param array $links An array of links from the plugin.
 *
 * @return array $links Array of links from the plugin plus settings link
 */
function btaialge_add_settings_link( $links ) {

	$settings_link = '<a href="options-general.php?page=btaialge_settings">' . __( 'Settings', 'bt-ai-alt-generator' ) . '</a>';
	array_unshift( $links, $settings_link );

	return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'btaialge_add_settings_link' );

/**
 * Add page settings to menu
 *
 * @since 1.0.0
 */
function btaialge_create_menu() {

	add_options_page(
		'BT AI ALT Settings',
		'BT AI Alt Text',
		'upload_files',
		'btaialge_settings',
		'btaialge_settings_page'
	);
}
add_action( 'admin_menu', 'btaialge_create_menu' );

/**
 * Enqueue Files + Localize Script
 *
 * @since 1.0.0
 *
 * @param string $hook Hook suffix for the current admin page.
 */
function btaialge_admin_enqueue_scripts( $hook ) {

	if ( 'settings_page_btaialge_settings' === $hook ) {
		wp_enqueue_script( 'btaialge-settings-js', plugins_url( '/assets/js/bt-ai-settings-page.js', __FILE__ ), array(), BTAIALGE_VERSION, true );
		return;
	}

	wp_enqueue_style( 'btaialge_css', plugins_url( '/assets/css/bt-ai-alt-generator.css', __FILE__ ), array(), BTAIALGE_VERSION );

	wp_enqueue_script( 'btaialge-js', plugins_url( '/assets/js/bt-ai-alt-generator.js', __FILE__ ), array( 'jquery' ), BTAIALGE_VERSION, true );

	$api_key = get_option( 'btaialge_api_key' );

	wp_localize_script(
		'btaialge-js',
		'btaialge',
		array(
			'ajax_url'    => admin_url( 'admin-ajax.php' ),
			'nonce'       => wp_create_nonce( 'btaialge_nonce_settings' ),
			'hasApiKey'   => ! empty( $api_key ),
			'settingsUrl' => admin_url( 'options-general.php?page=btaialge_settings' ),
			'texts'       => array(
				'alt_text'         => __( 'Alternative text', 'bt-ai-alt-generator' ),
				'wait'             => __( 'Please wait ...', 'bt-ai-alt-generator' ),
				'success'          => __( 'ALT text successfully inserted', 'bt-ai-alt-generator' ),
				'error'            => __( 'Error', 'bt-ai-alt-generator' ),
				'overwrite_notice' => __( 'Existing texts will be overwritten during generation!', 'bt-ai-alt-generator' ),
				'api_key_notice'   => __( 'Please enter an API-Key first:', 'bt-ai-alt-generator' ),
				'settings'         => __( 'Settings', 'bt-ai-alt-generator' ),
				'btn_label'        => __( 'Generate ALT-Text with AI', 'bt-ai-alt-generator' ),
			),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'btaialge_admin_enqueue_scripts' );

/**
 * Client-side API Call to Server
 *
 * @since 1.0.0
 */
function btaialge_ajax_handler() {

	if ( ! check_ajax_referer( 'btaialge_nonce_settings', 'btaialge_nonce', false ) ) {                            // Check nonce.
		wp_send_json_error( __( 'Security check failed! Please reload the page.', 'bt-ai-alt-generator' ) );
	}

	if ( ! current_user_can( 'upload_files' ) ) {
		wp_send_json_error( __( 'You do not have permission to perform this action.', 'bt-ai-alt-generator' ) );   // Check permission.
	}

	$api_key = get_option( 'btaialge_api_key' );                                                                   // Check API-key.
	if ( empty( $api_key ) ) {
		wp_send_json_error( __( 'Please enter your API-Key in the plugin settings!', 'bt-ai-alt-generator' ) );
	}

	if ( ! isset( $_POST['post_id'] ) ) {                                                                          // Check post_id.
		wp_send_json_error( __( 'Error! Post-Id missing', 'bt-ai-alt-generator' ) );
	}

	$post_id = intval( $_POST['post_id'] );

	$model  = get_option( 'btaialge_model', BTAIALGE_AI_MODELS[0]['model'] );                                      // Retrieve fields from the database.
	$prompt = get_option( 'btaialge_prompt', __( 'Describe the image in a maximum of 150 characters in English for image ALT attribute that is SEO-optimised and for an accessible website.', 'bt-ai-alt-generator' ) );

	$image_data = btaialge_get_scaled_base64( $post_id );
	if ( ! $image_data ) {
		wp_send_json_error( __( 'Image processing failed. Please check if the file is a valid image, not too large and has permitted filetype (jpg, jpeg, png, webp, avif, gif)!', 'bt-ai-alt-generator' ) );
	}

	$server_url = BTAIALGE_SERVER_URL_GENERATE;                                                                    // Server URL.

	$payload = array(
		'api_key'      => $api_key,
		'language'     => get_locale(),
		'image_base64' => $image_data,
		'model'        => $model,
		'prompt'       => $prompt,
	);

	$response = wp_remote_post(
		$server_url,
		array(
			'timeout'     => 60,
			'headers'     => array( 'Content-Type' => 'application/json; charset=utf-8' ),
			'data_format' => 'body',
			'body'        => wp_json_encode( $payload, JSON_UNESCAPED_UNICODE ),
		)
	);

	if ( is_wp_error( $response ) ) {
		wp_send_json_error( __( 'Server not available: ', 'bt-ai-alt-generator' ) . $response->get_error_message() );
	}

	$data = json_decode( wp_remote_retrieve_body( $response ), true );

	if ( isset( $data['alt_text'] ) ) {
		wp_send_json_success( $data['alt_text'] );
	} else {
		wp_send_json_error( $data['message'] ?? __( 'Unknown error from the server', 'bt-ai-alt-generator' ) );
	}
}
add_action( 'wp_ajax_btaialge_generate_alt', 'btaialge_ajax_handler' );

/**
 * Convert image to base64
 *
 * @since 1.0.0
 *
 * @param int $post_id Post-Id.
 * @param int $max_width Maximum allowed width of the image.
 *
 * @return boolean|string The Base64-encoded image data.
 */
function btaialge_get_scaled_base64( $post_id, $max_width = BTAIALGE_IMG_MAX_WIDTH ) {

	$supported_mimes = array(
		'image/jpeg',
		'image/jpg',
		'image/png',
		'image/gif',
		'image/webp',
		'image/avif',
	);

	$mime_type = get_post_mime_type( $post_id );

	if ( empty( $mime_type ) || '' === $mime_type ) {
		$file_path = get_attached_file( $post_id );                                                 // Get file path.

		if ( $file_path ) {
			$file_info = wp_check_filetype( $file_path );
			$mime_type = $file_info['type'];
		}
	}

	if ( ! in_array( $mime_type, $supported_mimes, true ) ) {
		return false;
	}

	$file_path = get_attached_file( $post_id );
	if ( ! $file_path || ! file_exists( $file_path ) ) {
		return false;
	}

	$editor = wp_get_image_editor( $file_path );                                                    // Load WordPress Image Editor.

	if ( ! is_wp_error( $editor ) ) {

		$size   = $editor->get_size();                                                              // Get image size.
		$orig_w = $size['width'];
		$orig_h = $size['height'];

		if ( $orig_w > $max_width ) {                                                               // Reduce image size if wider than BTAIALGE_IMG_MAX_WIDTH.
			$aspect_ratio = $orig_h / $orig_w;
			$new_w        = $max_width;
			$new_h        = round( $new_w * $aspect_ratio );

			$editor->resize( $new_w, $new_h, false );                                               // Resize without cropping.
		}

		ob_start();                                                                                 // Image in memory - temporary stream.
		$editor->set_quality( 80 );
		$mime_type = 'image/webp';                                                                  // webp to reduce size.
		$editor->stream( $mime_type );
		$image_data = ob_get_clean();

		return base64_encode( $image_data );                                                        // base64 required for the API transfer of image data.
	}

	return false;
}

/**
 * Register settings
 *
 * @since 1.0.0
 */
function btaialge_register_settings() {

	register_setting( 'btaialge-group', 'btaialge_api_key', array( 'sanitize_callback' => 'btaialge_validate_api_key' ) );
	register_setting( 'btaialge-group', 'btaialge_model', array( 'sanitize_callback' => 'sanitize_text_field' ) );
	register_setting( 'btaialge-group', 'btaialge_prompt', array( 'sanitize_callback' => 'sanitize_textarea_field' ) );
}
add_action( 'admin_init', 'btaialge_register_settings' );

/**
 * Plugin Setting Page
 *
 * @since 1.0.0
 *
 * @param string $api_key API-Key.
 * @param string $ret Return text.
 *
 * @return string API-Key or message or error message.
 */
function btaialge_validate_api_key( $api_key, $ret = '' ) {

	$api_key = sanitize_text_field( $api_key );

	if ( empty( $api_key ) ) {
		return $api_key;
	}

	$server_url = BTAIALGE_SERVER_URL_KEY_INFO;                                                     // Server URL.

	$payload = array(
		'api_key'  => $api_key,
		'language' => get_locale(),
	);

	$response = wp_remote_post(
		$server_url,
		array(
			'timeout'     => 60,
			'headers'     => array( 'Content-Type' => 'application/json; charset=utf-8' ),
			'data_format' => 'body',
			'body'        => wp_json_encode( $payload, JSON_UNESCAPED_UNICODE ),
		)
	);

	if ( is_wp_error( $response ) ) {

		if ( 'txt' === $ret ) {
			return '';
		}

		add_settings_error(
			'btaialge_api_key',
			'api_key_invalid',
			__( 'Server not available: ', 'bt-ai-alt-generator' ) . $response->get_error_message(),
			'error'
		);
	}

	$data = json_decode( wp_remote_retrieve_body( $response ), true );

	if ( is_wp_error( $response ) || ( isset( $data['status'] ) && 200 !== $data['status'] ) || ! isset( $data['success'] ) ) {

		$code = '';
		if ( isset( $data['code'] ) ) {
			$code = $data['code'] . ' - ';
		}

		$msg = '';
		if ( isset( $data['message'] ) ) {
			$msg = $data['message'];
		}

		$txt = __( 'The API key could not be verified. ', 'bt-ai-alt-generator' ) . $code . $msg;
		if ( 'txt' === $ret ) {
			return $txt;
		}

		add_settings_error(
			'btaialge_api_key',
			'api_key_invalid',
			$txt,
			'error'
		);
	} elseif ( ! $data['success'] ) {

		$msg = '';
		if ( isset( $data['msg'] ) ) {
			$msg = $data['msg'];
		}

		$txt = __( 'The API key is invalid. ', 'bt-ai-alt-generator' ) . $msg;
		if ( 'txt' === $ret ) {
			return $txt;
		}

		add_settings_error(
			'btaialge_api_key',
			'api_key_invalid',
			$txt,
			'error'
		);
	} else {

		$msg = '';
		if ( isset( $data['msg'] ) ) {
			$msg = $data['msg'];
		}

		$txt = __( 'API key successfully verified! ', 'bt-ai-alt-generator' ) . $msg;
		if ( 'txt' === $ret ) {
			return $txt;
		}

		add_settings_error(
			'btaialge_api_key',
			'api_key_valid',
			$txt,
			'updated'
		);
	}

	return $api_key;
}

/**
 * Plugin Setting Page
 *
 * @since 1.0.0
 */
function btaialge_settings_page() {

	$api_key     = esc_attr( get_option( 'btaialge_api_key' ) );
	$api_key_txt = btaialge_validate_api_key( $api_key, 'txt' );

	?>
	<div id="btaialge_settings" class="wrap">
		<div>
			<img src="<?php echo esc_url( plugins_url( '/assets/images/bt-ai-logo.svg', __FILE__ ) ); ?>" width="80px" height="80px" alt="BT AI ALT Text Generator" title="BT AI ALT Text Generator">
			<h1>BT AI ALT Text Generator (German AI)</h1>
			<h2><?php echo esc_html( __( 'Settings', 'bt-ai-alt-generator' ) ); ?></h2>
		</div>
		
		<form method="post" action="options.php">
			<?php settings_fields( 'btaialge-group' ); ?>
			<table class="form-table btaialge_set_table">
				<tr valign="top">
					<th scope="row">API-Key</th>
					<td>
						<input type="text" name="btaialge_api_key" value="<?php echo esc_html( $api_key ); ?>" class="regular-text" />
						<p class="description"><?php echo esc_html( __( 'API-Key for your website', 'bt-ai-alt-generator' ) ); ?> (<a href="<?php echo esc_html( BTAIALGE_SERVER_URL . 'shop/' ); ?>" title="<?php echo esc_html( __( 'Get API-Key ', 'bt-ai-alt-generator' ) ); ?>" target="_blank"><?php echo esc_html( __( 'Get API-Key ', 'bt-ai-alt-generator' ) ); ?></a>) <strong><span id="bt_attgga_api_key_txt"><?php echo esc_html( $api_key_txt ); ?></span></strong></p>
					</td>
				</tr>
				<tr valign="top">
					<th scope="row"><?php echo esc_html( __( 'AI model', 'bt-ai-alt-generator' ) ); ?></th>
					<td>
						<select name="btaialge_model">
						<?php
						foreach ( BTAIALGE_AI_MODELS as $model ) {
							echo '<option value="' . esc_attr( $model['model'] ) . '" ' . selected( get_option( 'btaialge_model', BTAIALGE_AI_MODELS[0]['model'] ), $model['model'], false ) . '>' . esc_html( $model['name'] ) . '</option>';
						}
						?>
						</select>
						<p class="description"><?php echo esc_html( __( 'Select the AI model for image analysis.', 'bt-ai-alt-generator' ) ); ?></p>
					</td>
				</tr>
				<tr valign="top">
					<th scope="row"><?php echo esc_html( __( 'AI prompt', 'bt-ai-alt-generator' ) ); ?></th>
					<td>
						<textarea id="btaialge_prompt" name="btaialge_prompt" rows="5" cols="80" class=""><?php echo esc_textarea( get_option( 'btaialge_prompt', __( 'Describe the image in a maximum of 150 characters in English for image ALT attribute that is SEO-optimised and for an accessible website.', 'bt-ai-alt-generator' ) ) ); ?></textarea>
						<p class="description">
							<?php echo esc_html( __( 'Prompt for AI to analyze the image.', 'bt-ai-alt-generator' ) . ' ' . __( 'Maximum 2.000 characters are passed on to the AI.', 'bt-ai-alt-generator' ) ); ?>
							(<strong><span id="btaialge_current_chars">0</span> / 2000 <?php echo esc_html( __( 'characters', 'bt-ai-alt-generator' ) ); ?></strong>)
						</p>
					</td>
				</tr>
			</table>
			
			<?php submit_button(); ?>
		</form>
	</div>
	<?php
}