<?php
/**
 * Plugin Name: B'SD Header Text
 * Description: Displays בס"ד at the top right corner of every page. Admin can customize the font size and color, with responsive margins for desktop, tablet, and mobile.
 * Version: 1.5
 * Author: Eliyahna
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

define('BSD_PLUGIN_VERSION', '1.5');
define('BSD_PLUGIN_DIR_URL', plugin_dir_url(__FILE__));
define('BSD_PLUGIN_DIR_PATH', plugin_dir_path(__FILE__));

/**
 * Enqueue styles and responsive dynamic inline CSS
 */
add_action('wp_enqueue_scripts', 'bsd_plugin_enqueue_styles');
function bsd_plugin_enqueue_styles() {
    wp_register_style('bsd-header-text-style', BSD_PLUGIN_DIR_URL . 'css/style.css', [], BSD_PLUGIN_VERSION);
    wp_enqueue_style('bsd-header-text-style');

    $color         = esc_attr(get_option('bsd_plugin_text_color', '#000000'));
    $font_size     = absint(get_option('bsd_plugin_font_size', 16));

    // Desktop margins (default)
    $top_desktop   = absint(get_option('bsd_plugin_top_margin_desktop', 25));
    $right_desktop = absint(get_option('bsd_plugin_right_margin_desktop', 25));

    // Tablet margins
    $top_tablet    = absint(get_option('bsd_plugin_top_margin_tablet', 20));
    $right_tablet  = absint(get_option('bsd_plugin_right_margin_tablet', 20));

    // Mobile margins
    $top_mobile    = absint(get_option('bsd_plugin_top_margin_mobile', 15));
    $right_mobile  = absint(get_option('bsd_plugin_right_margin_mobile', 15));

    $custom_css = "
        .bsd-plugin-header-text {
            color: {$color};
            font-size: {$font_size}px;
            position: fixed;
            z-index: 9999;
            font-family: inherit;
            direction: rtl;
            background: transparent;
            padding: 0;
            margin: 0;
            line-height: 1.2;
            pointer-events: none;

            top: {$top_desktop}px;
            right: {$right_desktop}px;
        }

        /* Tablet: 768px to 1024px */
        @media (max-width: 1024px) and (min-width: 768px) {
            .bsd-plugin-header-text {
                top: {$top_tablet}px !important;
                right: {$right_tablet}px !important;
            }
        }

        /* Mobile: below 768px */
        @media (max-width: 767px) {
            .bsd-plugin-header-text {
                top: {$top_mobile}px !important;
                right: {$right_mobile}px !important;
            }
        }
    ";

    wp_add_inline_style('bsd-header-text-style', $custom_css);
}

/**
 * Display the בס"ד text in footer
 */
add_action('wp_footer', 'bsd_plugin_display_text');
function bsd_plugin_display_text() {
    echo '<div class="bsd-plugin-header-text" aria-hidden="true">' . esc_html('בס"ד') . '</div>';
}

/**
 * Register settings including responsive margins
 */
add_action('admin_init', 'bsd_plugin_settings_init');
function bsd_plugin_settings_init() {
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_text_color', [
        'sanitize_callback' => 'sanitize_hex_color',
        'default'           => '#000000',
    ]);

    register_setting('bsd_plugin_settings_group', 'bsd_plugin_font_size', [
        'sanitize_callback' => 'absint',
        'default'           => 16,
    ]);

    // Desktop margins
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_top_margin_desktop', [
        'sanitize_callback' => 'absint',
        'default'           => 25,
    ]);
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_right_margin_desktop', [
        'sanitize_callback' => 'absint',
        'default'           => 25,
    ]);

    // Tablet margins
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_top_margin_tablet', [
        'sanitize_callback' => 'absint',
        'default'           => 20,
    ]);
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_right_margin_tablet', [
        'sanitize_callback' => 'absint',
        'default'           => 20,
    ]);

    // Mobile margins
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_top_margin_mobile', [
        'sanitize_callback' => 'absint',
        'default'           => 15,
    ]);
    register_setting('bsd_plugin_settings_group', 'bsd_plugin_right_margin_mobile', [
        'sanitize_callback' => 'absint',
        'default'           => 15,
    ]);
}

/**
 * Add admin menu page
 */
add_action('admin_menu', 'bsd_plugin_add_admin_menu');
function bsd_plugin_add_admin_menu() {
    add_options_page(
        __('B\'SD Header Text Settings', 'bsd-header-text'),
        __('B\'SD Header Text', 'bsd-header-text'),
        'manage_options',
        'bsd-plugin-settings',
        'bsd_plugin_settings_page'
    );
}

/**
 * Render the plugin settings page
 */
function bsd_plugin_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php esc_html_e("B'SD Header Text Settings", 'bsd-header-text'); ?></h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('bsd_plugin_settings_group');
            do_settings_sections('bsd-plugin-settings');
            ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th colspan="3"><h2><?php esc_html_e('Text Appearance', 'bsd-header-text'); ?></h2></th>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="bsd_plugin_text_color"><?php esc_html_e('Text Color', 'bsd-header-text'); ?></label>
                    </th>
                    <td colspan="2">
                        <input type="color" id="bsd_plugin_text_color" name="bsd_plugin_text_color"
                               value="<?php echo esc_attr(get_option('bsd_plugin_text_color', '#000000')); ?>">
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="bsd_plugin_font_size"><?php esc_html_e('Font Size (px)', 'bsd-header-text'); ?></label>
                    </th>
                    <td colspan="2">
                        <input type="number" id="bsd_plugin_font_size" name="bsd_plugin_font_size"
                               value="<?php echo esc_attr(get_option('bsd_plugin_font_size', 16)); ?>" min="1" max="100">
                    </td>
                </tr>

                <tr><th colspan="3"><h2><?php esc_html_e('Margins (px) for Different Devices', 'bsd-header-text'); ?></h2></th></tr>

                <tr>
                    <th scope="row"><?php esc_html_e('Desktop (≥ 1025px)', 'bsd-header-text'); ?></th>
                    <td>
                        <label for="bsd_plugin_top_margin_desktop"><?php esc_html_e('Top Margin', 'bsd-header-text'); ?></label><br>
                        <input type="number" id="bsd_plugin_top_margin_desktop" name="bsd_plugin_top_margin_desktop"
                               value="<?php echo esc_attr(get_option('bsd_plugin_top_margin_desktop', 25)); ?>" min="0" max="500">
                    </td>
                    <td>
                        <label for="bsd_plugin_right_margin_desktop"><?php esc_html_e('Right Margin', 'bsd-header-text'); ?></label><br>
                        <input type="number" id="bsd_plugin_right_margin_desktop" name="bsd_plugin_right_margin_desktop"
                               value="<?php echo esc_attr(get_option('bsd_plugin_right_margin_desktop', 25)); ?>" min="0" max="500">
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e('Tablet (768px - 1024px)', 'bsd-header-text'); ?></th>
                    <td>
                        <label for="bsd_plugin_top_margin_tablet"><?php esc_html_e('Top Margin', 'bsd-header-text'); ?></label><br>
                        <input type="number" id="bsd_plugin_top_margin_tablet" name="bsd_plugin_top_margin_tablet"
                               value="<?php echo esc_attr(get_option('bsd_plugin_top_margin_tablet', 20)); ?>" min="0" max="500">
                    </td>
                    <td>
                        <label for="bsd_plugin_right_margin_tablet"><?php esc_html_e('Right Margin', 'bsd-header-text'); ?></label><br>
                        <input type="number" id="bsd_plugin_right_margin_tablet" name="bsd_plugin_right_margin_tablet"
                               value="<?php echo esc_attr(get_option('bsd_plugin_right_margin_tablet', 20)); ?>" min="0" max="500">
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e('Mobile (< 768px)', 'bsd-header-text'); ?></th>
                    <td>
                        <label for="bsd_plugin_top_margin_mobile"><?php esc_html_e('Top Margin', 'bsd-header-text'); ?></label><br>
                        <input type="number" id="bsd_plugin_top_margin_mobile" name="bsd_plugin_top_margin_mobile"
                               value="<?php echo esc_attr(get_option('bsd_plugin_top_margin_mobile', 15)); ?>" min="0" max="500">
                    </td>
                    <td>
                        <label for="bsd_plugin_right_margin_mobile"><?php esc_html_e('Right Margin', 'bsd-header-text'); ?></label><br>
                        <input type="number" id="bsd_plugin_right_margin_mobile" name="bsd_plugin_right_margin_mobile"
                               value="<?php echo esc_attr(get_option('bsd_plugin_right_margin_mobile', 15)); ?>" min="0" max="500">
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}
