<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://github.com/husenmemon
 * @since      1.0.0
 *
 * @package    Breadcrumbs_Made_Easy
 * @subpackage Breadcrumbs_Made_Easy/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Breadcrumbs_Made_Easy
 * @subpackage Breadcrumbs_Made_Easy/public
 * @author     Husen Memon <husen@conicplex.com>
 */
class Breadcrumbs_Made_Easy_Public
{

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version)
	{

		$this->plugin_name = $plugin_name;
		$this->version = $version;

		add_shortcode('easy_breadcrumbs', array($this, 'bmecp_breadcrumbs'));

		$settings = get_option('bmecp_settings', array(
			'display_on' => '',
		));

		$display_on = isset($settings['display_on']) ? esc_attr($settings['display_on']) : '';

		switch ($display_on) {
			case 'before_content':
				//add_action('bmecp_before_content', array($this, 'bmecp_output_breadcrumbs'));
				add_filter('the_content', [$this, 'bmecp_prepend_breadcrumbs']);
				break;
			case 'after_content':
				add_filter('the_content', [$this, 'bmecp_append_breadcrumbs']);
				break;
			case 'before_title':
				add_filter('the_title', [$this, 'bmecp_prepend_breadcrumbs'], 10, 2);
				break;
			case 'after_title':
				add_filter('the_title', [$this, 'bmecp_append_breadcrumbs']);
				break;
			default:
				// Do not auto insert
				break;
		}
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles()
	{

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Breadcrumbs_Made_Easy_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Breadcrumbs_Made_Easy_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/breadcrumbs-made-easy-public.css', array(), $this->version, 'all');

		$settings = get_option('bmecp_settings', array(
			'font_size' => '20',
			'font_color' => '#66686B',
			'separator_color' => '#66686B',
			'highlight_last_item' => 1,
		));

		$font_size = esc_attr($settings['font_size'] ?: '20');
		$font_color = esc_html($settings['font_color'] ?: '#66686B');
		$separator_color = esc_html($settings['separator_color'] ?: '#66686B');
		$highlight_last_item = esc_attr($settings['highlight_last_item']);
		$current_page_font_color = esc_html($settings['current_page_font_color'] ?: '#66686B');

		// Enqueue inline style
		$custom_css = "
			.breadcrumbs-bmecp {
				font-size: {$font_size}px;
				color: {$font_color};
			}
			.breadcrumbs-bmecp a, .breadcrumbs-bmecp a:hover, .breadcrumbs-bmecp a:active, .breadcrumbs-bmecp a:visited {
				color: {$font_color};
			}
			.breadcrumb-sep{
				color: {$separator_color};
			}
			.breadcrumbs-bmecp .current{
				font-size: {$font_size}px;
				color: {$current_page_font_color};
			}
		";

		if ($highlight_last_item) {
			$custom_css .= "
				.breadcrumbs-bmecp .current {
					font-weight: bold;
				}
			";
		} else {
			$custom_css .= "
				.breadcrumbs-bmecp .current {
					font-weight: normal;
				}
			";
		}
		wp_add_inline_style($this->plugin_name, $custom_css);
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts()
	{

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Breadcrumbs_Made_Easy_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Breadcrumbs_Made_Easy_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		//wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/breadcrumbs-made-easy-public.js', array('jquery'), $this->version, false);
	}

	public function bmecp_breadcrumbs()
	{

		$current_settings = get_option('bmecp_settings', array(
			'show_on_home' => 0,
			'show_current' => 1,
			'separator' => '>',
			'highlight_last_item' => 1,
			'home_label' => 'Home',
			'search_result_label' => 'Search results for',
			'archive_label' => 'Archive',
			'tag_label' => 'Posts tagged',
			'author_label' => 'Author page for',
			'error_label' => 'Error 404',
			'font_size' => '20',
		));

		$highlight_last_item = esc_attr($current_settings['highlight_last_item']);
		$search_result_label = $current_settings['search_result_label'] ?: 'Search results for';
		$archive_label = esc_html($current_settings['archive_label'] ?: 'Archive');
		$tag_label = esc_html($current_settings['tag_label'] ?: 'Posts tagged');
		$author_label = esc_html($current_settings['author_label'] ?: 'Author page for');
		$error_label = esc_html($current_settings['error_label'] ?: 'Error 404');
		$font_size = esc_attr($current_settings['font_size'] ?: '20');

		$separator = esc_html($current_settings['separator'] ?: '>');

		$showOnHome = esc_attr($current_settings['show_on_home']);
		$delimiter = '<span class="breadcrumb-sep">' . esc_html($separator) . '</span>';
		$home = esc_html($current_settings['home_label'] ?: get_bloginfo('name') ?: 'Home'); // Default to 'Home' if empty

		// Force showing current item name in breadcrumb
		$showCurrent = esc_attr($current_settings['show_current']);
		$before = '<span class="current">';
		$after = '</span>';

		global $post;

		$homeLink = esc_url(trailingslashit(get_bloginfo('url')));

		// If is_home() or is_front_page()
		if (is_home() || is_front_page()) {

			if (!is_front_page() && is_home()) {
				// blog page
				$posts_page_id = get_option('page_for_posts'); // the page you selected
				$page_title = get_the_title($posts_page_id);
				echo '<div id="crumbs" class="breadcrumbs-bmecp crumbs">'
					. '<a href="' . esc_url($homeLink) . '">' . wp_kses_post($home) . '</a>'
					. ' ' . wp_kses_post($delimiter) . ' '
					. wp_kses_post($before) . esc_html($page_title) . wp_kses_post($after)
					. '</div>';
			} else {
				// front page
				if ($showOnHome == 1) {
					echo '<div id="crumbs" class="breadcrumbs-bmecp crumbs">'
						. '<a href="' . esc_url($homeLink) . '">' . wp_kses_post($home) . '</a>'
						. '</div>';
				}
			}
		} else {
			echo '<div id="crumbs" class="breadcrumbs-bmecp crumbs">'
				. '<a href="' . esc_url($homeLink) . '">' . wp_kses_post($home) . '</a> ' . wp_kses_post($delimiter) . ' ';

			if (is_category()) {
				$cat_id = get_query_var('cat');
				$thisCat = get_category($cat_id, false);

				if ($cat_id && !is_wp_error($thisCat) && $thisCat) {
					if (!empty($thisCat->parent)) {
						$parent_cats = get_category_parents($thisCat->parent, true, ' ' . wp_kses_post($delimiter) . ' ');
						if (!is_wp_error($parent_cats)) {
							echo esc_html($parent_cats);
						}
					}
					echo wp_kses_post($before) . esc_html(single_cat_title('', false)) . wp_kses_post($after);
				} else {
					echo wp_kses_post($before) . esc_html__('Category', 'breadcrumbs-made-easy') . wp_kses_post($after);
				}
			} elseif (is_search()) {
				echo wp_kses_post($before) . wp_kses_post($search_result_label) . ' "' . esc_html(get_search_query()) . '"' . wp_kses_post($after);
			} elseif (is_day()) {
				echo '<a href="' . esc_url(get_year_link(get_the_time('Y'))) . '">' . esc_html(get_the_time('Y')) . '</a> '
					. wp_kses_post($delimiter) . ' ';
				echo '<a href="' . esc_url(get_month_link(get_the_time('Y'), get_the_time('m'))) . '">'
					. esc_html(get_the_time('F')) . '</a> '
					. wp_kses_post($delimiter) . ' ';
				echo wp_kses_post($before) . esc_html(get_the_time('d')) . wp_kses_post($after);
			} elseif (is_month()) {
				echo '<a href="' . esc_url(get_year_link(get_the_time('Y'))) . '">' . esc_html(get_the_time('Y')) . '</a> '
					. wp_kses_post($delimiter) . ' ';
				echo wp_kses_post($before) . esc_html(get_the_time('F')) . wp_kses_post($after);
			} elseif (is_year()) {
				echo wp_kses_post($before) . esc_html(get_the_time('Y')) . wp_kses_post($after);
			} elseif (is_single() && !is_attachment()) {
				if (get_post_type() !== 'post') {
					$post_type = get_post_type_object(get_post_type());
					if (is_object($post_type) && isset($post_type->rewrite['slug'])) {
						echo '<a href="' . esc_url($homeLink . $post_type->rewrite['slug']) . '/">'
							. esc_html($post_type->labels->singular_name)
							. '</a>';
					} else {
						echo '<a href="' . esc_url($homeLink) . '">' . esc_html__('Custom Post', 'breadcrumbs-made-easy') . '</a>';
					}
					if ($showCurrent) {
						echo ' ' . wp_kses_post($delimiter) . ' ' . wp_kses_post($before) . esc_html(get_the_title()) . wp_kses_post($after);
					}
				} else {
					$cat = get_the_category();
					if (!empty($cat) && !is_wp_error($cat)) {
						$cat = $cat[0];
						$parents = get_category_parents($cat, true, ' ' . wp_kses_post($delimiter) . ' ');
						if (!is_wp_error($parents)) {
							if (!$showCurrent) {
								$parents = preg_replace("#^(.+)\s$delimiter\s$#", "$1", $parents);
							}
							echo wp_kses_post($parents);
						}
					}
					if ($showCurrent) {
						echo wp_kses_post($before) . esc_html(get_the_title()) . wp_kses_post($after);
					}
				}
			} elseif (!is_single() && !is_page() && get_post_type() !== 'post' && !is_404()) {
				$post_type = get_post_type_object(get_post_type());
				if (is_object($post_type) && isset($post_type->labels->name)) {
					/* echo '<pre>';
					print_r(get_queried_object());
					echo '</pre>'; */
					if (isset(get_queried_object()->name) && is_tax()) {
						echo '<a href="' . esc_url(get_post_type_archive_link(get_post_type())) . '">' . esc_html($post_type->labels->name) . '</a>' . ' ' . wp_kses_post($delimiter)  . wp_kses_post($before) . esc_html(get_queried_object()->name) . wp_kses_post($after);
					} else {
						echo wp_kses_post($before) . esc_html($post_type->labels->name) . wp_kses_post($after);
					}
				} else {
					echo wp_kses_post($before) . wp_kses_post($archive_label) . wp_kses_post($after);
				}
			} elseif (is_attachment()) {
				$parent = null;
				if (isset($post->post_parent)) {
					$parent = get_post($post->post_parent);
				}
				if ($parent && !is_wp_error($parent)) {
					$cat = get_the_category($parent->ID);
					if (!empty($cat) && !is_wp_error($cat)) {
						$cat = $cat[0];
						$parents = get_category_parents($cat, true, ' ' . wp_kses_post($delimiter) . ' ');
						if (!is_wp_error($parents)) {
							echo wp_kses_post($parents);
						}
					}
					echo '<a href="' . esc_url(get_permalink($parent)) . '">'
						. esc_html($parent->post_title) . '</a>';
					if ($showCurrent) {
						echo ' ' . wp_kses_post($delimiter) . ' ' . wp_kses_post($before) . esc_html(get_the_title()) . wp_kses_post($after);
					}
				} else {
					if ($showCurrent) {
						echo wp_kses_post($before) . esc_html(get_the_title()) . wp_kses_post($after);
					}
				}
			} elseif (is_page() && !$post->post_parent) {
				if ($showCurrent) {
					echo wp_kses_post($before) . esc_html(get_the_title()) . wp_kses_post($after);
				}
			} elseif (is_page() && $post->post_parent) {
				$parent_id = $post->post_parent;
				$breadcrumbs = array();
				while ($parent_id) {
					$page = get_page($parent_id);
					if (!is_wp_error($page) && $page) {
						$breadcrumbs[] = '<a href="' . esc_url(get_permalink($page->ID)) . '">'
							. esc_html(get_the_title($page->ID)) . '</a>';
						$parent_id = $page->post_parent;
					} else {
						break;
					}
				}
				$breadcrumbs = array_reverse($breadcrumbs);
				for ($i = 0; $i < count($breadcrumbs); $i++) {
					echo wp_kses_post($breadcrumbs[$i]);
					if ($i !== count($breadcrumbs) - 1) {
						echo ' ' . wp_kses_post($delimiter) . ' ';
					}
				}
				if ($showCurrent) {
					echo ' ' . wp_kses_post($delimiter) . ' ' . wp_kses_post($before) . esc_html(get_the_title()) . wp_kses_post($after);
				}
			} elseif (is_tag()) {
				echo wp_kses_post($before) . wp_kses_post($tag_label) . ' "' . esc_html(single_tag_title('', false)) . '"' . wp_kses_post($after);
			} elseif (is_author()) {
				global $author;
				$userdata = get_userdata($author);
				if ($userdata && !is_wp_error($userdata)) {
					echo esc_html($author_label) . wp_kses_post($delimiter) . ' ';
					echo wp_kses_post($before) . esc_html($userdata->display_name) . wp_kses_post($after);
				} else {
					echo wp_kses_post($before) . esc_html($author_label) . wp_kses_post($after);
				}
			} elseif (is_404()) {
				echo wp_kses_post($before) . wp_kses_post($error_label) . wp_kses_post($after);
			}

			echo '</div>'; // close #crumbs
		}
	}

	public function bmecp_output_breadcrumbs()
	{
		do_shortcode('[easy_breadcrumbs]');
	}

	public function bmecp_prepend_breadcrumbs($content, $post_id)
	{
		// Ensure it's only applied on singular pages or posts in the main loop
		if (
			is_singular() &&
			is_main_query() &&
			in_the_loop() &&
			($post_id === get_queried_object_id())
		) {
			remove_filter('the_title', [$this, 'bmecp_prepend_breadcrumbs'], 10, 2);
			// Prepend your custom breadcrumb shortcode output
			ob_start();
			do_shortcode('[easy_breadcrumbs]');
			$breadcrumbs = ob_get_clean();
			$content = $breadcrumbs . $content;

			add_filter('the_title', [$this, 'bmecp_prepend_breadcrumbs'], 10, 2);
		}
		return $content;
	}

	public function bmecp_append_breadcrumbs($content)
	{
		// Ensure it's only applied on singular pages or posts in the main loop
		if (
			is_singular() &&
			is_main_query() &&
			in_the_loop() &&
			(get_the_ID() === get_queried_object_id())
		) {
			remove_filter('the_title', [$this, 'bmecp_append_breadcrumbs']);
			// Prepend your custom breadcrumb shortcode output
			ob_start();
			do_shortcode('[easy_breadcrumbs]');
			$breadcrumbs = ob_get_clean();
			$content .= $breadcrumbs;

			add_filter('the_title', [$this, 'bmecp_append_breadcrumbs']);
		}
		return $content;
	}
}
