<?php
/**
 * Main Plugin Class for BrandismTech Popup Notification
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class BTPN_Popup_Notification {
    const VERSION      = '1.0.0';
    const OPTION_GROUP = 'btpn_settings_group';

    public function __construct() {
        add_action( 'admin_menu',               [ $this, 'add_admin_menu' ] );
        add_action( 'admin_init',               [ $this, 'register_settings' ] );
        add_action( 'wp_footer',                [ $this, 'display_popup' ] );
        add_action( 'admin_enqueue_scripts',    [ $this, 'enqueue_admin_assets' ] );
        add_action( 'wp_enqueue_scripts',       [ $this, 'enqueue_frontend_assets' ] );
        add_action( 'wp_logout',                [ $this, 'clear_popup_transient' ] );
    }

    public function enqueue_admin_assets( $hook ) {
        if ( 'toplevel_page_btpn-settings' !== $hook ) {
            return;
        }

        wp_enqueue_style(
            'btpn-admin-styles',
            BTPN_PLUGIN_URL . 'assets/css/admin-styles.css',
            [],
            self::VERSION
        );

        wp_enqueue_script(
            'btpn-admin-scripts',
            BTPN_PLUGIN_URL . 'assets/js/admin-scripts.js',
            [ 'jquery' ],
            self::VERSION,
            true
        );
    }

    public function enqueue_frontend_assets() {
        if ( ! $this->should_display_popup() ) {
            return;
        }

        wp_enqueue_style(
            'btpn-frontend-styles',
            BTPN_PLUGIN_URL . 'assets/css/frontend-styles.css',
            [],
            self::VERSION
        );

        wp_enqueue_script(
            'btpn-frontend-scripts',
            BTPN_PLUGIN_URL . 'assets/js/frontend-scripts.js',
            [ 'jquery' ],
            self::VERSION,
            true
        );
    }

    public function add_admin_menu() {
        add_menu_page(
            __( 'Popup Settings', 'brandismtech-popup-notification' ),
            __( 'Popup', 'brandismtech-popup-notification' ),
            'manage_options',
            'btpn-settings',
            [ $this, 'render_admin_page' ],
            'dashicons-megaphone',
            42
        );
    }

    public function register_settings() {
        register_setting( self::OPTION_GROUP, 'btpn_enabled', [
            'type'              => 'boolean',
            'default'           => true,
            'sanitize_callback' => [ $this, 'sanitize_checkbox' ],
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_url', [
            'type'              => 'string',
            'sanitize_callback' => [ $this, 'sanitize_url_path' ],
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_content', [
            'type'              => 'string',
            'sanitize_callback' => 'wp_kses_post',
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_start_datetime', [
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_end_datetime', [
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_show_frequency', [
            'type'              => 'string',
            'sanitize_callback' => [ $this, 'sanitize_show_frequency' ],
            'default'           => 'always',
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_custom_interval', [
            'type'              => 'integer',
            'sanitize_callback' => [ $this, 'sanitize_custom_interval' ],
            'default'           => 30,
        ] );

        register_setting( self::OPTION_GROUP, 'btpn_user_visibility', [
            'type'              => 'string',
            'sanitize_callback' => [ $this, 'sanitize_user_visibility' ],
            'default'           => 'logged_in',
        ] );
    }

    public function sanitize_checkbox( $input ) {
        return ( $input == 1 ? 1 : 0 );
    }

    public function sanitize_url_path( $input ) {
        $input = sanitize_text_field( $input );
        $input = trim( $input );

        if ( ! empty( $input ) && '/' !== $input[0] ) {
            $input = '/' . $input;
        }
        if ( ! empty( $input ) && '/' !== substr( $input, -1 ) ) {
            $input .= '/';
        }
        return $input;
    }

    public function sanitize_show_frequency( $input ) {
        $valid = [ 'always', 'once_per_session', 'custom_interval' ];
        return in_array( $input, $valid, true ) ? $input : 'always';
    }

    public function sanitize_custom_interval( $input ) {
        return max( 1, absint( $input ) );
    }

    public function sanitize_user_visibility( $input ) {
        $valid = [ 'logged_in', 'not_logged_in', 'both' ];
        return in_array( $input, $valid, true ) ? $input : 'logged_in';
    }

    public function clear_popup_transient() {
        $user_id = get_current_user_id();
        delete_transient( 'btpn_popup_shown_' . $user_id );
    }

    private function should_display_popup() {
        $visibility = get_option( 'btpn_user_visibility', 'logged_in' );
        $logged_in  = is_user_logged_in();

        if ( 'logged_in' === $visibility && ! $logged_in ) return false;
        if ( 'not_logged_in' === $visibility && $logged_in ) return false;

        if ( ! get_option( 'btpn_enabled', true ) ) return false;

        $url = get_option( 'btpn_url', '' );
        if ( empty( $url ) ) return false;

        $current = isset( $_SERVER['REQUEST_URI'] )
            ? trailingslashit( wp_parse_url( sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ), PHP_URL_PATH ) )
            : '';
        if ( trailingslashit( $url ) !== $current ) return false;

        $now = current_time( 'Y-m-d\TH:i' );
        $start = get_option( 'btpn_start_datetime', '' );
        $end   = get_option( 'btpn_end_datetime', '' );

        if ( $start && $now < $start ) return false;
        if ( $end && $now > $end ) return false;

        if ( empty( get_option( 'btpn_content', '' ) ) ) return false;

        if ( $logged_in ) {
            $freq      = get_option( 'btpn_show_frequency', 'always' );
            $user_id   = get_current_user_id();
            $transient = 'btpn_popup_shown_' . $user_id;

            if ( 'once_per_session' === $freq && get_transient( $transient ) ) return false;
            if ( 'custom_interval' === $freq && get_transient( $transient ) ) return false;

            if ( 'once_per_session' === $freq ) {
                set_transient( $transient, true, DAY_IN_SECONDS );
            } elseif ( 'custom_interval' === $freq ) {
                $minutes = get_option( 'btpn_custom_interval', 30 );
                set_transient( $transient, true, $minutes * MINUTE_IN_SECONDS );
            }
        }

        return true;
    }

    public function display_popup() {
        if ( ! $this->should_display_popup() ) return;

        $content       = get_option( 'btpn_content', '' );
        $content_clean = str_replace( "\xC2\xA0", ' ', $content );

        include BTPN_PLUGIN_DIR . 'templates/popup-frontend.php';
    }

    public function render_admin_page() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        $enabled          = get_option( 'btpn_enabled', true );
        $url              = get_option( 'btpn_url', '' );
        $content          = get_option( 'btpn_content', '' );
        $start_datetime   = get_option( 'btpn_start_datetime', '' );
        $end_datetime     = get_option( 'btpn_end_datetime', '' );
        $show_frequency   = get_option( 'btpn_show_frequency', 'always' );
        $custom_interval  = get_option( 'btpn_custom_interval', 30 );
        $user_visibility  = get_option( 'btpn_user_visibility', 'logged_in' );

        include BTPN_PLUGIN_DIR . 'templates/admin-page.php';
    }
}