// js/main.js

// --- Utility Functions for PCM to WAV Conversion ---

/** Converts Base64 audio data string to an ArrayBuffer. */
function base64ToArrayBuffer(base64) {
    const binaryString = atob(base64);
    const len = binaryString.length;
    const bytes = new Uint8Array(len);
    for (let i = 0; i < len; i++) {
        bytes[i] = binaryString.charCodeAt(i);
    }
    return bytes.buffer;
}

/** Writes a 32-bit integer to the DataView at the specified offset. */
function writeInt32(dataview, offset, value) {
    dataview.setUint32(offset, value, true); // true for little-endian
}

/** Writes a 16-bit integer to the DataView at the specified offset. */
function writeInt16(dataview, offset, value) {
    dataview.setUint16(offset, value, true); // true for little-endian
}

/** Writes a string (char codes) to the DataView at the specified offset. */
function writeString(dataview, offset, string) {
    for (let i = 0; i < string.length; i++) {
        dataview.setUint8(offset + i, string.charCodeAt(i));
    }
}

/** Converts raw PCM 16-bit audio data into a playable WAV Blob. */
function pcmToWav(pcm16, sampleRate) {
    const numChannels = 1;
    const buffer = new ArrayBuffer(44 + pcm16.byteLength);
    const view = new DataView(buffer);

    // RIFF chunk descriptor
    writeString(view, 0, 'RIFF');
    writeInt32(view, 4, 36 + pcm16.byteLength); // Chunk size
    writeString(view, 8, 'WAVE');

    // FMT sub-chunk
    writeString(view, 12, 'fmt ');
    writeInt32(view, 16, 16); // Sub-chunk 1 size (16 for PCM)
    writeInt16(view, 20, 1);  // Audio format (1 for PCM)
    writeInt16(view, 22, numChannels); // Number of channels
    writeInt32(view, 24, sampleRate); // Sample rate
    writeInt32(view, 28, sampleRate * numChannels * 2); // Byte rate
    writeInt16(view, 32, numChannels * 2); // Block align
    writeInt16(view, 34, 16); // Bits per sample

    // DATA sub-chunk
    writeString(view, 36, 'data');
    writeInt32(view, 40, pcm16.byteLength); // Sub-chunk 2 size (data size)

    // Write the PCM data itself
    let offset = 44;
    const pcmBytes = new Uint8Array(pcm16.buffer);
    for (let i = 0; i < pcmBytes.length; i++, offset++) {
        view.setUint8(offset, pcmBytes[i]);
    }

    return new Blob([view], { type: 'audio/wav' });
}

// --- Core Application Logic ---

let isGenerating = false;

async function generateVoice() {
    if (isGenerating) return;

    // Get UI elements
    const textInput = document.getElementById('text-input').value.trim();
    const voiceName = document.getElementById('voice-select').value;
    const useSsml = document.getElementById('ssml-toggle').checked;
    const rate = parseFloat(document.getElementById('rate-slider').value);
    const pitch = parseFloat(document.getElementById('pitch-slider').value);
    const audioPlayer = document.getElementById('audio-player');
    const loadingIndicator = document.getElementById('loading-indicator');
    const errorMessage = document.getElementById('error-message');
    const downloadContainer = document.getElementById('download-container');
    const generateBtn = document.getElementById('generate-btn');

    if (textInput.length === 0) {
        errorMessage.textContent = 'Please enter text to convert.';
        return;
    }

    // Reset UI
    isGenerating = true;
    generateBtn.disabled = true;
    errorMessage.textContent = '';
    audioPlayer.src = '';
    audioPlayer.style.display = 'none';
    loadingIndicator.style.display = 'flex';
    downloadContainer.innerHTML = '';

    // Clear any previous Pro buttons (handled by Pro script)
    document.dispatchEvent(new CustomEvent('brand_voice_generator_bvgbmo:resetUI'));

    const formData = new FormData();
    formData.append('action', 'brand_voice_generator_bvgbmo_proxy');
    formData.append('nonce', brand_voice_generator_bvgbmo_data.nonce);
    formData.append('text', textInput);
    formData.append('voice', voiceName);
    formData.append('ssml', useSsml);
    formData.append('rate', rate);
    formData.append('pitch', pitch);

    try {
        const response = await fetch(brand_voice_generator_bvgbmo_data.ajax_url, {
            method: 'POST',
            body: formData,
        });

        const result = await response.json();

        if (!response.ok || result.success === false) {
            throw new Error(result.data.message || 'An unknown error occurred.');
        }

        const audioData = result.data.audioData;
        const mimeType = result.data.mimeType;

        if (!audioData || !mimeType || !mimeType.startsWith("audio/L16")) {
            throw new Error('Failed to receive valid audio data from the API.');
        }

        const rateMatch = mimeType.match(/rate=(\d+)/);
        const sampleRate = rateMatch ? parseInt(rateMatch[1], 10) : 24000;

        const pcmData = base64ToArrayBuffer(audioData);
        const pcm16 = new Int16Array(pcmData);
        const wavBlob = pcmToWav(pcm16, sampleRate);

        const audioUrl = URL.createObjectURL(wavBlob);
        audioPlayer.src = audioUrl;
        audioPlayer.style.display = 'block';
        audioPlayer.play();

        // Create download button (Standard Free Feature)
        const downloadLink = document.createElement('a');
        downloadLink.href = audioUrl;
        downloadLink.download = `voiceover-${Date.now()}.wav`;
        downloadLink.textContent = 'Download WAV';
        downloadLink.className = "ml-4 inline-flex items-center bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-lg shadow transition duration-200";
        downloadContainer.appendChild(downloadLink);

        // --- Dispatch Event for Pro Script ---
        const event = new CustomEvent('brand_voice_generator_bvgbmo:voiceGenerated', {
            detail: {
                audioData: audioData, // Base64 Raw Data
                textInput: textInput,
                container: downloadContainer
            }
        });
        document.dispatchEvent(event);

    } catch (e) {
        errorMessage.textContent = `Error: ${e.message}`;
        console.error('Generation failed:', e);
    } finally {
        loadingIndicator.style.display = 'none';
        isGenerating = false;
        generateBtn.disabled = false;
    }
}

document.addEventListener('DOMContentLoaded', function() {
    if (document.getElementById('generate-btn')) {
        const generateBtn = document.getElementById('generate-btn');
        generateBtn.addEventListener('click', generateVoice);

        const rateSlider = document.getElementById('rate-slider');
        const pitchSlider = document.getElementById('pitch-slider');

        if (rateSlider && document.getElementById('rate-value')) {
            document.getElementById('rate-value').textContent = parseFloat(rateSlider.value).toFixed(1);
            rateSlider.addEventListener('input', e => document.getElementById('rate-value').textContent = parseFloat(e.target.value).toFixed(1));
        }

        if (pitchSlider && document.getElementById('pitch-value')) {
             document.getElementById('pitch-value').textContent = parseInt(pitchSlider.value);
            pitchSlider.addEventListener('input', e => document.getElementById('pitch-value').textContent = parseInt(e.target.value));
        }
    }
});
