<?php
/**
 * Plugin Name:       Brand Voice Generator
 * Description:       Integrates the Gemini TTS API into a private page for team voice-over generation.
 * Version:           1.0.6
 * Author:            Michael Ogolor
 * Author URI:        https://shorturl.at/wyUYh
 * LinkedIn:          https://linkedin.com/in/michael-ogolor
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       brand-voice-generator
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// ----------------------------------------------------------------------
// 1. FREEMIUS INTEGRATION
// ----------------------------------------------------------------------

if ( ! function_exists( 'brand_voice_generator_bvgbmo_fs' ) ) {
    /**
     * Create a helper function for easy SDK access.
     *
     * @return Freemius
     */
    function brand_voice_generator_bvgbmo_fs() {
        global $brand_voice_generator_bvgbmo_fs;

        if ( ! isset( $brand_voice_generator_bvgbmo_fs ) ) {
            // SAFEGUARD: Check if the file exists before requiring to avoid fatal errors.
            $fs_path = dirname( __FILE__ ) . '/vendor/freemius/start.php';
            if ( ! file_exists( $fs_path ) ) {
                return null; // Return null if SDK is missing.
            }

            require_once $fs_path;

            $brand_voice_generator_bvgbmo_fs = fs_dynamic_init( array(
                'id'                  => '21966',
                'slug'                => 'brand-voice-generator',
                'type'                => 'plugin',
                'public_key'          => 'pk_1ae0a2ec7041db9ae617da1b7bf3d',
                'is_premium'          => false, // FALSE for Repo version
                'premium_suffix'      => 'Pro',
                'has_premium_version' => true,
                'has_addons'          => false,
                'has_paid_plans'      => true,
                'is_org_compliant'    => true,
                'menu'                => array(
                    'slug'           => 'brand-voice-generator',
                    'first-path'     => 'admin.php?page=brand-voice-generator',
                    'support'        => false,
                    'pricing'        => true,
                    'account'        => true,
                    'contact'        => true,
                ),
            ) );
        }
        return $brand_voice_generator_bvgbmo_fs;
    }

    brand_voice_generator_bvgbmo_fs();
    do_action( 'brand_voice_generator_bvgbmo_fs_loaded' );
}

/**
 * Helper function to check for Pro features safely.
 */
function brand_voice_generator_bvgbmo_is_pro() {
    $fs = brand_voice_generator_bvgbmo_fs();
    // Check if FS is loaded AND is an object.
    if ( $fs && is_object( $fs ) ) {
        // Returns true if the user has a valid license for ANY paid plan (or active trial).
        return $fs->can_use_premium_code();
    }
    return false;
}

// ----------------------------------------------------------------------
// 2. MAIN PLUGIN CLASS
// ----------------------------------------------------------------------

class BrandVoiceGenerator {

    function __construct() {
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_shortcode('brand_voice_generator', [$this, 'render_shortcode']);

        // AJAX Hook
        add_action('wp_ajax_brand_voice_generator_bvgbmo_proxy', [$this, 'handle_ajax_proxy']);

        // Load Pro Module if available
        if ( file_exists( plugin_dir_path( __FILE__ ) . 'pro/class-bvg-pro.php' ) ) {
            require_once plugin_dir_path( __FILE__ ) . 'pro/class-bvg-pro.php';
        }
    }

    public function add_admin_menu() {
        // Register Top Level Menu
        add_menu_page(
            __('Brand Voice', 'brand-voice-generator'),
            __('Brand Voice', 'brand-voice-generator'),
            'manage_options',
            'brand-voice-generator',
            [$this, 'settings_page_html'],
            'dashicons-microphone',
            25
        );

        // Rename Submenu
        add_submenu_page(
            'brand-voice-generator',
            __('Generator', 'brand-voice-generator'),
            __('Generator', 'brand-voice-generator'),
            'manage_options',
            'brand-voice-generator',
            [$this, 'settings_page_html']
        );
    }

    public function register_settings() {
        register_setting('brand_voice_generator_bvgbmo_settings_group', 'brand_voice_generator_bvgbmo_api_key', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('brand_voice_generator_bvgbmo_settings_group', 'brand_voice_generator_bvgbmo_allowed_roles', ['sanitize_callback' => [$this, 'sanitize_array_map']]);
    }

    public function sanitize_array_map($input) {
        $safe_values = [];
        if (is_array($input)) {
            foreach ($input as $value) {
                $safe_values[] = sanitize_text_field($value);
            }
        }
        return $safe_values;
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_brand-voice-generator') {
            return;
        }

        wp_enqueue_style(
            'bvg-admin-style',
            plugin_dir_url(__FILE__) . 'css/admin.css',
            [],
            '1.0.6'
        );
    }

    public function settings_page_html() {
        if (!current_user_can('manage_options')) return;

        $is_pro = brand_voice_generator_bvgbmo_is_pro();
        $fs = brand_voice_generator_bvgbmo_fs();
        ?>
        <div class="wrap bvg-admin-wrapper">
            <div class="bvg-admin-header">
                <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
                <span class="bvg-version-badge">v1.0.6</span>
            </div>

            <form action="options.php" method="post" class="bvg-admin-form">
                <?php
                settings_fields('brand_voice_generator_bvgbmo_settings_group');
                do_settings_sections('brand_voice_generator_bvgbmo_settings_group');
                ?>

                <div class="bvg-admin-section">
                    <h2 class="bvg-section-title"><?php esc_html_e('🔌 API Configuration', 'brand-voice-generator'); ?></h2>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label for="brand_voice_generator_bvgbmo_api_key"><?php esc_html_e('Google API Key', 'brand-voice-generator'); ?></label></th>
                            <td><input type="text" id="brand_voice_generator_bvgbmo_api_key" name="brand_voice_generator_bvgbmo_api_key" value="<?php echo esc_attr(get_option('brand_voice_generator_bvgbmo_api_key')); ?>" class="regular-text" placeholder="AIza..." />
                                <p class="description"><?php esc_html_e('Enter the API key for your brand\'s Google Cloud account.', 'brand-voice-generator'); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="bvg-admin-section">
                    <h2 class="bvg-section-title"><?php esc_html_e('👥 Access Control', 'brand-voice-generator'); ?></h2>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label for="brand_voice_generator_bvgbmo_allowed_roles"><?php esc_html_e('Allowed User Roles', 'brand-voice-generator'); ?></label></th>
                            <td>
                                <div class="bvg-roles-grid">
                                    <?php global $wp_roles; $roles = $wp_roles->roles; ?>
                                    <?php foreach ($roles as $role => $details): ?>
                                        <label class="bvg-role-item">
                                            <input type="checkbox" name="brand_voice_generator_bvgbmo_allowed_roles[]" value="<?php echo esc_attr($role); ?>" <?php checked(in_array($role, (array)get_option('brand_voice_generator_bvgbmo_allowed_roles', []))); ?>>
                                            <?php echo esc_html(translate_user_role($details['name'])); ?>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                                <p class="description"><?php esc_html_e('Select which user roles can access the generator.', 'brand-voice-generator'); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <?php do_action( 'brand_voice_generator_bvgbmo_settings_after_roles' ); ?>

                <?php if ( ! $is_pro ): ?>
                    <div class="bvg-admin-notice bvg-info">
                        <div class="bvg-notice-content">
                            <h3><?php esc_html_e('Pro Feature: Workflow Tools', 'brand-voice-generator'); ?></h3>
                            <p>
                                <?php esc_html_e('Need to save your audio files? Upgrade to Pro to unlock the Generation History, Media Library saving, and data management tools.', 'brand-voice-generator'); ?>
                            </p>
                        </div>
                        <div class="bvg-notice-action">
                            <?php
                            if ( $fs && is_object( $fs ) && method_exists( $fs, 'get_upgrade_url' ) ) :
                            ?>
                                <a href="<?php echo esc_url($fs->get_upgrade_url()); ?>" target="_blank" class="button button-primary bvg-upgrade-btn"><?php esc_html_e('Upgrade to Pro', 'brand-voice-generator'); ?></a>
                            <?php else : ?>
                                <a href="#" class="button button-primary" disabled><?php esc_html_e('Upgrade to Pro', 'brand-voice-generator'); ?></a>
                                <br><small><em>Error: Licensing SDK not loaded. Please check your vendor folder.</em></small>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="bvg-submit-wrapper">
                    <?php submit_button(esc_html__('Save Settings', 'brand-voice-generator'), 'primary large'); ?>
                </div>
            </form>
        </div>
        <?php
    }

    public function enqueue_assets() {
        if (is_singular() && has_shortcode(get_post(get_the_ID())->post_content, 'brand_voice_generator')) {
            $api_key = get_option('brand_voice_generator_bvgbmo_api_key');
            if (empty($api_key)) return;

            wp_enqueue_style( 'bvg-google-fonts', 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&family=Outfit:wght@600;700;800&display=swap', [], '1.0.6' );
            wp_enqueue_style( 'bvg-style', plugin_dir_url(__FILE__) . 'css/style.css', [], '1.0.6' );

            // Main JS
            wp_enqueue_script( 'bvg-main-js', plugin_dir_url(__FILE__) . 'js/main.js', ['jquery'], '1.0.6', true );

            // Pro JS (Only if file exists)
            if ( brand_voice_generator_bvgbmo_is_pro() && file_exists( plugin_dir_path( __FILE__ ) . 'js/main-pro.js' ) ) {
                wp_enqueue_script( 'bvg-main-pro-js', plugin_dir_url(__FILE__) . 'js/main-pro.js', ['bvg-main-js'], '1.0.6', true );
            }

            $is_pro = brand_voice_generator_bvgbmo_is_pro();

            wp_localize_script('bvg-main-js', 'brand_voice_generator_bvgbmo_data', [
                'ajax_url'      => admin_url('admin-ajax.php'),
                'nonce'         => wp_create_nonce('brand_voice_generator_bvgbmo_nonce'),
                'media_nonce'   => $is_pro ? wp_create_nonce('brand_voice_generator_bvgbmo_media_nonce') : '',
                'is_pro'        => $is_pro
            ]);
        }
    }

    public function render_shortcode() {
        $allowed_roles = (array) get_option('brand_voice_generator_bvgbmo_allowed_roles', ['administrator', 'editor']);
        $current_user_roles = wp_get_current_user()->roles;

        if ( ! is_user_logged_in() || ! array_intersect($allowed_roles, $current_user_roles) ) {
            return '<p>' . esc_html__('You do not have permission to use this tool.', 'brand-voice-generator') . '</p>';
        }

        ob_start();
        include(plugin_dir_path(__FILE__) . 'templates/generator-page.php');
        return ob_get_clean();
    }

    public function handle_ajax_proxy() {
        check_ajax_referer('brand_voice_generator_bvgbmo_nonce', 'nonce');

        $allowed_roles = (array) get_option('brand_voice_generator_bvgbmo_allowed_roles', ['administrator', 'editor']);
        $current_user = wp_get_current_user();
        if ( ! array_intersect($allowed_roles, $current_user->roles) ) {
            wp_send_json_error(['message' => esc_html__('Unauthorized: You do not have permission to use this tool.', 'brand-voice-generator')]);
            return;
        }

        // 1. Hook for Pro Plugin to enforce Custom Role Limits (Allowed in Pro versions)
        $pro_validation = apply_filters( 'brand_voice_generator_bvgbmo_validate_generation_limit', null, $current_user );
        if ( is_wp_error( $pro_validation ) ) {
            wp_send_json_error(['message' => $pro_validation->get_error_message()]);
            return;
        }

        $api_key = get_option('brand_voice_generator_bvgbmo_api_key');
        if (empty($api_key)) {
            wp_send_json_error(['message' => esc_html__('API Key is not configured.', 'brand-voice-generator')]);
        }

        $text_input = isset($_POST['text']) ? sanitize_textarea_field(wp_unslash($_POST['text'])) : '';
        $voice_name = isset($_POST['voice']) ? sanitize_text_field(wp_unslash($_POST['voice'])) : 'Kore';
        $raw_ssml   = isset($_POST['ssml']) ? sanitize_text_field(wp_unslash($_POST['ssml'])) : 'false';
        $use_ssml   = filter_var($raw_ssml, FILTER_VALIDATE_BOOLEAN);
        $rate       = isset($_POST['rate']) ? floatval(wp_unslash($_POST['rate'])) : 1.0;
        $pitch      = isset($_POST['pitch']) ? floatval(wp_unslash($_POST['pitch'])) : 0.0;

        if (empty($text_input)) {
            wp_send_json_error(['message' => esc_html__('Text input cannot be empty.', 'brand-voice-generator')]);
        }

        // --- Gemini TTS API Call ---
        $final_prompt = $use_ssml ? "<speak>{$text_input}</speak>" : $text_input;
        $model_id = 'gemini-2.5-flash-preview-tts';
        $api_url = "https://generativelanguage.googleapis.com/v1beta/models/{$model_id}:generateContent?key=" . $api_key;

        $speech_config = [
            'voiceConfig' => ['prebuiltVoiceConfig' => ['voiceName' => $voice_name]],
        ];

        if ($rate !== 1.0) $speech_config['speakingRate'] = $rate;
        if ($pitch !== 0.0) $speech_config['pitch'] = $pitch;

        $payload = [
            'contents' => [['parts' => [['text' => $final_prompt]]]],
            'generationConfig' => [
                'responseModalities' => ["AUDIO"],
                'speechConfig' => $speech_config,
            ],
            'model' => $model_id
        ];

        // --- Execute API Call ---
        $response = wp_remote_post($api_url, [
            'method'    => 'POST',
            'headers'   => ['Content-Type' => 'application/json'],
            'body'      => json_encode($payload),
            'timeout'   => 45,
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => esc_html__('Failed to connect to Google API: ', 'brand-voice-generator') . esc_html($response->get_error_message())]);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (isset($data['error'])) {
            $error_msg = isset($data['error']['message']) ? $data['error']['message'] : 'Unknown API Error';
            wp_send_json_error(['message' => esc_html__('API Error: ', 'brand-voice-generator') . esc_html($error_msg)]);
        }

        $audio_data = $data['candidates'][0]['content']['parts'][0]['inlineData']['data'] ?? null;
        $mime_type = $data['candidates'][0]['content']['parts'][0]['inlineData']['mimeType'] ?? null;

        if (!$audio_data || !$mime_type) {
            wp_send_json_error(['message' => esc_html__('No audio returned. The model may have refused the request.', 'brand-voice-generator')]);
        }

        // Pass data for Pro logging if active (via hook)
        do_action( 'brand_voice_generator_bvgbmo_generation_success', get_current_user_id(), $text_input, $voice_name, $use_ssml, $audio_data );

        wp_send_json_success([
            'audioData' => $audio_data,
            'mimeType'  => $mime_type
        ]);
    }
}

new BrandVoiceGenerator();

/**
 * Cleanup function on uninstall
 */
function brand_voice_generator_bvgbmo_uninstall_cleanup() {
    if ( ! function_exists('brand_voice_generator_bvgbmo_fs') || ! brand_voice_generator_bvgbmo_fs()->is_paying() ) {
        delete_option( 'brand_voice_generator_bvgbmo_api_key' );
        delete_option( 'brand_voice_generator_bvgbmo_allowed_roles' );
        delete_option( 'brand_voice_generator_bvgbmo_role_settings' );
        delete_option( 'brand_voice_generator_bvgbmo_scrubber_days' );
    }
}

$fs_global = brand_voice_generator_bvgbmo_fs();
if ( $fs_global && is_object( $fs_global ) && method_exists( $fs_global, 'add_action' ) ) {
    $fs_global->add_action('after_uninstall', 'brand_voice_generator_bvgbmo_uninstall_cleanup');
}
