<?php
/**
 * Plugin Name: BotSubmit
 * Description: Automatically submits URLs to multiple indexing services (SpeedyIndex, Link Indexing Bot, IndexBotik) for fast search engine indexing. Supports Google, Yandex, and Bing.
 * Version: 1.2.0
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * Author: Наумов про Affiliate SEO
 * Author URI: https://t.me/naumov_top
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: botsubmit
 */

if ( ! defined('ABSPATH') ) exit;

class BotSubmit {
    const OPTION_API_KEY     = 'botsubmit_api_key';
    const OPTION_GLOBAL_LOG  = 'botsubmit_global_log';
    const OPTION_SEND_TYPES  = 'botsubmit_send_types'; // ['posts'=>1,'pages'=>0,'categories'=>0,'tags'=>0]
    const OPTION_SEND_UPDATE = 'botsubmit_send_update'; // при редактировании
    const OPTION_SERVICES    = 'botsubmit_services'; // выбранные сервисы ['speedyindex'=>1,'linkindexing'=>0,'indexbotik'=>0]
    const OPTION_API_KEYS    = 'botsubmit_api_keys'; // API ключи для каждого сервиса
    const OPTION_SERVICE_PARAMS = 'botsubmit_service_params'; // дополнительные параметры для сервисов
    
    const MAX_LOG_ROWS       = 1000;
    const LOG_PER_PAGE       = 50;
    
    // API URLs для разных сервисов
    const API_URL_SPEEDYINDEX = 'https://api.speedyindex.com/v2/google/url';
    const API_URL_LINKINDEXING = 'https://link-indexing-bot.ru/api/tasks/new';
    const API_URL_INDEXBOTIK = 'https://ivoque.de/api/link-submit.php';


    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'on_activate']);
        register_deactivation_hook(__FILE__, [$this, 'on_deactivate']);

        // Страница настроек в Tools
        add_action('admin_menu', [$this, 'add_settings_page']);

        // Публикация постов/страниц/CPT (только первая публикация)
        add_action('transition_post_status', [$this, 'handle_post_publish'], 10, 3);

        // Таксономии: категории / теги (архивы терминов при создании/редактировании)
        add_action('created_category',  [$this, 'handle_created_category'],  10, 3);
        add_action('edited_category',   [$this, 'handle_edited_category'],   10, 3);
        add_action('created_post_tag',  [$this, 'handle_created_post_tag'],  10, 3);
        add_action('edited_post_tag',   [$this, 'handle_edited_post_tag'],   10, 3);

        // Ссылка "Settings" в списке плагинов
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_action_links']);
		// Редактирование опубликованных постов/страниц
		add_action('post_updated', [$this, 'handle_updated_post'], 10, 3);
		
		// Bulk Actions для массовой отправки
		add_filter('bulk_actions-edit-post', [$this, 'register_bulk_action']);
		add_filter('bulk_actions-edit-page', [$this, 'register_bulk_action']);
		add_filter('handle_bulk_actions-edit-post', [$this, 'handle_bulk_action'], 10, 3);
		add_filter('handle_bulk_actions-edit-page', [$this, 'handle_bulk_action'], 10, 3);
		
		// Добавляем bulk actions для custom post types
		add_action('admin_init', [$this, 'register_cpt_bulk_actions']);
		
		add_action('admin_notices', [$this, 'bulk_action_admin_notice']);
		
		// AJAX для повторной отправки из лога
		add_action('wp_ajax_botsubmit_resend_url', [$this, 'ajax_resend_url']);
		
		// Мета-бокс в редакторе постов
		add_action('add_meta_boxes', [$this, 'add_post_meta_box']);
		add_action('save_post', [$this, 'save_post_meta_box'], 10, 1);
		
		// Admin scripts
		add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
		
		// WP Cron для фоновой отправки
		add_action('botsubmit_send_url_cron', [$this, 'process_cron_send'], 10, 2);

    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our settings page
        if ($hook !== 'tools_page_botsubmit-settings') {
            return;
        }
        
        wp_enqueue_script('jquery');
        
        // Custom CSS styles
        $custom_css = "
            /* BotSubmit Custom Styles */
            :root {
                --botsubmit-bg: #334354;
                --botsubmit-accent: #f27f26;
                --botsubmit-text: #fff;
                --botsubmit-muted: #a0a8ab;
            }
            
            /* Header */
            .botsubmit-header {
                background: var(--botsubmit-bg);
                padding: 20px 25px;
                margin: -10px -20px 20px -22px;
            }
            .botsubmit-header h1 {
                color: var(--botsubmit-text);
                margin: 0;
                font-size: 28px;
                display: flex;
                align-items: center;
                gap: 12px;
            }
            .botsubmit-header h1 .botsubmit-logo {
                color: var(--botsubmit-accent);
            }
            
            /* Tabs */
            .botsubmit-tabs {
                background: var(--botsubmit-bg);
                padding: 0 25px;
                margin: -20px -20px 25px -22px;
                border-bottom: 3px solid var(--botsubmit-accent);
            }
            .botsubmit-tabs .nav-tab {
                background: transparent;
                border: none;
                color: var(--botsubmit-muted);
                padding: 15px 20px;
                font-size: 14px;
                font-weight: 500;
                margin: 0;
                transition: all 0.2s ease;
            }
            .botsubmit-tabs .nav-tab:hover {
                color: var(--botsubmit-text);
                background: rgba(255,255,255,0.05);
            }
            .botsubmit-tabs .nav-tab-active,
            .botsubmit-tabs .nav-tab-active:hover {
                background: var(--botsubmit-accent);
                color: var(--botsubmit-text);
                border-radius: 6px 6px 0 0;
            }
            
            /* Service cards */
            .botsubmit-service-card {
                background: #fff;
                border: 1px solid #e0e0e0;
                border-radius: 8px;
                padding: 20px;
                margin-bottom: 15px;
                transition: all 0.2s ease;
                border-left: 4px solid transparent;
            }
            .botsubmit-service-card:hover {
                border-left-color: var(--botsubmit-accent);
                box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            }
            .botsubmit-service-card.active {
                border-left-color: var(--botsubmit-accent);
                background: #fdf8f4;
            }
            .botsubmit-service-card .service-header {
                display: flex;
                align-items: center;
                gap: 12px;
            }
            .botsubmit-service-card .service-header label {
                font-size: 16px;
                font-weight: 600;
                color: var(--botsubmit-bg);
                cursor: pointer;
            }
            .botsubmit-service-card .service-link {
                color: var(--botsubmit-accent);
                text-decoration: none;
                font-size: 13px;
                font-weight: 500;
            }
            .botsubmit-service-card .service-link:hover {
                text-decoration: underline;
            }
            
            /* Form inputs */
            .botsubmit-input {
                border: 2px solid #e0e0e0;
                border-radius: 6px;
                padding: 10px 14px;
                font-size: 14px;
                transition: border-color 0.2s ease;
            }
            .botsubmit-input:focus {
                border-color: var(--botsubmit-accent);
                outline: none;
                box-shadow: 0 0 0 3px rgba(242, 127, 38, 0.15);
            }
            
            /* Buttons */
            .botsubmit-btn {
                background: var(--botsubmit-accent);
                color: var(--botsubmit-text);
                border: none;
                padding: 12px 24px;
                font-size: 14px;
                font-weight: 600;
                border-radius: 6px;
                cursor: pointer;
                transition: all 0.2s ease;
            }
            .botsubmit-btn:hover {
                background: #e06f1a;
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(242, 127, 38, 0.3);
            }
            .botsubmit-btn-secondary {
                background: var(--botsubmit-bg);
            }
            .botsubmit-btn-secondary:hover {
                background: #2a3744;
            }
            
            /* Section headers */
            .botsubmit-section-title {
                color: var(--botsubmit-bg);
                font-size: 18px;
                font-weight: 600;
                margin: 30px 0 15px 0;
                padding-bottom: 10px;
                border-bottom: 2px solid var(--botsubmit-accent);
                display: inline-block;
            }
            
            /* Checkboxes styling */
            .botsubmit-checkbox-group {
                background: #f8f9fa;
                padding: 15px 20px;
                border-radius: 8px;
                border: 1px solid #e0e0e0;
            }
            .botsubmit-checkbox-group input[type=\"checkbox\"],
            .botsubmit-service-card input[type=\"checkbox\"] {
                margin: 0;
            }
            .botsubmit-checkbox-group label {
                display: flex;
                align-items: center;
                gap: 8px;
                padding: 8px 0;
                cursor: pointer;
                transition: color 0.2s ease;
            }
            .botsubmit-checkbox-group label:hover {
                color: var(--botsubmit-accent);
            }
            
            /* Log table */
            .botsubmit-log-table {
                border-collapse: separate;
                border-spacing: 0;
                border-radius: 8px;
                overflow: hidden;
                box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            }
            .botsubmit-log-table thead th {
                background: var(--botsubmit-bg);
                color: var(--botsubmit-text) !important;
                padding: 14px 16px;
                font-weight: 600;
                text-align: left;
                border: none;
            }
            .botsubmit-log-table tbody tr {
                transition: background 0.2s ease;
            }
            .botsubmit-log-table tbody tr:hover {
                background: #f8f9fa;
            }
            .botsubmit-log-table tbody td {
                padding: 12px 16px;
                border-bottom: 1px solid #e0e0e0;
            }
            .botsubmit-log-table tbody tr:last-child td {
                border-bottom: none;
            }
            .botsubmit-log-table .log-success {
                background: #f0fdf4;
            }
            .botsubmit-log-table .log-error {
                background: #fef2f2;
            }
            
            /* Status badges */
            .botsubmit-badge {
                display: inline-block;
                padding: 4px 10px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: 600;
            }
            .botsubmit-badge-success {
                background: #dcfce7;
                color: #166534;
            }
            .botsubmit-badge-error {
                background: #fee2e2;
                color: #991b1b;
            }
            
            /* Footer */
            .botsubmit-footer {
                background: var(--botsubmit-bg);
                padding: 20px 25px;
                margin: 40px -20px -20px -22px;
                text-align: center;
            }
            .botsubmit-footer a {
                color: var(--botsubmit-accent);
                text-decoration: none;
                font-weight: 600;
            }
            .botsubmit-footer a:hover {
                text-decoration: underline;
            }
            .botsubmit-footer p {
                color: var(--botsubmit-muted);
                margin: 0;
            }
            
            /* Resend button */
            .botsubmit-resend-btn {
                background: var(--botsubmit-accent);
                color: var(--botsubmit-text);
                border: none;
                padding: 6px 12px;
                font-size: 12px;
                border-radius: 4px;
                cursor: pointer;
                transition: all 0.2s ease;
            }
            .botsubmit-resend-btn:hover {
                background: #e06f1a;
            }
            
            /* Clear log button */
            .botsubmit-clear-btn {
                background: transparent;
                color: var(--botsubmit-bg);
                border: 2px solid var(--botsubmit-bg);
                padding: 10px 20px;
                font-size: 14px;
                font-weight: 500;
                border-radius: 6px;
                cursor: pointer;
                transition: all 0.2s ease;
            }
            .botsubmit-clear-btn:hover {
                background: var(--botsubmit-bg);
                color: var(--botsubmit-text);
            }
        ";
        
        wp_register_style('botsubmit-admin-styles', false, array(), '1.1.0');
        wp_enqueue_style('botsubmit-admin-styles');
        wp_add_inline_style('botsubmit-admin-styles', $custom_css);
        
        // Settings tab script
        $settings_script = "
            jQuery(document).ready(function($) {
                function toggleServiceFields(serviceId) {
                    var checkbox = $('input[name=\"botsubmit_services[' + serviceId + ']\"]');
                    var card = checkbox.closest('.botsubmit-service-card');
                    var fieldsContainer = card.find('.botsubmit-service-fields');
                    var apiKeyInput = card.find('input[name=\"botsubmit_api_keys[' + serviceId + ']\"]');
                    
                    if (checkbox.is(':checked')) {
                        fieldsContainer.slideDown(200);
                        card.addClass('active');
                        apiKeyInput.attr('required', true);
                    } else {
                        fieldsContainer.slideUp(200);
                        card.removeClass('active');
                        apiKeyInput.attr('required', false);
                    }
                }
                
                $('input[name^=\"botsubmit_services[\"]').each(function() {
                    var match = $(this).attr('name').match(/botsubmit_services\[([^\]]+)\]/);
                    if (match) {
                        toggleServiceFields(match[1]);
                    }
                });
                
                $('input[name^=\"botsubmit_services[\"]').on('change', function() {
                    var match = $(this).attr('name').match(/botsubmit_services\[([^\]]+)\]/);
                    if (match) {
                        toggleServiceFields(match[1]);
                    }
                });
                
                // Form validation
                $('form').on('submit', function(e) {
                    var hasError = false;
                    $('input[name^=\"botsubmit_services[\"]').each(function() {
                        if ($(this).is(':checked')) {
                            var match = $(this).attr('name').match(/botsubmit_services\[([^\]]+)\]/);
                            if (match) {
                                var apiKeyInput = $('input[name=\"botsubmit_api_keys[' + match[1] + ']\"]');
                                if (!apiKeyInput.val().trim()) {
                                    apiKeyInput.css('border-color', '#f27f26');
                                    hasError = true;
                                } else {
                                    apiKeyInput.css('border-color', '#e0e0e0');
                                }
                            }
                        }
                    });
                    
                    if (hasError) {
                        alert('Please fill in API Key for all enabled services');
                        e.preventDefault();
                        return false;
                    }
                });
            });
        ";
        
        wp_register_script('botsubmit-settings', '', array('jquery'), '1.1.0', true);
        wp_enqueue_script('botsubmit-settings');
        wp_add_inline_script('botsubmit-settings', $settings_script);
        
        // Log tab script - localized strings
        $log_script = "
            jQuery(document).ready(function($) {
                $('.botsubmit-resend-btn').on('click', function() {
                    var btn = $(this);
                    var url = btn.data('url');
                    var service = btn.data('service');
                    var nonce = btn.data('nonce');
                    
                    btn.prop('disabled', true).text(botsubmit_ajax.sending);
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'botsubmit_resend_url',
                            url: url,
                            service: service,
                            nonce: nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                btn.text(botsubmit_ajax.sent).css('background-color', '#46b450');
                                setTimeout(function() {
                                    location.reload();
                                }, 1500);
                            } else {
                                btn.text(botsubmit_ajax.error).css('background-color', '#dc3232');
                                alert(response.data || botsubmit_ajax.error_sending);
                                btn.prop('disabled', false);
                            }
                        },
                        error: function() {
                            btn.text(botsubmit_ajax.error).css('background-color', '#dc3232');
                            alert(botsubmit_ajax.network_error);
                            btn.prop('disabled', false);
                        }
                    });
                });
            });
        ";
        
        wp_register_script('botsubmit-log', '', array('jquery'), '1.1.0', true);
        wp_enqueue_script('botsubmit-log');
        wp_add_inline_script('botsubmit-log', $log_script);
        
        // Localize strings for JS
        wp_localize_script('botsubmit-log', 'botsubmit_ajax', array(
            'sending' => __('Sending...', 'botsubmit'),
            'sent' => __('Sent!', 'botsubmit'),
            'error' => __('Error', 'botsubmit'),
            'error_sending' => __('Error sending URL', 'botsubmit'),
            'network_error' => __('Network error', 'botsubmit'),
        ));
    }

    public function on_activate() {
        if (get_option(self::OPTION_GLOBAL_LOG) === false) {
            add_option(self::OPTION_GLOBAL_LOG, []);
        }
        // Старый API ключ - оставляем для обратной совместимости
        if (get_option(self::OPTION_API_KEY) === false) {
            add_option(self::OPTION_API_KEY, '');
        }
        if (get_option(self::OPTION_SEND_TYPES) === false) {
            add_option(self::OPTION_SEND_TYPES, [
                'posts'      => 1, // по умолчанию — только посты
                'pages'      => 0,
                'categories' => 0,
                'tags'       => 0,
            ]);
        }
		if (get_option(self::OPTION_SEND_UPDATE) === false) {
    		add_option(self::OPTION_SEND_UPDATE, [
       			'posts'      => 0,
        		'pages'      => 0,
        		'categories' => 0,
        		'tags'       => 0,
    		]);
		}
		
		// Новые опции для множественных сервисов
		if (get_option(self::OPTION_SERVICES) === false) {
    		add_option(self::OPTION_SERVICES, [
       			'speedyindex'   => 0, // по умолчанию выключен
        		'linkindexing'  => 0,
        		'indexbotik'    => 0,
    		]);
		}
		
		if (get_option(self::OPTION_API_KEYS) === false) {
    		add_option(self::OPTION_API_KEYS, [
       			'speedyindex'   => '',
        		'linkindexing'  => '',
        		'indexbotik'    => '',
    		]);
		}
		
		// Дополнительные параметры для сервисов (user_id и т.д.)
		if (get_option(self::OPTION_SERVICE_PARAMS) === false) {
    		add_option(self::OPTION_SERVICE_PARAMS, [
       			'linkindexing_user_id'  => '',
       			'linkindexing_searchengine' => 'google',
       			'linkindexing_se_type' => 'normal',
       			'indexbotik_engine' => 'universal',
    		]);
		}
		
		// Миграция старого API ключа в новую структуру
		$old_api_key = get_option(self::OPTION_API_KEY, '');
		$api_keys = get_option(self::OPTION_API_KEYS, []);
		if (!empty($old_api_key) && empty($api_keys['speedyindex'])) {
		    $api_keys['speedyindex'] = $old_api_key;
		    update_option(self::OPTION_API_KEYS, $api_keys);
		}
    }
    
    /**
     * Очистка при деактивации плагина
     */
    public function on_deactivate() {
        // Удаляем все запланированные cron задачи
        wp_clear_scheduled_hook('botsubmit_send_url_cron');
    }

    /* =======================
     *          UI
     * ======================= */

   /**
     * Получить список доступных сервисов индексации
     */
    private function get_available_services() {
        return [
            'linkindexing' => [
                'name' => 'Link Indexing Bot',
                'url' => 'https://t.me/Link_indexing_bot?start=437061',
                'api_url' => self::API_URL_LINKINDEXING,
                'extra_fields' => [
                    'user_id' => [
                        'label' => 'Telegram User ID',
                        'type' => 'text',
                        'description' => 'Get it from /help command in bot',
                        'required' => true,
                    ],
                    'searchengine' => [
                        'label' => 'Search Engine',
                        'type' => 'select',
                        'options' => [
                            'google' => 'Google',
                            'yandex' => 'Yandex',
                            'bing' => 'Bing',
                        ],
                        'default' => 'google',
                    ],
                    'se_type' => [
                        'label' => 'Indexing Type',
                        'type' => 'select',
                        'options' => [
                            'normal' => 'Normal',
                            'hard' => 'Hard',
                        ],
                        'default' => 'normal',
                    ],
                ],
            ],
            'indexbotik' => [
                'name' => 'IndexBotik',
                'url' => 'https://t.me/InderixingBot?start=CF2F3E3B26',
                'api_url' => self::API_URL_INDEXBOTIK,
                'extra_fields' => [
                    'engine' => [
                        'label' => 'Engine',
                        'type' => 'select',
                        'options' => [
                            'universal' => 'Universal (Google)',
                            'yandex' => 'Yandex',
                        ],
                        'default' => 'universal',
                        'description' => 'Select search engine',
                    ],
                ],
            ],
            'speedyindex' => [
                'name' => 'SpeedyIndex',
                'url' => 'https://t.me/SpeedyBotSubmit?start=437061',
                'api_url' => self::API_URL_SPEEDYINDEX,
                'extra_fields' => [
                    'searchengine' => [
                        'label' => 'Search Engine',
                        'type' => 'select',
                        'options' => [
                            'google' => 'Google (temporarily unavailable)',
                            'yandex' => 'Yandex',
                        ],
                        'disabled_options' => ['google'],
                        'default' => 'yandex',
                    ],
                ],
            ],
        ];
    }

   /**
     * Получить список публичных custom post types
     */
    private function get_custom_post_types() {
        $args = [
            'public'   => true,
            '_builtin' => false
        ];
        
        $post_types = get_post_types($args, 'objects');
        $custom_types = [];
        
        foreach ($post_types as $post_type) {
            $custom_types[$post_type->name] = $post_type->label;
        }
        
        return $custom_types;
    }

   public function add_action_links($links) {
    $url = add_query_arg(['page'=>'botsubmit-settings','tab'=>'settings'], admin_url('tools.php'));
    $links[] = '<a href="' . esc_url($url) . '">' . esc_html__('Settings', 'botsubmit') . '</a>';
    return $links;
}


    public function add_settings_page() {
        add_management_page(
            __('BotSubmit', 'botsubmit'),
            __('BotSubmit', 'botsubmit'),
            'manage_options',
            'botsubmit-settings',
            [$this, 'render_settings_page']
        );
    }

    public function render_settings_page() {
    if (! current_user_can('manage_options')) return;

    $base_url   = admin_url('tools.php?page=botsubmit-settings');
    $active_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'settings';
    if (!in_array($active_tab, ['settings','log'], true)) $active_tab = 'settings';
    
    // Динамический тайтл страницы
    $page_title = $active_tab === 'log' ? 'BotSubmit - Logs' : 'BotSubmit - Settings';
    ?><script>document.title = '<?php echo esc_js($page_title); ?> ‹ <?php echo esc_js(get_bloginfo('name')); ?> — WordPress';</script><?php

    // === Обработка POST ===
    // Сохранение настроек
    if (isset($_POST['botsubmit_save']) && check_admin_referer('botsubmit_save_settings', 'botsubmit_nonce')) {
        // Сохранение старого API ключа для обратной совместимости
        $key = isset($_POST['botsubmit_api_key']) ? sanitize_text_field(wp_unslash($_POST['botsubmit_api_key'])) : '';
        update_option(self::OPTION_API_KEY, $key);
        
        // Сохранение выбранных сервисов
        $services = [];
        $available_services = $this->get_available_services();
        foreach ($available_services as $service_id => $service_info) {
            $services[$service_id] = !empty($_POST['botsubmit_services'][$service_id]) ? 1 : 0;
        }
        update_option(self::OPTION_SERVICES, $services);
        
        // Сохранение API ключей для каждого сервиса
        $api_keys = [];
        foreach ($available_services as $service_id => $service_info) {
            $api_keys[$service_id] = isset($_POST['botsubmit_api_keys'][$service_id]) 
                ? sanitize_text_field(wp_unslash($_POST['botsubmit_api_keys'][$service_id])) 
                : '';
        }
        update_option(self::OPTION_API_KEYS, $api_keys);
        
        // Сохранение дополнительных параметров сервисов
        if (isset($_POST['botsubmit_service_params']) && is_array($_POST['botsubmit_service_params'])) {
            $service_params = [];
            $post_params = map_deep( wp_unslash( $_POST['botsubmit_service_params'] ), 'sanitize_text_field' ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            foreach ($post_params as $key => $value) {
                $service_params[sanitize_key($key)] = $value;
            }
            update_option(self::OPTION_SERVICE_PARAMS, $service_params);
        }

        // Чекбоксы типов отправки
        $types = (array) get_option(self::OPTION_SEND_TYPES, []);
        $types['posts']      = !empty($_POST['botsubmit_types']['posts']) ? 1 : 0;
        $types['pages']      = !empty($_POST['botsubmit_types']['pages']) ? 1 : 0;
        $types['categories'] = !empty($_POST['botsubmit_types']['categories']) ? 1 : 0;
        $types['tags']       = !empty($_POST['botsubmit_types']['tags']) ? 1 : 0;
        
        // Добавляем custom post types
        $custom_types = $this->get_custom_post_types();
        foreach ($custom_types as $cpt_name => $cpt_label) {
            $types['cpt_' . $cpt_name] = !empty($_POST['botsubmit_types']['cpt_' . $cpt_name]) ? 1 : 0;
        }
        
        update_option(self::OPTION_SEND_TYPES, $types);

        // Чекбоксы: при редактировании
        $types_upd = (array) get_option(self::OPTION_SEND_UPDATE, []);
        $types_upd['posts']      = !empty($_POST['botsubmit_update']['posts']) ? 1 : 0;
        $types_upd['pages']      = !empty($_POST['botsubmit_update']['pages']) ? 1 : 0;
        $types_upd['categories'] = !empty($_POST['botsubmit_update']['categories']) ? 1 : 0;
        $types_upd['tags']       = !empty($_POST['botsubmit_update']['tags']) ? 1 : 0;
        
        // Добавляем custom post types для редактирования
        foreach ($custom_types as $cpt_name => $cpt_label) {
            $types_upd['cpt_' . $cpt_name] = !empty($_POST['botsubmit_update']['cpt_' . $cpt_name]) ? 1 : 0;
        }
        
        update_option(self::OPTION_SEND_UPDATE, $types_upd);

        add_settings_error('botsubmit_messages', 'botsubmit_saved', __('Settings saved.', 'botsubmit'), 'updated');
        // Остаёмся на вкладке "Настройки"
        $active_tab = 'settings';
    }

    // Очистка лога
    if (isset($_POST['botsubmit_clear_log']) && check_admin_referer('botsubmit_clear_log', 'botsubmit_clear_nonce')) {
        update_option(self::OPTION_GLOBAL_LOG, []);
        add_settings_error('botsubmit_messages', 'botsubmit_log_cleared', __('Log cleared.', 'botsubmit'), 'updated');
        // Остаёмся на вкладке "Лог"
        $active_tab = 'log';
    }

    settings_errors('botsubmit_messages');

    // Данные для рендера
    $api_key   = get_option(self::OPTION_API_KEY, '');
    $types     = wp_parse_args(get_option(self::OPTION_SEND_TYPES, []), ['posts'=>0,'pages'=>0,'categories'=>0,'tags'=>0]);
    $types_upd = wp_parse_args(get_option(self::OPTION_SEND_UPDATE, []), ['posts'=>0,'pages'=>0,'categories'=>0,'tags'=>0]);
    $logs      = get_option(self::OPTION_GLOBAL_LOG, []);
    if (!is_array($logs)) $logs = [];
    
    // Данные для сервисов индексации
    $available_services = $this->get_available_services();
    $selected_services = get_option(self::OPTION_SERVICES, []);
    $api_keys_services = get_option(self::OPTION_API_KEYS, []);
    $service_params = get_option(self::OPTION_SERVICE_PARAMS, []);

    ?>
    <div class="wrap">
        <div class="botsubmit-header">
            <h1><span class="botsubmit-logo">●</span> BotSubmit</h1>
        </div>

        <div class="botsubmit-tabs nav-tab-wrapper">
            <a href="<?php echo esc_url(add_query_arg('tab','settings',$base_url)); ?>"
               class="nav-tab <?php echo $active_tab==='settings' ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e('Settings', 'botsubmit'); ?>
            </a>
            <a href="<?php echo esc_url(add_query_arg('tab','log',$base_url)); ?>"
               class="nav-tab <?php echo $active_tab==='log' ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e('Submission Log', 'botsubmit'); ?>
            </a>
        </div>

        <?php if ($active_tab === 'settings'): ?>

            <form method="post" action="<?php echo esc_url(add_query_arg('tab','settings',$base_url)); ?>">
                <?php wp_nonce_field('botsubmit_save_settings', 'botsubmit_nonce'); ?>

                <h2 class="botsubmit-section-title"><?php esc_html_e('Indexing Services', 'botsubmit'); ?></h2>
                <p class="description" style="margin-bottom: 20px;">
                    <?php esc_html_e('Select services to send URLs to. Multiple services can be enabled simultaneously.', 'botsubmit'); ?>
                </p>
                
                <?php foreach ($available_services as $service_id => $service_info): 
                    $is_checked = !empty($selected_services[$service_id]) ? 1 : 0;
                    $service_api_key = isset($api_keys_services[$service_id]) ? $api_keys_services[$service_id] : '';
                ?>
                <div class="botsubmit-service-card <?php echo $is_checked ? 'active' : ''; ?>">
                    <div class="service-header">
                        <input type="checkbox" 
                               name="botsubmit_services[<?php echo esc_attr($service_id); ?>]" 
                               value="1" 
                               id="service_<?php echo esc_attr($service_id); ?>"
                               <?php checked(1, $is_checked); ?>>
                        <label for="service_<?php echo esc_attr($service_id); ?>">
                            <?php echo esc_html($service_info['name']); ?>
                        </label>
                        <?php if (!empty($service_info['url'])): ?>
                        <a href="<?php echo esc_url($service_info['url']); ?>" target="_blank" class="service-link">
                            <?php esc_html_e('Get API Key', 'botsubmit'); ?> →
                        </a>
                        <?php endif; ?>
                    </div>
                    
                    <div class="botsubmit-service-fields" style="display: none; margin-top: 15px;">
                        <label style="display: block; margin-bottom: 6px; font-weight: 600; color: #334354;">
                            <?php esc_html_e('API Key', 'botsubmit'); ?>
                            <span style="color: #f27f26;">*</span>
                        </label>
                        <input type="text" 
                               name="botsubmit_api_keys[<?php echo esc_attr($service_id); ?>]" 
                               value="<?php echo esc_attr($service_api_key); ?>" 
                               class="regular-text botsubmit-input" 
                               placeholder="<?php esc_attr_e('Enter API Key', 'botsubmit'); ?>" />
                        
                        <?php if (!empty($service_info['extra_fields'])): ?>
                            <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #e0e0e0;">
                                <?php foreach ($service_info['extra_fields'] as $field_id => $field_info): 
                                    $param_key = $service_id . '_' . $field_id;
                                    $field_value = isset($service_params[$param_key]) ? $service_params[$param_key] : ($field_info['default'] ?? '');
                                ?>
                                <p style="margin: 12px 0;">
                                    <label style="display: block; margin-bottom: 6px; font-weight: 600; color: #334354;">
                                        <?php echo esc_html($field_info['label']); ?>
                                        <?php if (!empty($field_info['required'])): ?>
                                            <span style="color: #f27f26;">*</span>
                                        <?php endif; ?>
                                    </label>
                                    
                                    <?php if ($field_info['type'] === 'text'): ?>
                                        <input type="text" 
                                               name="botsubmit_service_params[<?php echo esc_attr($param_key); ?>]" 
                                               value="<?php echo esc_attr($field_value); ?>" 
                                               class="regular-text botsubmit-input" />
                                    <?php elseif ($field_info['type'] === 'select'): ?>
                                    <select name="botsubmit_service_params[<?php echo esc_attr($param_key); ?>]" class="regular-text botsubmit-input">
                                        <?php 
                                        $disabled_options = isset($field_info['disabled_options']) ? $field_info['disabled_options'] : [];
                                        foreach ($field_info['options'] as $opt_value => $opt_label): 
                                            $is_disabled = in_array($opt_value, $disabled_options, true);
                                        ?>
                                            <option value="<?php echo esc_attr($opt_value); ?>" <?php selected($field_value, $opt_value); ?> <?php echo $is_disabled ? 'disabled' : ''; ?>>
                                                <?php echo esc_html($opt_label); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                <?php endif; ?>
                                
                                <?php if (!empty($field_info['description'])): ?>
                                    <br><span class="description" style="color: #a0a8ab;"><?php echo esc_html($field_info['description']); ?></span>
                                <?php endif; ?>
                            </p>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>

                <h2 class="botsubmit-section-title" style="margin-top:2em;"><?php esc_html_e('Send to Indexing Services', 'botsubmit'); ?></h2>
                
                <div style="display: flex; gap: 30px; flex-wrap: wrap;">
                    <div class="botsubmit-checkbox-group" style="flex: 1; min-width: 250px;">
                        <h4 style="margin: 0 0 15px 0; color: #334354;"><?php esc_html_e('On Publish:', 'botsubmit'); ?></h4>
                        <label><input type="checkbox" name="botsubmit_types[posts]" value="1" <?php checked(1, (int)$types['posts']); ?>> <?php esc_html_e('Posts', 'botsubmit'); ?></label>
                        <label><input type="checkbox" name="botsubmit_types[pages]" value="1" <?php checked(1, (int)$types['pages']); ?>> <?php esc_html_e('Pages', 'botsubmit'); ?></label>
                        <?php
                        $custom_types = $this->get_custom_post_types();
                        if (!empty($custom_types)) {
                            foreach ($custom_types as $cpt_name => $cpt_label) {
                                $checked = !empty($types['cpt_' . $cpt_name]) ? 1 : 0;
                                ?>
                                <label>
                                    <input type="checkbox" name="botsubmit_types[cpt_<?php echo esc_attr($cpt_name); ?>]" value="1" <?php checked(1, $checked); ?>>
                                    <?php echo esc_html($cpt_label); ?>
                                </label>
                                <?php
                            }
                        }
                        ?>
                        <label><input type="checkbox" name="botsubmit_types[categories]" value="1" <?php checked(1, (int)$types['categories']); ?>> <?php esc_html_e('Categories', 'botsubmit'); ?></label>
                        <label><input type="checkbox" name="botsubmit_types[tags]" value="1" <?php checked(1, (int)$types['tags']); ?>> <?php esc_html_e('Tags', 'botsubmit'); ?></label>
                    </div>
                    
                    <div class="botsubmit-checkbox-group" style="flex: 1; min-width: 250px;">
                        <h4 style="margin: 0 0 15px 0; color: #334354;"><?php esc_html_e('On Update:', 'botsubmit'); ?></h4>
                        <label><input type="checkbox" name="botsubmit_update[posts]" value="1" <?php checked(1, (int)$types_upd['posts']); ?>> <?php esc_html_e('Posts', 'botsubmit'); ?></label>
                        <label><input type="checkbox" name="botsubmit_update[pages]" value="1" <?php checked(1, (int)$types_upd['pages']); ?>> <?php esc_html_e('Pages', 'botsubmit'); ?></label>
                        <?php
                        if (!empty($custom_types)) {
                            foreach ($custom_types as $cpt_name => $cpt_label) {
                                $checked_upd = !empty($types_upd['cpt_' . $cpt_name]) ? 1 : 0;
                                ?>
                                <label>
                                    <input type="checkbox" name="botsubmit_update[cpt_<?php echo esc_attr($cpt_name); ?>]" value="1" <?php checked(1, $checked_upd); ?>>
                                    <?php echo esc_html($cpt_label); ?>
                                </label>
                                <?php
                            }
                        }
                        ?>
                        <label><input type="checkbox" name="botsubmit_update[categories]" value="1" <?php checked(1, (int)$types_upd['categories']); ?>> <?php esc_html_e('Categories', 'botsubmit'); ?></label>
                        <label><input type="checkbox" name="botsubmit_update[tags]" value="1" <?php checked(1, (int)$types_upd['tags']); ?>> <?php esc_html_e('Tags', 'botsubmit'); ?></label>
                    </div>
                </div>

                <p class="submit" style="margin-top: 30px;">
                    <button type="submit" name="botsubmit_save" class="botsubmit-btn">
                        <?php esc_html_e('Save Settings', 'botsubmit'); ?>
                    </button>
                </p>
            </form>

        <?php else: // === LOG TAB === ?>

            <?php
            // Пагинация
            $total_logs = count($logs);
            $per_page = self::LOG_PER_PAGE;
            $total_pages = ceil($total_logs / $per_page);
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Pagination parameter, no action taken
            $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
            if ($current_page > $total_pages && $total_pages > 0) {
                $current_page = $total_pages;
            }
            
            // Получаем записи для текущей страницы (в обратном порядке - новые сверху)
            $logs_reversed = array_reverse($logs);
            $offset = ($current_page - 1) * $per_page;
            $logs_page = array_slice($logs_reversed, $offset, $per_page);
            ?>

            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                <form method="post" action="<?php echo esc_url(add_query_arg('tab','log',$base_url)); ?>">
                    <?php wp_nonce_field('botsubmit_clear_log', 'botsubmit_clear_nonce'); ?>
                    <button type="submit" name="botsubmit_clear_log" class="botsubmit-clear-btn">
                        <?php esc_html_e('Clear Log', 'botsubmit'); ?>
                    </button>
                </form>
                
                <span style="color: #666;">
                    <?php 
                    /* translators: %d: total number of records */
                    printf(esc_html__('Total: %d records', 'botsubmit'), intval($total_logs)); 
                    ?>
                </span>
            </div>

            <table class="widefat botsubmit-log-table">
                <thead>
                    <tr>
                        <th style="width:180px;"><?php esc_html_e('Time', 'botsubmit'); ?></th>
                        <th><?php esc_html_e('URL', 'botsubmit'); ?></th>
                        <th><?php esc_html_e('Response/Error', 'botsubmit'); ?></th>
                        <th style="width:120px;"><?php esc_html_e('Actions', 'botsubmit'); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php if (empty($logs_page)): ?>
                    <tr><td colspan="4"><?php esc_html_e('No records yet.', 'botsubmit'); ?></td></tr>
                <?php else: ?>
                    <?php foreach ($logs_page as $row):
                        $parts = explode(' — ', $row, 3);
                        $ts  = $parts[0] ?? '';
                        $url = $parts[1] ?? '';
                        $msg = $parts[2] ?? '';
                        
                        // Определяем есть ли ошибка
                        $has_error = (
                            stripos($msg, 'Ошибка') !== false || 
                            stripos($msg, 'Error') !== false || 
                            stripos($msg, 'Пропуск') !== false ||
                            stripos($msg, 'Skip') !== false ||
                            stripos($msg, 'не задан') !== false ||
                            stripos($msg, 'not set') !== false ||
                            stripos($msg, 'Ожидает реализации') !== false ||
                            stripos($msg, 'Awaiting implementation') !== false
                        );
                        
                        // Парсим сообщение для определения сервисов с ошибками
                        $services_with_errors = [];
                        // Разбиваем по переносам строк (новый формат) или по | (старый формат для совместимости)
                        $services_msgs = preg_split('/[\n|]/', $msg);
                        foreach ($services_msgs as $service_msg) {
                            $service_msg = trim($service_msg);
                            // Пропускаем пустые строки и контекст (POST UPDATE, PAGE PUBLISH и т.д.)
                            if (empty($service_msg)) continue;
                            if (preg_match('/^(POST|PAGE|CATEGORY|TAG|BULK|MANUAL)\s+(UPDATE|PUBLISH|CREATED|EDITED|ACTION|RESEND)/i', $service_msg)) continue;
                            
                            // Парсим строки вида "ServiceName: response"
                            if (preg_match('/^([^:]+):\s*(.+)$/s', $service_msg, $matches)) {
                                $service_name = trim($matches[1]);
                                $service_response = trim($matches[2]);
                                
                                // Пропускаем если название сервиса содержит слова контекста
                                if (preg_match('/^(POST|PAGE|CATEGORY|TAG|BULK|MANUAL)/i', $service_name)) continue;
                                
                                $is_error = (
                                    stripos($service_response, 'Ошибка') !== false || 
                                    stripos($service_response, 'Error') !== false || 
                                    stripos($service_response, 'не задан') !== false ||
                                    stripos($service_response, 'not set') !== false ||
                                    stripos($service_response, 'Ожидает') !== false ||
                                    stripos($service_response, 'Awaiting') !== false ||
                                    (stripos($service_response, 'HTTP') !== false && !preg_match('/HTTP\s+2\d\d/', $service_response))
                                );
                                
                                if ($is_error) {
                                    $service_id = '';
                                    if (stripos($service_name, 'Speedy') !== false) $service_id = 'speedyindex';
                                    elseif (stripos($service_name, 'Link') !== false) $service_id = 'linkindexing';
                                    elseif (stripos($service_name, 'Index') !== false && stripos($service_name, 'Botik') !== false) $service_id = 'indexbotik';
                                    
                                    if ($service_id) {
                                        $services_with_errors[] = [
                                            'id' => $service_id,
                                            'name' => $service_name
                                        ];
                                    }
                                }
                            }
                        }
                    ?>
                    <tr class="<?php echo $has_error ? 'log-error' : 'log-success'; ?>">
                        <td><?php echo esc_html($ts); ?></td>
                        <td><a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer" style="color: #334354;"><?php echo esc_html($url); ?></a></td>
                        <td>
                            <?php if ($has_error): ?>
                                <span class="botsubmit-badge botsubmit-badge-error" style="margin-bottom: 5px; display: inline-block;">Error</span><br>
                            <?php else: ?>
                                <span class="botsubmit-badge botsubmit-badge-success" style="margin-bottom: 5px; display: inline-block;">Success</span><br>
                            <?php endif; ?>
                            <code style="white-space:pre-wrap;display:block;max-height:150px;overflow:auto;background:#f8f9fa;padding:10px;border-radius:6px;font-size:12px;border:1px solid #e0e0e0;"><?php echo esc_html($msg); ?></code>
                        </td>
                        <td>
                            <?php if (!empty($services_with_errors)): ?>
                                <?php foreach ($services_with_errors as $error_service): ?>
                                    <button type="button" 
                                            class="botsubmit-resend-btn" 
                                            data-url="<?php echo esc_attr($url); ?>" 
                                            data-service="<?php echo esc_attr($error_service['id']); ?>"
                                            data-nonce="<?php echo esc_attr(wp_create_nonce('botsubmit_resend_' . $error_service['id'])); ?>"
                                            style="margin-bottom:5px;width:100%;">
                                        <?php 
                                        /* translators: %s: Service name */
                                        printf(esc_html__('Resend to %s', 'botsubmit'), esc_html($error_service['name'])); 
                                        ?>
                                    </button><br>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <span style="color:#a0a8ab;">—</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>
            
            <?php if ($total_pages > 1): ?>
            <div class="botsubmit-pagination" style="margin-top: 20px; display: flex; justify-content: center; align-items: center; gap: 5px;">
                <?php
                $pagination_base = add_query_arg('tab', 'log', $base_url);
                
                // Кнопка "Первая"
                if ($current_page > 1): ?>
                    <a href="<?php echo esc_url(add_query_arg('paged', 1, $pagination_base)); ?>" class="button" title="<?php esc_attr_e('First page', 'botsubmit'); ?>">&laquo;</a>
                <?php endif; ?>
                
                <?php // Кнопка "Предыдущая"
                if ($current_page > 1): ?>
                    <a href="<?php echo esc_url(add_query_arg('paged', $current_page - 1, $pagination_base)); ?>" class="button">&lsaquo;</a>
                <?php endif; ?>
                
                <?php
                // Номера страниц
                $range = 2;
                $start = max(1, $current_page - $range);
                $end = min($total_pages, $current_page + $range);
                
                if ($start > 1): ?>
                    <a href="<?php echo esc_url(add_query_arg('paged', 1, $pagination_base)); ?>" class="button">1</a>
                    <?php if ($start > 2): ?>
                        <span style="padding: 0 5px;">...</span>
                    <?php endif; ?>
                <?php endif; ?>
                
                <?php for ($i = $start; $i <= $end; $i++): ?>
                    <?php if ($i == $current_page): ?>
                        <span class="button button-primary" style="pointer-events: none;"><?php echo esc_html($i); ?></span>
                    <?php else: ?>
                        <a href="<?php echo esc_url(add_query_arg('paged', $i, $pagination_base)); ?>" class="button"><?php echo esc_html($i); ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php if ($end < $total_pages): ?>
                    <?php if ($end < $total_pages - 1): ?>
                        <span style="padding: 0 5px;">...</span>
                    <?php endif; ?>
                    <a href="<?php echo esc_url(add_query_arg('paged', $total_pages, $pagination_base)); ?>" class="button"><?php echo esc_html($total_pages); ?></a>
                <?php endif; ?>
                
                <?php // Кнопка "Следующая"
                if ($current_page < $total_pages): ?>
                    <a href="<?php echo esc_url(add_query_arg('paged', $current_page + 1, $pagination_base)); ?>" class="button">&rsaquo;</a>
                <?php endif; ?>
                
                <?php // Кнопка "Последняя"
                if ($current_page < $total_pages): ?>
                    <a href="<?php echo esc_url(add_query_arg('paged', $total_pages, $pagination_base)); ?>" class="button" title="<?php esc_attr_e('Last page', 'botsubmit'); ?>">&raquo;</a>
                <?php endif; ?>
                
                <span style="margin-left: 10px; color: #666;">
                    <?php 
                    /* translators: 1: current page number, 2: total pages */
                    printf(esc_html__('Page %1$d of %2$d', 'botsubmit'), intval($current_page), intval($total_pages)); 
                    ?>
                </span>
            </div>
            <?php endif; ?>

        <?php endif; ?>

        <div class="botsubmit-footer">
            <p>
                <?php echo wp_kses_post(__('If this plugin helped you, subscribe to my ', 'botsubmit')); ?>
                <a href="https://t.me/naumov_top" target="_blank">Telegram channel</a> 💙
            </p>
        </div>
    </div>
    <?php
}

    /* =======================
     *      Post Meta Box
     * ======================= */

    /**
     * Добавление мета-бокса в редактор постов/страниц
     */
    public function add_post_meta_box() {
        $post_types = ['post', 'page'];
        
        // Добавляем custom post types
        $custom_types = $this->get_custom_post_types();
        foreach ($custom_types as $cpt_name => $cpt_label) {
            $post_types[] = $cpt_name;
        }
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'botsubmit_sender_meta_box',
                __('BotSubmit', 'botsubmit'),
                [$this, 'render_post_meta_box'],
                $post_type,
                'side',
                'default'
            );
        }
    }

    /**
     * Рендер мета-бокса
     */
    public function render_post_meta_box($post) {
        wp_nonce_field('botsubmit_meta_box', 'botsubmit_meta_box_nonce');
        
        // Определяем статус поста
        $is_published = in_array($post->post_status, ['publish', 'private', 'future'], true);
        
        // Получаем текущие настройки поста
        $send_on_publish = get_post_meta($post->ID, '_botsubmit_send_on_publish', true);
        $send_on_update = get_post_meta($post->ID, '_botsubmit_send_on_update', true);
        
        // Получаем глобальные настройки
        $global_types = get_option(self::OPTION_SEND_TYPES, []);
        $global_update = get_option(self::OPTION_SEND_UPDATE, []);
        
        $post_type = $post->post_type;
        // Маппинг типа поста на ключ в настройках
        if ($post_type === 'post') {
            $type_key = 'posts';
        } elseif ($post_type === 'page') {
            $type_key = 'pages';
        } else {
            $type_key = 'cpt_' . $post_type;
        }
        
        // Определяем значения по умолчанию из глобальных настроек
        $default_publish = !empty($global_types[$type_key]) ? 1 : 0;
        $default_update = !empty($global_update[$type_key]) ? 1 : 0;
        
        // Если мета не установлена, используем глобальные настройки
        if ($send_on_publish === '') {
            $send_on_publish = $default_publish;
        }
        if ($send_on_update === '') {
            $send_on_update = $default_update;
        }
        
        ?>
        <div style="padding: 10px 0;">
            <p style="margin-bottom: 15px; font-size: 13px; color: #666;">
                <?php esc_html_e('Override indexing settings for this post only:', 'botsubmit'); ?>
            </p>
            
            <?php if (!$is_published): ?>
                <!-- Показываем только Send on Publish для неопубликованных постов -->
                <p>
                    <label>
                        <input type="checkbox" 
                               name="botsubmit_send_on_publish" 
                               value="1" 
                               <?php checked(1, (int)$send_on_publish); ?>>
                        <?php esc_html_e('Send on Publish', 'botsubmit'); ?>
                    </label>
                    <?php if ($default_publish): ?>
                        <br><span style="font-size: 11px; color: #999;">
                            <?php esc_html_e('(enabled in global settings)', 'botsubmit'); ?>
                        </span>
                    <?php endif; ?>
                </p>
            <?php else: ?>
                <!-- Показываем только Send on Update для опубликованных постов -->
                <p>
                    <label>
                        <input type="checkbox" 
                               name="botsubmit_send_on_update" 
                               value="1" 
                               <?php checked(1, (int)$send_on_update); ?>>
                        <?php esc_html_e('Send on Update', 'botsubmit'); ?>
                    </label>
                    <?php if ($default_update): ?>
                        <br><span style="font-size: 11px; color: #999;">
                            <?php esc_html_e('(enabled in global settings)', 'botsubmit'); ?>
                    </span>
                <?php endif; ?>
            </p>
            <?php endif; ?>
            
            <p style="margin-top: 15px; padding-top: 10px; border-top: 1px solid #ddd; font-size: 12px; color: #666;">
                <a href="<?php echo esc_url(admin_url('tools.php?page=botsubmit-settings')); ?>" target="_blank">
                    <?php esc_html_e('Global Settings', 'botsubmit'); ?> →
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Сохранение данных мета-бокса
     */
    public function save_post_meta_box($post_id) {
        // Проверки
        if (!isset($_POST['botsubmit_meta_box_nonce'])) {
            return;
        }
        
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['botsubmit_meta_box_nonce'])), 'botsubmit_meta_box')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Сохраняем настройки
        $send_on_publish = !empty($_POST['botsubmit_send_on_publish']) ? 1 : 0;
        $send_on_update = !empty($_POST['botsubmit_send_on_update']) ? 1 : 0;
        
        update_post_meta($post_id, '_botsubmit_send_on_publish', $send_on_publish);
        update_post_meta($post_id, '_botsubmit_send_on_update', $send_on_update);
    }


    /* =======================
     *     Handlers + Send
     * ======================= */

    /**
     * Обработчик публикации постов/страниц/CPT (только первая публикация)
     */
    public function handle_post_publish($new_status, $old_status, $post) {
        // Только переход В статус publish (не из publish в publish)
        if ($new_status !== 'publish' || $old_status === 'publish') {
            return;
        }
        
        // Пропускаем служебные типы
        $excluded_types = ['nav_menu_item', 'revision', 'attachment', 'customize_changeset', 'oembed_cache', 'user_request', 'wp_block'];
        if (in_array($post->post_type, $excluded_types, true)) {
            return;
        }
        
        // Проверяем настройки конкретного поста
        $post_setting = get_post_meta($post->ID, '_botsubmit_send_on_publish', true);
        
        if ($post_setting === '') {
            // Если настройка не установлена, используем глобальные
            $types = get_option(self::OPTION_SEND_TYPES, []);
            
            if ($post->post_type === 'post' && empty($types['posts'])) return;
            if ($post->post_type === 'page' && empty($types['pages'])) return;
            
            // Custom post types
            if (!in_array($post->post_type, ['post', 'page'], true)) {
                if (empty($types['cpt_' . $post->post_type])) return;
            }
        } elseif (empty($post_setting)) {
            // Настройка явно отключена для этого поста
            return;
        }
        
        $url = get_permalink($post->ID);
        if (!$url) {
            return;
        }
        
        $ctx = strtoupper($post->post_type) . ' PUBLISH';
        $this->send_url_to_indexing($url, $ctx);
    }

    // Категории
    public function handle_created_category($term_id, $tt_id = 0, $taxonomy = 'category') {
        $this->maybe_send_term($term_id, $taxonomy, 'CATEGORY CREATED');
    }
    public function handle_edited_category($term_id, $tt_id = 0, $taxonomy = 'category') {
		$upd = (array) get_option(self::OPTION_SEND_UPDATE, []);
		if (empty($upd['categories'])) return;
        $this->maybe_send_term($term_id, $taxonomy, 'CATEGORY EDITED');
    }

    // Теги
    public function handle_created_post_tag($term_id, $tt_id = 0, $taxonomy = 'post_tag') {
        $this->maybe_send_term($term_id, $taxonomy, 'TAG CREATED');
    }
    public function handle_edited_post_tag($term_id, $tt_id = 0, $taxonomy = 'post_tag') {
        $upd = (array) get_option(self::OPTION_SEND_UPDATE, []);
		if (empty($upd['tags'])) return;
		$this->maybe_send_term($term_id, $taxonomy, 'TAG EDITED');
    }

    private function maybe_send_term($term_id, $taxonomy, $verb) {
        $types = get_option(self::OPTION_SEND_TYPES, []);
        $need_category = !empty($types['categories']);
        $need_tag      = !empty($types['tags']);

        if ($taxonomy === 'category' && ! $need_category) return;
        if ($taxonomy === 'post_tag' && ! $need_tag) return;

        $link = get_term_link((int)$term_id, $taxonomy);
        if (is_wp_error($link)) {
            $this->save_global_log('(term '.$taxonomy.' #'.$term_id.')', 'Error: ' . $link->get_error_message());
            return;
        }

        $this->send_url_to_indexing($link, $verb);
    }

    /**
     * Планирование отправки URL через WP Cron (асинхронно)
     */
    private function send_url_to_indexing($url, $context = '') {
        // Проверяем есть ли хотя бы один активный сервис с API ключом
        $selected_services = get_option(self::OPTION_SERVICES, []);
        $api_keys = get_option(self::OPTION_API_KEYS, []);
        
        $has_active_service = false;
        foreach ($selected_services as $service_id => $is_enabled) {
            if (!empty($is_enabled)) {
                $api_key = isset($api_keys[$service_id]) ? trim($api_keys[$service_id]) : '';
                if (!empty($api_key)) {
                    $has_active_service = true;
                    break;
                }
            }
        }
        
        // Если нет активных сервисов - не планируем задачу и не пишем в лог
        if (!$has_active_service) {
            return;
        }
        
        // Планируем выполнение через 1 секунду (фоновая отправка)
        wp_schedule_single_event(time() + 1, 'botsubmit_send_url_cron', [$url, $context]);
        
        // Запускаем cron немедленно если возможно (spawn cron)
        spawn_cron();
    }
    
    /**
     * Обработчик cron задачи - реальная отправка URL
     */
    public function process_cron_send($url, $context = '') {
        $selected_services = get_option(self::OPTION_SERVICES, []);
        $api_keys = get_option(self::OPTION_API_KEYS, []);
        $available_services = $this->get_available_services();
        
        $all_logs = [];
        
        // Проходим по всем выбранным сервисам
        foreach ($selected_services as $service_id => $is_enabled) {
            if (empty($is_enabled)) {
                continue; // Сервис не выбран
            }
            
            if (!isset($available_services[$service_id])) {
                continue; // Неизвестный сервис
            }
            
            $api_key = isset($api_keys[$service_id]) ? trim($api_keys[$service_id]) : '';
            if (empty($api_key)) {
                continue; // Пропускаем без записи в лог
            }
            
            // Отправляем в конкретный сервис
            $result = $this->send_to_service($url, $service_id, $api_key, $available_services[$service_id]);
            $all_logs[] = $available_services[$service_id]['name'] . ': ' . $result;
        }
        
        // Если нет результатов - не пишем в лог
        if (empty($all_logs)) {
            return;
        }
        
        // Сохраняем лог со всеми результатами
        $msg = ($context ? $context . "\n" : '') . implode("\n", $all_logs);
        $this->save_global_log($url, $msg);
    }
    
    /**
     * Отправка URL в конкретный сервис индексации
     */
    private function send_to_service($url, $service_id, $api_key, $service_info) {
        // Здесь будет логика отправки в зависимости от сервиса
        // Пока используем базовую реализацию для SpeedyIndex
        
        switch ($service_id) {
            case 'speedyindex':
                return $this->send_to_speedyindex($url, $api_key);
            
            case 'linkindexing':
                return $this->send_to_linkindexing($url, $api_key);
            
            case 'indexbotik':
                return $this->send_to_indexbotik($url, $api_key);
            
            default:
                return 'Unknown service';
        }
    }
    
    /**
     * Отправка в SpeedyIndex
     */
    private function send_to_speedyindex($url, $api_key) {
        // Получаем выбранную поисковую систему
        $service_params = get_option(self::OPTION_SERVICE_PARAMS, []);
        $searchengine = isset($service_params['speedyindex_searchengine']) ? $service_params['speedyindex_searchengine'] : 'google';
        
        // Формируем URL API в зависимости от выбранной поисковой системы
        $api_url = 'https://api.speedyindex.com/v2/' . $searchengine . '/url';
        
        $args = [
            'body'        => wp_json_encode(['url' => $url]),
            'headers'     => [
                'Authorization' => $api_key,
                'Content-Type'  => 'application/json'
            ],
            'timeout'     => 15,
            'blocking'    => true,
            'data_format' => 'body'
        ];

        $response = wp_remote_post($api_url, $args);
        
        if (is_wp_error($response)) {
            return 'Error: ' . $response->get_error_message();
        }
        
        return 'HTTP ' . intval(wp_remote_retrieve_response_code($response)) . ': ' . wp_remote_retrieve_body($response);
    }
    
    /**
     * Отправка в Link Indexing Bot
     */
    private function send_to_linkindexing($url, $api_key) {
        // Получаем дополнительные параметры
        $service_params = get_option(self::OPTION_SERVICE_PARAMS, []);
        $user_id = isset($service_params['linkindexing_user_id']) ? trim($service_params['linkindexing_user_id']) : '';
        $searchengine = isset($service_params['linkindexing_searchengine']) ? $service_params['linkindexing_searchengine'] : 'google';
        $se_type = isset($service_params['linkindexing_se_type']) ? $service_params['linkindexing_se_type'] : 'normal';
        
        // Проверяем обязательные параметры
        if (empty($user_id)) {
            return 'Error: Telegram User ID not set';
        }
        
        // Формируем тело запроса
        $body = [
            'api_key' => $api_key,
            'user_id' => $user_id,
            'links' => $url, // один URL
            'searchengine' => $searchengine,
            'se_type' => $se_type,
        ];
        
        $args = [
            'body'        => wp_json_encode($body),
            'headers'     => [
                'Content-Type' => 'application/json',
            ],
            'timeout'     => 15,
            'blocking'    => true,
            'data_format' => 'body',
        ];
        
        $response = wp_remote_post(self::API_URL_LINKINDEXING, $args);
        
        if (is_wp_error($response)) {
            return 'Error: ' . $response->get_error_message();
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Пытаемся декодировать JSON ответ
        $response_data = json_decode($response_body, true);
        
        if ($response_data && isset($response_data['status'])) {
            if ($response_data['status'] == 201) {
                // Успешная отправка
                $task_id = isset($response_data['data']['task_id']) ? $response_data['data']['task_id'] : 'N/A';
                return 'Success: Task #' . $task_id . ' created';
            } else {
                // Ошибка от API
                $error_msg = isset($response_data['msg']) ? $response_data['msg'] : 'Unknown error';
                return 'Error: ' . $error_msg;
            }
        }
        
        // Если не удалось распарсить ответ
        return 'HTTP ' . $response_code . ': ' . $response_body;
    }
    
    /**
     * Отправка в IndexBotik
     */
    private function send_to_indexbotik($url, $api_key) {
        // Получаем дополнительные параметры
        $service_params = get_option(self::OPTION_SERVICE_PARAMS, []);
        $engine = isset($service_params['indexbotik_engine']) ? $service_params['indexbotik_engine'] : 'universal';
        
        // Формируем тело запроса
        $body = [
            'api_key' => $api_key,
            'engine' => $engine,
            'links' => [$url], // массив с одним URL
        ];
        
        $args = [
            'body'        => wp_json_encode($body),
            'headers'     => [
                'Content-Type' => 'application/json',
            ],
            'timeout'     => 15,
            'blocking'    => true,
            'data_format' => 'body',
        ];
        
        $response = wp_remote_post(self::API_URL_INDEXBOTIK, $args);
        
        if (is_wp_error($response)) {
            return 'Error: ' . $response->get_error_message();
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Пытаемся декодировать JSON ответ
        $response_data = json_decode($response_body, true);
        
        if ($response_data) {
            if (isset($response_data['status']) && $response_data['status'] === 'success') {
                // Успешная отправка
                $accepted = isset($response_data['accepted']) ? $response_data['accepted'] : 1;
                return 'Success: ' . $accepted . ' link(s) accepted';
            } elseif (isset($response_data['error'])) {
                // Ошибка от API
                return 'Error: ' . $response_data['error'];
            }
        }
        
        // Если не удалось распарсить ответ
        return 'HTTP ' . $response_code . ': ' . $response_body;
    }


    /* =======================
     *          Log
     * ======================= */

    private function save_logs_post($post_ID, $message) {
        // Локальный лог в метаполе поста/страницы
        update_post_meta($post_ID, '_botsubmit_log', $message);
        // И в общий лог — для соответствия таблице (в ней нет колонки "Источник")
        $this->save_global_log(get_permalink($post_ID), $message);
    }

    private function save_global_log($url, $message) {
        $logs = get_option(self::OPTION_GLOBAL_LOG, []);
        if (!is_array($logs)) $logs = [];
        $logs[] = date_i18n('Y-m-d H:i:s') . ' — ' . $url . ' — ' . $message;
        if (count($logs) > self::MAX_LOG_ROWS) {
            $logs = array_slice($logs, -self::MAX_LOG_ROWS);
        }
        update_option(self::OPTION_GLOBAL_LOG, $logs);
    }
	public function handle_updated_post($post_ID, $post_after, $post_before) {
    if (wp_is_post_autosave($post_ID) || wp_is_post_revision($post_ID)) return;
    if (!isset($post_after->post_status) || $post_after->post_status !== 'publish') return;
    
    // Пропускаем первую публикацию (она обрабатывается в handle_post_publish)
    if (!isset($post_before->post_status) || $post_before->post_status !== 'publish') return;
    
    // Исключаем элементы меню и другие служебные типы постов
    $excluded_types = ['nav_menu_item', 'revision', 'attachment', 'customize_changeset', 'oembed_cache', 'user_request', 'wp_block'];
    if (in_array($post_after->post_type, $excluded_types, true)) return;

    // Проверяем настройки конкретного поста
    $post_setting = get_post_meta($post_ID, '_botsubmit_send_on_update', true);
    
    if ($post_setting === '') {
        // Если настройка не установлена, используем глобальные
        $upd = (array) get_option(self::OPTION_SEND_UPDATE, []);
        
        // Проверяем настройки для стандартных типов
        if ($post_after->post_type === 'post' && empty($upd['posts'])) return;
        if ($post_after->post_type === 'page' && empty($upd['pages'])) return;
        
        // Проверяем настройки для custom post types
        if (!in_array($post_after->post_type, ['post', 'page'], true)) {
            if (empty($upd['cpt_' . $post_after->post_type])) return;
        }
    } elseif (empty($post_setting)) {
        // Настройка явно отключена для этого поста
        return;
    }

    $url = get_permalink($post_ID);
    if (!$url) return $this->save_logs_post($post_ID, 'Error: could not get permalink');

    $ctx = strtoupper($post_after->post_type) . ' UPDATE';
    $this->send_url_to_indexing($url, $ctx);
}

    /* =======================
     *      Bulk Actions
     * ======================= */

    /**
     * Регистрация bulk action в списке постов/страниц
     */
    public function register_bulk_action($bulk_actions) {
        $bulk_actions['botsubmit_send_to_index'] = __('Send to Indexing Bot', 'botsubmit');
        return $bulk_actions;
    }

    /**
     * Регистрация bulk actions для custom post types
     */
    public function register_cpt_bulk_actions() {
        $custom_types = $this->get_custom_post_types();
        
        foreach ($custom_types as $cpt_name => $cpt_label) {
            add_filter('bulk_actions-edit-' . $cpt_name, [$this, 'register_bulk_action']);
            add_filter('handle_bulk_actions-edit-' . $cpt_name, [$this, 'handle_bulk_action'], 10, 3);
        }
    }

    /**
     * Обработка bulk action
     */
    public function handle_bulk_action($redirect_to, $doaction, $post_ids) {
        if ($doaction !== 'botsubmit_send_to_index') {
            return $redirect_to;
        }

        // Проверяем наличие хотя бы одного настроенного сервиса
        $selected_services = get_option(self::OPTION_SERVICES, []);
        $api_keys = get_option(self::OPTION_API_KEYS, []);
        
        $has_configured_service = false;
        foreach ($selected_services as $service_id => $is_enabled) {
            if (!empty($is_enabled) && !empty($api_keys[$service_id])) {
                $has_configured_service = true;
                break;
            }
        }
        
        if (!$has_configured_service) {
            $redirect_to = add_query_arg('botsubmit_bulk_error', 'no_api_key', $redirect_to);
            return $redirect_to;
        }

        $sent_count = 0;
        $error_count = 0;

        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if (!$post || $post->post_status !== 'publish') {
                continue;
            }

            $url = get_permalink($post_id);
            if (!$url) {
                $error_count++;
                continue;
            }

            $this->send_url_to_indexing($url, 'BULK ACTION');
            $sent_count++;
        }

        $redirect_to = add_query_arg('botsubmit_bulk_sent', $sent_count, $redirect_to);
        if ($error_count > 0) {
            $redirect_to = add_query_arg('botsubmit_bulk_errors', $error_count, $redirect_to);
        }

        return $redirect_to;
    }

    /**
     * Показ уведомления после bulk action
     */
    public function bulk_action_admin_notice() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WordPress handles nonce for bulk actions
        if (!isset($_REQUEST['botsubmit_bulk_sent']) && !isset($_REQUEST['botsubmit_bulk_error'])) {
            return;
        }

        // Error: нет API ключа
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WordPress handles nonce for bulk actions
        if (isset($_REQUEST['botsubmit_bulk_error']) && sanitize_text_field(wp_unslash($_REQUEST['botsubmit_bulk_error'])) === 'no_api_key') {
            ?>
            <div class="notice notice-error is-dismissible">
                <p>
                    <strong><?php esc_html_e('BotSubmit:', 'botsubmit'); ?></strong>
                    <?php esc_html_e('No service configured. Please configure at least one indexing service with API key.', 'botsubmit'); ?>
                    <a href="<?php echo esc_url(admin_url('tools.php?page=botsubmit-settings')); ?>">
                        <?php esc_html_e('Go to Settings', 'botsubmit'); ?>
                    </a>
                </p>
            </div>
            <?php
            return;
        }

        // Успешная отправка
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WordPress handles nonce for bulk actions
        if (isset($_REQUEST['botsubmit_bulk_sent'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- WordPress handles nonce for bulk actions, intval sanitizes
            $sent_count = intval(wp_unslash($_REQUEST['botsubmit_bulk_sent']));
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- WordPress handles nonce for bulk actions, intval sanitizes
            $error_count = isset($_REQUEST['botsubmit_bulk_errors']) ? intval(wp_unslash($_REQUEST['botsubmit_bulk_errors'])) : 0;

            if ($sent_count > 0) {
                ?>
                <div class="notice notice-success is-dismissible">
                    <p>
                        <strong><?php esc_html_e('BotSubmit:', 'botsubmit'); ?></strong>
                        <?php
                        echo esc_html(sprintf(
                            /* translators: %d: number of URLs sent */
                            _n(
                                'Sent %d URL for indexing.',
                                'Sent %d URLs for indexing.',
                                $sent_count,
                                'botsubmit'
                            ),
                            $sent_count
                        ));
                        ?>
                        <?php if ($error_count > 0): ?>
                            <?php
                            echo ' ' . esc_html(sprintf(
                                /* translators: %d: number of errors */
                                _n(
                                    'Error processing %d item.',
                                    'Errors processing %d items.',
                                    $error_count,
                                    'botsubmit'
                                ),
                                $error_count
                            ));
                            ?>
                        <?php endif; ?>
                    </p>
                </div>
                <?php
            }
        }
    }
    
    /**
     * AJAX обработчик для повторной отправки URL в конкретный сервис
     */
    public function ajax_resend_url() {
        // Проверка прав доступа
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Access denied', 'botsubmit'));
        }
        
        $url = isset($_POST['url']) ? esc_url_raw(wp_unslash($_POST['url'])) : '';
        $service_id = isset($_POST['service']) ? sanitize_key(wp_unslash($_POST['service'])) : '';
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        
        if (empty($url) || empty($service_id)) {
            wp_send_json_error(__('Missing parameters', 'botsubmit'));
        }
        
        // Проверка nonce
        if (!wp_verify_nonce($nonce, 'botsubmit_resend_' . $service_id)) {
            wp_send_json_error(__('Security check failed', 'botsubmit'));
        }
        
        // Проверяем что сервис существует
        $available_services = $this->get_available_services();
        if (!isset($available_services[$service_id])) {
            wp_send_json_error(__('Unknown service', 'botsubmit'));
        }
        
        // Получаем API ключ
        $api_keys = get_option(self::OPTION_API_KEYS, []);
        $api_key = isset($api_keys[$service_id]) ? trim($api_keys[$service_id]) : '';
        
        if (empty($api_key)) {
            wp_send_json_error(__('API key not set for this service', 'botsubmit'));
        }
        
        // Отправляем в сервис
        $result = $this->send_to_service($url, $service_id, $api_key, $available_services[$service_id]);
        
        // Логируем
        $msg = 'MANUAL RESEND → ' . $available_services[$service_id]['name'] . ': ' . $result;
        $this->save_global_log($url, $msg);
        
        // Проверяем успешность
        $is_success = (
            stripos($result, 'Ошибка') === false && 
            stripos($result, 'Error') === false && 
            stripos($result, 'не задан') === false &&
            stripos($result, 'not set') === false &&
            stripos($result, 'Ожидает') === false &&
            (stripos($result, 'HTTP') === false || preg_match('/HTTP\s+2\d\d/', $result))
        );
        
        if ($is_success) {
            wp_send_json_success([
                'message' => __('URL successfully sent', 'botsubmit'),
                'result' => $result
            ]);
        } else {
            wp_send_json_error($result);
        }
    }

}

new BotSubmit();
