<?php
/**
 * Helper Functions for Admin Translation
 * 
 * These functions make it easier to use translations in the admin templates.
 */

if (!defined('ABSPATH')) exit;

/**
 * Get translated text for admin dashboard
 *
 * @param string $key Translation key in dot notation (e.g., 'admin.dashboard_title')
 * @param array $replacements Optional array of key-value pairs for string replacements
 * @return string Translated text
 */
function botamichat_admin_text($key, $replacements = []) {
    return Botami_Language_Manager::get_admin_text($key, $replacements);
}

/**
 * Echo translated text for admin dashboard
 *
 * @param string $key Translation key in dot notation (e.g., 'admin.dashboard_title')
 * @param array $replacements Optional array of key-value pairs for string replacements
 */
function botamichat_admin_e($key, $replacements = []) {
    echo esc_html(botamichat_admin_text($key, $replacements));
}

/**
 * Get the current WordPress language
 * 
 * @return string Language code ('en' or 'fr')
 */
function botami_get_current_language() {
    return Botami_Language_Manager::get_site_language();
}

/**
 * Get translated attribute for admin dashboard with attribute escaping
 *
 * @param string $key Translation key in dot notation (e.g., 'admin.dashboard_title')
 * @param array $replacements Optional array of key-value pairs for string replacements
 * @return string Translated and escaped attribute
 */
function botamichat_admin_attr_e($key, $replacements = []) {
    echo esc_attr(botamichat_admin_text($key, $replacements));
}

/**
 * Get translated text from JSON storage based on user's browser language
 * 
 * @param string $option_name The base option name (e.g., 'botami_placeholder_text')
 * @param string $fallback_text Fallback text if no translation found
 * @return string The translated text
 */
function botami_get_translated_text($option_name, $fallback_text = '') {
    // Get user's browser language
    $user_lang = Botami_Language_Manager::get_user_language();
    
    // Use WordPress site language as default instead of hardcoded 'en'
    $default_language = Botami_Language_Manager::get_site_language();

    // Get enabled languages
    $default_enabled_languages = json_encode([$default_language]);
    $enabled_languages_json = get_option('botami_enabled_languages', $default_enabled_languages);
    $enabled_languages = json_decode($enabled_languages_json, true);
    if (!is_array($enabled_languages)) {
        $enabled_languages = [$default_language];
    }

    // Check if user's language is enabled
    if (!in_array($user_lang, $enabled_languages)) {
        // Fall back to input language or default site language
        $user_lang = get_option('botami_input_language', $default_language);
    }
    
    // Get translations JSON
    $translations_json = get_option($option_name . '_translations', '{}');
    $translations = json_decode($translations_json, true);
    if (!is_array($translations)) {
        $translations = [];
    }
    
    // Return translation if exists
    if (isset($translations[$user_lang]) && !empty($translations[$user_lang])) {
        return $translations[$user_lang];
    }
    
    // Fall back to the base option (original value)
    $base_value = get_option($option_name, '');
    if (!empty($base_value)) {
        return $base_value;
    }
    
    // Finally, use the provided fallback
    return $fallback_text;
}