<?php
/**
 * Plugin Name: Botailor ModelContext Integrator for AI
 * Plugin URI: https://wordpress.org/botailor-modelcontext-integrator-for-ai
 * Description: Expose selected posts & pages as AI-readable endpoints via the Model Context Protocol (MCP).
 * Version: 1.0.0
 * Author: Hamid Azad
 * Author URI: https://github.com/hamidhosenazad
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: botailor-modelcontext-integrator-for-ai
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 *
 * @package MCIA_API
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MCIA_VERSION', '1.0.6');
define('MCIA_PLUGIN_FILE', __FILE__);
define('MCIA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MCIA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MCIA_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class
 */
class MCIA_Content_API {

    /**
     * Single instance of the plugin
     *
     * @var MCIA_Content_API
     */
    private static $instance = null;

    /**
     * Admin class instance
     *
     * @var MCIA_Admin
     */
    public $admin;

    /**
     * REST API class instance
     *
     * @var MCIA_REST
     */
    public $rest;

    /**
     * Get single instance of the plugin
     *
     * @return MCIA_Content_API
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
        $this->load_dependencies();
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action('init', array($this, 'init'));
        add_action('template_redirect', array($this, 'handle_well_known_manifest'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Load plugin dependencies
     */
    private function load_dependencies() {
        require_once MCIA_PLUGIN_DIR . 'includes/class-mcp-admin.php';
        require_once MCIA_PLUGIN_DIR . 'includes/class-mcp-rest.php';
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Initialize admin interface
        if (is_admin()) {
            $this->admin = new MCIA_Admin();
        }

        // Initialize REST API
        $this->rest = new MCIA_REST();
    }


    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        $default_options = array(
            'enabled_posts' => array(),
            'enabled_pages' => array(),
            'auto_enable_new' => false
        );

        add_option('mcia_content_api_options', $default_options);

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Handle .well-known/mcp-manifest.json requests
     */
    public function handle_well_known_manifest() {
        // Check if this is a request for the well-known manifest
        if (isset($_SERVER['REQUEST_URI']) && strpos(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])), '/.well-known/mcp-manifest.json') !== false) {
            // Get exposed posts and pages directly (bypass authentication for well-known endpoint)
            // Note: meta_query can be slow on large sites, but is necessary for filtering
            $posts = get_posts(array(
                'post_type' => array('post', 'page'),
                'post_status' => 'publish',
                'posts_per_page' => -1,
                'meta_query' => array(
                    array(
                        'key' => '_mcia_expose',
                        'value' => '1',
                        'compare' => '='
                    )
                )
            ));
            
            $endpoints = array();
            foreach ($posts as $post) {
                $description = get_post_meta($post->ID, '_mcia_description', true);
                if (empty($description)) {
                    $description = wp_trim_words($post->post_excerpt, 20);
                    if (empty($description)) {
                        $description = wp_trim_words(wp_strip_all_tags($post->post_content), 20);
                    }
                }
                
                $endpoints[] = array(
                    'url' => rest_url('mcia/v1/' . $post->post_name),
                    'slug' => $post->post_name,
                    'title' => $post->post_title,
                    'description' => $description
                );
            }
            
            $manifest_data = array(
                'name' => 'Botailor ModelContext Integrator for AI',
                'version' => MCIA_VERSION,
                'generated_time' => current_time('c'),
                'endpoints' => $endpoints
            );
            
            // Set proper headers
            header('Content-Type: application/json');
            header('Cache-Control: public, max-age=3600'); // Cache for 1 hour
            
            // Output the manifest
            echo json_encode($manifest_data, JSON_PRETTY_PRINT);
            exit;
        }
    }

    /**
     * Get plugin options
     *
     * @return array
     */
    public static function get_options() {
        return get_option('mcia_content_api_options', array());
    }

    /**
     * Update plugin options
     *
     * @param array $options
     * @return bool
     */
    public static function update_options($options) {
        return update_option('mcia_content_api_options', $options);
    }
}

/**
 * Initialize the plugin
 */
function mcia_content_api() {
    return MCIA_Content_API::get_instance();
}

// Start the plugin
mcia_content_api();
