<?php
/**
 * REST API endpoints for Botailor ModelContext Integrator for AI
 *
 * @package MCIA_API
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * REST API class for Botailor ModelContext Integrator for AI
 */
class MCIA_REST {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Manifest endpoint
        register_rest_route('mcia/v1', '/manifest', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_manifest'),
            'permission_callback' => array($this, 'check_permissions')
        ));

        // Master endpoints list (legacy)
        register_rest_route('mcia/v1', '/endpoints', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_endpoints'),
            'permission_callback' => array($this, 'check_permissions')
        ));

        // Individual post/page endpoints (READ)
        register_rest_route('mcia/v1', '/(?P<slug>[a-zA-Z0-9-_]+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_content'),
            'permission_callback' => array($this, 'check_permissions'),
            'args' => array(
                'slug' => array(
                    'required' => true,
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // CREATE post
        register_rest_route('mcia/v1', '/posts', array(
            'methods' => 'POST',
            'callback' => array($this, 'create_post'),
            'permission_callback' => array($this, 'check_create_post_permissions'),
            'args' => array(
                'title' => array(
                    'required' => true,
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'content' => array(
                    'required' => true,
                    'sanitize_callback' => 'wp_kses_post'
                ),
                'excerpt' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_textarea_field'
                ),
                'status' => array(
                    'required' => false,
                    'default' => 'publish',
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // UPDATE post
        register_rest_route('mcia/v1', '/posts/(?P<id>\d+)', array(
            'methods' => 'PUT',
            'callback' => array($this, 'update_post'),
            'permission_callback' => array($this, 'check_update_post_permissions'),
            'args' => array(
                'id' => array(
                    'required' => true,
                    'sanitize_callback' => 'absint'
                ),
                'title' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'content' => array(
                    'required' => false,
                    'sanitize_callback' => 'wp_kses_post'
                ),
                'excerpt' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_textarea_field'
                ),
                'status' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // DELETE post
        register_rest_route('mcia/v1', '/posts/(?P<id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array($this, 'delete_post'),
            'permission_callback' => array($this, 'check_delete_post_permissions'),
            'args' => array(
                'id' => array(
                    'required' => true,
                    'sanitize_callback' => 'absint'
                ),
                'force' => array(
                    'required' => false,
                    'default' => false,
                    'sanitize_callback' => 'rest_sanitize_boolean'
                )
            )
        ));

        // CREATE page
        register_rest_route('mcia/v1', '/pages', array(
            'methods' => 'POST',
            'callback' => array($this, 'create_page'),
            'permission_callback' => array($this, 'check_create_page_permissions'),
            'args' => array(
                'title' => array(
                    'required' => true,
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'content' => array(
                    'required' => true,
                    'sanitize_callback' => 'wp_kses_post'
                ),
                'excerpt' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_textarea_field'
                ),
                'status' => array(
                    'required' => false,
                    'default' => 'publish',
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // UPDATE page
        register_rest_route('mcia/v1', '/pages/(?P<id>\d+)', array(
            'methods' => 'PUT',
            'callback' => array($this, 'update_page'),
            'permission_callback' => array($this, 'check_update_page_permissions'),
            'args' => array(
                'id' => array(
                    'required' => true,
                    'sanitize_callback' => 'absint'
                ),
                'title' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'content' => array(
                    'required' => false,
                    'sanitize_callback' => 'wp_kses_post'
                ),
                'excerpt' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_textarea_field'
                ),
                'status' => array(
                    'required' => false,
                    'sanitize_callback' => 'sanitize_text_field'
                )
            )
        ));

        // DELETE page
        register_rest_route('mcia/v1', '/pages/(?P<id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array($this, 'delete_page'),
            'permission_callback' => array($this, 'check_delete_page_permissions'),
            'args' => array(
                'id' => array(
                    'required' => true,
                    'sanitize_callback' => 'absint'
                ),
                'force' => array(
                    'required' => false,
                    'default' => false,
                    'sanitize_callback' => 'rest_sanitize_boolean'
                )
            )
        ));
    }

    /**
     * Check permissions for REST API access
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_permissions($request) {
        // Always check if user has manage_options capability first (WordPress best practice)
        if (current_user_can('manage_options')) {
            return true;
        }
        
        // Get authentication settings
        $options = get_option('mcia_content_api_options', array());
        $require_auth = isset($options['require_auth']) ? $options['require_auth'] : true; // Default to true (mandatory)
        
        // If authentication is not required, allow access
        if (!$require_auth) {
            return true;
        }
        
        // Check if user is logged in via WordPress session (cookie authentication)
        $user = wp_get_current_user();
        if ($user->ID > 0) {
            // User is logged in via WordPress cookies - allow access
            return true;
        }
        
        // User is not logged in, try application password authentication
        $auth_username = isset($options['auth_username']) ? $options['auth_username'] : '';
        $auth_app_password = isset($options['auth_app_password']) ? $options['auth_app_password'] : '';
        
        if (!empty($auth_username) && !empty($auth_app_password)) {
            // Check Basic Auth headers
            $auth_header = $request->get_header('authorization');
            if ($auth_header) {
                $auth_data = base64_decode(substr($auth_header, 6)); // Remove 'Basic '
                list($username, $password) = explode(':', $auth_data, 2);
                
                if ($username === $auth_username && $password === $auth_app_password) {
                    return true;
                }
            }
        }
        
        return new WP_Error(
            'rest_authentication_required',
            __('Authentication required to access API endpoints.', 'botailor-modelcontext-integrator-for-ai'),
            array('status' => 401)
        );
    }

    /**
     * Check permissions for creating posts
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_create_post_permissions($request) {
        // Check if user is authenticated
        if (!is_user_logged_in()) {
            return new WP_Error(
                'rest_not_logged_in',
                __('You must be logged in to create posts.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 401)
            );
        }
        
        // Check if user can publish posts
        if (!current_user_can('publish_posts')) {
            return new WP_Error(
                'rest_cannot_create',
                __('You do not have permission to create posts.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        return true;
    }

    /**
     * Check permissions for updating posts
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_update_post_permissions($request) {
        // Check if user is authenticated
        if (!is_user_logged_in()) {
            return new WP_Error(
                'rest_not_logged_in',
                __('You must be logged in to update posts.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 401)
            );
        }
        
        $post_id = $request->get_param('id');
        
        // Check if user can edit this specific post
        if (!current_user_can('edit_post', $post_id)) {
            return new WP_Error(
                'rest_cannot_edit',
                __('You do not have permission to edit this post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        // Check plugin-specific permissions
        $allowed_operations = get_post_meta($post_id, '_mcia_permissions', true);
        if (!empty($allowed_operations) && !in_array('update', $allowed_operations)) {
            return new WP_Error(
                'rest_forbidden',
                __('This post does not allow update operations via the API.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        return true;
    }

    /**
     * Check permissions for deleting posts
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_delete_post_permissions($request) {
        // Check if user is authenticated
        if (!is_user_logged_in()) {
            return new WP_Error(
                'rest_not_logged_in',
                __('You must be logged in to delete posts.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 401)
            );
        }
        
        $post_id = $request->get_param('id');
        
        // Check if user can delete this specific post
        if (!current_user_can('delete_post', $post_id)) {
            return new WP_Error(
                'rest_cannot_delete',
                __('You do not have permission to delete this post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        // Check plugin-specific permissions
        $allowed_operations = get_post_meta($post_id, '_mcia_permissions', true);
        if (!empty($allowed_operations) && !in_array('delete', $allowed_operations)) {
            return new WP_Error(
                'rest_forbidden',
                __('This post does not allow delete operations via the API.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        return true;
    }

    /**
     * Check permissions for creating pages
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_create_page_permissions($request) {
        // Check if user is authenticated
        if (!is_user_logged_in()) {
            return new WP_Error(
                'rest_not_logged_in',
                __('You must be logged in to create pages.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 401)
            );
        }
        
        // Check if user can publish pages
        if (!current_user_can('publish_pages')) {
            return new WP_Error(
                'rest_cannot_create',
                __('You do not have permission to create pages.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        return true;
    }

    /**
     * Check permissions for updating pages
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_update_page_permissions($request) {
        // Check if user is authenticated
        if (!is_user_logged_in()) {
            return new WP_Error(
                'rest_not_logged_in',
                __('You must be logged in to update pages.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 401)
            );
        }
        
        $page_id = $request->get_param('id');
        
        // Check if user can edit this specific page
        if (!current_user_can('edit_page', $page_id)) {
            return new WP_Error(
                'rest_cannot_edit',
                __('You do not have permission to edit this page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        // Check plugin-specific permissions
        $allowed_operations = get_post_meta($page_id, '_mcia_permissions', true);
        if (!empty($allowed_operations) && !in_array('update', $allowed_operations)) {
            return new WP_Error(
                'rest_forbidden',
                __('This page does not allow update operations via the API.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        return true;
    }

    /**
     * Check permissions for deleting pages
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_delete_page_permissions($request) {
        // Check if user is authenticated
        if (!is_user_logged_in()) {
            return new WP_Error(
                'rest_not_logged_in',
                __('You must be logged in to delete pages.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 401)
            );
        }
        
        $page_id = $request->get_param('id');
        
        // Check if user can delete this specific page
        if (!current_user_can('delete_page', $page_id)) {
            return new WP_Error(
                'rest_cannot_delete',
                __('You do not have permission to delete this page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        // Check plugin-specific permissions
        $allowed_operations = get_post_meta($page_id, '_mcia_permissions', true);
        if (!empty($allowed_operations) && !in_array('delete', $allowed_operations)) {
            return new WP_Error(
                'rest_forbidden',
                __('This page does not allow delete operations via the API.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }
        
        return true;
    }

    /**
     * Get manifest
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function get_manifest($request) {
        // Get all posts and pages that are exposed via MCP
        $exposed_content = $this->get_exposed_content();
        
        $manifest = array(
            'name' => 'Botailor ModelContext Integrator for AI',
            'version' => MCIA_VERSION,
            'generated_time' => current_time('c', true), // UTC timestamp
            'endpoints' => array()
        );

        foreach ($exposed_content as $post) {
            $description = $this->get_custom_description($post);
            if (empty($description)) {
                $description = $this->get_content_description($post);
            }

            $manifest['endpoints'][] = array(
                'url' => rest_url('mcia/v1/' . $post->post_name),
                'slug' => $post->post_name,
                'title' => $post->post_title,
                'description' => $description
            );
        }

        return new WP_REST_Response($manifest, 200);
    }

    /**
     * Get all available endpoints (legacy method)
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function get_endpoints($request) {
        $exposed_content = $this->get_exposed_content();
        $endpoints = array();

        foreach ($exposed_content as $post) {
            $description = $this->get_custom_description($post);
            if (empty($description)) {
                $description = $this->get_content_description($post);
            }

            $endpoints[] = array(
                'title' => $post->post_title,
                'slug' => $post->post_name,
                'type' => $post->post_type,
                'description' => $description,
                'url' => rest_url('mcia/v1/' . $post->post_name)
            );
        }

        // Sort by title
        usort($endpoints, function($a, $b) {
            return strcmp($a['title'], $b['title']);
        });

        return new WP_REST_Response($endpoints, 200);
    }

    /**
     * Get individual content
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_content($request) {
        $slug = $request->get_param('slug');
        
        if (empty($slug)) {
            return new WP_Error(
                'invalid_slug',
                __('Invalid slug provided', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 400)
            );
        }

        // Find the post or page by slug
        $post = get_page_by_path($slug, OBJECT, array('post', 'page'));
        
        if (!$post) {
            return new WP_Error(
                'content_not_found',
                __('Content not found', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 404)
            );
        }

        // Check if this content is exposed via MCP
        $is_exposed = get_post_meta($post->ID, '_mcia_expose', true);
        
        if (!$is_exposed) {
            return new WP_Error(
                'content_not_available',
                __('This content is not available via the API', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }

        // Prepare content
        $content = $this->prepare_content($post);

        return new WP_REST_Response($content, 200);
    }

    /**
     * Prepare content for API response
     *
     * @param WP_Post $post
     * @return array
     */
    private function prepare_content($post) {
        // Get the content
        $content = $post->post_content;
        
        // Apply WordPress filters
        $content = apply_filters('the_content', $content);
        
        // Clean all newlines from HTML content for cleaner JSON
        $content = preg_replace('/\n+/', ' ', $content);
        $content = preg_replace('/\s+/', ' ', $content);
        $content = trim($content);
        
        // Strip HTML tags for clean text
        $content_text = wp_strip_all_tags($content);
        
        // Clean whitespace from text content
        $content_text = preg_replace('/\s+/', ' ', $content_text);
        $content_text = trim($content_text);
        
        // Get custom description or fallback to auto-generated
        $description = $this->get_custom_description($post);
        if (empty($description)) {
            $description = $this->get_content_description($post);
        }

        // Get featured image if available
        $featured_image = '';
        if (has_post_thumbnail($post->ID)) {
            $featured_image = get_the_post_thumbnail_url($post->ID, 'full');
        }

        // Get post meta
        $meta = get_post_meta($post->ID);
        $clean_meta = array();
        foreach ($meta as $key => $value) {
            if (is_array($value) && count($value) === 1) {
                $clean_meta[$key] = $value[0];
            } else {
                $clean_meta[$key] = $value;
            }
        }

        return array(
            'title' => $post->post_title,
            'content' => $content_text,
            'content_html' => $content,
            'description' => $description,
            'slug' => $post->post_name,
            'type' => $post->post_type,
            'status' => $post->post_status,
            'date' => $post->post_date,
            'modified' => $post->post_modified,
            'author' => get_the_author_meta('display_name', $post->post_author),
            'featured_image' => $featured_image,
            'meta' => $clean_meta,
            'url' => get_permalink($post->ID)
        );
    }

    /**
     * Get content description
     *
     * @param WP_Post $post
     * @return string
     */
    private function get_content_description($post) {
        if (!empty($post->post_excerpt)) {
            return $post->post_excerpt;
        }

        $content = wp_strip_all_tags($post->post_content);
        return wp_trim_words($content, 20, '...');
    }

    /**
     * Get custom description from post meta
     *
     * @param WP_Post $post
     * @return string
     */
    private function get_custom_description($post) {
        $custom_description = get_post_meta($post->ID, '_mcia_description', true);
        return !empty($custom_description) ? $custom_description : '';
    }

    /**
     * Get all exposed content (posts and pages with _mcia_expose = true)
     * Excludes WooCommerce content (reserved for pro version)
     *
     * @return array
     */
    private function get_exposed_content() {
        // Get posts that are exposed (excluding WooCommerce products)
        // Note: meta_query can be slow on large sites, but is necessary for filtering
        $posts = get_posts(array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                array(
                    'key' => '_mcia_expose',
                    'value' => '1',
                    'compare' => '='
                )
            )
        ));

        // Get pages that are exposed (excluding WooCommerce pages)
        // Note: meta_query can be slow on large sites, but is necessary for filtering
        $pages = get_posts(array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                array(
                    'key' => '_mcia_expose',
                    'value' => '1',
                    'compare' => '='
                )
            )
        ));

        // Filter out WooCommerce content
        $pages = array_filter($pages, array($this, 'is_not_woocommerce_content'));
        $posts = array_filter($posts, array($this, 'is_not_woocommerce_content'));

        // Combine and sort by title
        $all_content = array_merge($posts, $pages);
        usort($all_content, function($a, $b) {
            return strcmp($a->post_title, $b->post_title);
        });

        return $all_content;
    }

    /**
     * Check if content is not WooCommerce related
     *
     * @param WP_Post $post
     * @return bool
     */
    private function is_not_woocommerce_content($post) {
        // Exclude WooCommerce pages by slug
        $woocommerce_pages = array(
            'shop',
            'cart', 
            'checkout',
            'my-account',
            'lost-password',
            'edit-address',
            'view-order',
            'orders',
            'downloads',
            'edit-account',
            'customer-logout'
        );
        
        if (in_array($post->post_name, $woocommerce_pages)) {
            return false;
        }
        
        // Exclude WooCommerce pages by template
        $template = get_page_template_slug($post->ID);
        if (strpos($template, 'woocommerce') !== false) {
            return false;
        }
        
        // Exclude pages with WooCommerce shortcodes
        if (strpos($post->post_content, '[woocommerce') !== false) {
            return false;
        }
        
        // Exclude WooCommerce products (if any posts are products)
        if ($post->post_type === 'product') {
            return false;
        }
        
        return true;
    }

    /**
     * Create a new post
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function create_post($request) {
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        $excerpt = $request->get_param('excerpt');
        $status = $request->get_param('status');

        // Validate required fields
        if (empty($title) || empty($content)) {
            return new WP_Error(
                'missing_required_fields',
                __('Title and content are required.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 400)
            );
        }

        // Create the post
        $post_data = array(
            'post_title' => $title,
            'post_content' => $content,
            'post_excerpt' => $excerpt,
            'post_status' => $status,
            'post_type' => 'post',
            'post_author' => get_current_user_id()
        );

        $post_id = wp_insert_post($post_data);

        if (is_wp_error($post_id)) {
            return $post_id;
        }

        if (!$post_id) {
            return new WP_Error(
                'post_creation_failed',
                __('Failed to create post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 500)
            );
        }

        // Set MCP expose meta to enabled by default
        update_post_meta($post_id, '_mcia_expose', '1');

        // Get the created post
        $post = get_post($post_id);
        $prepared_content = $this->prepare_content($post);

        return new WP_REST_Response($prepared_content, 201);
    }

    /**
     * Update an existing post
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function update_post($request) {
        $post_id = $request->get_param('id');
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        $excerpt = $request->get_param('excerpt');
        $status = $request->get_param('status');

        // Check if post exists
        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'post') {
            return new WP_Error(
                'post_not_found',
                __('Post not found.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 404)
            );
        }

        // Check if user can edit this post (only for logged-in WordPress users)
        $user = wp_get_current_user();
        if ($user->ID > 0 && !current_user_can('edit_post', $post_id)) {
            return new WP_Error(
                'rest_forbidden',
                __('You do not have permission to edit this post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }

        // Prepare update data
        $update_data = array('ID' => $post_id);
        
        if (!empty($title)) {
            $update_data['post_title'] = $title;
        }
        
        if (!empty($content)) {
            $update_data['post_content'] = $content;
        }
        
        if (!empty($excerpt)) {
            $update_data['post_excerpt'] = $excerpt;
        }
        
        if (!empty($status)) {
            $update_data['post_status'] = $status;
        }

        $result = wp_update_post($update_data);

        if (is_wp_error($result)) {
            return $result;
        }

        if (!$result) {
            return new WP_Error(
                'post_update_failed',
                __('Failed to update post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 500)
            );
        }

        // Get the updated post
        $updated_post = get_post($post_id);
        $prepared_content = $this->prepare_content($updated_post);

        return new WP_REST_Response($prepared_content, 200);
    }

    /**
     * Delete a post
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function delete_post($request) {
        $post_id = $request->get_param('id');
        $force = $request->get_param('force');

        // Check if post exists
        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'post') {
            return new WP_Error(
                'post_not_found',
                __('Post not found.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 404)
            );
        }

        // Check if user can delete this post (only for logged-in WordPress users)
        $user = wp_get_current_user();
        if ($user->ID > 0 && !current_user_can('delete_post', $post_id)) {
            return new WP_Error(
                'rest_forbidden',
                __('You do not have permission to delete this post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }

        // Delete the post
        $result = wp_delete_post($post_id, $force);

        if (!$result) {
            return new WP_Error(
                'post_deletion_failed',
                __('Failed to delete post.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 500)
            );
        }

        return new WP_REST_Response(array(
            'message' => __('Post deleted successfully.', 'botailor-modelcontext-integrator-for-ai'),
            'id' => $post_id,
            'deleted' => true
        ), 200);
    }

    /**
     * Create a new page
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function create_page($request) {
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        $excerpt = $request->get_param('excerpt');
        $status = $request->get_param('status');

        // Validate required fields
        if (empty($title) || empty($content)) {
            return new WP_Error(
                'missing_required_fields',
                __('Title and content are required.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 400)
            );
        }

        // Create the page
        $page_data = array(
            'post_title' => $title,
            'post_content' => $content,
            'post_excerpt' => $excerpt,
            'post_status' => $status,
            'post_type' => 'page',
            'post_author' => get_current_user_id()
        );

        $page_id = wp_insert_post($page_data);

        if (is_wp_error($page_id)) {
            return $page_id;
        }

        if (!$page_id) {
            return new WP_Error(
                'page_creation_failed',
                __('Failed to create page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 500)
            );
        }

        // Set MCP expose meta to enabled by default
        update_post_meta($page_id, '_mcia_expose', '1');

        // Get the created page
        $page = get_post($page_id);
        $prepared_content = $this->prepare_content($page);

        return new WP_REST_Response($prepared_content, 201);
    }

    /**
     * Update an existing page
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function update_page($request) {
        $page_id = $request->get_param('id');
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        $excerpt = $request->get_param('excerpt');
        $status = $request->get_param('status');

        // Check if page exists
        $page = get_post($page_id);
        if (!$page || $page->post_type !== 'page') {
            return new WP_Error(
                'page_not_found',
                __('Page not found.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 404)
            );
        }

        // Check if user can edit this page (only for logged-in WordPress users)
        $user = wp_get_current_user();
        if ($user->ID > 0 && !current_user_can('edit_post', $page_id)) {
            return new WP_Error(
                'rest_forbidden',
                __('You do not have permission to edit this page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }

        // Prepare update data
        $update_data = array('ID' => $page_id);
        
        if (!empty($title)) {
            $update_data['post_title'] = $title;
        }
        
        if (!empty($content)) {
            $update_data['post_content'] = $content;
        }
        
        if (!empty($excerpt)) {
            $update_data['post_excerpt'] = $excerpt;
        }
        
        if (!empty($status)) {
            $update_data['post_status'] = $status;
        }

        $result = wp_update_post($update_data);

        if (is_wp_error($result)) {
            return $result;
        }

        if (!$result) {
            return new WP_Error(
                'page_update_failed',
                __('Failed to update page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 500)
            );
        }

        // Get the updated page
        $updated_page = get_post($page_id);
        $prepared_content = $this->prepare_content($updated_page);

        return new WP_REST_Response($prepared_content, 200);
    }

    /**
     * Delete a page
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function delete_page($request) {
        $page_id = $request->get_param('id');
        $force = $request->get_param('force');

        // Check if page exists
        $page = get_post($page_id);
        if (!$page || $page->post_type !== 'page') {
            return new WP_Error(
                'page_not_found',
                __('Page not found.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 404)
            );
        }

        // Check if user can delete this page (only for logged-in WordPress users)
        $user = wp_get_current_user();
        if ($user->ID > 0 && !current_user_can('delete_post', $page_id)) {
            return new WP_Error(
                'rest_forbidden',
                __('You do not have permission to delete this page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 403)
            );
        }

        // Delete the page
        $result = wp_delete_post($page_id, $force);

        if (!$result) {
            return new WP_Error(
                'page_deletion_failed',
                __('Failed to delete page.', 'botailor-modelcontext-integrator-for-ai'),
                array('status' => 500)
            );
        }

        return new WP_REST_Response(array(
            'message' => __('Page deleted successfully.', 'botailor-modelcontext-integrator-for-ai'),
            'id' => $page_id,
            'deleted' => true
        ), 200);
    }
}
