<?php
/**
 * Admin interface for Botailor ModelContext Integrator for AI
 *
 * @package MCIA_API
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Admin class for Botailor ModelContext Integrator for AI
 */
class MCIA_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_mcia_save_post_meta', array($this, 'ajax_save_post_meta'));
        add_action('wp_ajax_mcia_save_page_meta', array($this, 'ajax_save_page_meta'));
        add_action('wp_ajax_mcia_save_permissions', array($this, 'ajax_save_permissions'));
        add_action('wp_ajax_mcia_search_content', array($this, 'ajax_search_content'));
        add_action('wp_ajax_mcia_bulk_toggle', array($this, 'ajax_bulk_toggle'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            __('Botailor ModelContext Integrator for AI Settings', 'botailor-modelcontext-integrator-for-ai'),
            __('Botailor ModelContext Integrator for AI', 'botailor-modelcontext-integrator-for-ai'),
            'manage_options',
            'botailor-modelcontext-integrator-for-ai',
            array($this, 'admin_page')
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('mcia_content_api_settings', 'mcia_content_api_options', array(
            'sanitize_callback' => array($this, 'sanitize_options')
        ));
    }

    /**
     * Sanitize options
     */
    public function sanitize_options($input) {
        $sanitized = array();
        
        // Authentication is always required (mandatory)
        $sanitized['require_auth'] = true;
        
        if (isset($input['auth_username'])) {
            $sanitized['auth_username'] = sanitize_text_field($input['auth_username']);
        }
        
        if (isset($input['auth_app_password'])) {
            $sanitized['auth_app_password'] = sanitize_text_field($input['auth_app_password']);
        }
        
        return $sanitized;
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook
     */
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_botailor-modelcontext-integrator-for-ai' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'mcia-admin-style',
            MCIA_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            MCIA_VERSION . '.' . time()
        );

        wp_enqueue_script(
            'mcia-admin-script',
            MCIA_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            MCIA_VERSION . '.' . time(),
            true
        );

        // Get authentication settings
        $options = get_option('mcia_content_api_options', array());
        $auth_username = isset($options['auth_username']) ? $options['auth_username'] : '';
        $auth_app_password = isset($options['auth_app_password']) ? $options['auth_app_password'] : '';
        
        // Get totals for pagination (called in enqueue_admin_scripts context)
        $posts_count = wp_count_posts('post');
        $pages_count = wp_count_posts('page');
        
        wp_localize_script('mcia-admin-script', 'mciaAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mcia_admin_nonce'),
            'restNonce' => wp_create_nonce('wp_rest'),
            'authUsername' => $auth_username,
            'authAppPassword' => $auth_app_password,
            'totalPosts' => isset($posts_count->publish) ? $posts_count->publish : 0,
            'totalPages' => isset($pages_count->publish) ? $pages_count->publish : 0,
            'strings' => array(
                'saveSuccess' => __('Settings saved successfully', 'botailor-modelcontext-integrator-for-ai'),
                'saveError' => __('Error saving settings', 'botailor-modelcontext-integrator-for-ai'),
                'confirmDisable' => __('Are you sure you want to disable this item?', 'botailor-modelcontext-integrator-for-ai'),
                'bulkToggleSuccess' => __('Bulk action completed successfully', 'botailor-modelcontext-integrator-for-ai')
            )
        ));
    }

    /**
     * Admin page HTML
     */
    public function admin_page() {
        // Get initial set of posts (25 by default, pagination handled by JavaScript)
        $initial_per_page = 25;
        
        // Get total count of posts
        $posts_query = new WP_Query(array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $initial_per_page,
            'orderby' => 'title',
            'order' => 'ASC'
        ));
        $posts = $posts_query->posts;
        $total_posts = $posts_query->found_posts;

        // Get total count of pages
        $pages_query = new WP_Query(array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'posts_per_page' => $initial_per_page,
            'orderby' => 'title',
            'order' => 'ASC'
        ));
        $pages = $pages_query->posts;
        $total_pages = $pages_query->found_posts;

        // Filter out WooCommerce specific pages
        $pages = array_filter($pages, function($page) {
            // Exclude WooCommerce pages
            $woocommerce_pages = array(
                'shop',
                'cart', 
                'checkout',
                'my-account',
                'lost-password',
                'edit-address',
                'view-order',
                'orders',
                'downloads',
                'edit-account',
                'customer-logout'
            );
            
            // Check if it's a WooCommerce page by slug
            if (in_array($page->post_name, $woocommerce_pages)) {
                return false;
            }
            
            // Check if it's a WooCommerce page by template
            $template = get_page_template_slug($page->ID);
            if (strpos($template, 'woocommerce') !== false) {
                return false;
            }
            
            // Check if it contains WooCommerce shortcodes
            if (strpos($page->post_content, '[woocommerce') !== false) {
                return false;
            }
            
            return true;
        });

        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Botailor ModelContext Integrator for AI Settings', 'botailor-modelcontext-integrator-for-ai'); ?></h1>
            
            <div class="mcp-admin-header">
                <p><?php esc_html_e('Configure which posts and pages should be exposed via the API endpoints. Set granular permissions (read, write, update, delete) for each item.', 'botailor-modelcontext-integrator-for-ai'); ?></p>
                <p>
                    <strong><?php esc_html_e('Manifest endpoint:', 'botailor-modelcontext-integrator-for-ai'); ?></strong>
                    <code><?php echo esc_url(rest_url('mcia/v1/manifest')); ?></code>
                </p>
                <p>
                    <strong><?php esc_html_e('Legacy endpoints list:', 'botailor-modelcontext-integrator-for-ai'); ?></strong>
                    <code><?php echo esc_url(rest_url('mcia/v1/endpoints')); ?></code>
                </p>
                <p>
                    <strong><?php esc_html_e('CRUD Endpoints:', 'botailor-modelcontext-integrator-for-ai'); ?></strong>
                </p>
                <ul>
                    <li><strong><?php esc_html_e('Posts:', 'botailor-modelcontext-integrator-for-ai'); ?></strong>
                        <ul>
                            <li><code>POST <?php echo esc_url(rest_url('mcia/v1/posts')); ?></code> - <?php esc_html_e('Create post', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                            <li><code>PUT <?php echo esc_url(rest_url('mcia/v1/posts/{id}')); ?></code> - <?php esc_html_e('Update post', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                            <li><code>DELETE <?php echo esc_url(rest_url('mcia/v1/posts/{id}')); ?></code> - <?php esc_html_e('Delete post', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        </ul>
                    </li>
                    <li><strong><?php esc_html_e('Pages:', 'botailor-modelcontext-integrator-for-ai'); ?></strong>
                        <ul>
                            <li><code>POST <?php echo esc_url(rest_url('mcia/v1/pages')); ?></code> - <?php esc_html_e('Create page', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                            <li><code>PUT <?php echo esc_url(rest_url('mcia/v1/pages/{id}')); ?></code> - <?php esc_html_e('Update page', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                            <li><code>DELETE <?php echo esc_url(rest_url('mcia/v1/pages/{id}')); ?></code> - <?php esc_html_e('Delete page', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        </ul>
                    </li>
                </ul>
            </div>

            <?php
            // Handle form submission
            if (isset($_POST['submit']) && isset($_POST['mcia_auth_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mcia_auth_nonce'])), 'mcia_auth_settings')) {
                $options = array(
                    'require_auth' => isset($_POST['require_auth']),
                    'auth_username' => isset($_POST['auth_username']) ? sanitize_text_field(wp_unslash($_POST['auth_username'])) : '',
                    'auth_app_password' => isset($_POST['auth_app_password']) ? sanitize_text_field(wp_unslash($_POST['auth_app_password'])) : ''
                );
                update_option('mcia_content_api_options', $options);
                echo '<div class="notice notice-success"><p>' . esc_html(__('Authentication settings saved successfully!', 'botailor-modelcontext-integrator-for-ai')) . '</p></div>';
            }
            
            $options = get_option('mcia_content_api_options', array());
            $require_auth = isset($options['require_auth']) ? $options['require_auth'] : true; // Default to true (mandatory)
            $auth_username = isset($options['auth_username']) ? $options['auth_username'] : '';
            $auth_app_password = isset($options['auth_app_password']) ? $options['auth_app_password'] : '';
            ?>

            <div class="mcp-auth-settings">
                <h2><?php esc_html_e('API Authentication', 'botailor-modelcontext-integrator-for-ai'); ?></h2>
                <p class="description" style="margin-bottom: 20px;">
                    <strong><?php esc_html_e('⚠️ Authentication is mandatory for security.', 'botailor-modelcontext-integrator-for-ai'); ?></strong> 
                    <?php esc_html_e('External API access requires valid credentials. Configure the username and password below.', 'botailor-modelcontext-integrator-for-ai'); ?>
                </p>
                
                <?php if (empty($auth_username) || empty($auth_app_password)) : ?>
                <div class="notice notice-warning inline">
                    <p><strong><?php esc_html_e('⚠️ API Not Configured:', 'botailor-modelcontext-integrator-for-ai'); ?></strong> <?php esc_html_e('Please set username and password below to enable API access.', 'botailor-modelcontext-integrator-for-ai'); ?></p>
                </div>
                <?php else : ?>
                <div class="notice notice-success inline">
                    <p><strong><?php esc_html_e('✓ API Configured:', 'botailor-modelcontext-integrator-for-ai'); ?></strong> <?php esc_html_e('External API access is active with authentication.', 'botailor-modelcontext-integrator-for-ai'); ?></p>
                </div>
                <?php endif; ?>
                
                <form method="post" action="">
                    <?php wp_nonce_field('mcia_auth_settings', 'mcia_auth_nonce'); ?>
                    <input type="hidden" name="require_auth" value="1" />
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="auth_username"><?php esc_html_e('API Username', 'botailor-modelcontext-integrator-for-ai'); ?> <span style="color: red;">*</span></label>
                            </th>
                            <td>
                                <input type="text" id="auth_username" name="auth_username" value="<?php echo esc_attr($auth_username); ?>" class="regular-text" placeholder="Enter username" required />
                                <p class="description">
                                    <?php esc_html_e('Username for API authentication (e.g., admin, api-user, etc.)', 'botailor-modelcontext-integrator-for-ai'); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="auth_app_password"><?php esc_html_e('API Password', 'botailor-modelcontext-integrator-for-ai'); ?> <span style="color: red;">*</span></label>
                            </th>
                            <td>
                                <input type="password" id="auth_app_password" name="auth_app_password" value="<?php echo esc_attr($auth_app_password); ?>" class="regular-text" placeholder="Enter a secure password" required />
                                <p class="description">
                                    <?php esc_html_e('Secure password for API authentication. Choose a strong password.', 'botailor-modelcontext-integrator-for-ai'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button(__('Save API Credentials', 'botailor-modelcontext-integrator-for-ai')); ?>
                </form>
                
                <?php if ($require_auth && !empty($auth_username) && !empty($auth_app_password)) : ?>
                <div class="mcp-auth-info">
                    <h3><?php esc_html_e('Authentication Information', 'botailor-modelcontext-integrator-for-ai'); ?></h3>
                    <p><strong><?php esc_html_e('API Endpoints now require authentication:', 'botailor-modelcontext-integrator-for-ai'); ?></strong></p>
                    <ul>
                        <li><code>GET <?php echo esc_url(rest_url('mcia/v1/manifest')); ?></code> - <?php esc_html_e('Get manifest', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>GET <?php echo esc_url(rest_url('mcia/v1/endpoints')); ?></code> - <?php esc_html_e('Get endpoints list', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>GET <?php echo esc_url(rest_url('mcia/v1/{slug}')); ?></code> - <?php esc_html_e('Get content by slug', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>POST <?php echo esc_url(rest_url('mcia/v1/posts')); ?></code> - <?php esc_html_e('Create post', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>PUT <?php echo esc_url(rest_url('mcia/v1/posts/{id}')); ?></code> - <?php esc_html_e('Update post', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>DELETE <?php echo esc_url(rest_url('mcia/v1/posts/{id}')); ?></code> - <?php esc_html_e('Delete post', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>POST <?php echo esc_url(rest_url('mcia/v1/pages')); ?></code> - <?php esc_html_e('Create page', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>PUT <?php echo esc_url(rest_url('mcia/v1/pages/{id}')); ?></code> - <?php esc_html_e('Update page', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                        <li><code>DELETE <?php echo esc_url(rest_url('mcia/v1/pages/{id}')); ?></code> - <?php esc_html_e('Delete page', 'botailor-modelcontext-integrator-for-ai'); ?></li>
                    </ul>
                    <p><strong><?php esc_html_e('Authentication Method:', 'botailor-modelcontext-integrator-for-ai'); ?></strong></p>
                    <p><?php esc_html_e('Use Basic Authentication with your WordPress username and application password:', 'botailor-modelcontext-integrator-for-ai'); ?></p>
                    <pre><code>curl -u "<?php echo esc_html($auth_username); ?>:<?php echo esc_html($auth_app_password); ?>" "<?php echo esc_url(rest_url('mcia/v1/manifest')); ?>"</code></pre>
                    
                    <p><strong><?php esc_html_e('CRUD Examples:', 'botailor-modelcontext-integrator-for-ai'); ?></strong></p>
                    <pre><code># Create a post
curl -u "<?php echo esc_html($auth_username); ?>:<?php echo esc_html($auth_app_password); ?>" \
  -X POST "<?php echo esc_url(rest_url('mcia/v1/posts')); ?>" \
  -H "Content-Type: application/json" \
  -d '{"title": "New Post", "content": "Post content", "status": "publish"}'

# Update a post
curl -u "<?php echo esc_html($auth_username); ?>:<?php echo esc_html($auth_app_password); ?>" \
  -X PUT "<?php echo esc_url(rest_url('mcia/v1/posts/123')); ?>" \
  -H "Content-Type: application/json" \
  -d '{"title": "Updated Title", "content": "Updated content"}'

# Delete a post
curl -u "<?php echo esc_html($auth_username); ?>:<?php echo esc_html($auth_app_password); ?>" \
  -X DELETE "<?php echo esc_url(rest_url('mcia/v1/posts/123')); ?>"</code></pre>
                </div>
                <?php endif; ?>
            </div>

            <div class="mcp-content-sections">
                <div class="mcp-section">
                    <h2><?php esc_html_e('Posts', 'botailor-modelcontext-integrator-for-ai'); ?></h2>
                    
                    <!-- Search and Pagination Controls -->
                    <div class="mcp-controls">
                        <div class="mcp-search-box">
                            <input type="text" id="search-posts" placeholder="<?php esc_attr_e('Search posts...', 'botailor-modelcontext-integrator-for-ai'); ?>" class="mcp-search-input" />
                            <button type="button" class="button mcp-search-btn" data-target="posts"><?php esc_html_e('Search', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                            <button type="button" class="button mcp-clear-search" data-target="posts"><?php esc_html_e('Clear', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                        </div>
                        <div class="mcp-pagination-controls">
                            <label for="posts-per-page"><?php esc_html_e('Per page:', 'botailor-modelcontext-integrator-for-ai'); ?></label>
                            <select id="posts-per-page" class="mcp-per-page-select">
                                <option value="10">10</option>
                                <option value="25" selected>25</option>
                                <option value="50">50</option>
                                <option value="100">100</option>
                            </select>
                            <div class="mcp-pagination-info">
                                <span class="mcp-pagination-text"><?php esc_html_e('Page', 'botailor-modelcontext-integrator-for-ai'); ?> <span id="posts-current-page">1</span> <?php esc_html_e('of', 'botailor-modelcontext-integrator-for-ai'); ?> <span id="posts-total-pages">1</span></span>
                            </div>
                            <div class="mcp-pagination-nav">
                                <button type="button" class="button mcp-prev-page" data-target="posts" disabled><?php esc_html_e('← Previous', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                                <button type="button" class="button mcp-next-page" data-target="posts" disabled><?php esc_html_e('Next →', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mcp-content-list">
                        <?php if (empty($posts)) : ?>
                            <p><?php esc_html_e('No published posts found.', 'botailor-modelcontext-integrator-for-ai'); ?></p>
                        <?php else : ?>
                             <div class="mcp-bulk-actions">
                                 <label for="bulk-action-posts"><?php esc_html_e('Bulk Action:', 'botailor-modelcontext-integrator-for-ai'); ?></label>
                                 <select id="bulk-action-posts" class="mcp-bulk-select">
                                     <option value="enabled" selected><?php esc_html_e('Enabled (Default)', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                     <option value="disabled"><?php esc_html_e('Disabled', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                 </select>
                                 <button type="button" class="button mcp-apply-bulk-posts"><?php esc_html_e('Apply to Selected', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                             </div>
                            <table class="wp-list-table widefat fixed striped posts-table">
                                <thead>
                                    <tr>
                                        <th class="manage-column column-cb check-column">
                                            <input type="checkbox" id="select-all-posts" />
                                        </th>
                                        <th class="manage-column column-id"><?php esc_html_e('ID', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-title"><?php esc_html_e('Title', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-slug"><?php esc_html_e('Slug', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-expose"><?php esc_html_e('Expose to API', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-permissions"><?php esc_html_e('Permissions', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-description"><?php esc_html_e('Description', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-preview"><?php esc_html_e('Preview', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                     <?php foreach ($posts as $post) : 
                                         $is_exposed = get_post_meta($post->ID, '_mcia_expose', true);
                                         // Default to enabled if no meta exists
                                         if ($is_exposed === '') {
                                             $is_exposed = '1';
                                             update_post_meta($post->ID, '_mcia_expose', '1');
                                         }
                                         $description = get_post_meta($post->ID, '_mcia_description', true);
                                         $endpoint_url = rest_url('mcia/v1/' . $post->post_name);
                                     ?>
                                        <tr data-post-id="<?php echo esc_attr($post->ID); ?>">
                                            <th class="check-column">
                                                <input type="checkbox" class="post-checkbox" />
                                            </th>
                                            <td class="column-id">
                                                <strong><?php echo esc_html($post->ID); ?></strong>
                                            </td>
                                            <td class="column-title">
                                                <strong><?php echo esc_html($post->post_title); ?></strong>
                                            </td>
                                             <td class="column-slug">
                                                 <code><?php echo esc_html($post->post_name); ?></code>
                                             </td>
                                             <td class="column-expose">
                                                 <div class="mcp-toggle-container">
                                                     <label class="mcp-toggle-switch">
                                                         <input type="checkbox" 
                                                                class="mcp-expose-toggle" 
                                                                data-post-id="<?php echo esc_attr($post->ID); ?>"
                                                                <?php checked($is_exposed, '1'); ?> />
                                                         <span class="mcp-toggle-slider"></span>
                                                     </label>
                                                     <span class="mcp-toggle-label">
                                                         <?php echo ($is_exposed === '1') ? esc_html(__('Enabled', 'botailor-modelcontext-integrator-for-ai')) : esc_html(__('Disabled', 'botailor-modelcontext-integrator-for-ai')); ?>
                                                     </span>
                                                 </div>
                                             </td>
                                             <td class="column-permissions">
                                                 <?php 
                                                 $permissions = get_post_meta($post->ID, '_mcia_permissions', true);
                                                 if (empty($permissions)) {
                                                     $permissions = array('read'); // Default to read only
                                                     update_post_meta($post->ID, '_mcia_permissions', $permissions);
                                                 }
                                                 ?>
                                                 <select class="mcp-permissions-select" 
                                                         data-post-id="<?php echo esc_attr($post->ID); ?>" 
                                                         multiple="multiple" 
                                                         size="4">
                                                     <option value="read" <?php selected(in_array('read', $permissions), true); ?>><?php esc_html_e('Read', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                     <option value="write" <?php selected(in_array('write', $permissions), true); ?>><?php esc_html_e('Write', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                     <option value="update" <?php selected(in_array('update', $permissions), true); ?>><?php esc_html_e('Update', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                     <option value="delete" <?php selected(in_array('delete', $permissions), true); ?>><?php esc_html_e('Delete', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                 </select>
                                             </td>
                                            <td class="column-description">
                                                <textarea class="mcp-description-input" 
                                                          data-post-id="<?php echo esc_attr($post->ID); ?>"
                                                          placeholder="<?php esc_attr_e('Custom description (optional)', 'botailor-modelcontext-integrator-for-ai'); ?>"
                                                          rows="2"><?php echo esc_textarea($description); ?></textarea>
                                            </td>
                                            <td class="column-preview">
                                                <?php if ($is_exposed) : ?>
                                                    <button type="button" 
                                                            class="button button-small mcp-view-endpoint" 
                                                            data-url="<?php echo esc_url($endpoint_url); ?>">
                                                        <?php esc_html_e('View JSON', 'botailor-modelcontext-integrator-for-ai'); ?>
                                                    </button>
                                                <?php else : ?>
                                                    <span class="mcp-disabled"><?php esc_html_e('Not exposed', 'botailor-modelcontext-integrator-for-ai'); ?></span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="mcp-section">
                    <h2><?php esc_html_e('Pages', 'botailor-modelcontext-integrator-for-ai'); ?></h2>
                    
                    <!-- Search and Pagination Controls -->
                    <div class="mcp-controls">
                        <div class="mcp-search-box">
                            <input type="text" id="search-pages" placeholder="<?php esc_attr_e('Search pages...', 'botailor-modelcontext-integrator-for-ai'); ?>" class="mcp-search-input" />
                            <button type="button" class="button mcp-search-btn" data-target="pages"><?php esc_html_e('Search', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                            <button type="button" class="button mcp-clear-search" data-target="pages"><?php esc_html_e('Clear', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                        </div>
                        <div class="mcp-pagination-controls">
                            <label for="pages-per-page"><?php esc_html_e('Per page:', 'botailor-modelcontext-integrator-for-ai'); ?></label>
                            <select id="pages-per-page" class="mcp-per-page-select">
                                <option value="10">10</option>
                                <option value="25" selected>25</option>
                                <option value="50">50</option>
                                <option value="100">100</option>
                            </select>
                            <div class="mcp-pagination-info">
                                <span class="mcp-pagination-text"><?php esc_html_e('Page', 'botailor-modelcontext-integrator-for-ai'); ?> <span id="pages-current-page">1</span> <?php esc_html_e('of', 'botailor-modelcontext-integrator-for-ai'); ?> <span id="pages-total-pages">1</span></span>
                            </div>
                            <div class="mcp-pagination-nav">
                                <button type="button" class="button mcp-prev-page" data-target="pages" disabled><?php esc_html_e('← Previous', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                                <button type="button" class="button mcp-next-page" data-target="pages" disabled><?php esc_html_e('Next →', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mcp-content-list">
                        <?php if (empty($pages)) : ?>
                            <p><?php esc_html_e('No published pages found.', 'botailor-modelcontext-integrator-for-ai'); ?></p>
                        <?php else : ?>
                             <div class="mcp-bulk-actions">
                                 <label for="bulk-action-pages"><?php esc_html_e('Bulk Action:', 'botailor-modelcontext-integrator-for-ai'); ?></label>
                                 <select id="bulk-action-pages" class="mcp-bulk-select">
                                     <option value="enabled" selected><?php esc_html_e('Enabled (Default)', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                     <option value="disabled"><?php esc_html_e('Disabled', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                 </select>
                                 <button type="button" class="button mcp-apply-bulk-pages"><?php esc_html_e('Apply to Selected', 'botailor-modelcontext-integrator-for-ai'); ?></button>
                             </div>
                            <table class="wp-list-table widefat fixed striped pages-table">
                                <thead>
                                    <tr>
                                        <th class="manage-column column-cb check-column">
                                            <input type="checkbox" id="select-all-pages" />
                                        </th>
                                        <th class="manage-column column-id"><?php esc_html_e('ID', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-title"><?php esc_html_e('Title', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-slug"><?php esc_html_e('Slug', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-expose"><?php esc_html_e('Expose to API', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-permissions"><?php esc_html_e('Permissions', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-description"><?php esc_html_e('Description', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                        <th class="manage-column column-preview"><?php esc_html_e('Preview', 'botailor-modelcontext-integrator-for-ai'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                     <?php foreach ($pages as $page) : 
                                         $is_exposed = get_post_meta($page->ID, '_mcia_expose', true);
                                         // Default to enabled if no meta exists
                                         if ($is_exposed === '') {
                                             $is_exposed = '1';
                                             update_post_meta($page->ID, '_mcia_expose', '1');
                                         }
                                         $description = get_post_meta($page->ID, '_mcia_description', true);
                                         $endpoint_url = rest_url('mcia/v1/' . $page->post_name);
                                     ?>
                                         <tr data-page-id="<?php echo esc_attr($page->ID); ?>">
                                             <th class="check-column">
                                                 <input type="checkbox" class="page-checkbox" />
                                             </th>
                                             <td class="column-id">
                                                 <strong><?php echo esc_html($page->ID); ?></strong>
                                             </td>
                                             <td class="column-title">
                                                 <strong><?php echo esc_html($page->post_title); ?></strong>
                                             </td>
                                             <td class="column-slug">
                                                 <code><?php echo esc_html($page->post_name); ?></code>
                                             </td>
                                             <td class="column-expose">
                                                 <div class="mcp-toggle-container">
                                                     <label class="mcp-toggle-switch">
                                                         <input type="checkbox" 
                                                                class="mcp-expose-toggle" 
                                                                data-page-id="<?php echo esc_attr($page->ID); ?>"
                                                                <?php checked($is_exposed, '1'); ?> />
                                                         <span class="mcp-toggle-slider"></span>
                                                     </label>
                                                     <span class="mcp-toggle-label">
                                                         <?php echo ($is_exposed === '1') ? esc_html(__('Enabled', 'botailor-modelcontext-integrator-for-ai')) : esc_html(__('Disabled', 'botailor-modelcontext-integrator-for-ai')); ?>
                                                     </span>
                                                 </div>
                                             </td>
                                             <td class="column-permissions">
                                                 <?php 
                                                 $permissions = get_post_meta($page->ID, '_mcia_permissions', true);
                                                 if (empty($permissions)) {
                                                     $permissions = array('read'); // Default to read only
                                                     update_post_meta($page->ID, '_mcia_permissions', $permissions);
                                                 }
                                                 ?>
                                                 <select class="mcp-permissions-select" 
                                                         data-page-id="<?php echo esc_attr($page->ID); ?>" 
                                                         multiple="multiple" 
                                                         size="4">
                                                     <option value="read" <?php selected(in_array('read', $permissions), true); ?>><?php esc_html_e('Read', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                     <option value="write" <?php selected(in_array('write', $permissions), true); ?>><?php esc_html_e('Write', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                     <option value="update" <?php selected(in_array('update', $permissions), true); ?>><?php esc_html_e('Update', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                     <option value="delete" <?php selected(in_array('delete', $permissions), true); ?>><?php esc_html_e('Delete', 'botailor-modelcontext-integrator-for-ai'); ?></option>
                                                 </select>
                                             </td>
                                            <td class="column-description">
                                                <textarea class="mcp-description-input" 
                                                          data-page-id="<?php echo esc_attr($page->ID); ?>"
                                                          placeholder="<?php esc_attr_e('Custom description (optional)', 'botailor-modelcontext-integrator-for-ai'); ?>"
                                                          rows="2"><?php echo esc_textarea($description); ?></textarea>
                                            </td>
                                            <td class="column-preview">
                                                <?php if ($is_exposed) : ?>
                                                    <button type="button" 
                                                            class="button button-small mcp-view-endpoint" 
                                                            data-url="<?php echo esc_url($endpoint_url); ?>">
                                                        <?php esc_html_e('View JSON', 'botailor-modelcontext-integrator-for-ai'); ?>
                                                    </button>
                                                <?php else : ?>
                                                    <span class="mcp-disabled"><?php esc_html_e('Not exposed', 'botailor-modelcontext-integrator-for-ai'); ?></span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

        </div>
        <?php
    }

    /**
     * AJAX handler for saving post meta
     */
    public function ajax_save_post_meta() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mcia_admin_nonce')) {
            wp_die(esc_html(__('Security check failed', 'botailor-modelcontext-integrator-for-ai')));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html(__('Insufficient permissions', 'botailor-modelcontext-integrator-for-ai')));
        }

        $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
        $expose = isset($_POST['expose']) && $_POST['expose'] === '1' ? true : false;
        $description = isset($_POST['description']) ? sanitize_textarea_field(wp_unslash($_POST['description'])) : '';


        // Update post meta
        $expose_result = update_post_meta($post_id, '_mcia_expose', $expose ? '1' : '0');
        $description_result = update_post_meta($post_id, '_mcia_description', $description);


        wp_send_json_success(array(
            'message' => __('Post settings saved successfully', 'botailor-modelcontext-integrator-for-ai'),
            'post_id' => $post_id,
            'expose' => $expose,
            'expose_result' => $expose_result
        ));
    }

    /**
     * AJAX handler for saving page meta
     */
    public function ajax_save_page_meta() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mcia_admin_nonce')) {
            wp_die(esc_html(__('Security check failed', 'botailor-modelcontext-integrator-for-ai')));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html(__('Insufficient permissions', 'botailor-modelcontext-integrator-for-ai')));
        }

        $page_id = isset($_POST['page_id']) ? absint($_POST['page_id']) : 0;
        $expose = isset($_POST['expose']) && $_POST['expose'] === '1' ? true : false;
        $description = isset($_POST['description']) ? sanitize_textarea_field(wp_unslash($_POST['description'])) : '';

        // Update post meta
        $expose_result = update_post_meta($page_id, '_mcia_expose', $expose ? '1' : '0');
        $description_result = update_post_meta($page_id, '_mcia_description', $description);


        wp_send_json_success(array(
            'message' => __('Page settings saved successfully', 'botailor-modelcontext-integrator-for-ai'),
            'page_id' => $page_id,
            'expose' => $expose,
            'expose_result' => $expose_result
        ));
    }

    /**
     * AJAX handler for saving permissions
     */
    public function ajax_save_permissions() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mcia_admin_nonce')) {
            wp_die(esc_html(__('Security check failed', 'botailor-modelcontext-integrator-for-ai')));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html(__('Insufficient permissions', 'botailor-modelcontext-integrator-for-ai')));
        }

        $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
        $permissions = isset($_POST['permissions']) ? array_map('sanitize_text_field', wp_unslash($_POST['permissions'])) : array();
        
        // Sanitize permissions
        $allowed_permissions = array('read', 'write', 'update', 'delete');
        $permissions = array_intersect($permissions, $allowed_permissions);
        
        // Ensure at least read permission is always present
        if (!in_array('read', $permissions)) {
            $permissions[] = 'read';
        }

        // Update post meta
        $result = update_post_meta($post_id, '_mcia_permissions', $permissions);

        wp_send_json_success(array(
            'message' => __('Permissions saved successfully', 'botailor-modelcontext-integrator-for-ai'),
            'post_id' => $post_id,
            'permissions' => $permissions,
            'result' => $result
        ));
    }

    /**
     * AJAX handler for search content
     */
    public function ajax_search_content() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mcia_admin_nonce')) {
            wp_die(esc_html(__('Security check failed', 'botailor-modelcontext-integrator-for-ai')));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html(__('Insufficient permissions', 'botailor-modelcontext-integrator-for-ai')));
        }

        $content_type = isset($_POST['content_type']) ? sanitize_text_field(wp_unslash($_POST['content_type'])) : 'posts';
        $search_term = isset($_POST['search_term']) ? sanitize_text_field(wp_unslash($_POST['search_term'])) : '';
        $page = isset($_POST['page']) ? absint($_POST['page']) : 1;
        $per_page = isset($_POST['per_page']) ? absint($_POST['per_page']) : 25;
        
        if ($per_page < 1 || $per_page > 100) {
            $per_page = 25;
        }
        
        if ($page < 1) {
            $page = 1;
        }

        $args = array(
            'post_type' => $content_type === 'pages' ? 'page' : 'post',
            'post_status' => 'publish',
            'posts_per_page' => $per_page,
            'paged' => $page
        );

        // Add search if term provided
        if (!empty($search_term)) {
            $args['s'] = $search_term;
        }

        // Filter out WooCommerce content
        $query = new WP_Query($args);
        $filtered_posts = array();
        
        foreach ($query->posts as $post) {
            if ($this->is_not_woocommerce_content($post)) {
                $filtered_posts[] = $post;
            }
        }

        // Prepare response data
        $response_data = array();
        foreach ($filtered_posts as $post) {
            $is_exposed = get_post_meta($post->ID, '_mcia_expose', true);
            if ($is_exposed === '') {
                $is_exposed = '1';
            }
            $description = get_post_meta($post->ID, '_mcia_description', true);
            $permissions = get_post_meta($post->ID, '_mcia_permissions', true);
            if (empty($permissions)) {
                $permissions = array('read');
            }
            
            $response_data[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'slug' => $post->post_name,
                'exposed' => $is_exposed === '1',
                'description' => $description,
                'permissions' => $permissions,
                'endpoint_url' => rest_url('mcia/v1/' . $post->post_name)
            );
        }

        wp_send_json_success(array(
            'posts' => $response_data,
            'total' => count($filtered_posts),
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil(count($filtered_posts) / $per_page)
        ));
    }

    /**
     * Check if content is not WooCommerce related
     */
    private function is_not_woocommerce_content($post) {
        // Filter out WooCommerce specific pages
        $woocommerce_pages = array(
            'shop', 'cart', 'checkout', 'my-account', 'lost-password',
            'edit-address', 'view-order', 'orders', 'downloads',
            'edit-account', 'customer-logout'
        );
        
        if (in_array($post->post_name, $woocommerce_pages)) {
            return false;
        }
        
        // Check for WooCommerce templates
        $template = get_page_template_slug($post->ID);
        if (strpos($template, 'woocommerce') !== false) {
            return false;
        }
        
        // Check for WooCommerce shortcodes
        if (strpos($post->post_content, '[woocommerce') !== false) {
            return false;
        }
        
        return true;
    }

    /**
     * AJAX handler for bulk toggle
     */
    public function ajax_bulk_toggle() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'mcia_admin_nonce')) {
            wp_die(esc_html(__('Security check failed', 'botailor-modelcontext-integrator-for-ai')));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html(__('Insufficient permissions', 'botailor-modelcontext-integrator-for-ai')));
        }

        $post_ids = isset($_POST['post_ids']) ? array_map('absint', wp_unslash($_POST['post_ids'])) : array();
        $enable = isset($_POST['enable']) ? (bool) $_POST['enable'] : false;

        foreach ($post_ids as $post_id) {
            update_post_meta($post_id, '_mcia_expose', $enable ? '1' : '0');
        }

        wp_send_json_success(array(
            'message' => sprintf(
                // translators: %d is the number of items updated
                _n('%d item updated', '%d items updated', count($post_ids), 'botailor-modelcontext-integrator-for-ai'),
                count($post_ids)
            )
        ));
    }
}