/**
 * Admin JavaScript for Botailor ModelContext Integrator for AI
 *
 * @package MCIA_API
 */

(function($) {
    'use strict';

    /**
     * MCIA Admin object
     */
    var MCIAAdmin = {
        
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.initBulkActions();
            this.initPagination();
        },

        /**
         * Initialize pagination on page load
         */
        initPagination: function() {
            // Calculate total pages based on current per-page value
            var postsPerPage = parseInt($('#posts-per-page').val()) || 25;
            var pagesPerPage = parseInt($('#pages-per-page').val()) || 25;
            
            var totalPosts = mciaAdmin.totalPosts || 0;
            var totalPages = mciaAdmin.totalPages || 0;
            
            var postsTotalPages = Math.ceil(totalPosts / postsPerPage);
            var pagesTotalPages = Math.ceil(totalPages / pagesPerPage);
            
            // Update pagination info
            $('#posts-total-pages').text(postsTotalPages);
            $('#pages-total-pages').text(pagesTotalPages);
            
            // Hide rows beyond the first page
            this.showPageRows('posts', 1, postsPerPage);
            this.showPageRows('pages', 1, pagesPerPage);
        },

        /**
         * Show only rows for the current page
         */
        showPageRows: function(target, page, perPage) {
            var $tbody = $('.' + target + '-table tbody');
            var $rows = $tbody.find('tr');
            
            var start = (page - 1) * perPage;
            var end = start + perPage;
            
            $rows.hide();
            $rows.slice(start, end).show();
        },

        /**
         * Bind events
         */
        bindEvents: function() {
            // Select all checkboxes
            $('#select-all-posts').on('change', this.toggleAllPosts);
            $('#select-all-pages').on('change', this.toggleAllPages);
            
            // Individual checkboxes
            $('.post-checkbox').on('change', this.updateSelectAllPosts);
            $('.page-checkbox').on('change', this.updateSelectAllPages);
            
            // Bulk actions
            $('.mcp-apply-bulk-posts').on('click', this.applyBulkPosts);
            $('.mcp-apply-bulk-pages').on('click', this.applyBulkPages);
            
            // Expose toggles
            $('.mcp-expose-toggle').on('change', this.handleExposeToggle);
            
            // Auto-save on toggle change
            $('.mcp-expose-toggle').on('change', this.autoSaveToggle);
            
            // Description inputs
            $('.mcp-description-input').on('blur', this.handleDescriptionChange);
            
            // Permissions multiselect
            $('.mcp-permissions-select').on('change', this.handlePermissionsChange);
            
            // Search functionality
            $('.mcp-search-btn').on('click', this.handleSearch);
            $('.mcp-clear-search').on('click', this.handleClearSearch);
            $('.mcp-search-input').on('keypress', this.handleSearchKeypress);
            
            // Pagination
            $('.mcp-per-page-select').on('change', this.handlePerPageChange);
            $('.mcp-prev-page').on('click', this.handlePrevPage);
            $('.mcp-next-page').on('click', this.handleNextPage);
            
            // View endpoint buttons (delegated event)
            $(document).on('click', '.mcp-view-endpoint', this.viewEndpoint);
        },

        /**
         * Initialize bulk actions
         */
        initBulkActions: function() {
            // Update select all checkboxes based on individual checkboxes
            this.updateSelectAllPosts();
            this.updateSelectAllPages();
        },

        /**
         * Toggle all posts
         */
        toggleAllPosts: function() {
            var isChecked = $(this).is(':checked');
            $('.post-checkbox').prop('checked', isChecked);
            MCIAAdmin.updateSelectAllPosts();
        },

        /**
         * Toggle all pages
         */
        toggleAllPages: function() {
            var isChecked = $(this).is(':checked');
            $('.page-checkbox').prop('checked', isChecked);
            MCIAAdmin.updateSelectAllPages();
        },

        /**
         * Update select all posts checkbox
         */
        updateSelectAllPosts: function() {
            var totalPosts = $('.post-checkbox').length;
            var checkedPosts = $('.post-checkbox:checked').length;
            
            if (checkedPosts === 0) {
                $('#select-all-posts').prop('indeterminate', false).prop('checked', false);
            } else if (checkedPosts === totalPosts) {
                $('#select-all-posts').prop('indeterminate', false).prop('checked', true);
            } else {
                $('#select-all-posts').prop('indeterminate', true);
            }
        },

        /**
         * Update select all pages checkbox
         */
        updateSelectAllPages: function() {
            var totalPages = $('.page-checkbox').length;
            var checkedPages = $('.page-checkbox:checked').length;
            
            if (checkedPages === 0) {
                $('#select-all-pages').prop('indeterminate', false).prop('checked', false);
            } else if (checkedPages === totalPages) {
                $('#select-all-pages').prop('indeterminate', false).prop('checked', true);
            } else {
                $('#select-all-pages').prop('indeterminate', true);
            }
        },

        /**
         * Apply bulk action to posts
         */
        applyBulkPosts: function(e) {
            e.preventDefault();
            var action = $('#bulk-action-posts').val();
            var enable = (action === 'enabled');
            MCIAAdmin.bulkTogglePosts(enable);
        },

        /**
         * Apply bulk action to pages
         */
        applyBulkPages: function(e) {
            e.preventDefault();
            var action = $('#bulk-action-pages').val();
            var enable = (action === 'enabled');
            MCIAAdmin.bulkTogglePages(enable);
        },

        /**
         * Bulk toggle posts
         */
        bulkTogglePosts: function(enable) {
            var postIds = [];
            $('.post-checkbox:checked').each(function() {
                var postId = $(this).closest('tr').data('post-id');
                if (postId) {
                    postIds.push(postId);
                }
            });

            if (postIds.length === 0) {
                MCIAAdmin.showNotice('Please select posts to update', 'error');
                return;
            }

            // Update all selected posts individually
            var completed = 0;
            var total = postIds.length;

            postIds.forEach(function(postId) {
                MCIAAdmin.ajaxRequest('mcia_save_post_meta', {
                    post_id: postId,
                    expose: enable ? '1' : '0',
                    description: ''
                }, function(response) {
                    completed++;
                    
                    // Update the UI for this specific post
                    $('.mcp-expose-toggle[data-post-id="' + postId + '"]').each(function() {
                        var $toggle = $(this);
                        var $row = $toggle.closest('tr');
                        $toggle.prop('checked', enable);
                        MCIAAdmin.updatePreviewLink($row, enable);
                        MCIAAdmin.updateToggleLabel($row, enable);
                    });
                    
                    if (completed === total) {
                        MCIAAdmin.showNotice('Bulk action completed successfully', 'success');
                    }
                });
            });
        },

        /**
         * Bulk toggle pages
         */
        bulkTogglePages: function(enable) {
            var pageIds = [];
            $('.page-checkbox:checked').each(function() {
                var pageId = $(this).closest('tr').data('page-id');
                if (pageId) {
                    pageIds.push(pageId);
                }
            });

            if (pageIds.length === 0) {
                MCIAAdmin.showNotice('Please select pages to update', 'error');
                return;
            }

            // Update all selected pages individually
            var completed = 0;
            var total = pageIds.length;

            pageIds.forEach(function(pageId) {
                MCIAAdmin.ajaxRequest('mcia_save_page_meta', {
                    page_id: pageId,
                    expose: enable ? '1' : '0',
                    description: ''
                }, function(response) {
                    completed++;
                    
                    // Update the UI for this specific page
                    $('.mcp-expose-toggle[data-page-id="' + pageId + '"]').each(function() {
                        var $toggle = $(this);
                        var $row = $toggle.closest('tr');
                        $toggle.prop('checked', enable);
                        MCIAAdmin.updatePreviewLink($row, enable);
                        MCIAAdmin.updateToggleLabel($row, enable);
                    });
                    
                    if (completed === total) {
                        MCIAAdmin.showNotice('Bulk action completed successfully', 'success');
                    }
                });
            });
        },

        /**
         * Handle expose toggle change
         */
        handleExposeToggle: function() {
            var $toggle = $(this);
            var $row = $toggle.closest('tr');
            var isEnabled = $toggle.is(':checked');
            
            MCIAAdmin.updatePreviewLink($row, isEnabled);
            MCIAAdmin.updateToggleLabel($row, isEnabled);
        },

        /**
         * Auto-save toggle changes
         */
        autoSaveToggle: function() {
            var $toggle = $(this);
            var $row = $toggle.closest('tr');
            var isEnabled = $toggle.is(':checked');
            var description = $row.find('.mcp-description-input').val();
            
            // Get the post/page ID from the toggle's data attributes
            var postId = $toggle.data('post-id');
            var pageId = $toggle.data('page-id');
            
            console.log('Toggle changed:', {
                postId: postId,
                pageId: pageId,
                isEnabled: isEnabled,
                description: description
            });
            
            if (postId) {
                MCIAAdmin.ajaxRequest('mcia_save_post_meta', {
                    post_id: postId,
                    expose: isEnabled ? '1' : '0',
                    description: description
                }, function(response) {
                    console.log('Post save response:', response);
                    MCIAAdmin.showNotice('Post settings saved', 'success');
                });
            } else if (pageId) {
                MCIAAdmin.ajaxRequest('mcia_save_page_meta', {
                    page_id: pageId,
                    expose: isEnabled ? '1' : '0',
                    description: description
                }, function(response) {
                    console.log('Page save response:', response);
                    MCIAAdmin.showNotice('Page settings saved', 'success');
                });
            } else {
                console.error('No post or page ID found for toggle');
            }
        },

        /**
         * Handle description change
         */
        handleDescriptionChange: function() {
            // Description changes are saved automatically on blur
            // This could be enhanced with auto-save functionality
        },

        /**
         * Handle permissions change
         */
        handlePermissionsChange: function() {
            var $select = $(this);
            var $row = $select.closest('tr');
            var postId = $select.data('post-id') || $select.data('page-id');
            var permissions = $select.val() || [];
            
            // Ensure read permission is always present
            if (permissions.indexOf('read') === -1) {
                permissions.push('read');
                $select.val(permissions);
            }
            
            MCIAAdmin.ajaxRequest('mcia_save_permissions', {
                post_id: postId,
                permissions: permissions
            }, function(response) {
                MCIAAdmin.showNotice('Permissions saved successfully', 'success');
            });
        },


        /**
         * Update preview link
         */
        updatePreviewLink: function($row, isEnabled) {
            var $previewCell = $row.find('.column-preview');
            
            if (isEnabled) {
                var slug = $row.find('.column-slug code').text();
                var endpointUrl = mciaAdmin.ajaxUrl.replace('admin-ajax.php', '') + 'wp-json/mcia/v1/' + slug;
                $previewCell.html('<button type="button" class="button button-small mcp-view-endpoint" data-url="' + endpointUrl + '">View JSON</button>');
            } else {
                $previewCell.html('<span class="mcp-disabled">Not exposed</span>');
            }
        },

        /**
         * Update toggle label
         */
        updateToggleLabel: function($row, isEnabled) {
            var $label = $row.find('.mcp-toggle-label');
            $label.text(isEnabled ? 'Enabled' : 'Disabled');
            
            if (isEnabled) {
                $label.removeClass('mcp-disabled').addClass('mcp-enabled');
            } else {
                $label.removeClass('mcp-enabled').addClass('mcp-disabled');
            }
        },

        /**
         * Handle search
         */
        handleSearch: function() {
            var $button = $(this);
            var target = $button.data('target');
            var $searchInput = $('#search-' + target);
            var searchTerm = $searchInput.val().trim();
            
            MCIAAdmin.searchContent(target, searchTerm, 1);
        },

        /**
         * Handle clear search
         */
        handleClearSearch: function() {
            var $button = $(this);
            var target = $button.data('target');
            var $searchInput = $('#search-' + target);
            
            $searchInput.val('');
            MCIAAdmin.searchContent(target, '', 1);
        },

        /**
         * Handle search keypress
         */
        handleSearchKeypress: function(e) {
            if (e.which === 13) { // Enter key
                var $input = $(this);
                var target = $input.attr('id').replace('search-', '');
                var searchTerm = $input.val().trim();
                
                MCIAAdmin.searchContent(target, searchTerm, 1);
            }
        },

        /**
         * Handle per page change
         */
        handlePerPageChange: function() {
            var $select = $(this);
            var target = $select.attr('id').replace('-per-page', '');
            var perPage = parseInt($select.val());
            
            // Recalculate total pages
            var totalItems = target === 'posts' ? mciaAdmin.totalPosts : mciaAdmin.totalPages;
            var totalPages = Math.ceil(totalItems / perPage);
            
            $('#' + target + '-total-pages').text(totalPages);
            $('#' + target + '-current-page').text(1);
            
            // Show first page with new per-page value
            MCIAAdmin.showPageRows(target, 1, perPage);
            MCIAAdmin.updatePaginationButtons(target, 1, totalPages);
        },

        /**
         * Handle previous page
         */
        handlePrevPage: function() {
            var $button = $(this);
            var target = $button.data('target');
            var currentPage = parseInt($('#' + target + '-current-page').text());
            
            if (currentPage > 1) {
                var newPage = currentPage - 1;
                var perPage = parseInt($('#' + target + '-per-page').val());
                var totalPages = parseInt($('#' + target + '-total-pages').text());
                
                $('#' + target + '-current-page').text(newPage);
                MCIAAdmin.showPageRows(target, newPage, perPage);
                MCIAAdmin.updatePaginationButtons(target, newPage, totalPages);
            }
        },

        /**
         * Handle next page
         */
        handleNextPage: function() {
            var $button = $(this);
            var target = $button.data('target');
            var currentPage = parseInt($('#' + target + '-current-page').text());
            var totalPages = parseInt($('#' + target + '-total-pages').text());
            
            if (currentPage < totalPages) {
                var newPage = currentPage + 1;
                var perPage = parseInt($('#' + target + '-per-page').val());
                
                $('#' + target + '-current-page').text(newPage);
                MCIAAdmin.showPageRows(target, newPage, perPage);
                MCIAAdmin.updatePaginationButtons(target, newPage, totalPages);
            }
        },

        /**
         * Update pagination button states
         */
        updatePaginationButtons: function(target, currentPage, totalPages) {
            var $prevBtn = $('.mcp-prev-page[data-target="' + target + '"]');
            var $nextBtn = $('.mcp-next-page[data-target="' + target + '"]');
            
            $prevBtn.prop('disabled', currentPage <= 1);
            $nextBtn.prop('disabled', currentPage >= totalPages);
        },

        /**
         * Search content via AJAX
         */
        searchContent: function(target, searchTerm, page, perPage) {
            if (typeof perPage === 'undefined') {
                perPage = parseInt($('#' + target + '-per-page').val());
            }
            
            MCIAAdmin.ajaxRequest('mcia_search_content', {
                content_type: target,
                search_term: searchTerm,
                page: page,
                per_page: perPage
            }, function(response) {
                MCIAAdmin.updateContentTable(target, response.data.posts);
                MCIAAdmin.updatePagination(target, response.data);
            });
        },

        /**
         * Update content table with search results
         */
        updateContentTable: function(target, posts) {
            var $tbody = $('.' + target + '-table tbody');
            $tbody.empty();
            
            if (posts.length === 0) {
                $tbody.html('<tr><td colspan="7" class="no-results">No results found.</td></tr>');
                return;
            }
            
            posts.forEach(function(post) {
                var permissionsHtml = '';
                post.permissions.forEach(function(perm) {
                    permissionsHtml += '<option value="' + perm + '" selected>' + perm.charAt(0).toUpperCase() + perm.slice(1) + '</option>';
                });
                
                var rowHtml = '<tr data-' + target.slice(0, -1) + '-id="' + post.id + '">' +
                    '<th class="check-column"><input type="checkbox" class="' + target.slice(0, -1) + '-checkbox" /></th>' +
                    '<td class="column-title"><strong>' + post.title + '</strong></td>' +
                    '<td class="column-slug"><code>' + post.slug + '</code></td>' +
                    '<td class="column-expose">' +
                        '<div class="mcp-toggle-container">' +
                            '<label class="mcp-toggle-switch">' +
                                '<input type="checkbox" class="mcp-expose-toggle" data-' + target.slice(0, -1) + '-id="' + post.id + '" ' + (post.exposed ? 'checked' : '') + ' />' +
                                '<span class="mcp-toggle-slider"></span>' +
                            '</label>' +
                            '<span class="mcp-toggle-label">' + (post.exposed ? 'Enabled' : 'Disabled') + '</span>' +
                        '</div>' +
                    '</td>' +
                    '<td class="column-permissions">' +
                        '<select class="mcp-permissions-select" data-' + target.slice(0, -1) + '-id="' + post.id + '" multiple="multiple" size="4">' +
                            '<option value="read"' + (post.permissions.indexOf('read') !== -1 ? ' selected' : '') + '>Read</option>' +
                            '<option value="write"' + (post.permissions.indexOf('write') !== -1 ? ' selected' : '') + '>Write</option>' +
                            '<option value="update"' + (post.permissions.indexOf('update') !== -1 ? ' selected' : '') + '>Update</option>' +
                            '<option value="delete"' + (post.permissions.indexOf('delete') !== -1 ? ' selected' : '') + '>Delete</option>' +
                        '</select>' +
                    '</td>' +
                    '<td class="column-description">' +
                        '<textarea class="mcp-description-input" data-' + target.slice(0, -1) + '-id="' + post.id + '" placeholder="Custom description (optional)" rows="2">' + (post.description || '') + '</textarea>' +
                    '</td>' +
                    '<td class="column-preview">' +
                        (post.exposed ? 
                            '<button type="button" class="button button-small mcp-view-endpoint" data-url="' + post.endpoint_url + '">View JSON</button>' :
                            '<span class="mcp-disabled">Not exposed</span>'
                        ) +
                    '</td>' +
                '</tr>';
                
                $tbody.append(rowHtml);
            });
            
            // Re-bind events for new content
            MCIAAdmin.bindEvents();
        },

        /**
         * Update pagination controls
         */
        updatePagination: function(target, data) {
            $('#' + target + '-current-page').text(data.page);
            $('#' + target + '-total-pages').text(data.total_pages);
            
            var $prevBtn = $('.mcp-prev-page[data-target="' + target + '"]');
            var $nextBtn = $('.mcp-next-page[data-target="' + target + '"]');
            
            $prevBtn.prop('disabled', data.page <= 1);
            $nextBtn.prop('disabled', data.page >= data.total_pages);
        },

        /**
         * View endpoint JSON
         */
        viewEndpoint: function(e) {
            e.preventDefault();
            var $button = $(this);
            var endpointUrl = $button.data('url');
            
            // Show loading state
            var originalText = $button.text();
            $button.text('Loading...').prop('disabled', true);
            
            // Use WordPress REST API nonce for authentication (already logged in as admin)
            var restNonce = mciaAdmin.restNonce || '';
            
            console.log('Endpoint URL:', endpointUrl);
            console.log('Using WordPress session authentication');
            
            var ajaxSettings = {
                url: endpointUrl,
                type: 'GET',
                dataType: 'json',
                xhrFields: {
                    withCredentials: true  // Send cookies with request
                },
                beforeSend: function(xhr) {
                    // Use WordPress REST API nonce for authentication
                    if (restNonce) {
                        xhr.setRequestHeader('X-WP-Nonce', restNonce);
                    }
                },
                success: function(data) {
                    $button.text(originalText).prop('disabled', false);
                    MCIAAdmin.showJsonModal(data, endpointUrl);
                },
                error: function(xhr, status, error) {
                    $button.text(originalText).prop('disabled', false);
                    console.error('Error loading endpoint:', xhr, status, error);
                    console.error('Response:', xhr.responseText);
                    
                    var errorMsg = 'Error loading endpoint data';
                    if (xhr.status === 401) {
                        errorMsg = 'Authentication failed. Your WordPress session may have expired. Try refreshing the page.';
                    } else if (xhr.status === 404) {
                        errorMsg = 'Endpoint not found.';
                    } else if (xhr.status === 0) {
                        errorMsg = 'Network error. Please check your internet connection.';
                    }
                    
                    MCIAAdmin.showNotice(errorMsg, 'error');
                }
            };
            
            $.ajax(ajaxSettings);
        },

        /**
         * Show JSON in modal
         */
        showJsonModal: function(data, url) {
            console.log('Showing JSON modal for:', url);
            console.log('Data:', data);
            
            var jsonString = JSON.stringify(data, null, 2);
            
            // Remove any existing modals first
            $('.mcp-json-modal').remove();
            
            var modal = $('<div class="mcp-json-modal">' +
                '<div class="mcp-json-modal-content">' +
                    '<div class="mcp-json-modal-header">' +
                        '<h3>Endpoint JSON Response</h3>' +
                        '<button type="button" class="mcp-close-modal" title="Close">×</button>' +
                    '</div>' +
                    '<div class="mcp-json-modal-body">' +
                        '<div style="margin-bottom: 10px;"><strong>Endpoint URL:</strong> <code>' + url + '</code></div>' +
                        '<pre><code>' + jsonString + '</code></pre>' +
                    '</div>' +
                    '<div class="mcp-json-modal-footer">' +
                        '<button type="button" class="button mcp-close-modal">Close</button>' +
                        '<a href="' + url + '" target="_blank" class="button button-primary">Open in New Tab</a>' +
                    '</div>' +
                '</div>' +
            '</div>');
            
            $('body').append(modal);
            
            // Bind close events
            modal.find('.mcp-close-modal').on('click', function() {
                console.log('Closing modal');
                modal.fadeOut(200, function() {
                    modal.remove();
                });
            });
            
            // Close on background click
            modal.on('click', function(e) {
                if ($(e.target).hasClass('mcp-json-modal')) {
                    console.log('Closing modal - background click');
                    modal.fadeOut(200, function() {
                        modal.remove();
                    });
                }
            });
            
            console.log('Modal appended to body');
        },

        /**
         * Show notice
         */
        showNotice: function(message, type) {
            var noticeClass = type === 'success' ? 'mcp-notice-success' : 'mcp-notice-error';
            var notice = $('<div class="mcp-notice ' + noticeClass + '"><p>' + message + '</p></div>');
            
            $('.wrap h1').after(notice);
            
            setTimeout(function() {
                notice.fadeOut(300, function() {
                    notice.remove();
                });
            }, 3000);
        },

        /**
         * AJAX request helper
         */
        ajaxRequest: function(action, data, callback) {
            data = data || {};
            data.action = action;
            data.nonce = mciaAdmin.nonce;
            
            $.ajax({
                url: mciaAdmin.ajaxUrl,
                type: 'POST',
                data: data,
                success: function(response) {
                    if (response.success) {
                        if (callback) callback(response.data);
                    } else {
                        MCIAAdmin.showNotice(response.data || mciaAdmin.strings.saveError, 'error');
                    }
                },
                error: function() {
                    MCIAAdmin.showNotice(mciaAdmin.strings.saveError, 'error');
                }
            });
        }
    };

        /**
         * Initialize when document is ready
         */
        $(document).ready(function() {
            MCIAAdmin.init();
            
            // Test function to verify toggles are working
            console.log('MCP Admin initialized');
            console.log('Found toggles:', $('.mcp-expose-toggle').length);
            $('.mcp-expose-toggle').each(function(index) {
                var $toggle = $(this);
                var postId = $toggle.data('post-id');
                var pageId = $toggle.data('page-id');
                console.log('Toggle ' + index + ':', {
                    postId: postId,
                    pageId: pageId,
                    checked: $toggle.is(':checked')
                });
            });
        });
 
})(jQuery);