<?php
/**
 * Collects update and patch posture security signals
 * 
 * Data collected:
 * - Pending WordPress updates
 * - Plugin updates available
 * - Theme updates available
 * - Auto-update configuration
 * 
 * Note: This collector is READ-ONLY. It reads existing WordPress update
 * configuration options but does NOT modify WordPress update behavior
 * or intercept the update process in any way.
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Updates_Collector {
    
    /**
     * Collect all update-related data
     */
    public function collect() {
        // Force update check refresh
        wp_version_check([], true);
        wp_update_plugins();
        wp_update_themes();
        
        return [
            'core' => $this->collect_core_updates(),
            'plugins' => $this->collect_plugin_updates(),
            'themes' => $this->collect_theme_updates(),
            'auto_updates' => $this->collect_auto_update_config(),
            'update_history' => $this->collect_update_history(),
        ];
    }
    
    /**
     * Collect WordPress core update status
     */
    private function collect_core_updates() {
        global $wp_version;
        
        $updates = get_site_transient('update_core');
        
        $result = [
            'current_version' => $wp_version,
            'updates_available' => [],
            'is_latest' => true,
            'update_type' => null,
        ];
        
        if (!empty($updates->updates)) {
            foreach ($updates->updates as $update) {
                if ($update->response === 'upgrade') {
                    $result['is_latest'] = false;
                    $result['updates_available'][] = [
                        'version' => $update->version,
                        'download_url' => $update->download ?? null,
                        'php_version_required' => $update->php_version ?? null,
                        'mysql_version_required' => $update->mysql_version ?? null,
                    ];
                    
                    // Determine update type
                    $current_parts = explode('.', $wp_version);
                    $new_parts = explode('.', $update->version);
                    
                    if ($current_parts[0] !== $new_parts[0]) {
                        $result['update_type'] = 'major';
                    } elseif ($current_parts[1] !== $new_parts[1]) {
                        $result['update_type'] = 'minor';
                    } else {
                        $result['update_type'] = 'security';
                    }
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Collect plugin update status
     */
    private function collect_plugin_updates() {
        $updates = get_site_transient('update_plugins');
        $active_plugins = get_option('active_plugins', []);
        
        $result = [
            'total_with_updates' => 0,
            'active_with_updates' => 0,
            'updates' => [],
        ];
        
        if (!empty($updates->response)) {
            $result['total_with_updates'] = count($updates->response);
            
            foreach ($updates->response as $plugin_file => $update_info) {
                $is_active = in_array($plugin_file, $active_plugins);
                
                if ($is_active) {
                    $result['active_with_updates']++;
                }
                
                $result['updates'][] = [
                    'plugin' => $plugin_file,
                    'slug' => $update_info->slug ?? null,
                    'current_version' => $update_info->Version ?? 'unknown',
                    'new_version' => $update_info->new_version,
                    'is_active' => $is_active,
                    'requires_php' => $update_info->requires_php ?? null,
                    'requires_wp' => $update_info->requires ?? null,
                    'tested_up_to' => $update_info->tested ?? null,
                    'compatibility' => $this->check_update_compatibility($update_info),
                ];
            }
        }
        
        // Check for plugins without updates available
        if (!empty($updates->no_update)) {
            $result['up_to_date_count'] = count($updates->no_update);
        }
        
        return $result;
    }
    
    /**
     * Collect theme update status
     */
    private function collect_theme_updates() {
        $updates = get_site_transient('update_themes');
        $current_theme = wp_get_theme();
        
        $result = [
            'total_with_updates' => 0,
            'active_theme_needs_update' => false,
            'updates' => [],
        ];
        
        if (!empty($updates->response)) {
            $result['total_with_updates'] = count($updates->response);
            
            foreach ($updates->response as $theme_slug => $update_info) {
                $is_active = $theme_slug === $current_theme->get_stylesheet() || 
                             $theme_slug === $current_theme->get_template();
                
                if ($is_active) {
                    $result['active_theme_needs_update'] = true;
                }
                
                $result['updates'][] = [
                    'theme' => $theme_slug,
                    'current_version' => $update_info['current_version'] ?? 'unknown',
                    'new_version' => $update_info['new_version'],
                    'is_active' => $is_active,
                    'requires_php' => $update_info['requires_php'] ?? null,
                    'requires_wp' => $update_info['requires'] ?? null,
                ];
            }
        }
        
        return $result;
    }
    
    /**
     * Collect auto-update configuration
     */
    private function collect_auto_update_config() {
        $config = [
            'core' => $this->get_core_auto_update_status(),
            'plugins' => $this->get_plugins_auto_update_status(),
            'themes' => $this->get_themes_auto_update_status(),
            'translations' => $this->get_translations_auto_update_status(),
        ];
        
        return $config;
    }
    
    /**
     * Get WordPress core auto-update status
     */
    private function get_core_auto_update_status() {
        // Check for constant
        if (defined('WP_AUTO_UPDATE_CORE')) {
            $setting = WP_AUTO_UPDATE_CORE;
            if ($setting === false) {
                return ['enabled' => false, 'type' => 'disabled'];
            }
            if ($setting === 'minor') {
                return ['enabled' => true, 'type' => 'minor'];
            }
            if ($setting === true || $setting === 'beta' || $setting === 'rc') {
                return ['enabled' => true, 'type' => 'all'];
            }
        }
        
        // Check filter
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
        $auto_update = apply_filters('allow_major_auto_core_updates', true);
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
        $minor_auto_update = apply_filters('allow_minor_auto_core_updates', true);
        
        if ($auto_update && $minor_auto_update) {
            return ['enabled' => true, 'type' => 'all'];
        } elseif ($minor_auto_update) {
            return ['enabled' => true, 'type' => 'minor'];
        }
        
        return ['enabled' => false, 'type' => 'disabled'];
    }
    
    /**
     * Get plugins auto-update status
     * 
     * Note: This function only reads update configuration, it does not modify update behavior.
     * phpcs:disable WordPress.WP.CronInterval -- Not modifying cron, just reading config
     */
    private function get_plugins_auto_update_status() {
        // phpcs:ignore WordPress.WP.CronInterval -- Reading existing option, not modifying update behavior
        $auto_updates = get_option('auto_update_plugins', []);
        $active_plugins = get_option('active_plugins', []);
        
        $enabled_count = 0;
        foreach ($active_plugins as $plugin) {
            if (in_array($plugin, $auto_updates)) {
                $enabled_count++;
            }
        }
        
        return [
            'total_enabled' => count($auto_updates),
            'active_plugins_with_auto_update' => $enabled_count,
            'active_plugins_total' => count($active_plugins),
            'coverage_percentage' => count($active_plugins) > 0 
                ? round(($enabled_count / count($active_plugins)) * 100) 
                : 0,
        ];
    }
    
    /**
     * Get themes auto-update status
     */
    private function get_themes_auto_update_status() {
        $auto_updates = get_option('auto_update_themes', []);
        $current_theme = wp_get_theme();
        
        $active_theme_auto_update = in_array($current_theme->get_stylesheet(), $auto_updates);
        
        // Check parent theme if child
        $parent_auto_update = null;
        if ($current_theme->parent()) {
            $parent_auto_update = in_array($current_theme->get_template(), $auto_updates);
        }
        
        return [
            'total_enabled' => count($auto_updates),
            'active_theme_auto_update' => $active_theme_auto_update,
            'parent_theme_auto_update' => $parent_auto_update,
        ];
    }
    
    /**
     * Get translations auto-update status
     */
    private function get_translations_auto_update_status() {
        // Translation auto-updates are on by default
        $enabled = !defined('AUTOMATIC_UPDATER_DISABLED') || !AUTOMATIC_UPDATER_DISABLED;
        
        // Check for filter
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
        $filtered = apply_filters('auto_update_translation', $enabled);
        
        return [
            'enabled' => $filtered,
        ];
    }
    
    /**
     * Collect recent update history
     */
    private function collect_update_history() {
        global $wpdb;
        
        // Get from WordPress update history option
        $update_history = get_option('auto_updater_log', []);
        
        // Get core update history from options
        $cache_key = 'boonrisk_update_history';
        $wp_options = wp_cache_get($cache_key, 'boonrisk-site-security-check-report');
        
        if (false === $wp_options) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cached below
            $wp_options = $wpdb->get_results(
                "SELECT option_name, option_value FROM {$wpdb->options} 
                 WHERE option_name LIKE '_site_transient_update_%'"
            );
            wp_cache_set($cache_key, $wp_options, 'boonrisk', HOUR_IN_SECONDS);
        }
        
        return [
            'auto_update_log' => array_slice($update_history, -10), // Last 10 entries
            'last_checked' => [
                'core' => get_site_transient('update_core') ? 
                    get_site_option('_site_transient_timeout_update_core') : null,
                'plugins' => get_site_transient('update_plugins') ? 
                    get_site_option('_site_transient_timeout_update_plugins') : null,
                'themes' => get_site_transient('update_themes') ? 
                    get_site_option('_site_transient_timeout_update_themes') : null,
            ],
        ];
    }
    
    /**
     * Check compatibility of an update
     */
    private function check_update_compatibility($update_info) {
        global $wp_version;
        
        $compatibility = [
            'php_compatible' => true,
            'wp_compatible' => true,
        ];
        
        // Check PHP compatibility
        if (!empty($update_info->requires_php)) {
            $compatibility['php_compatible'] = version_compare(
                phpversion(), 
                $update_info->requires_php, 
                '>='
            );
        }
        
        // Check WordPress compatibility
        if (!empty($update_info->requires)) {
            $compatibility['wp_compatible'] = version_compare(
                $wp_version, 
                $update_info->requires, 
                '>='
            );
        }
        
        return $compatibility;
    }
}

