<?php
/**
 * Collects plugins and themes security signals
 * 
 * Data collected:
 * - Active plugins with versions
 * - Active theme info
 * - Must-use plugins
 * - Drop-ins
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Plugins_Collector {
    
    // Known security plugins for detection
    private $known_security_plugins = [
        'wordfence' => 'wordfence/wordfence.php',
        'sucuri' => 'sucuri-scanner/sucuri.php',
        'ithemes_security' => 'better-wp-security/better-wp-security.php',
        'all_in_one_security' => 'all-in-one-wp-security-and-firewall/wp-security.php',
        'jetpack' => 'jetpack/jetpack.php',
        'defender' => 'defender-security/wp-defender.php',
        'bulletproof_security' => 'bulletproof-security/bulletproof-security.php',
        'wp_cerber' => 'wp-cerber/wp-cerber.php',
        'secupress' => 'secupress/secupress.php',
    ];
    
    // Known 2FA plugins
    private $known_2fa_plugins = [
        'google_authenticator' => 'google-authenticator/google-authenticator.php',
        'two_factor' => 'two-factor/two-factor.php',
        'wordfence_2fa' => 'wordfence/wordfence.php',
        'duo_two_factor' => 'duo-two-factor/duo-web.php',
        'wp_2fa' => 'wp-2fa/wp-2fa.php',
        'miniOrange' => 'miniorange-2-factor-authentication/miniorange_2_factor_settings.php',
    ];
    
    /**
     * Collect all plugins and themes data
     */
    public function collect() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        return [
            'active_plugins' => $this->collect_active_plugins(),
            'all_plugins_count' => count(get_plugins()),
            'must_use_plugins' => $this->collect_must_use_plugins(),
            'dropins' => $this->collect_dropins(),
            'theme' => $this->collect_theme_info(),
            'security_plugins' => $this->detect_security_plugins(),
            '2fa_plugins' => $this->detect_2fa_plugins(),
            'caching_plugins' => $this->detect_caching_plugins(),
        ];
    }
    
    /**
     * Collect active plugins info
     */
    private function collect_active_plugins() {
        $active_plugins = get_option('active_plugins', []);
        $all_plugins = get_plugins();
        $plugins = [];
        
        foreach ($active_plugins as $plugin_file) {
            if (!isset($all_plugins[$plugin_file])) {
                continue;
            }
            
            $plugin_data = $all_plugins[$plugin_file];
            
            $plugins[] = [
                'file' => $plugin_file,
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'author' => $plugin_data['Author'],
                'plugin_uri' => $plugin_data['PluginURI'],
                'text_domain' => $plugin_data['TextDomain'],
                'network_activated' => is_multisite() ? is_plugin_active_for_network($plugin_file) : false,
                'slug' => $this->extract_slug($plugin_file),
            ];
        }
        
        return $plugins;
    }
    
    /**
     * Collect must-use plugins
     */
    private function collect_must_use_plugins() {
        $mu_plugins = get_mu_plugins();
        $plugins = [];
        
        foreach ($mu_plugins as $file => $data) {
            $plugins[] = [
                'file' => $file,
                'name' => $data['Name'],
                'version' => $data['Version'],
            ];
        }
        
        return $plugins;
    }
    
    /**
     * Collect drop-ins
     */
    private function collect_dropins() {
        $dropins = get_dropins();
        $dropin_info = [];
        
        // Known drop-in files
        $known_dropins = [
            'advanced-cache.php' => 'Advanced caching drop-in',
            'db.php' => 'Database drop-in',
            'db-error.php' => 'Database error handler',
            'install.php' => 'Custom installer',
            'maintenance.php' => 'Maintenance mode handler',
            'object-cache.php' => 'Object caching drop-in',
            'php-error.php' => 'PHP error handler',
            'fatal-error-handler.php' => 'Fatal error handler',
        ];
        
        foreach ($dropins as $file => $data) {
            $dropin_info[] = [
                'file' => $file,
                'name' => $data['Name'] ?: ($known_dropins[$file] ?? 'Unknown drop-in'),
                'description' => $data['Description'],
            ];
        }
        
        // Also check for existence of common drop-ins
        foreach ($known_dropins as $file => $description) {
            if (file_exists(WP_CONTENT_DIR . '/' . $file) && !isset($dropins[$file])) {
                $dropin_info[] = [
                    'file' => $file,
                    'name' => $description,
                    'description' => 'Present but not in drop-ins list',
                ];
            }
        }
        
        return $dropin_info;
    }
    
    /**
     * Collect active theme info
     */
    private function collect_theme_info() {
        $theme = wp_get_theme();
        $parent = $theme->parent();
        
        return [
            'name' => $theme->get('Name'),
            'version' => $theme->get('Version'),
            'author' => $theme->get('Author'),
            'theme_uri' => $theme->get('ThemeURI'),
            'template' => $theme->get_template(),
            'stylesheet' => $theme->get_stylesheet(),
            'is_child_theme' => $theme->parent() !== false,
            'parent_theme' => $parent ? [
                'name' => $parent->get('Name'),
                'version' => $parent->get('Version'),
            ] : null,
            'text_domain' => $theme->get('TextDomain'),
            'requires_wp' => $theme->get('RequiresWP'),
            'requires_php' => $theme->get('RequiresPHP'),
        ];
    }
    
    /**
     * Detect installed security plugins
     */
    private function detect_security_plugins() {
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($this->known_security_plugins as $name => $file) {
            if (in_array($file, $active_plugins)) {
                $detected[] = [
                    'identifier' => $name,
                    'file' => $file,
                    'is_active' => true,
                ];
            }
        }
        
        return [
            'detected' => $detected,
            'has_security_plugin' => !empty($detected),
        ];
    }
    
    /**
     * Detect 2FA plugins
     */
    private function detect_2fa_plugins() {
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($this->known_2fa_plugins as $name => $file) {
            if (in_array($file, $active_plugins)) {
                $detected[] = [
                    'identifier' => $name,
                    'file' => $file,
                ];
            }
        }
        
        return [
            'detected' => $detected,
            'has_2fa' => !empty($detected),
        ];
    }
    
    /**
     * Detect caching plugins
     */
    private function detect_caching_plugins() {
        $known_caching = [
            'wp_super_cache' => 'wp-super-cache/wp-cache.php',
            'w3_total_cache' => 'w3-total-cache/w3-total-cache.php',
            'wp_fastest_cache' => 'wp-fastest-cache/wpFastestCache.php',
            'litespeed_cache' => 'litespeed-cache/litespeed-cache.php',
            'wp_rocket' => 'wp-rocket/wp-rocket.php',
            'autoptimize' => 'autoptimize/autoptimize.php',
            'cache_enabler' => 'cache-enabler/cache-enabler.php',
            'hummingbird' => 'hummingbird-performance/wp-hummingbird.php',
            'sg_optimizer' => 'sg-cachepress/sg-cachepress.php',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($known_caching as $name => $file) {
            if (in_array($file, $active_plugins)) {
                $detected[] = $name;
            }
        }
        
        // Check for object cache drop-in
        $has_object_cache = file_exists(WP_CONTENT_DIR . '/object-cache.php');
        
        return [
            'plugins' => $detected,
            'has_page_cache' => !empty($detected),
            'has_object_cache' => $has_object_cache,
        ];
    }
    
    /**
     * Extract plugin slug from file path
     */
    private function extract_slug($plugin_file) {
        $parts = explode('/', $plugin_file);
        return $parts[0];
    }
}

