<?php
/**
 * Extended Collector for Additional Security Signals
 * 
 * Collects additional evidence required for full control evaluation:
 * - Backup plugins
 * - Bot protection plugins
 * - Image optimization plugins
 * - Activity log plugins
 * - Inactive plugins/themes
 * - User activity status
 * - File permissions
 * - WP-Cron events
 * - Production environment indicators
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Extended_Collector {
    
    /**
     * Collect all extended security signals
     */
    public function collect() {
        return [
            'backup_plugins' => $this->detect_backup_plugins(),
            'bot_protection' => $this->detect_bot_protection(),
            'image_optimization' => $this->detect_image_optimization(),
            'activity_log' => $this->detect_activity_log(),
            'inactive_plugins' => $this->collect_inactive_plugins(),
            'themes' => $this->collect_theme_info(),
            'inactive_admins' => $this->detect_inactive_admins(),
            'filesystem' => $this->collect_filesystem_permissions(),
            'cron' => $this->collect_cron_events(),
            'production' => $this->detect_production_environment(),
            'webhooks' => $this->detect_webhook_support(),
        ];
    }
    
    /**
     * Detect backup plugins
     */
    private function detect_backup_plugins() {
        $known_backup_plugins = [
            'updraftplus/updraftplus.php' => 'UpdraftPlus',
            'backwpup/backwpup.php' => 'BackWPup',
            'duplicator/duplicator.php' => 'Duplicator',
            'backupbuddy/backupbuddy.php' => 'BackupBuddy',
            'jetpack/jetpack.php' => 'Jetpack Backup',
            'blogvault-real-time-backup/blogvault.php' => 'BlogVault',
            'backup-backup/backup-backup.php' => 'Backup Migration',
            'all-in-one-wp-migration/all-in-one-wp-migration.php' => 'All-in-One WP Migration',
            'wp-db-backup/wp-db-backup.php' => 'WP-DB-Backup',
            'total-upkeep/flavor.php' => 'Total Upkeep',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($known_backup_plugins as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $detected[] = $name;
            }
        }
        
        // Check for Jetpack Backup specifically (requires plan)
        if (in_array('jetpack/jetpack.php', $active_plugins)) {
            // Jetpack backup requires a paid plan, we can only detect presence
            $has_jetpack_backup = class_exists('Jetpack') && method_exists('Jetpack', 'is_module_active');
        }
        
        return [
            'has_backup_plugin' => !empty($detected),
            'providers' => $detected,
        ];
    }
    
    /**
     * Detect bot protection plugins
     */
    private function detect_bot_protection() {
        $known_bot_protection = [
            'wordfence/wordfence.php' => 'Wordfence',
            'sucuri-scanner/sucuri.php' => 'Sucuri',
            'better-wp-security/better-wp-security.php' => 'iThemes Security',
            'wp-cerber/wp-cerber.php' => 'WP Cerber',
            'all-in-one-wp-security-and-firewall/wp-security.php' => 'All In One WP Security',
            'advanced-nocaptcha-recaptcha/advanced-nocaptcha-recaptcha.php' => 'Advanced noCaptcha reCaptcha',
            'google-captcha/google-captcha.php' => 'reCaptcha by BestWebSoft',
            'really-simple-captcha/really-simple-captcha.php' => 'Really Simple CAPTCHA',
            'login-recaptcha/login-recaptcha.php' => 'Login reCAPTCHA',
            'jetpack/jetpack.php' => 'Jetpack Protect',
            'akismet/akismet.php' => 'Akismet',
            'antispam-bee/antispam_bee.php' => 'Antispam Bee',
            'hcaptcha-for-forms-and-more/hcaptcha.php' => 'hCaptcha',
            'cloudflare/cloudflare.php' => 'Cloudflare',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($known_bot_protection as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $detected[] = $name;
            }
        }
        
        return [
            'has_bot_protection' => !empty($detected),
            'providers' => $detected,
        ];
    }
    
    /**
     * Detect image optimization plugins
     */
    private function detect_image_optimization() {
        $known_image_optimization = [
            'wp-smushit/wp-smush.php' => 'Smush',
            'shortpixel-image-optimiser/wp-shortpixel.php' => 'ShortPixel',
            'ewww-image-optimizer/ewww-image-optimizer.php' => 'EWWW Image Optimizer',
            'imagify/imagify.php' => 'Imagify',
            'tiny-compress-images/tiny-compress-images.php' => 'TinyPNG',
            'optimus/optimus.php' => 'Optimus',
            'wp-optimize/wp-optimize.php' => 'WP-Optimize',
            'regenerate-thumbnails/regenerate-thumbnails.php' => 'Regenerate Thumbnails',
            'enable-media-replace/enable-media-replace.php' => 'Enable Media Replace',
            'kraken-image-optimizer/kraken.php' => 'Kraken Image Optimizer',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($known_image_optimization as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $detected[] = $name;
            }
        }
        
        return [
            'has_plugin' => !empty($detected),
            'providers' => $detected,
        ];
    }
    
    /**
     * Detect activity log plugins
     */
    private function detect_activity_log() {
        $known_activity_log = [
            'wp-security-audit-log/wp-security-audit-log.php' => 'WP Activity Log',
            'simple-history/index.php' => 'Simple History',
            'activity-log/activity-log.php' => 'Activity Log',
            'stream/stream.php' => 'Stream',
            'user-activity-log/user_activity_log.php' => 'User Activity Log',
            'aryo-activity-log/aryo-activity-log.php' => 'Activity Log',
            'sucuri-scanner/sucuri.php' => 'Sucuri (includes audit log)',
            'wordfence/wordfence.php' => 'Wordfence (includes activity)',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        $detected = [];
        
        foreach ($known_activity_log as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $detected[] = $name;
            }
        }
        
        return [
            'has_logging' => !empty($detected),
            'providers' => $detected,
        ];
    }
    
    /**
     * Collect inactive plugins info
     */
    private function collect_inactive_plugins() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', []);
        
        $inactive = [];
        foreach ($all_plugins as $file => $data) {
            if (!in_array($file, $active_plugins)) {
                $inactive[] = [
                    'file' => $file,
                    'name' => $data['Name'],
                    'version' => $data['Version'],
                ];
            }
        }
        
        return [
            'inactive_count' => count($inactive),
            'inactive_list' => $inactive,
        ];
    }
    
    /**
     * Collect theme info including inactive themes
     */
    private function collect_theme_info() {
        $all_themes = wp_get_themes();
        $active_theme = wp_get_theme();
        $active_stylesheet = $active_theme->get_stylesheet();
        
        $inactive_count = 0;
        $inactive_list = [];
        
        foreach ($all_themes as $stylesheet => $theme) {
            if ($stylesheet !== $active_stylesheet && $stylesheet !== $active_theme->get_template()) {
                $inactive_count++;
                $inactive_list[] = [
                    'stylesheet' => $stylesheet,
                    'name' => $theme->get('Name'),
                    'version' => $theme->get('Version'),
                ];
            }
        }
        
        // Determine theme source trust level
        $theme_uri = $active_theme->get('ThemeURI');
        $author_uri = $active_theme->get('AuthorURI');
        $text_domain = $active_theme->get('TextDomain');
        
        $source = 'unknown';
        $repository = null;
        
        // Check if it's a WordPress.org theme
        if (strpos($theme_uri, 'wordpress.org') !== false || 
            strpos($author_uri, 'wordpress.org') !== false ||
            $this->is_wordpress_org_theme($active_stylesheet)) {
            $source = 'wordpress.org';
            $repository = 'wordpress.org';
        } elseif (strpos($theme_uri, 'developer') !== false || 
                  strpos($theme_uri, 'theme') === false && 
                  empty($theme_uri)) {
            $source = 'custom';
        } elseif (strpos($theme_uri, 'themeforest') !== false || 
                  strpos($theme_uri, 'envato') !== false) {
            $source = 'themeforest';
        }
        
        return [
            'inactive_count' => $inactive_count,
            'inactive_list' => $inactive_list,
            'active' => [
                'name' => $active_theme->get('Name'),
                'stylesheet' => $active_stylesheet,
                'source' => $source,
                'repository' => $repository,
                'theme_uri' => $theme_uri,
            ],
        ];
    }
    
    /**
     * Check if theme is from WordPress.org
     */
    private function is_wordpress_org_theme($stylesheet) {
        // Check update transient for WordPress.org themes
        $update_themes = get_site_transient('update_themes');
        if ($update_themes && isset($update_themes->no_update[$stylesheet])) {
            return true;
        }
        if ($update_themes && isset($update_themes->response[$stylesheet])) {
            return true;
        }
        return false;
    }
    
    /**
     * Detect inactive admin users
     */
    private function detect_inactive_admins() {
        $inactive_days_threshold = 90; // Days without login to be considered inactive
        $cutoff_date = strtotime("-{$inactive_days_threshold} days");
        
        $admins = get_users([
            'role' => 'administrator',
            'fields' => ['ID', 'user_login', 'display_name'],
        ]);
        
        $inactive_admins = [];
        
        foreach ($admins as $admin) {
            $last_login = get_user_meta($admin->ID, 'last_login', true);
            
            // If no last_login meta, check user registered date as fallback
            if (empty($last_login)) {
                $user = get_userdata($admin->ID);
                $registered = strtotime($user->user_registered);
                
                // If registered before cutoff and never logged in since, consider inactive
                if ($registered < $cutoff_date) {
                    $inactive_admins[] = [
                        'user_id' => $admin->ID,
                        'username' => $admin->user_login,
                        'last_activity' => 'Never recorded',
                    ];
                }
            } elseif (strtotime($last_login) < $cutoff_date) {
                $inactive_admins[] = [
                    'user_id' => $admin->ID,
                    'username' => $admin->user_login,
                    'last_activity' => $last_login,
                ];
            }
        }
        
        return [
            'inactive_admins' => $inactive_admins,
            'inactive_days_threshold' => $inactive_days_threshold,
        ];
    }
    
    /**
     * Collect filesystem permissions
     */
    private function collect_filesystem_permissions() {
        $wp_content = WP_CONTENT_DIR;
        $plugin_dir = WP_PLUGIN_DIR;
        $upload_dir = wp_upload_dir();
        $upload_basedir = $upload_dir['basedir'];
        
        return [
            'wp_content_writable' => wp_is_writable($wp_content),
            'plugin_dir_writable' => wp_is_writable($plugin_dir),
            'upload_dir_writable' => wp_is_writable($upload_basedir),
            'wp_config_writable' => wp_is_writable(ABSPATH . 'wp-config.php'),
            'htaccess_writable' => wp_is_writable(ABSPATH . '.htaccess'),
        ];
    }
    
    /**
     * Collect WP-Cron events
     */
    private function collect_cron_events() {
        $crons = _get_cron_array();
        $events = [];
        $suspicious_events = [];
        
        // Known core cron hooks
        $core_hooks = [
            'wp_version_check',
            'wp_update_plugins',
            'wp_update_themes',
            'wp_scheduled_delete',
            'wp_scheduled_auto_draft_delete',
            'delete_expired_transients',
            'wp_privacy_delete_old_export_files',
            'wp_site_health_scheduled_check',
            'recovery_mode_clean_expired_keys',
        ];
        
        if (!empty($crons)) {
            foreach ($crons as $timestamp => $cron_hooks) {
                foreach ($cron_hooks as $hook => $details) {
                    $is_core = in_array($hook, $core_hooks) || strpos($hook, 'wp_') === 0;
                    
                    $event = [
                        'hook' => $hook,
                        'timestamp' => $timestamp,
                        'is_core' => $is_core,
                    ];
                    
                    $events[] = $event;
                    
                    // Flag suspicious hooks (non-core, non-common patterns)
                    if (!$is_core && 
                        strpos($hook, 'action_scheduler') === false &&
                        strpos($hook, 'wc_') === false &&
                        strpos($hook, 'jetpack') === false &&
                        strpos($hook, 'updraft') === false &&
                        strpos($hook, 'boonrisk-site-security-check-report') === false) {
                        // Could be suspicious, flag for review
                        $suspicious_events[] = $hook;
                    }
                }
            }
        }
        
        return [
            'events' => array_slice($events, 0, 50), // Limit to 50 events
            'total_events' => count($events),
            'suspicious_events' => array_unique($suspicious_events),
        ];
    }
    
    /**
     * Detect production environment indicators
     */
    private function detect_production_environment() {
        $is_production = true;
        $debug_artifacts = [];
        $staging_indicators = [];
        
        // Check for debug mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $debug_artifacts[] = 'WP_DEBUG enabled';
        }
        if (defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY) {
            $debug_artifacts[] = 'WP_DEBUG_DISPLAY enabled';
        }
        if (defined('SCRIPT_DEBUG') && SCRIPT_DEBUG) {
            $debug_artifacts[] = 'SCRIPT_DEBUG enabled';
        }
        
        // Check for staging indicators in URL
        $site_url = get_site_url();
        $staging_patterns = ['staging', 'dev.', 'test.', 'local.', '.local', 'localhost', '.test', '.dev'];
        foreach ($staging_patterns as $pattern) {
            if (stripos($site_url, $pattern) !== false) {
                $staging_indicators[] = "URL contains '{$pattern}'";
                $is_production = false;
            }
        }
        
        // Check for development plugins
        $dev_plugins = [
            'query-monitor/query-monitor.php',
            'debug-bar/debug-bar.php',
            'developer/developer.php',
            'wp-crontrol/wp-crontrol.php',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        foreach ($dev_plugins as $plugin) {
            if (in_array($plugin, $active_plugins)) {
                $debug_artifacts[] = 'Development plugin active: ' . basename(dirname($plugin));
            }
        }
        
        // Check for common staging/dev constants
        if (defined('WP_ENVIRONMENT_TYPE') && WP_ENVIRONMENT_TYPE !== 'production') {
            $staging_indicators[] = 'WP_ENVIRONMENT_TYPE is ' . WP_ENVIRONMENT_TYPE;
            $is_production = false;
        }
        
        return [
            'is_production' => $is_production && empty($staging_indicators),
            'debug_artifacts' => $debug_artifacts,
            'staging_indicators' => $staging_indicators,
        ];
    }
    
    /**
     * Detect webhook support
     */
    private function detect_webhook_support() {
        $active_plugins = get_option('active_plugins', []);
        
        $webhook_plugins = [
            'woocommerce/woocommerce.php' => 'WooCommerce Webhooks',
            'wp-webhooks/wp-webhooks.php' => 'WP Webhooks',
            'automator/automator.php' => 'Uncanny Automator',
            'zapier/zapier.php' => 'Zapier for WordPress',
        ];
        
        $detected = [];
        foreach ($webhook_plugins as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $detected[] = $name;
            }
        }
        
        // WooCommerce has built-in webhook support
        $has_woocommerce = in_array('woocommerce/woocommerce.php', $active_plugins);
        $wc_webhooks_count = 0;
        
        if ($has_woocommerce && class_exists('WC_Data_Store')) {
            // Count configured WooCommerce webhooks
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time count query for webhooks
            $wc_webhooks_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}wc_webhooks WHERE status = 'active'");
        }
        
        return [
            'supported' => !empty($detected) || $has_woocommerce,
            'configured_count' => $wc_webhooks_count,
            'providers' => $detected,
        ];
    }
}

