<?php
/**
 * Collects environment-related security signals
 * 
 * Data collected:
 * - PHP version and EOL status
 * - WordPress version
 * - Server software indicators
 * - Hosting environment signals
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Environment_Collector {
    
    // PHP EOL dates (approximate, check php.net for exact dates)
    private $php_eol_dates = [
        '5.6' => '2018-12-31',
        '7.0' => '2019-01-10',
        '7.1' => '2019-12-01',
        '7.2' => '2020-11-30',
        '7.3' => '2021-12-06',
        '7.4' => '2022-11-28',
        '8.0' => '2023-11-26',
        '8.1' => '2024-11-25',
        '8.2' => '2025-12-08',
        '8.3' => '2026-11-23',
        '8.4' => '2027-11-23',
    ];
    
    /**
     * Collect all environment data
     */
    public function collect() {
        return [
            'php' => $this->collect_php_info(),
            'wordpress' => $this->collect_wordpress_info(),
            'server' => $this->collect_server_info(),
            'hosting' => $this->collect_hosting_signals(),
            'database' => $this->collect_database_info(),
        ];
    }
    
    /**
     * Collect PHP version and security-relevant info
     */
    private function collect_php_info() {
        $version = phpversion();
        $major_minor = implode('.', array_slice(explode('.', $version), 0, 2));
        
        $eol_date = $this->php_eol_dates[$major_minor] ?? null;
        $is_eol = $eol_date ? strtotime($eol_date) < time() : null;
        
        return [
            'version' => $version,
            'major_minor' => $major_minor,
            'eol_date' => $eol_date,
            'is_eol' => $is_eol,
            'sapi' => php_sapi_name(),
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size'),
            'display_errors' => ini_get('display_errors'),
            'expose_php' => ini_get('expose_php'),
            'allow_url_fopen' => ini_get('allow_url_fopen'),
            'allow_url_include' => ini_get('allow_url_include'),
            'disabled_functions' => $this->get_disabled_functions(),
            'loaded_extensions' => $this->get_security_relevant_extensions(),
        ];
    }
    
    /**
     * Get list of disabled functions
     */
    private function get_disabled_functions() {
        $disabled = ini_get('disable_functions');
        if (empty($disabled)) {
            return [];
        }
        return array_map('trim', explode(',', $disabled));
    }
    
    /**
     * Get security-relevant PHP extensions
     */
    private function get_security_relevant_extensions() {
        $relevant = ['openssl', 'sodium', 'curl', 'mbstring', 'imagick', 'gd'];
        $loaded = [];
        
        foreach ($relevant as $ext) {
            $loaded[$ext] = extension_loaded($ext);
        }
        
        return $loaded;
    }
    
    /**
     * Collect WordPress version info
     */
    private function collect_wordpress_info() {
        global $wp_version;
        
        return [
            'version' => $wp_version,
            'is_multisite' => is_multisite(),
            'locale' => get_locale(),
            'timezone' => get_option('timezone_string') ?: 'Not set',
            'abspath' => ABSPATH,
            'content_dir' => WP_CONTENT_DIR,
            'plugin_dir' => WP_PLUGIN_DIR,
        ];
    }
    
    /**
     * Collect server software info
     */
    private function collect_server_info() {
        return [
            'software' => isset($_SERVER['SERVER_SOFTWARE']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unknown',
            'protocol' => isset($_SERVER['SERVER_PROTOCOL']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_PROTOCOL'])) : 'Unknown',
            'https' => is_ssl(),
            'server_ip' => isset($_SERVER['SERVER_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_ADDR'])) : null,
            'document_root' => isset($_SERVER['DOCUMENT_ROOT']) ? sanitize_text_field(wp_unslash($_SERVER['DOCUMENT_ROOT'])) : null,
        ];
    }
    
    /**
     * Detect hosting environment signals
     */
    private function collect_hosting_signals() {
        $signals = [
            'type' => 'unknown',
            'indicators' => [],
        ];
        
        // Check for CloudLinux (shared hosting indicator)
        if (file_exists('/etc/cloudlinux')) {
            $signals['indicators'][] = 'cloudlinux';
            $signals['type'] = 'shared';
        }
        
        // Check for cPanel
        if (file_exists('/usr/local/cpanel')) {
            $signals['indicators'][] = 'cpanel';
        }
        
        // Check for Plesk
        if (file_exists('/usr/local/psa')) {
            $signals['indicators'][] = 'plesk';
        }
        
        // Check for common managed WordPress hosts
        $server_software = isset($_SERVER['SERVER_SOFTWARE']) ? strtolower(sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE']))) : '';
        
        if (strpos($server_software, 'litespeed') !== false) {
            $signals['indicators'][] = 'litespeed';
        }
        
        if (defined('WPE_APIKEY')) {
            $signals['indicators'][] = 'wp_engine';
            $signals['type'] = 'managed';
        }
        
        if (defined('KINSTAGING_PATH') || getenv('KINSTA_CACHE_ZONE')) {
            $signals['indicators'][] = 'kinsta';
            $signals['type'] = 'managed';
        }
        
        if (getenv('PANTHEON_SITE')) {
            $signals['indicators'][] = 'pantheon';
            $signals['type'] = 'managed';
        }
        
        if (getenv('FLYWHEEL_CONFIG_DIR')) {
            $signals['indicators'][] = 'flywheel';
            $signals['type'] = 'managed';
        }
        
        // Check for containerized environments
        if (file_exists('/.dockerenv') || getenv('DOCKER_CONTAINER')) {
            $signals['indicators'][] = 'docker';
            $signals['type'] = 'containerized';
        }
        
        return $signals;
    }
    
    /**
     * Collect database info
     */
    private function collect_database_info() {
        global $wpdb;
        
        $db_version = $wpdb->db_version();
        $db_server_info = $wpdb->db_server_info();
        
        // Detect if MariaDB or MySQL
        $is_mariadb = stripos($db_server_info, 'mariadb') !== false;
        
        return [
            'version' => $db_version,
            'server_info' => $db_server_info,
            'is_mariadb' => $is_mariadb,
            'charset' => $wpdb->charset,
            'collate' => $wpdb->collate,
            'prefix' => $wpdb->prefix,
            'prefix_is_default' => $wpdb->prefix === 'wp_',
        ];
    }
}

