<?php
/**
 * Collects WordPress configuration security signals
 * 
 * Data collected:
 * - XML-RPC status
 * - REST API exposure
 * - File editor status
 * - Debug mode flags
 * - Security-relevant constants
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Config_Collector {
    
    /**
     * Collect all configuration data
     */
    public function collect() {
        return [
            'xmlrpc' => $this->collect_xmlrpc_status(),
            'rest_api' => $this->collect_rest_api_exposure(),
            'file_editor' => $this->is_file_editor_enabled(),
            'debug' => $this->collect_debug_settings(),
            'security_constants' => $this->collect_security_constants(),
            'permalinks' => $this->collect_permalink_info(),
            'users' => $this->collect_user_info(),
            'content_security' => $this->collect_content_security(),
        ];
    }
    
    /**
     * Check XML-RPC status
     */
    private function collect_xmlrpc_status() {
        $xmlrpc_url = site_url('xmlrpc.php');
        $file_exists = file_exists(ABSPATH . 'xmlrpc.php');
        
        // Check if XML-RPC is filtered off
        $is_filtered = false;
        if (has_filter('xmlrpc_enabled')) {
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
            $is_filtered = !apply_filters('xmlrpc_enabled', true);
        }
        
        return [
            'file_exists' => $file_exists,
            'url' => $xmlrpc_url,
            'is_filtered_disabled' => $is_filtered,
            'is_effectively_enabled' => $file_exists && !$is_filtered,
        ];
    }
    
    /**
     * Collect REST API exposure information
     */
    private function collect_rest_api_exposure() {
        $rest_url = rest_url();
        
        // Check if REST API is disabled
        $is_disabled = false;
        if (has_filter('rest_authentication_errors')) {
            // Some plugins disable REST API via this filter
            $is_disabled = true;
        }
        
        // Check exposed namespaces
        $namespaces = [];
        if (function_exists('rest_get_server')) {
            $server = rest_get_server();
            $namespaces = $server->get_namespaces();
        }
        
        // Check user enumeration possibility
        $users_endpoint_public = $this->check_users_endpoint_public();
        
        return [
            'url' => $rest_url,
            'prefix' => rest_get_url_prefix(),
            'namespaces' => $namespaces,
            'has_authentication_filter' => has_filter('rest_authentication_errors'),
            'users_endpoint_public' => $users_endpoint_public,
        ];
    }
    
    /**
     * Check if users endpoint is publicly accessible
     */
    private function check_users_endpoint_public() {
        // Check if there are any restrictions on /wp-json/wp/v2/users
        $current_user = wp_get_current_user();
        
        // Check if authentication is required for users endpoint
        $require_auth = has_filter('rest_user_query');
        
        return [
            'likely_exposed' => !$require_auth,
            'has_query_filter' => $require_auth,
        ];
    }
    
    /**
     * Check if file editor is enabled
     */
    private function is_file_editor_enabled() {
        $disallowed = defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT;
        $mods_disallowed = defined('DISALLOW_FILE_MODS') && DISALLOW_FILE_MODS;
        
        return [
            'disallow_file_edit' => $disallowed,
            'disallow_file_mods' => $mods_disallowed,
            'is_editor_accessible' => !$disallowed && !$mods_disallowed,
        ];
    }
    
    /**
     * Collect debug mode settings
     */
    private function collect_debug_settings() {
        return [
            'wp_debug' => defined('WP_DEBUG') && WP_DEBUG,
            'wp_debug_log' => defined('WP_DEBUG_LOG') && WP_DEBUG_LOG,
            'wp_debug_display' => defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY,
            'script_debug' => defined('SCRIPT_DEBUG') && SCRIPT_DEBUG,
            'savequeries' => defined('SAVEQUERIES') && SAVEQUERIES,
            'concatenate_scripts' => defined('CONCATENATE_SCRIPTS') ? CONCATENATE_SCRIPTS : true,
        ];
    }
    
    /**
     * Collect security-relevant constants
     */
    private function collect_security_constants() {
        return [
            'force_ssl_admin' => defined('FORCE_SSL_ADMIN') && FORCE_SSL_ADMIN,
            'force_ssl_login' => defined('FORCE_SSL_LOGIN') && FORCE_SSL_LOGIN,
            'autosave_interval' => defined('AUTOSAVE_INTERVAL') ? AUTOSAVE_INTERVAL : 60,
            'empty_trash_days' => defined('EMPTY_TRASH_DAYS') ? EMPTY_TRASH_DAYS : 30,
            'wp_post_revisions' => defined('WP_POST_REVISIONS') ? WP_POST_REVISIONS : true,
            'cookie_domain' => defined('COOKIE_DOMAIN') ? COOKIE_DOMAIN : '',
            'cookiepath' => defined('COOKIEPATH') ? COOKIEPATH : '',
            'sitecookiepath' => defined('SITECOOKIEPATH') ? SITECOOKIEPATH : '',
            'admin_cookie_path' => defined('ADMIN_COOKIE_PATH') ? ADMIN_COOKIE_PATH : '',
            'automatic_updater_disabled' => defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED,
            'wp_auto_update_core' => defined('WP_AUTO_UPDATE_CORE') ? WP_AUTO_UPDATE_CORE : true,
        ];
    }
    
    /**
     * Collect permalink structure info
     */
    private function collect_permalink_info() {
        $structure = get_option('permalink_structure');
        
        return [
            'structure' => $structure,
            'is_plain' => empty($structure),
            'has_index_php' => strpos($structure, 'index.php') !== false,
        ];
    }
    
    /**
     * Collect basic user information for security assessment
     */
    private function collect_user_info() {
        global $wpdb;
        
        // Count users by role
        $user_counts = count_users();
        
        // Check for default admin user
        $has_admin_user = username_exists('admin');
        
        // Check for user with ID 1
        $user_one = get_userdata(1);
        $user_one_is_admin = $user_one && in_array('administrator', $user_one->roles);
        
        // Count administrators
        $admin_count = isset($user_counts['avail_roles']['administrator']) 
            ? $user_counts['avail_roles']['administrator'] 
            : 0;
        
        return [
            'total_users' => $user_counts['total_users'],
            'roles_count' => $user_counts['avail_roles'],
            'admin_count' => $admin_count,
            'has_default_admin_username' => $has_admin_user !== false,
            'user_id_1_is_admin' => $user_one_is_admin,
            'user_registration_open' => get_option('users_can_register'),
            'default_role' => get_option('default_role'),
        ];
    }
    
    /**
     * Collect content security settings
     */
    private function collect_content_security() {
        return [
            'blog_public' => get_option('blog_public'),
            'comments_open' => get_option('default_comment_status') === 'open',
            'pingbacks_open' => get_option('default_ping_status') === 'open',
            'require_name_email' => get_option('require_name_email'),
            'comment_registration' => get_option('comment_registration'),
            'comment_moderation' => get_option('comment_moderation'),
            'comment_previously_approved' => get_option('comment_previously_approved'),
            'uploads_use_yearmonth_folders' => get_option('uploads_use_yearmonth_folders'),
        ];
    }
}

