<?php
/**
 * Collects authentication surface security signals
 * 
 * Data collected:
 * - Login exposure indicators
 * - Rate-limiting signals
 * - 2FA presence detection
 * - Session configuration
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Auth_Collector {
    
    /**
     * Collect all authentication-related data
     */
    public function collect() {
        return [
            'login_exposure' => $this->collect_login_exposure(),
            'rate_limiting' => $this->detect_rate_limiting(),
            '2fa_signals' => $this->detect_2fa_signals(),
            'session_config' => $this->collect_session_config(),
            'password_policy' => $this->detect_password_policy(),
            'login_security' => $this->collect_login_security(),
        ];
    }
    
    /**
     * Collect login page exposure indicators
     */
    private function collect_login_exposure() {
        $login_url = wp_login_url();
        $default_login = site_url('wp-login.php');
        
        // Check if login URL has been changed (common security measure)
        $is_default_login = strpos($login_url, 'wp-login.php') !== false;
        
        // Check for login page protection plugins
        $has_login_protection = $this->detect_login_protection();
        
        // Check if registration is enabled
        $registration_enabled = get_option('users_can_register');
        
        // Check for common login slugs
        $common_slugs = ['wp-admin', 'admin', 'login', 'wp-login.php'];
        $accessible_slugs = [];
        
        foreach ($common_slugs as $slug) {
            // We only report that these are standard locations
            $accessible_slugs[$slug] = true;
        }
        
        return [
            'login_url' => $login_url,
            'is_default_location' => $is_default_login,
            'registration_enabled' => (bool) $registration_enabled,
            'lost_password_enabled' => true, // WP default
            'has_login_protection' => $has_login_protection,
            'standard_endpoints' => $accessible_slugs,
        ];
    }
    
    /**
     * Detect login page protection mechanisms
     */
    private function detect_login_protection() {
        $protection_detected = false;
        
        // Check for common login protection plugins
        $protection_plugins = [
            'wps-hide-login/wps-hide-login.php',
            'rename-wp-login/rename-wp-login.php',
            'sf-move-login/sf-move-login.php',
            'lockdown-wp-admin/lockdown-wp-admin.php',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        
        foreach ($protection_plugins as $plugin) {
            if (in_array($plugin, $active_plugins)) {
                $protection_detected = true;
                break;
            }
        }
        
        return $protection_detected;
    }
    
    /**
     * Detect rate limiting mechanisms
     */
    private function detect_rate_limiting() {
        $signals = [
            'has_rate_limiting' => false,
            'detected_mechanisms' => [],
        ];
        
        // Check for login attempt limiting plugins
        $rate_limit_plugins = [
            'limit-login-attempts-reloaded/limit-login-attempts-reloaded.php' => 'Limit Login Attempts Reloaded',
            'login-lockdown/loginlockdown.php' => 'Login LockDown',
            'wordfence/wordfence.php' => 'Wordfence',
            'better-wp-security/better-wp-security.php' => 'iThemes Security',
            'sucuri-scanner/sucuri.php' => 'Sucuri',
            'wp-cerber/wp-cerber.php' => 'WP Cerber',
            'all-in-one-wp-security-and-firewall/wp-security.php' => 'All In One WP Security',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        
        foreach ($rate_limit_plugins as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $signals['has_rate_limiting'] = true;
                $signals['detected_mechanisms'][] = $name;
            }
        }
        
        // Check for fail2ban integration
        if (file_exists('/etc/fail2ban')) {
            $signals['detected_mechanisms'][] = 'fail2ban (server-level)';
        }
        
        return $signals;
    }
    
    /**
     * Detect 2FA implementation signals
     */
    private function detect_2fa_signals() {
        $signals = [
            'has_2fa' => false,
            'coverage' => 'unknown',
            'methods' => [],
        ];
        
        // Check for 2FA plugins
        $twofa_plugins = [
            'two-factor/two-factor.php' => 'Two-Factor',
            'google-authenticator/google-authenticator.php' => 'Google Authenticator',
            'wp-2fa/wp-2fa.php' => 'WP 2FA',
            'duo-two-factor/duo-web.php' => 'Duo Two-Factor',
            'miniorange-2-factor-authentication/miniorange_2_factor_settings.php' => 'miniOrange 2FA',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        
        foreach ($twofa_plugins as $plugin => $name) {
            if (in_array($plugin, $active_plugins)) {
                $signals['has_2fa'] = true;
                $signals['methods'][] = $name;
            }
        }
        
        // Check if Wordfence 2FA is likely enabled
        if (in_array('wordfence/wordfence.php', $active_plugins)) {
            $wf_options = get_option('wordfence_options', []);
            if (isset($wf_options['loginSec_enableTwoFactor']) && $wf_options['loginSec_enableTwoFactor']) {
                $signals['has_2fa'] = true;
                $signals['methods'][] = 'Wordfence 2FA';
            }
        }
        
        // Check if Jetpack SSO is enabled
        if (in_array('jetpack/jetpack.php', $active_plugins)) {
            if (class_exists('Jetpack') && Jetpack::is_module_active('sso')) {
                $signals['has_2fa'] = true;
                $signals['methods'][] = 'Jetpack SSO';
            }
        }
        
        return $signals;
    }
    
    /**
     * Collect session configuration
     */
    private function collect_session_config() {
        // WordPress uses cookies, not PHP sessions typically
        return [
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
            'auth_cookie_expiration' => apply_filters('auth_cookie_expiration', 2 * DAY_IN_SECONDS, 0, false),
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
            'auth_cookie_expiration_remember' => apply_filters('auth_cookie_expiration', 14 * DAY_IN_SECONDS, 0, true),
            'secure_cookie' => is_ssl() || (defined('FORCE_SSL_ADMIN') && FORCE_SSL_ADMIN),
            'cookie_httponly' => true, // WordPress default
            'session_tokens_table' => $this->check_session_tokens(),
        ];
    }
    
    /**
     * Check session tokens storage
     */
    private function check_session_tokens() {
        // Check if a custom session handler is in use
        $session_handler = get_option('wp_session_handler', 'default');
        
        return [
            'handler' => $session_handler,
            'is_default' => $session_handler === 'default',
        ];
    }
    
    /**
     * Detect password policy enforcement
     */
    private function detect_password_policy() {
        $policy = [
            'has_policy_plugin' => false,
            'detected_plugins' => [],
            'minimum_password_strength' => $this->get_minimum_password_strength(),
        ];
        
        // Check for password policy plugins
        $policy_plugins = [
            'password-policy-manager/password-policy-manager.php',
            'force-strong-passwords/force-strong-passwords.php',
            'password-bcrypt/password-bcrypt.php',
            'better-wp-security/better-wp-security.php', // Has password requirements
            'wordfence/wordfence.php', // Has password requirements
        ];
        
        $active_plugins = get_option('active_plugins', []);
        
        foreach ($policy_plugins as $plugin) {
            if (in_array($plugin, $active_plugins)) {
                $policy['has_policy_plugin'] = true;
                $policy['detected_plugins'][] = $plugin;
            }
        }
        
        return $policy;
    }
    
    /**
     * Get minimum password strength indicator
     */
    private function get_minimum_password_strength() {
        // WordPress 4.3+ has native password strength meter
        // Check if any plugins enforce minimum strength
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Checking core hook value
        $min_strength = apply_filters('wp_password_strength_meter_cutoff', 3);
        
        $strength_labels = [
            0 => 'Very Weak',
            1 => 'Weak',
            2 => 'Medium',
            3 => 'Strong',
            4 => 'Very Strong',
        ];
        
        return [
            'level' => $min_strength,
            'label' => $strength_labels[$min_strength] ?? 'Unknown',
            'is_enforced' => $min_strength >= 3,
        ];
    }
    
    /**
     * Collect additional login security settings
     */
    private function collect_login_security() {
        return [
            'remember_me_enabled' => true, // WordPress default
            'login_message_filtered' => has_filter('login_message'),
            'login_errors_filtered' => has_filter('login_errors'),
            'authenticate_filtered' => has_filter('authenticate'),
            'wp_authenticate_user_filtered' => has_filter('wp_authenticate_user'),
            'login_form_action_hooks' => [
                'login_form' => has_action('login_form'),
                'login_enqueue_scripts' => has_action('login_enqueue_scripts'),
            ],
        ];
    }
}

