<?php
/**
 * Local Assessment Engine
 * 
 * Performs security posture checks entirely locally without any external API calls.
 * This allows the plugin to provide value immediately without registration.
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Local_Assessment {
    
    private $collector;
    
    // PHP EOL dates for local evaluation
    private $php_eol_dates = [
        '5.6' => '2018-12-31',
        '7.0' => '2019-01-10',
        '7.1' => '2019-12-01',
        '7.2' => '2020-11-30',
        '7.3' => '2021-12-06',
        '7.4' => '2022-11-28',
        '8.0' => '2023-11-26',
        '8.1' => '2024-11-25',
        '8.2' => '2025-12-08',
        '8.3' => '2026-11-23',
        '8.4' => '2027-11-23',
    ];
    
    public function __construct($collector) {
        $this->collector = $collector;
    }
    
    /**
     * Run all local assessment checks
     * 
     * @return array Assessment results with summary
     */
    public function run() {
        $checks = [];
        
        // Security - Account security
        $checks[] = $this->check_two_factor_auth();
        $checks[] = $this->check_login_rate_limiting();
        $checks[] = $this->check_admin_username();
        $checks[] = $this->check_user_registration();
        $checks[] = $this->check_excessive_admins();
        $checks[] = $this->check_inactive_privileged_users();
        
        // Security - Public access & endpoints
        $checks[] = $this->check_xmlrpc();
        $checks[] = $this->check_file_editor();
        
        // Security - Updates & maintenance
        $checks[] = $this->check_plugin_updates();
        $checks[] = $this->check_theme_updates();
        $checks[] = $this->check_auto_updates();
        
        // Security - Server security settings
        $checks[] = $this->check_dangerous_php_functions();
        
        // Security - Data protection
        $checks[] = $this->check_db_prefix();
        
        // Security - Bot & abuse protection
        $checks[] = $this->check_bot_protection();
        
        // Security - File & code changes
        $checks[] = $this->check_file_write_permissions();
        
        // Security - Background tasks
        $checks[] = $this->check_wp_cron_visibility();
        
        // Trust - Secure connection
        $checks[] = $this->check_https();
        
        // Readiness - Core software health
        $checks[] = $this->check_php_version();
        $checks[] = $this->check_wordpress_version();
        $checks[] = $this->check_debug_mode();
        
        // Readiness - Backups & recovery
        $checks[] = $this->check_backup_mechanism();
        
        // Readiness - Site hygiene
        $checks[] = $this->check_inactive_plugins();
        $checks[] = $this->check_inactive_themes();
        $checks[] = $this->check_plugin_maintenance();
        $checks[] = $this->check_theme_source_trust();
        
        // Readiness - Performance & speed
        $checks[] = $this->check_caching_plugin();
        $checks[] = $this->check_image_optimization();
        
        // Readiness - Site changes & stability
        $checks[] = $this->check_update_policy_consistency();
        
        // Readiness - Monitoring & alerts
        $checks[] = $this->check_security_logging();
        
        // Readiness - Environment setup
        $checks[] = $this->check_production_isolation();
        
        // Calculate summary
        $summary = $this->calculate_summary($checks);
        
        // Get top risks (prioritized by impact)
        $top_risks = $this->get_top_risks($checks);
        
        return [
            'checks' => $checks,
            'summary' => $summary,
            'top_risks' => $top_risks,
            'generated_at' => current_time('c'),
        ];
    }
    
    /**
     * Get top 3-5 risks prioritized by impact
     * 
     * @param array $checks All assessment checks
     * @return array Top risks with prioritization
     */
    private function get_top_risks($checks) {
        // Impact weights for prioritization (higher = more important)
        $impact_weights = [
            'https' => 100,                      // Data transmission security
            'php_version' => 95,                 // EOL = no security patches
            'wordpress_version' => 90,           // Known vulnerabilities
            'plugin_updates' => 85,              // #1 attack vector
            'dangerous_functions' => 80,         // RCE risk
            'debug_mode' => 75,                  // Information disclosure
            'file_editor' => 70,                 // Code injection risk
            'file_write_permissions' => 68,      // File modification risk
            'theme_updates' => 65,
            'backup' => 60,                      // Recovery capability
            'two_factor' => 55,                  // Account security
            'login_rate_limiting' => 53,         // Brute-force protection
            'excessive_admins' => 52,            // Attack surface
            'inactive_privileged_users' => 51,   // Dormant accounts
            'admin_username' => 50,
            'xmlrpc' => 45,
            'bot_protection' => 40,
            'plugin_count' => 30,
            'user_registration' => 25,
            'auto_updates' => 20,
            'wp_cron' => 15,
            'db_prefix' => 10,
            'inactive_plugins' => 9,
            'inactive_themes' => 8,
            'plugin_maintenance' => 7,
            'theme_source' => 6,
            'caching_plugin' => 5,
            'image_optimization' => 4,
            'update_policy' => 3,
            'security_logging' => 2,
            'production_isolation' => 1,
        ];
        
        // Filter to only failed checks
        $failed_checks = array_filter($checks, function($check) {
            return $check['status'] !== 'pass';
        });
        
        // Sort by impact weight
        usort($failed_checks, function($a, $b) use ($impact_weights) {
            $weight_a = $impact_weights[$a['id']] ?? 0;
            $weight_b = $impact_weights[$b['id']] ?? 0;
            
            // High risk items first within same weight
            if ($weight_a === $weight_b) {
                return $a['status'] === 'high_risk' ? -1 : 1;
            }
            
            return $weight_b - $weight_a;
        });
        
        // Return top 5 (or fewer if less issues)
        return array_slice($failed_checks, 0, 5);
    }
    
    /**
     * Check PHP version and EOL status (Local: configuration-based risk signals)
     * 
     * Local mode: Shows unsupported/older status as risk indicator
     * Cloud mode: Can add vulnerability intelligence as additional evidence
     */
    private function check_php_version() {
        $version = phpversion();
        $major_minor = implode('.', array_slice(explode('.', $version), 0, 2));
        $eol_date = $this->php_eol_dates[$major_minor] ?? null;
        $is_eol = $eol_date ? strtotime($eol_date) < time() : false;
        
        if ($is_eol) {
            return [
                'id' => 'php_version',
                'name' => __('Unsupported PHP Version', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => $version,
                'message' => sprintf(
                    /* translators: 1: PHP version (e.g., 7.4), 2: end of life date (e.g., Nov 2022) */
                    __('PHP %1$s is no longer supported (end of life: %2$s). This version no longer receives updates.', 'boonrisk-site-security-check-report'),
                    $major_minor,
                    $eol_date
                ),
                'recommendation' => __('Contact your hosting provider to upgrade to PHP 8.1 or higher.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Unsupported PHP versions no longer receive updates, which may leave known issues unaddressed.', 'boonrisk-site-security-check-report'),
                'evidence_depth' => 'local',
                'cloud_enhancement' => __('Detailed vulnerability intelligence is available with an optional web connection.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        // Check if it's an older but still supported version
        if (version_compare($version, '8.1', '<')) {
            return [
                'id' => 'php_version',
                'name' => __('PHP Version', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => $version,
                /* translators: %s: PHP version */
                'message' => sprintf(__('PHP %s is supported but a newer version is recommended.', 'boonrisk-site-security-check-report'), $version),
                'recommendation' => __('Consider upgrading to PHP 8.1+ for improved performance and longer support.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Newer PHP versions include improvements and have longer support timelines.', 'boonrisk-site-security-check-report'),
                'evidence_depth' => 'local',
            ];
        }
        
        return [
            'id' => 'php_version',
            'name' => __('PHP Version', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => $version,
            /* translators: %s: PHP version */
            'message' => sprintf(__('PHP %s is current and supported.', 'boonrisk-site-security-check-report'), $version),
            'recommendation' => null,
            'why_it_matters' => __('Using a supported PHP version ensures continued updates and support.', 'boonrisk-site-security-check-report'),
            'evidence_depth' => 'local',
        ];
    }
    
    /**
     * Check WordPress version (Local: configuration-based risk signals)
     * 
     * Local mode: Shows outdated status as risk indicator
     * Cloud mode: Can add vulnerability intelligence as additional evidence
     */
    private function check_wordpress_version() {
        global $wp_version;
        
        // Check for available updates
        $update_data = get_site_transient('update_core');
        $has_update = false;
        $latest_version = $wp_version;
        
        if (!empty($update_data->updates)) {
            foreach ($update_data->updates as $update) {
                if ($update->response === 'upgrade') {
                    $has_update = true;
                    $latest_version = $update->version;
                    break;
                }
            }
        }
        
        if ($has_update) {
            return [
                'id' => 'wordpress_version',
                'name' => __('Outdated WordPress', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => $wp_version,
                'message' => sprintf(
                    /* translators: 1: Current WordPress version, 2: Available WordPress version */
                    __('WordPress %1$s is no longer up to date. Version %2$s is available.', 'boonrisk-site-security-check-report'),
                    $wp_version,
                    $latest_version
                ),
                'recommendation' => __('Review the available update in Dashboard → Updates.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Outdated WordPress versions may contain unresolved issues that are addressed in newer releases.', 'boonrisk-site-security-check-report'),
                'evidence_depth' => 'local',
                'cloud_enhancement' => __('Detailed vulnerability intelligence is available with an optional web connection.', 'boonrisk-site-security-check-report'),
                'details' => [
                    'current_version' => $wp_version,
                    'available_version' => $latest_version,
                ],
            ];
        }
        
        return [
            'id' => 'wordpress_version',
            'name' => __('WordPress Version', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => $wp_version,
            /* translators: %s: WordPress version */
            'message' => sprintf(__('WordPress %s is up to date.', 'boonrisk-site-security-check-report'), $wp_version),
            'recommendation' => null,
            'why_it_matters' => __('Running an up-to-date WordPress version reduces the likelihood of known issues.', 'boonrisk-site-security-check-report'),
            'evidence_depth' => 'local',
        ];
    }
    
    /**
     * Check debug mode
     */
    private function check_debug_mode() {
        $debug_enabled = defined('WP_DEBUG') && WP_DEBUG;
        $debug_display = defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY;
        
        if ($debug_enabled && $debug_display) {
            return [
                'id' => 'debug_mode',
                'name' => __('Debug Mode', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => __('Enabled with display', 'boonrisk-site-security-check-report'),
                'message' => __('Debug mode is enabled and errors are displayed publicly.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('In wp-config.php, set WP_DEBUG to false or WP_DEBUG_DISPLAY to false.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Debug output can reveal sensitive information like file paths and database structure to attackers.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        if ($debug_enabled) {
            return [
                'id' => 'debug_mode',
                'name' => __('Debug Mode', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Enabled', 'boonrisk-site-security-check-report'),
                'message' => __('Debug mode is enabled but display is off.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Disable WP_DEBUG in production unless actively debugging.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Debug mode can impact performance and may log sensitive information.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'debug_mode',
            'name' => __('Debug Mode', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Disabled', 'boonrisk-site-security-check-report'),
            'message' => __('Debug mode is properly disabled.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Keeping debug mode off in production prevents information leakage.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check file editor status
     */
    private function check_file_editor() {
        $disabled = defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT;
        
        if (!$disabled) {
            return [
                'id' => 'file_editor',
                'name' => __('File Editor', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Enabled', 'boonrisk-site-security-check-report'),
                'message' => __('The built-in theme and plugin editor is enabled.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Add define(\'DISALLOW_FILE_EDIT\', true); to wp-config.php', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('If an admin account is compromised, attackers can directly inject malicious code through the editor.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'file_editor',
            'name' => __('File Editor', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Disabled', 'boonrisk-site-security-check-report'),
            'message' => __('The file editor is disabled.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Disabling the file editor reduces the attack surface if an admin account is compromised.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check XML-RPC status
     */
    private function check_xmlrpc() {
        $file_exists = file_exists(ABSPATH . 'xmlrpc.php');
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core WordPress hook
        $is_filtered = has_filter('xmlrpc_enabled') && !apply_filters('xmlrpc_enabled', true);
        
        if ($file_exists && !$is_filtered) {
            return [
                'id' => 'xmlrpc',
                'name' => __('XML-RPC', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Enabled', 'boonrisk-site-security-check-report'),
                'message' => __('XML-RPC endpoint is accessible.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Disable XML-RPC if not using Jetpack, mobile apps, or XML-RPC clients.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('XML-RPC can be used for brute force amplification attacks and DDoS.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'xmlrpc',
            'name' => __('XML-RPC', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Disabled or filtered', 'boonrisk-site-security-check-report'),
            'message' => __('XML-RPC is disabled or filtered.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Disabling unused endpoints reduces your attack surface.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check HTTPS status
     */
    private function check_https() {
        $is_ssl = is_ssl();
        $force_ssl = defined('FORCE_SSL_ADMIN') && FORCE_SSL_ADMIN;
        
        if (!$is_ssl) {
            return [
                'id' => 'https',
                'name' => __('HTTPS', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => __('Not active', 'boonrisk-site-security-check-report'),
                'message' => __('Site is not served over HTTPS.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Enable HTTPS through your hosting provider and install an SSL certificate.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Without HTTPS, all data including passwords is transmitted in plain text.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        if (!$force_ssl) {
            return [
                'id' => 'https',
                'name' => __('HTTPS', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Active', 'boonrisk-site-security-check-report'),
                'message' => __('HTTPS is active but FORCE_SSL_ADMIN is not set.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Add define(\'FORCE_SSL_ADMIN\', true); to wp-config.php', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('FORCE_SSL_ADMIN ensures the admin area always uses HTTPS.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'https',
            'name' => __('HTTPS', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Enforced', 'boonrisk-site-security-check-report'),
            'message' => __('HTTPS is active and enforced for admin.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('HTTPS encrypts all data in transit, protecting against eavesdropping.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check plugin updates (Local: configuration-based risk signals)
     * 
     * Local mode: Shows outdated status as risk indicator
     * Cloud mode: Can add vulnerability intelligence as additional evidence
     */
    private function check_plugin_updates() {
        $updates = get_site_transient('update_plugins');
        $pending = !empty($updates->response) ? count($updates->response) : 0;
        
        // Get list of outdated plugins for detail
        $outdated_plugins = [];
        if (!empty($updates->response)) {
            foreach ($updates->response as $plugin_file => $plugin_data) {
                $plugin_info = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_file);
                $outdated_plugins[] = [
                    'name' => $plugin_info['Name'] ?? $plugin_file,
                    'current_version' => $plugin_info['Version'] ?? 'unknown',
                    'available_version' => $plugin_data->new_version ?? 'unknown',
                ];
            }
        }
        
        if ($pending > 5) {
            return [
                'id' => 'plugin_updates',
                'name' => __('Outdated Plugins', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                /* translators: %d: number of outdated plugins */
                'value' => sprintf(__('%d outdated', 'boonrisk-site-security-check-report'), $pending),
                /* translators: %d: number of outdated plugins */
                'message' => sprintf(__('%d plugins are no longer up to date and may introduce security risk.', 'boonrisk-site-security-check-report'), $pending),
                'recommendation' => __('Review available updates in Dashboard → Updates, or consider replacing unmaintained plugins.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Outdated plugins are a common source of site issues and security incidents.', 'boonrisk-site-security-check-report'),
                'evidence_depth' => 'local',
                'cloud_enhancement' => __('Detailed vulnerability intelligence is available with an optional web connection.', 'boonrisk-site-security-check-report'),
                'details' => $outdated_plugins,
            ];
        }
        
        if ($pending > 0) {
            return [
                'id' => 'plugin_updates',
                'name' => __('Outdated Plugins', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                /* translators: %d: number of outdated plugins */
                'value' => sprintf(__('%d outdated', 'boonrisk-site-security-check-report'), $pending),
                /* translators: %d: number of outdated plugins */
                'message' => sprintf(__('%d plugin(s) are no longer up to date.', 'boonrisk-site-security-check-report'), $pending),
                'recommendation' => __('Review available updates or consider replacing unmaintained plugins.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Plugin updates often include fixes for known issues.', 'boonrisk-site-security-check-report'),
                'evidence_depth' => 'local',
                'cloud_enhancement' => __('Detailed vulnerability intelligence is available with an optional web connection.', 'boonrisk-site-security-check-report'),
                'details' => $outdated_plugins,
            ];
        }
        
        return [
            'id' => 'plugin_updates',
            'name' => __('Plugin Updates', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('All current', 'boonrisk-site-security-check-report'),
            'message' => __('All plugins are up to date.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Keeping plugins updated reduces the likelihood of known issues.', 'boonrisk-site-security-check-report'),
            'evidence_depth' => 'local',
        ];
    }
    
    /**
     * Check theme updates (Local: configuration-based risk signals)
     * 
     * Local mode: Shows outdated status as risk indicator
     * Cloud mode: Can add vulnerability intelligence as additional evidence
     */
    private function check_theme_updates() {
        $updates = get_site_transient('update_themes');
        $current_theme = wp_get_theme();
        $theme_slug = $current_theme->get_stylesheet();
        
        $active_needs_update = isset($updates->response[$theme_slug]);
        
        if ($active_needs_update) {
            $theme_update = $updates->response[$theme_slug];
            return [
                'id' => 'theme_updates',
                'name' => __('Outdated Theme', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Update available', 'boonrisk-site-security-check-report'),
                /* translators: %s: theme name */
                'message' => sprintf(__('Active theme "%s" is no longer up to date and may introduce security risk.', 'boonrisk-site-security-check-report'), $current_theme->get('Name')),
                'recommendation' => __('Review the available update in Dashboard → Updates, or consider replacing an unmaintained theme.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Outdated themes can contain unresolved issues that affect site security.', 'boonrisk-site-security-check-report'),
                'evidence_depth' => 'local',
                'cloud_enhancement' => __('Detailed vulnerability intelligence is available with an optional web connection.', 'boonrisk-site-security-check-report'),
                'details' => [
                    'name' => $current_theme->get('Name'),
                    'current_version' => $current_theme->get('Version'),
                    'available_version' => $theme_update['new_version'] ?? 'unknown',
                ],
            ];
        }
        
        return [
            'id' => 'theme_updates',
            'name' => __('Theme Updates', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Current', 'boonrisk-site-security-check-report'),
            'message' => __('Active theme is up to date.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Keeping your theme updated reduces the likelihood of known issues.', 'boonrisk-site-security-check-report'),
            'evidence_depth' => 'local',
        ];
    }
    
    /**
     * Check plugin count
     */
    private function check_plugin_count() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', []);
        $total = count($all_plugins);
        $active = count($active_plugins);
        $inactive = $total - $active;
        
        if ($inactive > 5) {
            return [
                'id' => 'plugin_count',
                'name' => __('Plugin Count', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                /* translators: 1: number of active plugins, 2: number of inactive plugins */
                'value' => sprintf(__('%1$d active, %2$d inactive', 'boonrisk-site-security-check-report'), $active, $inactive),
                /* translators: %d: number of inactive plugins */
                'message' => sprintf(__('%d inactive plugins installed.', 'boonrisk-site-security-check-report'), $inactive),
                'recommendation' => __('Delete inactive plugins you don\'t need.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Inactive plugins can still contain vulnerabilities and increase attack surface.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        if ($active > 30) {
            return [
                'id' => 'plugin_count',
                'name' => __('Plugin Count', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                /* translators: %d: number of active plugins */
                'value' => sprintf(__('%d active', 'boonrisk-site-security-check-report'), $active),
                /* translators: %d: number of active plugins */
                'message' => sprintf(__('%d active plugins - higher than typical.', 'boonrisk-site-security-check-report'), $active),
                'recommendation' => __('Review if all plugins are necessary. Each plugin adds potential risk.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('More plugins mean more code to audit and more potential vulnerabilities.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'plugin_count',
            'name' => __('Plugin Count', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            /* translators: 1: number of active plugins, 2: number of inactive plugins */
            'value' => sprintf(__('%1$d active, %2$d inactive', 'boonrisk-site-security-check-report'), $active, $inactive),
            'message' => __('Plugin count is reasonable.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Keeping plugin count manageable reduces your attack surface.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check for default admin username
     */
    private function check_admin_username() {
        $admin_exists = username_exists('admin');
        
        if ($admin_exists) {
            return [
                'id' => 'admin_username',
                'name' => __('Admin Username', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Default "admin" exists', 'boonrisk-site-security-check-report'),
                'message' => __('A user with username "admin" exists.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Create a new admin account with a unique username and delete the "admin" user.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('The "admin" username is the first guess in brute force attacks.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'admin_username',
            'name' => __('Admin Username', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('No default', 'boonrisk-site-security-check-report'),
            'message' => __('No default "admin" username found.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Using unique admin usernames makes brute force attacks harder.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check user registration setting
     */
    private function check_user_registration() {
        $open = get_option('users_can_register');
        $default_role = get_option('default_role');
        
        if ($open && $default_role !== 'subscriber') {
            return [
                'id' => 'user_registration',
                'name' => __('User Registration', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                /* translators: %s: default user role */
                'value' => sprintf(__('Open (default: %s)', 'boonrisk-site-security-check-report'), $default_role),
                /* translators: %s: default user role */
                'message' => sprintf(__('Registration is open with default role "%s".', 'boonrisk-site-security-check-report'), $default_role),
                'recommendation' => __('Set default role to "Subscriber" or disable registration if not needed.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Open registration with elevated roles can allow unauthorized access.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        if ($open) {
            return [
                'id' => 'user_registration',
                'name' => __('User Registration', 'boonrisk-site-security-check-report'),
                'status' => 'pass',
                'value' => __('Open (Subscriber)', 'boonrisk-site-security-check-report'),
                'message' => __('Registration is open but limited to subscriber role.', 'boonrisk-site-security-check-report'),
                'recommendation' => null,
                'why_it_matters' => __('Subscriber role has minimal permissions.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'user_registration',
            'name' => __('User Registration', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Disabled', 'boonrisk-site-security-check-report'),
            'message' => __('Public registration is disabled.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Closed registration prevents unauthorized account creation.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check database prefix
     */
    private function check_db_prefix() {
        global $wpdb;
        
        if ($wpdb->prefix === 'wp_') {
            return [
                'id' => 'db_prefix',
                'name' => __('Database Prefix', 'boonrisk-site-security-check-report'),
                'status' => 'pass', // Low priority, just informational
                'value' => 'wp_',
                'message' => __('Using default database prefix.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Consider using a custom prefix for new installations (low priority).', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Custom prefix adds minor defense-in-depth against SQL injection.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'db_prefix',
            'name' => __('Database Prefix', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => $wpdb->prefix,
            'message' => __('Using custom database prefix.', 'boonrisk-site-security-check-report'),
            'recommendation' => null,
            'why_it_matters' => __('Custom prefix makes SQL injection attacks slightly harder.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Calculate overall summary with human-readable explanation
     */
    private function calculate_summary($checks) {
        $high_risk = 0;
        $warnings = 0;
        $passed = 0;
        
        // Collect issue categories for explanation
        $issue_categories = [];
        
        foreach ($checks as $check) {
            switch ($check['status']) {
                case 'high_risk':
                    $high_risk++;
                    $issue_categories[] = $this->get_issue_category($check['id']);
                    break;
                case 'warning':
                    $warnings++;
                    if (count($issue_categories) < 3) {
                        $issue_categories[] = $this->get_issue_category($check['id']);
                    }
                    break;
                case 'pass':
                    $passed++;
                    break;
            }
        }
        
        $total = count($checks);
        $issue_categories = array_unique($issue_categories);
        
        // Determine overall posture with human-readable explanation
        if ($high_risk > 0) {
            $posture = 'high';
            $posture_label = __('High', 'boonrisk-site-security-check-report');
            $posture_explanation = $this->build_posture_explanation('high', $issue_categories, $high_risk, $warnings);
        } elseif ($warnings > 3) {
            $posture = 'medium';
            $posture_label = __('Medium', 'boonrisk-site-security-check-report');
            $posture_explanation = $this->build_posture_explanation('medium', $issue_categories, $high_risk, $warnings);
        } elseif ($warnings > 0) {
            $posture = 'low';
            $posture_label = __('Low', 'boonrisk-site-security-check-report');
            $posture_explanation = $this->build_posture_explanation('low', $issue_categories, $high_risk, $warnings);
        } else {
            $posture = 'secure';
            $posture_label = __('Low', 'boonrisk-site-security-check-report');
            $posture_explanation = __('Your site has a good security posture. All assessed configuration areas passed.', 'boonrisk-site-security-check-report');
        }
        
        return [
            'posture' => $posture,
            'posture_label' => $posture_label,
            'posture_explanation' => $posture_explanation,
            'high_risk_count' => $high_risk,
            'warning_count' => $warnings,
            'passed_count' => $passed,
            'total_count' => $total,
        ];
    }
    
    /**
     * Get human-readable category for an issue
     */
    private function get_issue_category($check_id) {
        $categories = [
            'php_version' => __('outdated runtime environment', 'boonrisk-site-security-check-report'),
            'wordpress_version' => __('outdated software', 'boonrisk-site-security-check-report'),
            'debug_mode' => __('exposed configuration settings', 'boonrisk-site-security-check-report'),
            'file_editor' => __('exposed configuration settings', 'boonrisk-site-security-check-report'),
            'xmlrpc' => __('unnecessary endpoints', 'boonrisk-site-security-check-report'),
            'https' => __('unencrypted data transmission', 'boonrisk-site-security-check-report'),
            'plugin_updates' => __('outdated components', 'boonrisk-site-security-check-report'),
            'theme_updates' => __('outdated components', 'boonrisk-site-security-check-report'),
            'plugin_count' => __('extended attack surface', 'boonrisk-site-security-check-report'),
            'admin_username' => __('predictable credentials', 'boonrisk-site-security-check-report'),
            'user_registration' => __('open registration settings', 'boonrisk-site-security-check-report'),
            'db_prefix' => __('default configuration', 'boonrisk-site-security-check-report'),
            'two_factor' => __('weak authentication', 'boonrisk-site-security-check-report'),
            'login_rate_limiting' => __('brute-force vulnerability', 'boonrisk-site-security-check-report'),
            'excessive_admins' => __('excessive privileges', 'boonrisk-site-security-check-report'),
            'inactive_privileged_users' => __('dormant accounts', 'boonrisk-site-security-check-report'),
            'auto_updates' => __('manual update process', 'boonrisk-site-security-check-report'),
            'dangerous_functions' => __('dangerous PHP functions', 'boonrisk-site-security-check-report'),
            'bot_protection' => __('missing bot protection', 'boonrisk-site-security-check-report'),
            'file_write_permissions' => __('excessive file permissions', 'boonrisk-site-security-check-report'),
            'wp_cron' => __('unreliable scheduled tasks', 'boonrisk-site-security-check-report'),
            'backup' => __('missing backups', 'boonrisk-site-security-check-report'),
            'inactive_plugins' => __('unused plugins', 'boonrisk-site-security-check-report'),
            'inactive_themes' => __('unused themes', 'boonrisk-site-security-check-report'),
            'plugin_maintenance' => __('unmaintained plugins', 'boonrisk-site-security-check-report'),
            'theme_source' => __('untrusted theme source', 'boonrisk-site-security-check-report'),
            'caching_plugin' => __('missing performance optimization', 'boonrisk-site-security-check-report'),
            'image_optimization' => __('missing image optimization', 'boonrisk-site-security-check-report'),
            'update_policy' => __('inconsistent update policy', 'boonrisk-site-security-check-report'),
            'security_logging' => __('missing security logging', 'boonrisk-site-security-check-report'),
            'production_isolation' => __('debug mode in production', 'boonrisk-site-security-check-report'),
        ];
        
        return $categories[$check_id] ?? __('configuration issue', 'boonrisk-site-security-check-report');
    }
    
    /**
     * Build human-readable posture explanation
     */
    private function build_posture_explanation($level, $categories, $high_risk, $warnings) {
        $categories = array_slice(array_unique($categories), 0, 2);
        $category_text = implode(' and ', $categories);
        
        if ($level === 'high') {
            return sprintf(
                /* translators: %s: Categories of security issues (e.g., "PHP version and WordPress updates") */
                __('Your site currently has a high security risk posture, based on %s. These issues should be reviewed.', 'boonrisk-site-security-check-report'),
                $category_text
            );
        } elseif ($level === 'medium') {
            return sprintf(
                /* translators: %s: Categories of security issues (e.g., "PHP version and WordPress updates") */
                __('Your site has a medium security risk posture, based on %s. Review the findings below for improvement opportunities.', 'boonrisk-site-security-check-report'),
                $category_text
            );
        } else {
            return sprintf(
                /* translators: %s: Categories of security issues (e.g., "PHP version and WordPress updates") */
                __('Your site has a low security risk posture with minor areas for improvement related to %s.', 'boonrisk-site-security-check-report'),
                $category_text
            );
        }
    }
    
    // ============================================
    // NEW CONTROL CHECKS (16 additional)
    // ============================================
    
    /**
     * Check Two-Factor Authentication (SEC-001)
     */
    private function check_two_factor_auth() {
        // Check for common 2FA plugins
        $twofa_plugins = [
            'two-factor/two-factor.php',
            'wordfence/wordfence.php',
            'google-authenticator/google-authenticator.php',
            'two-factor-authentication/two-factor-authentication.php',
        ];
        
        $has_2fa = false;
        foreach ($twofa_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_2fa = true;
                break;
            }
        }
        
        if (!$has_2fa) {
            return [
                'id' => 'two_factor',
                'name' => __('Two-Factor Authentication for admins', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No two-factor authentication plugin detected for admin accounts.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Enable 2FA plugin', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Prevents account takeover', 'boonrisk-site-security-check-report'),
                'risk_if_missing' => __('Admin compromise', 'boonrisk-site-security-check-report'),
                'how_to_fix' => __('Enable 2FA plugin', 'boonrisk-site-security-check-report'),
                'effort' => 'Medium',
                'risk_impact' => 'High',
                'finding_type' => 'Risk',
            ];
        }
        
        return [
            'id' => 'two_factor',
            'name' => __('Two-Factor Authentication for admins', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Enabled', 'boonrisk-site-security-check-report'),
            'message' => __('Two-factor authentication plugin is active.', 'boonrisk-site-security-check-report'),
            'why_it_matters' => __('Prevents account takeover', 'boonrisk-site-security-check-report'),
            'risk_impact' => 'High',
            'finding_type' => 'Risk',
        ];
    }
    
    /**
     * Check Login Rate Limiting
     */
    private function check_login_rate_limiting() {
        // Check for rate limiting plugins
        $rate_limit_plugins = [
            'limit-login-attempts-reloaded/limit-login-attempts-reloaded.php',
            'wordfence/wordfence.php',
            'login-lockdown/loginlockdown.php',
        ];
        
        $has_rate_limiting = false;
        foreach ($rate_limit_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_rate_limiting = true;
                break;
            }
        }
        
        if (!$has_rate_limiting) {
            return [
                'id' => 'login_rate_limiting',
                'name' => __('Login Rate Limiting', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No login rate limiting detected.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Install a plugin like Limit Login Attempts Reloaded to block brute-force attacks.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Rate limiting blocks automated brute-force login attempts.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'login_rate_limiting',
            'name' => __('Login Rate Limiting', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Enabled', 'boonrisk-site-security-check-report'),
            'message' => __('Login rate limiting is active.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Excessive Administrator Accounts
     */
    private function check_excessive_admins() {
        $admin_count = count(get_users(['role' => 'administrator']));
        
        if ($admin_count > 3) {
            return [
                'id' => 'excessive_admins',
                'name' => __('Administrator Accounts', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => sprintf(__('%d admins', 'boonrisk-site-security-check-report'), $admin_count),
                'message' => sprintf(__('%d administrator accounts detected. This increases the attack surface.', 'boonrisk-site-security-check-report'), $admin_count),
                'recommendation' => __('Review administrator accounts and reduce to the minimum required.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Each admin account is a potential entry point for attackers.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'excessive_admins',
            'name' => __('Administrator Accounts', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => sprintf(__('%d admins', 'boonrisk-site-security-check-report'), $admin_count),
            'message' => __('Administrator account count is reasonable.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Inactive Privileged Users
     */
    private function check_inactive_privileged_users() {
        $privileged_roles = ['administrator', 'editor'];
        $inactive_threshold = 90 * DAY_IN_SECONDS; // 90 days
        $inactive_users = [];
        
        foreach ($privileged_roles as $role) {
            $users = get_users(['role' => $role]);
            foreach ($users as $user) {
                $last_login = get_user_meta($user->ID, 'last_login', true);
                if (!$last_login) continue;
                
                if ((time() - $last_login) > $inactive_threshold) {
                    $inactive_users[] = $user->user_login;
                }
            }
        }
        
        if (count($inactive_users) > 0) {
            return [
                'id' => 'inactive_privileged_users',
                'name' => __('Inactive Privileged Users', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => sprintf(__('%d inactive', 'boonrisk-site-security-check-report'), count($inactive_users)),
                'message' => sprintf(__('%d privileged user(s) have been inactive for over 90 days.', 'boonrisk-site-security-check-report'), count($inactive_users)),
                'recommendation' => __('Remove or downgrade inactive privileged accounts.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Dormant privileged accounts are common entry points for attackers.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'inactive_privileged_users',
            'name' => __('Inactive Privileged Users', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('None', 'boonrisk-site-security-check-report'),
            'message' => __('No inactive privileged users detected.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Automatic Updates
     */
    private function check_auto_updates() {
        $auto_updates_enabled = get_option('auto_update_core_major', false) || 
                                get_option('auto_update_core_minor', true);
        
        if (!$auto_updates_enabled) {
            return [
                'id' => 'auto_updates',
                'name' => __('Automatic Updates', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Disabled', 'boonrisk-site-security-check-report'),
                'message' => __('Automatic updates are not enabled.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Enable automatic updates for at least minor WordPress releases.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Auto-updates reduce the window of exposure to known vulnerabilities.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'auto_updates',
            'name' => __('Automatic Updates', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Enabled', 'boonrisk-site-security-check-report'),
            'message' => __('Automatic updates are enabled.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Dangerous PHP Functions
     */
    private function check_dangerous_php_functions() {
        $dangerous_functions = ['eval', 'exec', 'system', 'shell_exec', 'passthru', 'proc_open', 'popen'];
        $disabled = ini_get('disable_functions');
        $disabled_array = array_map('trim', explode(',', $disabled));
        
        $enabled_dangerous = array_diff($dangerous_functions, $disabled_array);
        
        if (count($enabled_dangerous) > 0) {
            $func_list = implode(', ', array_values($enabled_dangerous));
            return [
                'id' => 'dangerous_functions',
                'name' => __('Dangerous PHP Functions', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => sprintf(__('%d enabled', 'boonrisk-site-security-check-report'), count($enabled_dangerous)),
                'message' => sprintf(__('Dangerous PHP functions are enabled and could be exploited: %s', 'boonrisk-site-security-check-report'), $func_list),
                'recommendation' => sprintf(__('Contact your hosting provider to disable these PHP functions: %s', 'boonrisk-site-security-check-report'), $func_list),
                'why_it_matters' => __('These functions can be used for remote code execution if exploited.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'dangerous_functions',
            'name' => __('Dangerous PHP Functions', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Disabled', 'boonrisk-site-security-check-report'),
            'message' => __('Dangerous PHP functions are properly disabled.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Bot Protection
     */
    private function check_bot_protection() {
        $bot_protection_plugins = [
            'wordfence/wordfence.php',
            'akismet/akismet.php',
            'recaptcha/recaptcha.php',
            'google-captcha/google-captcha.php',
        ];
        
        $has_bot_protection = false;
        foreach ($bot_protection_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_bot_protection = true;
                break;
            }
        }
        
        if (!$has_bot_protection) {
            return [
                'id' => 'bot_protection',
                'name' => __('Bot Protection', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No bot protection mechanism detected.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Install a CAPTCHA or bot protection plugin.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Bot protection prevents automated abuse and spam.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'bot_protection',
            'name' => __('Bot Protection', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Enabled', 'boonrisk-site-security-check-report'),
            'message' => __('Bot protection is active.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check File Write Permissions
     */
    private function check_file_write_permissions() {
        $critical_files = [
            ABSPATH . 'wp-config.php',
            ABSPATH . '.htaccess',
        ];
        
        $writable_files = [];
        foreach ($critical_files as $file) {
            if (file_exists($file) && wp_is_writable($file)) {
                $writable_files[] = basename($file);
            }
        }
        
        if (count($writable_files) > 0) {
            return [
                'id' => 'file_write_permissions',
                'name' => __('File Write Permissions', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => sprintf(__('%d writable', 'boonrisk-site-security-check-report'), count($writable_files)),
                'message' => __('Critical files have excessive write permissions.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Set critical files to read-only (chmod 444).', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Writable critical files can be modified by attackers.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'file_write_permissions',
            'name' => __('File Write Permissions', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Restricted', 'boonrisk-site-security-check-report'),
            'message' => __('File write permissions are properly restricted.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check WP-Cron Visibility
     */
    private function check_wp_cron_visibility() {
        $cron_disabled = defined('DISABLE_WP_CRON') && DISABLE_WP_CRON;
        
        if (!$cron_disabled) {
            return [
                'id' => 'wp_cron',
                'name' => __('WP-Cron Execution', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Default', 'boonrisk-site-security-check-report'),
                'message' => __('WP-Cron is using default execution (may be unreliable).', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Consider using a system cron for more reliable scheduled tasks.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Proper cron execution ensures scheduled tasks run reliably.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'wp_cron',
            'name' => __('WP-Cron Execution', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('System cron', 'boonrisk-site-security-check-report'),
            'message' => __('WP-Cron is properly configured with system cron.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Backup Mechanism
     */
    private function check_backup_mechanism() {
        $backup_plugins = [
            'updraftplus/updraftplus.php',
            'backwpup/backwpup.php',
            'duplicator/duplicator.php',
            'all-in-one-wp-migration/all-in-one-wp-migration.php',
        ];
        
        $has_backup = false;
        foreach ($backup_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_backup = true;
                break;
            }
        }
        
        if (!$has_backup) {
            return [
                'id' => 'backup',
                'name' => __('Backup Mechanism', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No backup plugin detected.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Install a backup plugin like UpdraftPlus or BackWPup.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Backups enable recovery from data loss or security incidents.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'backup',
            'name' => __('Backup Mechanism', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Detected', 'boonrisk-site-security-check-report'),
            'message' => __('Backup plugin is active.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Inactive Plugins
     */
    private function check_inactive_plugins() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', []);
        $inactive_count = count($all_plugins) - count($active_plugins);
        
        if ($inactive_count > 0) {
            return [
                'id' => 'inactive_plugins',
                'name' => __('Inactive Plugins', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => sprintf(__('%d inactive', 'boonrisk-site-security-check-report'), $inactive_count),
                'message' => sprintf(__('%d inactive plugin(s) detected.', 'boonrisk-site-security-check-report'), $inactive_count),
                'recommendation' => __('Remove unused plugins to reduce attack surface.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Inactive plugins can still be exploited if they contain vulnerabilities.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'inactive_plugins',
            'name' => __('Inactive Plugins', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('None', 'boonrisk-site-security-check-report'),
            'message' => __('No inactive plugins detected.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Inactive Themes
     */
    private function check_inactive_themes() {
        $all_themes = wp_get_themes();
        $active_theme = wp_get_theme();
        $inactive_count = count($all_themes) - 1; // Minus active theme
        
        if ($inactive_count > 1) {
            return [
                'id' => 'inactive_themes',
                'name' => __('Inactive Themes', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => sprintf(__('%d inactive', 'boonrisk-site-security-check-report'), $inactive_count),
                'message' => sprintf(__('%d inactive theme(s) detected.', 'boonrisk-site-security-check-report'), $inactive_count),
                'recommendation' => __('Remove unused themes (keep one default theme as fallback).', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Inactive themes can contain vulnerabilities.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'inactive_themes',
            'name' => __('Inactive Themes', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Minimal', 'boonrisk-site-security-check-report'),
            'message' => __('Theme count is reasonable.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Plugin Maintenance Status
     */
    private function check_plugin_maintenance() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $active_plugins = get_option('active_plugins', []);
        $all_plugins = get_plugins();
        $stale_plugins = [];
        
        foreach ($active_plugins as $plugin_path) {
            if (isset($all_plugins[$plugin_path])) {
                // Check if plugin hasn't been updated in 2+ years (rough heuristic)
                // In a real implementation, you'd query WordPress.org API
                $plugin_data = $all_plugins[$plugin_path];
                // For now, we'll just check if version is very old
                if (isset($plugin_data['Version']) && version_compare($plugin_data['Version'], '1.0', '<')) {
                    $stale_plugins[] = $plugin_data['Name'];
                }
            }
        }
        
        if (count($stale_plugins) > 0) {
            return [
                'id' => 'plugin_maintenance',
                'name' => __('Plugin Maintenance', 'boonrisk-site-security-check-report'),
                'status' => 'high_risk',
                'value' => sprintf(__('%d stale', 'boonrisk-site-security-check-report'), count($stale_plugins)),
                'message' => __('Some plugins may be abandoned or unmaintained.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Review plugin maintenance status and replace abandoned plugins.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Unmaintained plugins become security liabilities.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'plugin_maintenance',
            'name' => __('Plugin Maintenance', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Active', 'boonrisk-site-security-check-report'),
            'message' => __('Plugins appear to be actively maintained.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Theme Source Trust
     */
    private function check_theme_source_trust() {
        $theme = wp_get_theme();
        $theme_uri = $theme->get('ThemeURI');
        
        // Check if theme is from WordPress.org or known trusted sources
        $is_trusted = (
            strpos($theme_uri, 'wordpress.org') !== false ||
            strpos($theme_uri, 'themeforest.net') !== false ||
            $theme->get('Author') === 'WordPress.org'
        );
        
        if (!$is_trusted && $theme_uri) {
            return [
                'id' => 'theme_source',
                'name' => __('Theme Source', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Unknown', 'boonrisk-site-security-check-report'),
                'message' => __('Theme source cannot be verified.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Use themes from trusted sources like WordPress.org.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Themes from unknown sources may contain malicious code.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'theme_source',
            'name' => __('Theme Source', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Trusted', 'boonrisk-site-security-check-report'),
            'message' => __('Theme is from a trusted source.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Caching Plugin
     */
    private function check_caching_plugin() {
        $caching_plugins = [
            'wp-rocket/wp-rocket.php',
            'w3-total-cache/w3-total-cache.php',
            'wp-super-cache/wp-cache.php',
            'litespeed-cache/litespeed-cache.php',
            'wp-fastest-cache/wpFastestCache.php',
        ];
        
        $has_caching = false;
        foreach ($caching_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_caching = true;
                break;
            }
        }
        
        if (!$has_caching) {
            return [
                'id' => 'caching_plugin',
                'name' => __('Caching Plugin', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No caching plugin detected.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Install a caching plugin to improve performance.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Caching improves site speed and reduces server load.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'caching_plugin',
            'name' => __('Caching Plugin', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Active', 'boonrisk-site-security-check-report'),
            'message' => __('Caching plugin is active.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Image Optimization
     */
    private function check_image_optimization() {
        $image_plugins = [
            'shortpixel-image-optimiser/wp-shortpixel.php',
            'wp-smushit/wp-smush.php',
            'imagify/imagify.php',
            'ewww-image-optimizer/ewww-image-optimizer.php',
        ];
        
        $has_image_opt = false;
        foreach ($image_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_image_opt = true;
                break;
            }
        }
        
        if (!$has_image_opt) {
            return [
                'id' => 'image_optimization',
                'name' => __('Image Optimization', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No image optimization plugin detected.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Install an image optimization plugin to improve performance.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Image optimization reduces page load times.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'image_optimization',
            'name' => __('Image Optimization', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Active', 'boonrisk-site-security-check-report'),
            'message' => __('Image optimization is active.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Update Policy Consistency
     */
    private function check_update_policy_consistency() {
        $core_auto = get_option('auto_update_core_major', false);
        $plugin_auto = get_option('auto_update_plugins', []);
        $theme_auto = get_option('auto_update_themes', []);
        
        // Check if policies are inconsistent
        $has_inconsistency = ($core_auto && empty($plugin_auto)) || (!$core_auto && !empty($plugin_auto));
        
        if ($has_inconsistency) {
            return [
                'id' => 'update_policy',
                'name' => __('Update Policy', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Inconsistent', 'boonrisk-site-security-check-report'),
                'message' => __('Update policies are inconsistent across components.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Align auto-update settings for core, plugins, and themes.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Inconsistent policies can leave components outdated.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'update_policy',
            'name' => __('Update Policy', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Consistent', 'boonrisk-site-security-check-report'),
            'message' => __('Update policies are consistent.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Security Logging
     */
    private function check_security_logging() {
        $logging_plugins = [
            'wordfence/wordfence.php',
            'wp-security-audit-log/wp-security-audit-log.php',
            'simple-history/index.php',
        ];
        
        $has_logging = false;
        foreach ($logging_plugins as $plugin) {
            if (is_plugin_active($plugin)) {
                $has_logging = true;
                break;
            }
        }
        
        if (!$has_logging) {
            return [
                'id' => 'security_logging',
                'name' => __('Security Logging', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Not detected', 'boonrisk-site-security-check-report'),
                'message' => __('No security activity logging detected.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Install a security logging plugin for audit trails.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Logging helps detect and investigate security incidents.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'security_logging',
            'name' => __('Security Logging', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Active', 'boonrisk-site-security-check-report'),
            'message' => __('Security logging is active.', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check Production Environment Isolation
     */
    private function check_production_isolation() {
        $debug_enabled = defined('WP_DEBUG') && WP_DEBUG;
        $debug_display = defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY;
        
        if ($debug_enabled || $debug_display) {
            return [
                'id' => 'production_isolation',
                'name' => __('Production Environment', 'boonrisk-site-security-check-report'),
                'status' => 'warning',
                'value' => __('Debug enabled', 'boonrisk-site-security-check-report'),
                'message' => __('Debug mode is enabled in production.', 'boonrisk-site-security-check-report'),
                'recommendation' => __('Disable debug mode in production environments.', 'boonrisk-site-security-check-report'),
                'why_it_matters' => __('Debug mode can expose sensitive information.', 'boonrisk-site-security-check-report'),
            ];
        }
        
        return [
            'id' => 'production_isolation',
            'name' => __('Production Environment', 'boonrisk-site-security-check-report'),
            'status' => 'pass',
            'value' => __('Isolated', 'boonrisk-site-security-check-report'),
            'message' => __('Production environment is properly isolated.', 'boonrisk-site-security-check-report'),
        ];
    }
}

