<?php
/**
 * Main collector class that orchestrates all data collection
 * 
 * Collects all security signals required for full control evaluation.
 * Supports both plugin scans (full data) and snapshot generation.
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Collector {
    
    private $collectors = [];
    
    public function __construct() {
        $this->collectors = [
            'environment' => new BoonRisk_Environment_Collector(),
            'config' => new BoonRisk_Config_Collector(),
            'plugins' => new BoonRisk_Plugins_Collector(),
            'auth' => new BoonRisk_Auth_Collector(),
            'updates' => new BoonRisk_Updates_Collector(),
            'extended' => new BoonRisk_Extended_Collector(),
        ];
    }
    
    /**
     * Collect all security signals
     * 
     * @return array Structured snapshot of all collected data
     */
    public function collect_all() {
        $snapshot = [
            'schema_version' => '2.0',
            'collected_at' => gmdate('c'),
            'site_url' => get_site_url(),
            'site_hash' => $this->generate_site_hash(),
            'agent_version' => BOONRISK_VERSION,
            'data' => [],
        ];
        
        foreach ($this->collectors as $key => $collector) {
            try {
                $snapshot['data'][$key] = $collector->collect();
            } catch (Exception $e) {
                $snapshot['data'][$key] = [
                    'error' => true,
                    'message' => $e->getMessage(),
                ];
            }
        }
        
        // Flatten extended data into main structure for easier access
        if (isset($snapshot['data']['extended']) && is_array($snapshot['data']['extended'])) {
            $extended = $snapshot['data']['extended'];
            
            // Merge backup plugins into plugins section
            if (isset($extended['backup_plugins'])) {
                $snapshot['data']['plugins']['backup'] = $extended['backup_plugins'];
            }
            
            // Merge bot protection into plugins section
            if (isset($extended['bot_protection'])) {
                $snapshot['data']['plugins']['bot_protection'] = $extended['bot_protection'];
            }
            
            // Merge image optimization into plugins section
            if (isset($extended['image_optimization'])) {
                $snapshot['data']['plugins']['image_optimization'] = $extended['image_optimization'];
            }
            
            // Merge activity log into plugins section
            if (isset($extended['activity_log'])) {
                $snapshot['data']['plugins']['activity_log'] = $extended['activity_log'];
            }
            
            // Merge inactive plugins info
            if (isset($extended['inactive_plugins'])) {
                $snapshot['data']['plugins']['inactive_count'] = $extended['inactive_plugins']['inactive_count'];
                $snapshot['data']['plugins']['inactive_list'] = $extended['inactive_plugins']['inactive_list'];
            }
            
            // Add themes section
            if (isset($extended['themes'])) {
                $snapshot['data']['themes'] = $extended['themes'];
            }
            
            // Add users inactive info
            if (isset($extended['inactive_admins'])) {
                $snapshot['data']['users'] = $extended['inactive_admins'];
            }
            
            // Add filesystem info
            if (isset($extended['filesystem'])) {
                $snapshot['data']['filesystem'] = $extended['filesystem'];
            }
            
            // Add cron info
            if (isset($extended['cron'])) {
                $snapshot['data']['cron'] = $extended['cron'];
            }
            
            // Merge production environment info into environment
            if (isset($extended['production'])) {
                $snapshot['data']['environment']['is_production'] = $extended['production']['is_production'];
                $snapshot['data']['environment']['debug_artifacts'] = $extended['production']['debug_artifacts'];
                $snapshot['data']['environment']['staging_indicators'] = $extended['production']['staging_indicators'];
            }
            
            // Add webhooks info
            if (isset($extended['webhooks'])) {
                $snapshot['data']['webhooks'] = $extended['webhooks'];
            }
        }
        
        // Add database prefix to config for easy access
        if (isset($snapshot['data']['environment']['database']['prefix'])) {
            $snapshot['data']['config']['database'] = [
                'prefix' => $snapshot['data']['environment']['database']['prefix'],
            ];
        }
        
        return $snapshot;
    }
    
    /**
     * Generate unique site hash for identification
     */
    private function generate_site_hash() {
        $site_data = get_site_url() . get_option('blogname') . DB_NAME;
        return hash('sha256', $site_data);
    }
    
    /**
     * Get summary of collected data for quick overview
     * 
     * @return array Summary statistics
     */
    public function get_summary() {
        $data = $this->collect_all();
        
        $summary = [
            'schema_version' => $data['schema_version'],
            'collected_at' => $data['collected_at'],
            'totals' => [
                'active_plugins' => count($data['data']['plugins']['active_plugins'] ?? []),
                'inactive_plugins' => $data['data']['plugins']['inactive_count'] ?? 0,
                'pending_plugin_updates' => $data['data']['updates']['plugins']['active_with_updates'] ?? 0,
                'pending_theme_updates' => $data['data']['updates']['themes']['active_theme_needs_update'] ? 1 : 0,
                'admin_users' => $data['data']['config']['users']['admin_count'] ?? 0,
                'inactive_admins' => count($data['data']['users']['inactive_admins'] ?? []),
            ],
            'security_signals' => [
                'has_2fa' => $data['data']['plugins']['2fa_plugins']['has_2fa'] ?? false,
                'has_rate_limiting' => $data['data']['auth']['rate_limiting']['has_rate_limiting'] ?? false,
                'has_backup' => $data['data']['plugins']['backup']['has_backup_plugin'] ?? false,
                'has_bot_protection' => $data['data']['plugins']['bot_protection']['has_bot_protection'] ?? false,
                'has_activity_log' => $data['data']['plugins']['activity_log']['has_logging'] ?? false,
                'has_caching' => $data['data']['plugins']['caching_plugins']['has_page_cache'] ?? false,
                'debug_enabled' => $data['data']['config']['debug']['wp_debug'] ?? false,
                'is_production' => $data['data']['environment']['is_production'] ?? true,
            ],
        ];
        
        return $summary;
    }
}
