<?php
/**
 * Auto Sync Handler
 * 
 * Handles automatic daily synchronization with BoonRisk dashboard.
 * Complies with WordPress.org guidelines:
 * - Opt-in only (disabled by default)
 * - Clear user consent required
 * - Transparent about data transmission
 * - User can disable at any time
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Auto_Sync {
    
    /**
     * Cron hook name
     */
    const CRON_HOOK = 'boonrisk_daily_sync';
    
    /**
     * Option names
     */
    const OPTION_ENABLED = 'boonrisk_auto_sync_enabled';
    const OPTION_LAST_HASH = 'boonrisk_last_assessment_hash';
    const OPTION_LAST_SYNC = 'boonrisk_last_auto_sync';
    const OPTION_CONSENT_GIVEN = 'boonrisk_auto_sync_consent';
    const OPTION_SYNC_HISTORY = 'boonrisk_sync_history';
    const MAX_HISTORY_ENTRIES = 10;
    
    /**
     * Collector instance
     */
    private $collector;
    
    /**
     * Local assessment instance
     */
    private $local_assessment;
    
    /**
     * API client instance
     */
    private $api_client;
    
    /**
     * Constructor
     */
    public function __construct($collector, $local_assessment, $api_client) {
        $this->collector = $collector;
        $this->local_assessment = $local_assessment;
        $this->api_client = $api_client;
        
        // Register cron hook
        add_action(self::CRON_HOOK, [$this, 'run_scheduled_sync']);
        
        // Handle activation/deactivation of auto-sync
        add_action('update_option_' . self::OPTION_ENABLED, [$this, 'handle_setting_change'], 10, 2);
    }
    
    /**
     * Check if auto-sync is enabled
     * 
     * @return bool True if enabled and has consent
     */
    public function is_enabled() {
        return (
            get_option(self::OPTION_ENABLED, false) &&
            get_option(self::OPTION_CONSENT_GIVEN, false) &&
            $this->api_client->has_api_key()
        );
    }
    
    /**
     * Enable auto-sync with consent
     * 
     * @return bool Success
     */
    public function enable($consent = false) {
        if (!$consent) {
            return false;
        }
        
        if (!$this->api_client->has_api_key()) {
            return false;
        }
        
        update_option(self::OPTION_CONSENT_GIVEN, true);
        update_option(self::OPTION_ENABLED, true);
        
        $this->schedule_cron();
        
        return true;
    }
    
    /**
     * Disable auto-sync
     * 
     * @return bool Success
     */
    public function disable() {
        update_option(self::OPTION_ENABLED, false);
        $this->unschedule_cron();
        return true;
    }
    
    /**
     * Schedule the daily cron job
     */
    public function schedule_cron() {
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            // Schedule for a random time to distribute load
            $random_offset = wp_rand(0, 3600 * 6); // Random offset up to 6 hours
            $first_run = strtotime('tomorrow 3:00am') + $random_offset;
            
            wp_schedule_event($first_run, 'daily', self::CRON_HOOK);
        }
    }
    
    /**
     * Unschedule the cron job
     */
    public function unschedule_cron() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }
    }
    
    /**
     * Handle setting change
     */
    public function handle_setting_change($old_value, $new_value) {
        if ($new_value && get_option(self::OPTION_CONSENT_GIVEN, false)) {
            $this->schedule_cron();
        } else {
            $this->unschedule_cron();
        }
    }
    
    /**
     * Run scheduled sync
     * Called by WP-Cron
     */
    public function run_scheduled_sync() {
        // Double-check conditions
        if (!$this->is_enabled()) {
            return;
        }
        
        // Check if there's a pending request from dashboard
        $has_pending_request = $this->check_pending_request();
        
        // Run local assessment
        $results = $this->local_assessment->run();
        
        // Calculate hash of results
        $current_hash = $this->calculate_hash($results);
        $last_hash = get_option(self::OPTION_LAST_HASH, '');
        $last_sync = get_option(self::OPTION_LAST_SYNC, 0);
        
        // Check if last sync was more than 23 hours ago (force daily sync)
        $hours_since_last_sync = (time() - $last_sync) / 3600;
        
        // Determine if we should send
        $should_send = false;
        $reason = '';
        
        if ($has_pending_request) {
            $should_send = true;
            $reason = 'pending_request';
        } elseif ($current_hash !== $last_hash) {
            $should_send = true;
            $reason = 'changes_detected';
        } elseif ($hours_since_last_sync >= 23) {
            // Always send at least once per day even if no changes
            $should_send = true;
            $reason = 'daily_heartbeat';
        }
        
        if ($should_send) {
            // Store local assessment
            update_option('boonrisk_last_local_assessment', [
                'timestamp' => time(),
                'results' => $results,
            ]);
            
            // Collect full snapshot and send
            $snapshot = $this->collector->collect_all();
            $result = $this->api_client->send_snapshot($snapshot);
            
            if ($result['success']) {
                // Update hash and timestamp
                update_option(self::OPTION_LAST_HASH, $current_hash);
                update_option(self::OPTION_LAST_SYNC, time());
                
                // Add to sync history
                $this->add_to_history('auto', $reason, true);
                
                // Clear pending request if there was one
                if ($has_pending_request) {
                    $this->clear_pending_request();
                }
            } else {
                // Record failed sync
                $error_msg = $result['message'] ?? 'Unknown error';
                $this->add_to_history('auto', $reason, false, $error_msg);
            }
        }
    }
    
    /**
     * Run manual sync (always sends)
     * 
     * @return array Result with success status
     */
    public function run_manual_sync() {
        if (!$this->api_client->has_api_key()) {
            return [
                'success' => false,
                'message' => __('API key not configured', 'boonrisk-site-security-check-report'),
            ];
        }
        
        // Run local assessment
        $results = $this->local_assessment->run();
        
        // Store local assessment
        update_option('boonrisk_last_local_assessment', [
            'timestamp' => time(),
            'results' => $results,
        ]);
        
        // Collect full snapshot and send
        $snapshot = $this->collector->collect_all();
        $result = $this->api_client->send_snapshot($snapshot);
        
        if ($result['success']) {
            // Update hash but NOT timestamp (to avoid disrupting auto-sync schedule)
            $current_hash = $this->calculate_hash($results);
            update_option(self::OPTION_LAST_HASH, $current_hash);
            // Note: We intentionally do NOT update OPTION_LAST_SYNC here
            // to avoid disrupting the automatic daily sync schedule
            
            // Add to sync history
            $this->add_to_history('manual', 'user_initiated', true);
            
            // Clear any pending request
            $this->clear_pending_request();
        } else {
            // Record failed sync
            $this->add_to_history('manual', 'user_initiated', false, $result['message'] ?? null);
        }
        
        return $result;
    }
    
    /**
     * Calculate hash of assessment results
     * 
     * @param array $results Assessment results
     * @return string Hash
     */
    private function calculate_hash($results) {
        // Create deterministic string from key data points
        $data_points = [];
        
        if (!empty($results['checks'])) {
            foreach ($results['checks'] as $check) {
                $data_points[] = $check['id'] . ':' . $check['status'] . ':' . ($check['value'] ?? '');
            }
        }
        
        sort($data_points);
        return md5(implode('|', $data_points));
    }
    
    /**
     * Check if dashboard has requested a scan
     * 
     * @return bool True if pending request exists
     */
    private function check_pending_request() {
        $result = $this->api_client->check_pending_request();
        return $result['success'] && !empty($result['pending']);
    }
    
    /**
     * Clear pending request after fulfilling it
     */
    private function clear_pending_request() {
        $this->api_client->clear_pending_request();
    }
    
    /**
     * Get sync status for display
     * 
     * @return array Status information
     */
    public function get_status() {
        $enabled = $this->is_enabled();
        $last_sync = get_option(self::OPTION_LAST_SYNC, 0);
        $next_scheduled = wp_next_scheduled(self::CRON_HOOK);
        $has_api_key = $this->api_client->has_api_key();
        $consent_given = get_option(self::OPTION_CONSENT_GIVEN, false);
        
        return [
            'enabled' => $enabled,
            'has_api_key' => $has_api_key,
            'consent_given' => $consent_given,
            'last_sync' => $last_sync,
            'last_sync_formatted' => $last_sync 
                ? human_time_diff($last_sync, time()) . ' ' . __('ago', 'boonrisk-site-security-check-report')
                : __('Never', 'boonrisk-site-security-check-report'),
            'next_scheduled' => $next_scheduled,
            'next_scheduled_formatted' => $next_scheduled 
                ? human_time_diff(time(), $next_scheduled)
                : __('Not scheduled', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Add entry to sync history
     * 
     * @param string $type Sync type (manual or auto)
     * @param string $reason Reason for sync
     * @param bool $success Whether sync was successful
     * @param string $error_message Optional error message
     */
    private function add_to_history($type, $reason, $success, $error_message = null) {
        $history = get_option(self::OPTION_SYNC_HISTORY, []);
        
        // Add new entry at the beginning
        array_unshift($history, [
            'timestamp' => time(),
            'type' => $type,
            'reason' => $reason,
            'success' => $success,
            'error_message' => $error_message,
        ]);
        
        // Keep only last MAX_HISTORY_ENTRIES
        $history = array_slice($history, 0, self::MAX_HISTORY_ENTRIES);
        
        update_option(self::OPTION_SYNC_HISTORY, $history);
    }
    
    /**
     * Get sync history
     * 
     * @return array Sync history entries
     */
    public function get_history() {
        return get_option(self::OPTION_SYNC_HISTORY, []);
    }
    
    /**
     * Clean up on plugin deactivation
     */
    public static function deactivate() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }
    }
}

