<?php
/**
 * API client for communicating with BoonRisk platform
 * 
 * Connects to BoonRisk API for:
 * - API key validation
 * - Scan submission
 * - App analysis
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_API_Client {
    
    /**
     * Base URL for BoonRisk API (unified backend)
     */
    private $api_url = 'https://api.boonrisk.com/v1/plugin';
    
    /**
     * API key for authentication
     */
    private $api_key;
    
    public function __construct() {
        // Use hardcoded endpoint; no user-configurable endpoint to reduce mistakes
        $this->api_key = get_option('boonrisk_api_key', '');
    }
    
    /**
     * Get the endpoint URL for a specific action
     * 
     * @param string $action API action (validate-key, submit-scan)
     * @return string Full URL to the endpoint
     */
    private function get_endpoint_url($action) {
        return trailingslashit($this->api_url) . $action;
    }
    
    /**
     * Make authenticated request to BoonRisk API
     * 
     * @param string $action API action to call
     * @param array  $data Data to send (for POST requests)
     * @param string $method HTTP method (GET or POST)
     * @return array Response data or error
     */
    private function request($action, $data = null, $method = 'POST') {
        if (empty($this->api_key)) {
            return [
                'success' => false,
                'message' => __('API key not configured', 'boonrisk-site-security-check-report'),
            ];
        }
        
        $url = $this->get_endpoint_url($action);
        
        $args = [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key'    => $this->api_key,
            ],
        ];
        
        if ($method === 'POST' && $data) {
            $args['body'] = wp_json_encode($data);
            $response = wp_remote_post($url, $args);
        } else {
            $response = wp_remote_get($url, $args);
        }
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            
            return [
                'success' => false,
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Connection error: %s', 'boonrisk-site-security-check-report'),
                    $error_message
                ),
            ];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status_code >= 200 && $status_code < 300) {
            return [
                'success' => true,
                'data'    => $body,
            ];
        }
        
        // Enhanced error handling with more context
        $error_message = $body['error'] ?? $body['message'] ?? __('Unknown error', 'boonrisk-site-security-check-report');
        
        // User-friendly error messages
        $user_message = $error_message;
        if ($status_code === 401 || $status_code === 403) {
            $user_message = __('Authentication failed. Please check your API key.', 'boonrisk-site-security-check-report');
        } elseif ($status_code === 404) {
            $user_message = __('API endpoint not found. Please update the plugin.', 'boonrisk-site-security-check-report');
        } elseif ($status_code >= 500) {
            $user_message = __('Server error. Please try again later.', 'boonrisk-site-security-check-report');
        }
        
        return [
            'success'     => false,
            'message'     => $user_message,
            'status_code' => $status_code,
            'raw_error'   => $error_message,
        ];
    }
    
    /**
     * Send snapshot to BoonRisk platform for app analysis
     * 
     * @param array $snapshot Collected security data
     * @return array Result with success status and message
     */
    public function send_snapshot($snapshot) {
        // Get local assessment results if available
        $local_assessment = get_option('boonrisk_last_local_assessment', null);
        $local_posture = null;
        $local_score = null;
        $local_checks = null;
        $local_top_risks = null;
        
        if ($local_assessment && isset($local_assessment['results'])) {
            $results = $local_assessment['results'];
            
            if (isset($results['summary'])) {
                $summary = $results['summary'];
            $local_posture = $summary['posture'] ?? 'medium';
            // Calculate approximate score
            $total = $summary['total_count'] ?? 12;
            $passed = $summary['passed_count'] ?? 0;
            $local_score = $total > 0 ? round(($passed / $total) * 100) : 50;
            }
            
            // Include full checks data for rich findings display
            $local_checks = $results['checks'] ?? null;
            $local_top_risks = $results['top_risks'] ?? null;
        }
        
        $data = [
            'snapshot'        => $snapshot,
            'agent_version'   => defined('BOONRISK_VERSION') ? BOONRISK_VERSION : '1.0.2',
            'local_posture'   => $local_posture,
            'local_score'     => $local_score,
            'local_checks'    => $local_checks,
            'local_top_risks' => $local_top_risks,
            'local_summary'   => $local_assessment['results']['summary'] ?? null,
            'site_url'        => get_site_url(),
            'site_name'       => get_bloginfo('name'),
        ];
        
        $result = $this->request('submit-scan', $data);
        
        if ($result['success']) {
            // Store scan ID for reference
            if (isset($result['data']['scan_id'])) {
                update_option('boonrisk_last_app_scan_id', $result['data']['scan_id']);
                update_option('boonrisk_last_app_scan_time', current_time('mysql'));
            }
            
            return [
                'success'       => true,
                'message'       => __('Sent to dashboard successfully', 'boonrisk-site-security-check-report'),
                'data'          => $result['data'] ?? null,
                'score'         => $result['data']['score'] ?? null,
                'risk_level'    => $result['data']['risk_level'] ?? null,
                'dashboard_url' => $result['data']['dashboard_url'] ?? 'https://app.boonrisk.com',
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['message'] ?? __('Failed to send to dashboard', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Test connection to BoonRisk platform
     * Validates the API key
     * 
     * @return array Result with success status
     */
    public function test_connection() {
        $result = $this->request('validate-key', null, 'GET');
        
        if ($result['success'] && isset($result['data']['valid']) && $result['data']['valid']) {
            return [
                'success'  => true,
                'message'  => __('Connection successful', 'boonrisk-site-security-check-report'),
                'site_id'  => $result['data']['site_id'] ?? null,
                'user_id'  => $result['data']['user_id'] ?? null,
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['data']['error'] ?? $result['message'] ?? __('Connection failed', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Check if API key is configured
     * 
     * @return bool True if API key exists
     */
    public function has_api_key() {
        return !empty($this->api_key);
    }
    
    /**
     * Get the configured API endpoint
     * 
     * @return string API endpoint URL
     */
    public function get_endpoint() {
        return $this->api_url;
    }
    
    /**
     * Check if dashboard has a pending scan request for this site
     * 
     * @return array Result with pending status
     */
    public function check_pending_request() {
        $result = $this->request('check-pending', null, 'GET');
        
        if ($result['success']) {
            return [
                'success' => true,
                'pending' => $result['data']['pending'] ?? false,
                'requested_at' => $result['data']['requested_at'] ?? null,
            ];
        }
        
        return [
            'success' => false,
            'pending' => false,
            'message' => $result['message'] ?? __('Failed to check pending requests', 'boonrisk-site-security-check-report'),
        ];
    }
    
    /**
     * Clear pending scan request after fulfilling it
     * 
     * @return array Result with success status
     */
    public function clear_pending_request() {
        $result = $this->request('clear-pending', ['fulfilled' => true], 'POST');
        
        return [
            'success' => $result['success'],
            'message' => $result['success'] 
                ? __('Request cleared', 'boonrisk-site-security-check-report')
                : ($result['message'] ?? __('Failed to clear request', 'boonrisk-site-security-check-report')),
        ];
    }
    
}
