<?php
/**
 * Plugin Name: BoonRisk – Site Security Check & Report
 * Plugin URI: https://boonrisk.com
 * Description: Safe, read-only security check for WordPress. Get a clear security report with risks explained and actionable recommendations. No scanning, blocking, or site changes.
 * Version: 1.0.2
 * Author: boonband
 * Author URI: https://boonrisk.com
 * License: GPL v2 or later
 * Text Domain: boonrisk-site-security-check-report
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

define('BOONRISK_VERSION', '1.0.2');
define('BOONRISK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BOONRISK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BOONRISK_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include core files
require_once BOONRISK_PLUGIN_DIR . 'includes/class-collector.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/class-api-client.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/class-local-assessment.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/class-auto-sync.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/collectors/class-environment-collector.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/collectors/class-config-collector.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/collectors/class-plugins-collector.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/collectors/class-auth-collector.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/collectors/class-updates-collector.php';
require_once BOONRISK_PLUGIN_DIR . 'includes/collectors/class-extended-collector.php';

// Admin pages
require_once BOONRISK_PLUGIN_DIR . 'admin/class-admin-pages.php';

/**
 * Main plugin class
 */
class BoonRisk_Agent {
    
    private static $instance = null;
    private $collector;
    private $api_client;
    private $local_assessment;
    private $auto_sync;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->collector = new BoonRisk_Collector();
        $this->api_client = new BoonRisk_API_Client();
        $this->local_assessment = new BoonRisk_Local_Assessment($this->collector);
        $this->auto_sync = new BoonRisk_Auto_Sync($this->collector, $this->local_assessment, $this->api_client);
        
        // Admin menu and pages
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        
        // AJAX handlers - all require explicit user action
        add_action('wp_ajax_boonrisk_local_assessment', [$this, 'ajax_local_assessment']);
        add_action('wp_ajax_boonrisk_test_connection', [$this, 'ajax_test_connection']);
        add_action('wp_ajax_boonrisk_app_analysis', [$this, 'ajax_app_analysis']);
        add_action('wp_ajax_boonrisk_save_api_key', [$this, 'ajax_save_api_key']);
        add_action('wp_ajax_boonrisk_remove_api_key', [$this, 'ajax_remove_api_key']);
        add_action('wp_ajax_boonrisk_toggle_auto_sync', [$this, 'ajax_toggle_auto_sync']);
        add_action('wp_ajax_boonrisk_manual_sync', [$this, 'ajax_manual_sync']);
        
        // Plugin action links
        add_filter('plugin_action_links_' . BOONRISK_PLUGIN_BASENAME, [$this, 'add_action_links']);
        
        // Auto-sync uses WP-Cron but only when explicitly enabled by user (opt-in)
    }
    
    /**
     * Add admin menu pages - Simplified UX
     * Streamlined menu structure with clear sections
     */
    public function add_admin_menu() {
        // Main menu - Dashboard
        add_menu_page(
            __('BoonRisk', 'boonrisk-site-security-check-report'),
            __('BoonRisk', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk',
            [$this, 'render_dashboard_page'],
            'dashicons-shield-alt',
            80
        );
        
        // Submenu: Dashboard
        add_submenu_page(
            'boonrisk',
            __('Dashboard', 'boonrisk-site-security-check-report'),
            __('Dashboard', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk',
            [$this, 'render_dashboard_page']
        );
        
        // Submenu: Local Assessment (visible in menu)
        add_submenu_page(
            'boonrisk',
            __('Local Assessment', 'boonrisk-site-security-check-report'),
            __('Local Assessment', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk-local',
            [$this, 'render_local_page']
        );
        
        // Submenu: Data & Monitoring (visible when connected)
        if (self::has_api_key()) {
            add_submenu_page(
                'boonrisk',
                __('Data & Monitoring', 'boonrisk-site-security-check-report'),
                __('Data & Monitoring', 'boonrisk-site-security-check-report'),
                'manage_options',
                'boonrisk-sync',
                [$this, 'render_sync_page']
            );
        }
        
        // Submenu: Settings
        add_submenu_page(
            'boonrisk',
            __('Settings', 'boonrisk-site-security-check-report'),
            __('Settings', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk-settings',
            [$this, 'render_settings_page']
        );
        
        // === Hidden pages (not in menu, accessible via links) ===
        
        // Full HTML report page
        add_submenu_page(
            null,
            __('Security Report', 'boonrisk-site-security-check-report'),
            __('Security Report', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk-report',
            [$this, 'render_local_report_page']
        );
        
        // Connect page (for initial connection setup)
        add_submenu_page(
            null,
            __('Connect to Dashboard', 'boonrisk-site-security-check-report'),
            __('Connect to Dashboard', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk-connect',
            [$this, 'render_connect_page']
        );
        
        // Legacy routes (redirects for old links)
        add_submenu_page(
            null,
            __('App Dashboard', 'boonrisk-site-security-check-report'),
            __('App Dashboard', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk-app',
            [$this, 'render_cloud_page']
        );
        
        add_submenu_page(
            null,
            __('Reports Archive', 'boonrisk-site-security-check-report'),
            __('Reports Archive', 'boonrisk-site-security-check-report'),
            'manage_options',
            'boonrisk-reports',
            [$this, 'render_reports_page']
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('boonrisk_settings', 'boonrisk_api_key', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'show_in_rest' => false,
        ]);
        
        register_setting('boonrisk_settings', 'boonrisk_site_id', [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'show_in_rest' => false,
        ]);
        
        register_setting('boonrisk_settings', 'boonrisk_app_consent', [
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean',
            'show_in_rest' => false,
        ]);
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'boonrisk') === false) {
            return;
        }
        
        wp_enqueue_style(
            'boonrisk-admin',
            BOONRISK_PLUGIN_URL . 'admin/css/admin.css',
            [],
            BOONRISK_VERSION
        );
        
        wp_enqueue_script(
            'boonrisk-admin',
            BOONRISK_PLUGIN_URL . 'admin/js/admin.js',
            ['jquery'],
            BOONRISK_VERSION,
            true
        );
        
        // Get auto-sync status for JS
        $auto_sync_enabled = $this->auto_sync ? $this->auto_sync->is_enabled() : false;
        
        wp_localize_script('boonrisk-admin', 'boonriskAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('boonrisk_nonce'),
            'hasApiKey' => !empty(get_option('boonrisk_api_key', '')),
            'autoSyncEnabled' => $auto_sync_enabled,
            'strings' => [
                'running' => __('Running assessment...', 'boonrisk-site-security-check-report'),
                'complete' => __('Assessment complete', 'boonrisk-site-security-check-report'),
                'error' => __('An error occurred', 'boonrisk-site-security-check-report'),
                'confirmRemoveKey' => __('Are you sure you want to remove the API key? Web dashboard features will be disabled.', 'boonrisk-site-security-check-report'),
                'enableAutoSync' => __('Enabling automatic updates...', 'boonrisk-site-security-check-report'),
                'disableAutoSync' => __('Disabling automatic updates...', 'boonrisk-site-security-check-report'),
            ],
        ]);
    }
    
    /**
     * Render page methods - delegate to admin pages class
     */
    public function render_dashboard_page() {
        BoonRisk_Admin_Pages::render_dashboard();
    }
    
    public function render_local_page() {
        BoonRisk_Admin_Pages::render_local($this->local_assessment);
    }
    
    public function render_local_report_page() {
        BoonRisk_Admin_Pages::render_local_report();
    }
    
    public function render_connect_page() {
        BoonRisk_Admin_Pages::render_connect();
    }
    
    public function render_sync_page() {
        BoonRisk_Admin_Pages::render_sync($this->auto_sync);
    }
    
    public function render_cloud_page() {
        // Redirect old route to new sync page
        wp_safe_redirect(admin_url('admin.php?page=boonrisk-sync'));
        exit;
    }
    
    public function render_reports_page() {
        BoonRisk_Admin_Pages::render_reports();
    }
    
    public function render_settings_page() {
        BoonRisk_Admin_Pages::render_settings();
    }
    
    /**
     * Add plugin action links
     */
    public function add_action_links($links) {
        $plugin_links = [
            '<a href="' . admin_url('admin.php?page=boonrisk-local') . '">' . __('Run Assessment', 'boonrisk-site-security-check-report') . '</a>',
            '<a href="' . admin_url('admin.php?page=boonrisk-settings') . '">' . __('Settings', 'boonrisk-site-security-check-report') . '</a>',
        ];
        return array_merge($plugin_links, $links);
    }
    
    /**
     * AJAX: Run local assessment (no external calls)
     */
    public function ajax_local_assessment() {
        try {
            check_ajax_referer('boonrisk_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            $results = $this->local_assessment->run();
            
            if (empty($results)) {
                wp_send_json_error(['message' => __('Assessment failed to generate results', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            update_option('boonrisk_last_local_assessment', [
                'timestamp' => current_time('timestamp'),
                'results' => $results,
            ]);
            
            wp_send_json_success([
                'message' => __('Local assessment complete', 'boonrisk-site-security-check-report'),
                'results' => $results,
            ]);
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error: %s', 'boonrisk-site-security-check-report'),
                    $e->getMessage()
                ),
            ]);
        }
    }
    
    /**
     * AJAX: Test connection (explicit user action only)
     */
    public function ajax_test_connection() {
        try {
            check_ajax_referer('boonrisk_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            $api_key = get_option('boonrisk_api_key', '');
            if (empty($api_key)) {
                wp_send_json_error(['message' => __('API key not configured', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            $result = $this->api_client->test_connection();
            
            if ($result['success']) {
                wp_send_json_success(['message' => __('Connected successfully', 'boonrisk-site-security-check-report')]);
            } else {
                wp_send_json_error(['message' => $result['message'] ?? __('Connection failed', 'boonrisk-site-security-check-report')]);
            }
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error: %s', 'boonrisk-site-security-check-report'),
                    $e->getMessage()
                ),
            ]);
        }
    }
    
    /**
     * AJAX: Run app analysis (explicit user action + consent required)
     */
    public function ajax_app_analysis() {
        try {
            check_ajax_referer('boonrisk_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            // Verify consent checkbox was checked
            if (empty($_POST['consent'])) {
                wp_send_json_error(['message' => __('Please confirm data sharing consent', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            $api_key = get_option('boonrisk_api_key', '');
            if (empty($api_key)) {
                wp_send_json_error(['message' => __('API key required for app analysis', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            // Check if auto_sync is available
            if (!$this->auto_sync) {
                wp_send_json_error(['message' => __('Plugin not properly initialized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            // Use auto_sync->run_manual_sync() which handles everything including history logging
            $result = $this->auto_sync->run_manual_sync();
            
            if ($result['success']) {
                wp_send_json_success([
                    'message' => $result['message'],
                    'data' => $result['data'] ?? null,
                ]);
            } else {
                wp_send_json_error([
                    'message' => $result['message'] ?? __('Failed to send to dashboard', 'boonrisk-site-security-check-report'),
                ]);
            }
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error: %s', 'boonrisk-site-security-check-report'),
                    $e->getMessage()
                ),
            ]);
        }
    }
    
    /**
     * AJAX: Save API key
     */
    public function ajax_save_api_key() {
        check_ajax_referer('boonrisk_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
        }
        
        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        
        if (empty($api_key)) {
            wp_send_json_error(['message' => __('API key cannot be empty', 'boonrisk-site-security-check-report')]);
        }
        
        update_option('boonrisk_api_key', $api_key);
        
        wp_send_json_success(['message' => __('API key saved', 'boonrisk-site-security-check-report')]);
    }
    
    /**
     * AJAX: Remove API key and reset ALL plugin data
     * This ensures "Reset" truly returns the plugin to fresh install state
     */
    public function ajax_remove_api_key() {
        check_ajax_referer('boonrisk_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
        }
        
        // Delete ALL BoonRisk options to ensure complete reset
        delete_option('boonrisk_api_key');
        delete_option('boonrisk_app_consent');
        delete_option('boonrisk_last_local_assessment');
        delete_option('boonrisk_last_app_scan_id');
        delete_option('boonrisk_last_app_scan_time');
        delete_option('boonrisk_last_assessment_hash');
        delete_option('boonrisk_last_auto_sync');
        delete_option('boonrisk_auto_sync_enabled');
        delete_option('boonrisk_auto_sync_consent');
        delete_option('boonrisk_sync_history');
        
        wp_send_json_success(['message' => __('All data reset successfully', 'boonrisk-site-security-check-report')]);
    }
    
    /**
     * Check if API key is configured
     */
    public static function has_api_key() {
        return !empty(get_option('boonrisk_api_key', ''));
    }
    
    /**
     * Get auto-sync instance (for admin pages)
     */
    public function get_auto_sync() {
        return $this->auto_sync;
    }
    
    /**
     * AJAX: Toggle auto-sync (requires explicit consent)
     */
    public function ajax_toggle_auto_sync() {
        try {
            check_ajax_referer('boonrisk_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            $enable = !empty($_POST['enable']);
            $consent = !empty($_POST['consent']);
            
            if ($enable) {
                // Enabling requires consent checkbox
                if (!$consent) {
                    wp_send_json_error(['message' => __('Please confirm consent to enable automatic sync', 'boonrisk-site-security-check-report')]);
                    return;
                }
                
                if (!$this->api_client->has_api_key()) {
                    wp_send_json_error(['message' => __('API key required for automatic sync', 'boonrisk-site-security-check-report')]);
                    return;
                }
                
                $result = $this->auto_sync->enable(true);
                
                if ($result) {
                    wp_send_json_success([
                        'message' => __('Automatic daily sync enabled', 'boonrisk-site-security-check-report'),
                        'status' => $this->auto_sync->get_status(),
                    ]);
                } else {
                    wp_send_json_error(['message' => __('Failed to enable auto-sync', 'boonrisk-site-security-check-report')]);
                }
            } else {
                $this->auto_sync->disable();
                wp_send_json_success([
                    'message' => __('Automatic sync disabled', 'boonrisk-site-security-check-report'),
                    'status' => $this->auto_sync->get_status(),
                ]);
            }
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error: %s', 'boonrisk-site-security-check-report'),
                    $e->getMessage()
                ),
            ]);
        }
    }
    
    /**
     * AJAX: Manual sync now (sends immediately)
     */
    public function ajax_manual_sync() {
        try {
            check_ajax_referer('boonrisk_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Unauthorized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            // Verify consent checkbox was checked
            if (empty($_POST['consent'])) {
                wp_send_json_error(['message' => __('Please confirm data sharing consent', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            // Check if auto_sync is initialized
            if (!$this->auto_sync) {
                wp_send_json_error(['message' => __('Plugin not properly initialized', 'boonrisk-site-security-check-report')]);
                return;
            }
            
            $result = $this->auto_sync->run_manual_sync();
            
            if ($result['success']) {
                wp_send_json_success([
                    'message' => $result['message'],
                    'status' => $this->auto_sync->get_status(),
                    'history' => $this->auto_sync->get_history(),
                ]);
            } else {
                wp_send_json_error([
                    'message' => $result['message'],
                    'history' => $this->auto_sync->get_history(),
                ]);
            }
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error: %s', 'boonrisk-site-security-check-report'),
                    $e->getMessage()
                ),
            ]);
        }
    }
}

// Initialize plugin
add_action('plugins_loaded', function() {
    BoonRisk_Agent::get_instance();
});

// Deactivation hook - cleanup scheduled events
register_deactivation_hook(__FILE__, function() {
    BoonRisk_Auto_Sync::deactivate();
});
