/**
 * BoonRisk Admin JavaScript
 * Handles all user interactions for the security posture plugin.
 * All external API calls require explicit user action.
 */

(function($) {
    'use strict';

    const BoonRisk = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            // Local Assessment
            $('#run-local-assessment').on('click', this.runLocalAssessment.bind(this));
            
            // Connect Page
            $('#api-key-form').on('submit', this.saveApiKey.bind(this));
            $('#toggle-api-key-visibility').on('click', this.toggleApiKeyVisibility);
            $('#test-connection').on('click', this.testConnection.bind(this));
            $('#remove-api-key').on('click', this.removeApiKey.bind(this));
            
            // App Analysis
            $('#app-analysis-form').on('submit', this.runAppAnalysis.bind(this));
            
            // Auto-Sync (on App Dashboard page)
            $('#enable-auto-sync').on('click', this.enableAutoSync.bind(this));
            $('#disable-auto-sync').on('click', this.disableAutoSync.bind(this));
            
            // Auto-Sync (on Settings page)
            $('#settings-enable-auto-sync').on('click', this.enableAutoSyncSettings.bind(this));
            $('#settings-disable-auto-sync').on('click', this.disableAutoSyncSettings.bind(this));
            
            // Reports
            $('#generate-report-form').on('submit', this.generateReport.bind(this));
            
            // Settings
            $('#reset-plugin-data').on('click', this.resetPluginData.bind(this));
            
            // Close modal on escape
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    $('.boonrisk-modal').hide();
                }
            });
        },

        /**
         * Run local assessment (no external calls)
         */
        runLocalAssessment: function(e) {
            e.preventDefault();
            
            const $button = $('#run-local-assessment');
            const $status = $('#assessment-status');
            const $results = $('#assessment-results');
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading').text(boonriskAdmin.strings.running).show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_local_assessment',
                    nonce: boonriskAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text(boonriskAdmin.strings.complete);
                        
                        // Reload page to show new results
                        location.reload();
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text(response.data.message || boonriskAdmin.strings.error);
                    }
                },
                error: function() {
                    $status.removeClass('loading').addClass('error')
                        .text(boonriskAdmin.strings.error);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Save API key
         */
        saveApiKey: function(e) {
            e.preventDefault();
            
            const $form = $('#api-key-form');
            const $button = $('#save-api-key');
            const $status = $('#api-key-status');
            const apiKey = $('#boonrisk_api_key').val().trim();
            
            if (!apiKey) {
                $status.removeClass('success loading').addClass('error')
                    .text('API key cannot be empty').show();
                return;
            }
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading').text('Saving...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_save_api_key',
                    nonce: boonriskAdmin.nonce,
                    api_key: apiKey
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('API key saved. Testing connection...');
                        
                        // Test connection after saving
                        BoonRisk.testConnectionAfterSave();
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text(response.data.message || 'Failed to save API key');
                    }
                },
                error: function() {
                    $status.removeClass('loading').addClass('error')
                        .text('Error saving API key');
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Test connection after saving key
         */
        testConnectionAfterSave: function() {
            const $status = $('#api-key-status');
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_test_connection',
                    nonce: boonriskAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('Connected successfully! Reloading...');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('Key saved but connection test failed: ' + (response.data.message || 'Unknown error'));
                    }
                },
                error: function() {
                    $status.removeClass('loading').addClass('error')
                        .text('Key saved but connection test failed');
                }
            });
        },

        /**
         * Toggle API key visibility
         */
        toggleApiKeyVisibility: function() {
            const $input = $('#boonrisk_api_key');
            const $icon = $(this).find('.dashicons');
            
            if ($input.attr('type') === 'password') {
                $input.attr('type', 'text');
                $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                $input.attr('type', 'password');
                $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        },

        /**
         * Test connection (explicit user action)
         */
        testConnection: function(e) {
            e.preventDefault();
            
            const $button = $('#test-connection');
            const $status = $('#connection-status');
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading').text('Testing connection...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_test_connection',
                    nonce: boonriskAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('✓ ' + response.data.message);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('✗ ' + (response.data.message || 'Connection failed'));
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Connection error';
                    
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (error) {
                        errorMessage = error;
                    }
                    
                    $status.removeClass('loading').addClass('error')
                        .text('✗ ' + errorMessage);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Remove API key
         */
        removeApiKey: function(e) {
            e.preventDefault();
            
            if (!confirm(boonriskAdmin.strings.confirmRemoveKey)) {
                return;
            }
            
            const $button = $('#remove-api-key');
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_remove_api_key',
                    nonce: boonriskAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.data.message || 'Failed to remove API key');
                    }
                },
                error: function() {
                    alert('Error removing API key');
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Run app analysis (explicit user action + consent)
         */
        runAppAnalysis: function(e) {
            e.preventDefault();
            
            const $form = $('#app-analysis-form');
            const $button = $('#run-app-analysis');
            const $status = $('#app-analysis-status');
            const $consent = $('#app-consent');
            
            if (!$consent.is(':checked')) {
                $status.removeClass('success loading').addClass('error')
                    .text('Please confirm data sharing consent').show();
                return;
            }
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading')
                .text('Sending to dashboard...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_app_analysis',
                    nonce: boonriskAdmin.nonce,
                    consent: 1
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('✓ ' + response.data.message);
                        
                        // Reload to show results
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('✗ ' + (response.data.message || 'Failed to send'));
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Error sending to dashboard';
                    
                    // Try to extract error message from response
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (xhr.statusText && xhr.statusText !== 'error') {
                        errorMessage = xhr.statusText;
                    } else if (error) {
                        errorMessage = error;
                    }
                    
                    $status.removeClass('loading').addClass('error')
                        .text('✗ ' + errorMessage);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Enable auto-sync (requires consent)
         */
        enableAutoSync: function(e) {
            e.preventDefault();
            
            const $button = $('#enable-auto-sync');
            const $status = $('#auto-sync-status');
            const $consent = $('#auto-sync-consent');
            
            if (!$consent.is(':checked')) {
                $status.removeClass('success loading').addClass('error')
                    .text('Please confirm consent to enable automatic updates').show();
                return;
            }
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading')
                .text('Enabling automatic updates...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_toggle_auto_sync',
                    nonce: boonriskAdmin.nonce,
                    enable: 1,
                    consent: 1
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('✓ ' + response.data.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('✗ ' + (response.data.message || 'Failed to enable'));
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Error enabling automatic updates';
                    
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (error) {
                        errorMessage = error;
                    }
                    
                    $status.removeClass('loading').addClass('error')
                        .text('✗ ' + errorMessage);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Disable auto-sync
         */
        disableAutoSync: function(e) {
            e.preventDefault();
            
            const $button = $('#disable-auto-sync');
            const $status = $('#auto-sync-status');
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading')
                .text('Disabling automatic updates...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_toggle_auto_sync',
                    nonce: boonriskAdmin.nonce,
                    enable: 0
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('✓ ' + response.data.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('✗ ' + (response.data.message || 'Failed to disable'));
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Error disabling automatic updates';
                    
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (error) {
                        errorMessage = error;
                    }
                    
                    $status.removeClass('loading').addClass('error')
                        .text('✗ ' + errorMessage);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },
        
        /**
         * Enable auto-sync from Settings page
         */
        enableAutoSyncSettings: function(e) {
            e.preventDefault();
            
            const $button = $('#settings-enable-auto-sync');
            const $status = $('#settings-auto-sync-status');
            const $consent = $('#settings-auto-sync-consent');
            
            if (!$consent.is(':checked')) {
                $status.removeClass('success loading').addClass('error')
                    .text('Please confirm consent to enable automatic sync').show();
                return;
            }
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading')
                .text('Enabling automatic sync...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_toggle_auto_sync',
                    nonce: boonriskAdmin.nonce,
                    enable: 1,
                    consent: 1
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('✓ ' + response.data.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('✗ ' + (response.data.message || 'Failed to enable'));
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Error enabling automatic sync';
                    
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (error) {
                        errorMessage = error;
                    }
                    
                    $status.removeClass('loading').addClass('error')
                        .text('✗ ' + errorMessage);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },
        
        /**
         * Disable auto-sync from Settings page
         */
        disableAutoSyncSettings: function(e) {
            e.preventDefault();
            
            const $button = $('#settings-disable-auto-sync');
            const $status = $('#settings-auto-sync-status');
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            $status.removeClass('success error').addClass('loading')
                .text('Disabling automatic sync...').show();
            
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_toggle_auto_sync',
                    nonce: boonriskAdmin.nonce,
                    enable: 0
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('loading').addClass('success')
                            .text('✓ ' + response.data.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $status.removeClass('loading').addClass('error')
                            .text('✗ ' + (response.data.message || 'Failed to disable'));
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Error disabling automatic sync';
                    
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (error) {
                        errorMessage = error;
                    }
                    
                    $status.removeClass('loading').addClass('error')
                        .text('✗ ' + errorMessage);
                },
                complete: function() {
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Generate report (explicit user action + consent)
         */
        generateReport: function(e) {
            e.preventDefault();
            
            const $form = $('#generate-report-form');
            const $consent = $('#report-consent');
            
            if (!$consent.is(':checked')) {
                alert('Please confirm data sharing consent');
                return;
            }
            
            const reportType = $form.find('input[name="report_type"]:checked').val();
            
            // For now, redirect to web dashboard for report generation
            const dashboardUrl = 'https://app.boonrisk.io/reports/generate?type=' + reportType;
            window.open(dashboardUrl, '_blank');
        },

        /**
         * Reset plugin data
         */
        resetPluginData: function(e) {
            e.preventDefault();
            
            if (!confirm('Are you sure you want to reset all BoonRisk data? This will remove your API key, all stored assessments, and return the plugin to fresh install state.')) {
                return;
            }
            
            const $button = $('#reset-plugin-data');
            
            $button.prop('disabled', true).addClass('boonrisk-loading');
            
            // Remove API key first
            $.ajax({
                url: boonriskAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'boonrisk_remove_api_key',
                    nonce: boonriskAdmin.nonce
                },
                success: function() {
                    location.reload();
                },
                error: function() {
                    alert('Error resetting data');
                    $button.prop('disabled', false).removeClass('boonrisk-loading');
                }
            });
        },

        /**
         * Show notice message
         */
        showNotice: function(message, type) {
            const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
            const $notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
            
            $('.boonrisk-admin h1').after($notice);
            
            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };

    $(document).ready(function() {
        BoonRisk.init();
    });

})(jQuery);
