<?php
/**
 * Admin Pages Renderer
 * 
 * Handles rendering of all admin pages with proper separation:
 * - Local Assessment: Works without API key
 * - Web Dashboard features: Require API key and explicit user consent
 */

if (!defined('ABSPATH')) {
    exit;
}

class BoonRisk_Admin_Pages {
    
    /**
     * Render Dashboard page (main landing page)
     * Consolidated: Overview + Quick Access to Assessment + Web Dashboard Link
     */
    public static function render_dashboard() {
        $has_api_key = BoonRisk_Agent::has_api_key();
        $last_local = get_option('boonrisk_last_local_assessment', null);
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-shield-alt"></span>
                <?php esc_html_e('WordPress Security Check', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <!-- What This Plugin Does block -->
            <div class="boonrisk-info-banner">
                <div class="boonrisk-info-banner-content">
                    <div class="boonrisk-info-col">
                        <strong><?php esc_html_e('What this plugin does', 'boonrisk-site-security-check-report'); ?></strong>
                        <p><?php esc_html_e('Check your WordPress security in minutes. Safe, read-only analysis that identifies risks and recommends fixes in plain language.', 'boonrisk-site-security-check-report'); ?></p>
                    </div>
                    <div class="boonrisk-info-col">
                        <strong><?php esc_html_e('What this plugin does not do', 'boonrisk-site-security-check-report'); ?></strong>
                        <p><?php esc_html_e('It does not block attacks, remove malware, scan files, or modify your site in any way.', 'boonrisk-site-security-check-report'); ?></p>
                    </div>
                </div>
            </div>
            
            <div class="boonrisk-overview">
                <!-- Welcome Card -->
                <div class="boonrisk-card boonrisk-welcome-card">
                    <h2><?php esc_html_e('Security & Readiness Check', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('BoonRisk evaluates your WordPress site across Security, Trust and Readiness domains through assessments, not enforcement. We explain risks and provide recommendations — you decide what to do.', 'boonrisk-site-security-check-report'); ?></p>
                    
                    <div class="boonrisk-local-notice">
                        <span class="dashicons dashicons-info"></span>
                        <?php esc_html_e('This local assessment runs entirely inside WordPress. No account or external service is required.', 'boonrisk-site-security-check-report'); ?>
                    </div>
                    
                    <div class="boonrisk-cta-buttons">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>" class="button button-primary button-hero">
                            <span class="dashicons dashicons-search"></span>
                            <?php esc_html_e('Check Site Security', 'boonrisk-site-security-check-report'); ?>
                        </a>
                        
                        <?php if ($last_local): ?>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-report')); ?>" class="button button-secondary button-hero">
                                <span class="dashicons dashicons-media-document"></span>
                                <?php esc_html_e('View Full Report', 'boonrisk-site-security-check-report'); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Quick Status -->
                <?php if ($last_local): ?>
                    <div class="boonrisk-card">
                        <h2><?php esc_html_e('Last Local Assessment', 'boonrisk-site-security-check-report'); ?></h2>
                        <?php
                        $summary = $last_local['results']['summary'] ?? null;
                        if ($summary):
                        ?>
                            <div class="boonrisk-posture-badge posture-<?php echo esc_attr($summary['posture']); ?>">
                                <?php echo esc_html($summary['posture_label']); ?>
                            </div>
                            <p><?php echo esc_html($summary['posture_explanation'] ?? ''); ?></p>
                            <p class="boonrisk-meta">
                                <?php 
                                printf(
                                    /* translators: %s: Date and time of assessment */
                                    esc_html__('Assessed on %s', 'boonrisk-site-security-check-report'),
                                    esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_local['timestamp']))
                                );
                                ?>
                            </p>
                            <div class="boonrisk-cta-buttons" style="margin-top: 16px;">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>" class="button">
                                <?php esc_html_e('View Details', 'boonrisk-site-security-check-report'); ?>
                            </a>
                                <?php if ($has_api_key): ?>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-sync')); ?>" class="button button-primary">
                                        <span class="dashicons dashicons-upload"></span>
                                        <?php esc_html_e('Send to Dashboard', 'boonrisk-site-security-check-report'); ?>
                                    </a>
                                <?php else: ?>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button">
                                        <span class="dashicons dashicons-admin-links"></span>
                                        <?php esc_html_e('Connect to Dashboard', 'boonrisk-site-security-check-report'); ?>
                                    </a>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Local Features (always available) -->
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('What\'s Included (Free, No Registration)', 'boonrisk-site-security-check-report'); ?></h2>
                    <div class="boonrisk-features-list">
                        <div class="boonrisk-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong><?php esc_html_e('Security Check Summary', 'boonrisk-site-security-check-report'); ?></strong>
                                <p><?php esc_html_e('Clear Low/Medium/High risk level with plain-language explanation.', 'boonrisk-site-security-check-report'); ?></p>
                            </div>
                        </div>
                        <div class="boonrisk-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong><?php esc_html_e('Top Risks (Prioritized)', 'boonrisk-site-security-check-report'); ?></strong>
                                <p><?php esc_html_e('Most impactful findings ranked by potential risk, not quantity.', 'boonrisk-site-security-check-report'); ?></p>
                            </div>
                        </div>
                        <div class="boonrisk-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong><?php esc_html_e('30 Configuration Checks', 'boonrisk-site-security-check-report'); ?></strong>
                                <p><?php esc_html_e('Security, trust, and readiness checks including 2FA, updates, backups, PHP version, HTTPS, and more.', 'boonrisk-site-security-check-report'); ?></p>
                            </div>
                        </div>
                        <div class="boonrisk-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong><?php esc_html_e('Explainable Results', 'boonrisk-site-security-check-report'); ?></strong>
                                <p><?php esc_html_e('Every finding includes "why it matters" and "what to do" guidance.', 'boonrisk-site-security-check-report'); ?></p>
                            </div>
                        </div>
                        <div class="boonrisk-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong><?php esc_html_e('Printable HTML Report', 'boonrisk-site-security-check-report'); ?></strong>
                                <p><?php esc_html_e('Full report viewable and printable directly from WordPress admin.', 'boonrisk-site-security-check-report'); ?></p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php if (!$has_api_key): ?>
                <!-- Optional Web Dashboard -->
                <div class="boonrisk-card boonrisk-optional-card">
                    <h2><?php esc_html_e('Optional: Web Dashboard', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('Connect to the BoonRisk web dashboard for additional features. This is entirely optional — local assessment is fully functional without it.', 'boonrisk-site-security-check-report'); ?></p>
                    <ul class="boonrisk-optional-features">
                        <li><?php esc_html_e('Advanced analysis with vulnerability database', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('PDF report generation', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Historical trend tracking', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button">
                        <?php esc_html_e('Connect to Web Dashboard', 'boonrisk-site-security-check-report'); ?>
                    </a>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Local Assessment page
     */
    public static function render_local($local_assessment) {
        $last_assessment = get_option('boonrisk_last_local_assessment', null);
        $has_api_key = BoonRisk_Agent::has_api_key();
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-search"></span>
                <?php esc_html_e('Local Assessment', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <!-- What This Plugin Does block -->
            <div class="boonrisk-info-banner">
                <div class="boonrisk-info-banner-content">
                    <div class="boonrisk-info-col">
                        <strong><?php esc_html_e('What this plugin does', 'boonrisk-site-security-check-report'); ?></strong>
                        <p><?php esc_html_e('Safe, read-only security check. Identifies configuration risks and recommends actions in plain language.', 'boonrisk-site-security-check-report'); ?></p>
                    </div>
                    <div class="boonrisk-info-col">
                        <strong><?php esc_html_e('What this plugin does not do', 'boonrisk-site-security-check-report'); ?></strong>
                        <p><?php esc_html_e('It does not block attacks, remove malware, scan files, or modify your site in any way.', 'boonrisk-site-security-check-report'); ?></p>
                    </div>
                </div>
            </div>
            
            <div class="boonrisk-local-notice">
                <span class="dashicons dashicons-info"></span>
                <?php esc_html_e('This local assessment runs entirely inside WordPress. No account or external service is required.', 'boonrisk-site-security-check-report'); ?>
            </div>
            
            <div class="boonrisk-local-assessment">
                <div class="boonrisk-card">
                    <div class="boonrisk-action-row">
                        <button type="button" class="button button-primary button-hero" id="run-local-assessment">
                            <span class="dashicons dashicons-update"></span>
                            <?php esc_html_e('Run Assessment Now', 'boonrisk-site-security-check-report'); ?>
                        </button>
                        
                        <?php if ($last_assessment): ?>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-report')); ?>" class="button button-secondary button-hero">
                                <span class="dashicons dashicons-media-document"></span>
                                <?php esc_html_e('View Full Report', 'boonrisk-site-security-check-report'); ?>
                            </a>
                            
                            <?php if (!$has_api_key): ?>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button button-primary button-hero">
                                    <span class="dashicons dashicons-admin-links"></span>
                                    <?php esc_html_e('Connect to Dashboard', 'boonrisk-site-security-check-report'); ?>
                                </a>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                    
                    <div id="assessment-status" class="boonrisk-status-message" style="display: none;"></div>
                </div>
                
                <!-- Results Container -->
                <div id="assessment-results">
                    <?php if ($last_assessment): ?>
                        <?php self::render_assessment_results($last_assessment['results']); ?>
                    <?php else: ?>
                        <div class="boonrisk-card boonrisk-empty-state">
                            <span class="dashicons dashicons-shield-alt"></span>
                            <h3><?php esc_html_e('No assessment yet', 'boonrisk-site-security-check-report'); ?></h3>
                            <p><?php esc_html_e('Click "Check Site Security" to run your first security check.', 'boonrisk-site-security-check-report'); ?></p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render assessment results
     */
    public static function render_assessment_results($results) {
        if (empty($results) || empty($results['checks'])) {
            return;
        }
        
        $summary = $results['summary'];
        $checks = $results['checks'];
        $top_risks = $results['top_risks'] ?? [];
        ?>
        
        <!-- Security Check Summary (Top Priority) -->
        <div class="boonrisk-card boonrisk-posture-summary-card">
            <div class="boonrisk-posture-header">
                <div class="boonrisk-posture-level posture-<?php echo esc_attr($summary['posture']); ?>">
                    <span class="posture-risk-label"><?php esc_html_e('Security Risk Posture', 'boonrisk-site-security-check-report'); ?></span>
                    <span class="posture-level-text"><?php echo esc_html($summary['posture_label']); ?></span>
                </div>
            </div>
            <p class="boonrisk-posture-explanation">
                <?php echo esc_html($summary['posture_explanation']); ?>
            </p>
            <p class="boonrisk-meta">
                <?php 
                printf(
                    /* translators: %s: Date and time of assessment */
                    esc_html__('Assessed on %s', 'boonrisk-site-security-check-report'),
                    esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($results['generated_at'])))
                );
                ?>
            </p>
        </div>
        
        <!-- Findings Overview (Priority #1) -->
        <div class="boonrisk-card boonrisk-stats-card">
            <h2><?php esc_html_e('Findings Overview', 'boonrisk-site-security-check-report'); ?></h2>
            <div class="boonrisk-stats-row">
                <div class="stat stat-high">
                    <span class="stat-value"><?php echo esc_html($summary['high_risk_count']); ?></span>
                    <span class="stat-label"><?php esc_html_e('High Risk', 'boonrisk-site-security-check-report'); ?></span>
                </div>
                <div class="stat stat-medium">
                    <span class="stat-value"><?php echo esc_html($summary['warning_count']); ?></span>
                    <span class="stat-label"><?php esc_html_e('Medium Risk', 'boonrisk-site-security-check-report'); ?></span>
                </div>
                <div class="stat stat-pass">
                    <span class="stat-value"><?php echo esc_html($summary['passed_count']); ?></span>
                    <span class="stat-label"><?php esc_html_e('Passed', 'boonrisk-site-security-check-report'); ?></span>
                </div>
                <div class="stat stat-total">
                    <span class="stat-value"><?php echo esc_html($summary['total_count']); ?></span>
                    <span class="stat-label"><?php esc_html_e('Total Checks', 'boonrisk-site-security-check-report'); ?></span>
                </div>
            </div>
        </div>
        
        <!-- Top Risks Section (Prioritized) -->
        <?php if (!empty($top_risks)): ?>
        <div class="boonrisk-card boonrisk-top-risks-card">
            <h2><?php esc_html_e('Top Risks', 'boonrisk-site-security-check-report'); ?></h2>
            <p class="boonrisk-section-description"><?php esc_html_e('These are the most impactful findings from your assessment, prioritized by potential risk.', 'boonrisk-site-security-check-report'); ?></p>
            
            <div class="boonrisk-top-risks-list">
                <?php foreach ($top_risks as $index => $risk): ?>
                    <div class="boonrisk-risk-item risk-<?php echo esc_attr($risk['status']); ?>">
                        <div class="risk-header">
                            <span class="risk-number"><?php echo esc_html($index + 1); ?></span>
                            <span class="risk-title"><?php echo esc_html($risk['name']); ?></span>
                            <span class="boonrisk-status-badge status-<?php echo esc_attr($risk['status']); ?>">
                                <?php echo $risk['status'] === 'high_risk' ? esc_html__('High', 'boonrisk-site-security-check-report') : esc_html__('Medium', 'boonrisk-site-security-check-report'); ?>
                            </span>
                        </div>
                        <div class="risk-body">
                            <p class="risk-finding"><?php echo esc_html($risk['message']); ?></p>
                            <div class="risk-details">
                                <div class="risk-why">
                                    <strong><?php esc_html_e('Why it matters:', 'boonrisk-site-security-check-report'); ?></strong>
                                    <p><?php echo esc_html($risk['why_it_matters']); ?></p>
                                </div>
                                <?php if (!empty($risk['recommendation'])): ?>
                                <div class="risk-action">
                                    <strong><?php esc_html_e('What to do:', 'boonrisk-site-security-check-report'); ?></strong>
                                    <p><?php echo esc_html($risk['recommendation']); ?></p>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- All Findings (Detailed) -->
        <div class="boonrisk-card">
            <h2><?php esc_html_e('All Findings', 'boonrisk-site-security-check-report'); ?></h2>
            
            <div class="boonrisk-findings-list">
                <?php foreach ($checks as $check): ?>
                    <div class="boonrisk-finding-item finding-<?php echo esc_attr($check['status']); ?>">
                        <div class="finding-header">
                            <span class="finding-name"><?php echo esc_html($check['name']); ?></span>
                            <span class="boonrisk-status-badge status-<?php echo esc_attr($check['status']); ?>">
                                <?php 
                                $status_labels = [
                                    'pass' => __('Pass', 'boonrisk-site-security-check-report'),
                                    'warning' => __('Medium', 'boonrisk-site-security-check-report'),
                                    'high_risk' => __('High', 'boonrisk-site-security-check-report'),
                                ];
                                echo esc_html($status_labels[$check['status']] ?? $check['status']);
                                ?>
                            </span>
                        </div>
                        <div class="finding-value">
                            <code><?php echo esc_html($check['value']); ?></code>
                        </div>
                        <div class="finding-details">
                            <p class="finding-message"><?php echo esc_html($check['message']); ?></p>
                            
                            <?php if ($check['status'] !== 'pass'): ?>
                                <div class="finding-explainer">
                                    <div class="explainer-item">
                                        <strong><?php esc_html_e('Why this matters:', 'boonrisk-site-security-check-report'); ?></strong>
                                        <p><?php echo esc_html($check['why_it_matters']); ?></p>
                                    </div>
                                    <?php if (!empty($check['recommendation'])): ?>
                                    <div class="explainer-item">
                                        <strong><?php esc_html_e('What you should do:', 'boonrisk-site-security-check-report'); ?></strong>
                                        <p><?php echo esc_html($check['recommendation']); ?></p>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($check['cloud_enhancement']) && !BoonRisk_Agent::has_api_key()): ?>
                                    <div class="explainer-item explainer-cloud-note">
                                        <p class="cloud-note-text"><?php echo esc_html($check['cloud_enhancement']); ?></p>
                                        <p class="cloud-note-local"><?php esc_html_e('Local assessment focuses on configuration and exposure risks.', 'boonrisk-site-security-check-report'); ?></p>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Connect (API Key) page
     */
    public static function render_connect() {
        $has_api_key = BoonRisk_Agent::has_api_key();
        $last_local = get_option('boonrisk_last_local_assessment', null);
        $has_local_assessment = !empty($last_local);
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-cloud"></span>
                <?php esc_html_e('Connect to Web Dashboard', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <?php if ($has_local_assessment): ?>
                <div class="boonrisk-success-card boonrisk-card">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <div>
                        <p><strong><?php esc_html_e('Local Assessment Complete', 'boonrisk-site-security-check-report'); ?></strong></p>
                        <p class="boonrisk-meta">
                            <?php esc_html_e('Connect with API key to sync your results to the web dashboard.', 'boonrisk-site-security-check-report'); ?>
                        </p>
                    </div>
                </div>
            <?php else: ?>
            <div class="boonrisk-local-notice">
                <span class="dashicons dashicons-info"></span>
                    <?php esc_html_e('Tip: Run a local assessment first, then connect to sync your results.', 'boonrisk-site-security-check-report'); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>"><?php esc_html_e('Run Assessment', 'boonrisk-site-security-check-report'); ?></a>
            </div>
            <?php endif; ?>
            
            <div class="boonrisk-connect">
                <div class="boonrisk-card">
                    <?php if ($has_api_key): ?>
                        <!-- Connected state -->
                        <div class="boonrisk-connected-status">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <span><?php esc_html_e('Connected', 'boonrisk-site-security-check-report'); ?></span>
                        </div>
                        
                        <p><?php esc_html_e('Your site is connected to the BoonRisk web dashboard.', 'boonrisk-site-security-check-report'); ?></p>
                        
                        <div class="boonrisk-actions">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-sync')); ?>" class="button button-primary">
                                <span class="dashicons dashicons-upload"></span>
                                <?php esc_html_e('Send to Dashboard', 'boonrisk-site-security-check-report'); ?>
                            </a>
                            <button type="button" class="button" id="test-connection">
                                <?php esc_html_e('Test Connection', 'boonrisk-site-security-check-report'); ?>
                            </button>
                            <button type="button" class="button button-link-delete" id="remove-api-key">
                                <?php esc_html_e('Remove API Key', 'boonrisk-site-security-check-report'); ?>
                            </button>
                        </div>
                        
                        <div id="connection-status" class="boonrisk-status-message" style="display: none;"></div>
                        
                    <?php else: ?>
                        <!-- Not connected state -->
                        <h2><?php esc_html_e('Enter API Key', 'boonrisk-site-security-check-report'); ?></h2>
                        
                        <div class="boonrisk-connect-explanation">
                            <p><?php esc_html_e('Connecting to the web dashboard enables extended analysis using external vulnerability intelligence and historical insights.', 'boonrisk-site-security-check-report'); ?></p>
                            <p><?php esc_html_e('This is optional. The local security check works perfectly without connecting.', 'boonrisk-site-security-check-report'); ?></p>
                        </div>
                        
                        <div class="boonrisk-benefits">
                            <h3><?php esc_html_e('Optional enhancements:', 'boonrisk-site-security-check-report'); ?></h3>
                            <ul>
                                <li><?php esc_html_e('Vulnerability intelligence from external sources', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('PDF reports for stakeholders', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Historical trend tracking', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Web dashboard access', 'boonrisk-site-security-check-report'); ?></li>
                            </ul>
                        </div>
                        
                        <div class="boonrisk-privacy-note">
                            <p><strong><?php esc_html_e('Privacy:', 'boonrisk-site-security-check-report'); ?></strong> <?php esc_html_e('No personal data is collected. Site configuration data is sent only when you explicitly send to dashboard.', 'boonrisk-site-security-check-report'); ?></p>
                        </div>
                        
                        <form id="api-key-form" class="boonrisk-api-key-form">
                            <div class="form-field">
                                <label for="boonrisk_api_key"><?php esc_html_e('API Key', 'boonrisk-site-security-check-report'); ?></label>
                                <div class="input-group">
                                    <input type="password" 
                                           id="boonrisk_api_key" 
                                           name="api_key" 
                                           class="regular-text"
                                           autocomplete="new-password"
                                           placeholder="<?php esc_attr_e('Paste your API key here', 'boonrisk-site-security-check-report'); ?>">
                                    <button type="button" class="button" id="toggle-api-key-visibility">
                                        <span class="dashicons dashicons-visibility"></span>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="boonrisk-actions">
                                <button type="submit" class="button button-primary" id="save-api-key">
                                    <?php esc_html_e('Save Key', 'boonrisk-site-security-check-report'); ?>
                                </button>
                            </div>
                            
                            <div id="api-key-status" class="boonrisk-status-message" style="display: none;"></div>
                        </form>
                        
                        <p class="boonrisk-help-link">
                            <a href="https://app.boonrisk.com/api-keys" target="_blank" rel="noopener">
                                <?php esc_html_e('Get your API key from the dashboard', 'boonrisk-site-security-check-report'); ?>
                                <span class="dashicons dashicons-external"></span>
                            </a>
                        </p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render App Dashboard page (legacy)
     */
    public static function render_cloud() {
        $has_api_key = BoonRisk_Agent::has_api_key();
        $last_local = get_option('boonrisk_last_local_assessment', null);
        
        // Check auto-sync status
        $auto_sync = null;
        $sync_status = null;
        $auto_sync_enabled = false;
        
        if ($has_api_key) {
            $auto_sync = BoonRisk_Agent::get_instance()->get_auto_sync();
            $sync_status = $auto_sync->get_status();
            $auto_sync_enabled = $sync_status['enabled'];
        }
        
        // If auto-sync is enabled, consider assessment available (cron runs fresh assessments)
        $has_local_assessment = !empty($last_local) || $auto_sync_enabled;
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-desktop"></span>
                <?php esc_html_e('Web Dashboard', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <!-- Step Progress Indicator -->
            <div class="boonrisk-steps-progress">
                <div class="boonrisk-step <?php echo $has_local_assessment ? 'step-complete' : 'step-active'; ?>">
                    <span class="step-number"><?php echo $has_local_assessment ? '✓' : '1'; ?></span>
                    <span class="step-label"><?php esc_html_e('Local Assessment', 'boonrisk-site-security-check-report'); ?></span>
                </div>
                <div class="boonrisk-step-connector <?php echo $has_local_assessment ? 'connector-active' : ''; ?>"></div>
                <div class="boonrisk-step <?php echo ($has_local_assessment && $has_api_key) ? 'step-active' : ($has_local_assessment ? 'step-next' : 'step-locked'); ?>">
                    <span class="step-number"><?php echo ($has_local_assessment && $has_api_key && $last_analysis) ? '✓' : '2'; ?></span>
                    <span class="step-label"><?php esc_html_e('Send to Dashboard', 'boonrisk-site-security-check-report'); ?></span>
                </div>
            </div>
            
            <?php if (!$has_local_assessment): ?>
                <!-- Step 1: Local Assessment Required -->
                <div class="boonrisk-card boonrisk-requirement-card">
                    <div class="boonrisk-requirement-icon">
                        <span class="dashicons dashicons-search"></span>
                    </div>
                    <h2><?php esc_html_e('Check Site Security First', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('Before sending to the dashboard, you need to run a local assessment or enable automatic sync below.', 'boonrisk-site-security-check-report'); ?></p>
                    
                    <div class="boonrisk-requirement-benefits">
                        <p><strong><?php esc_html_e('Local Assessment gives you:', 'boonrisk-site-security-check-report'); ?></strong></p>
                        <ul>
                            <li><?php esc_html_e('Instant security check and risk overview', 'boonrisk-site-security-check-report'); ?></li>
                            <li><?php esc_html_e('30 security & readiness checks', 'boonrisk-site-security-check-report'); ?></li>
                            <li><?php esc_html_e('Prioritized risk findings', 'boonrisk-site-security-check-report'); ?></li>
                            <li><?php esc_html_e('Printable HTML report', 'boonrisk-site-security-check-report'); ?></li>
                        </ul>
                    </div>
                    
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>" class="button button-primary button-hero">
                        <span class="dashicons dashicons-search"></span>
                        <?php esc_html_e('Run Local Assessment', 'boonrisk-site-security-check-report'); ?>
                    </a>
                    
                    <p class="boonrisk-note">
                        <span class="dashicons dashicons-info"></span>
                        <?php esc_html_e('Tip: Enable automatic sync below to monitor your site continuously without manual assessments.', 'boonrisk-site-security-check-report'); ?>
                    </p>
                </div>
                
            <?php elseif ($auto_sync_enabled && empty($last_local)): ?>
                <!-- Monitoring Mode Active (auto-sync enabled, no manual assessment yet) -->
                <div class="boonrisk-card boonrisk-success-card">
                    <div class="boonrisk-monitoring-status">
                        <span class="dashicons dashicons-update-alt" style="color: #2271b1; font-size: 48px; display: block; margin-bottom: 16px;"></span>
                        <h2 style="margin-top: 0;"><?php esc_html_e('Monitoring Mode Active', 'boonrisk-site-security-check-report'); ?></h2>
                        <p><?php esc_html_e('Your site is being monitored automatically. The plugin will run fresh security assessments and send data once per day.', 'boonrisk-site-security-check-report'); ?></p>
                        
                        <div class="boonrisk-monitoring-info" style="background: #f0f0f1; padding: 16px; border-radius: 4px; margin-top: 16px;">
                            <p style="margin: 0 0 8px 0;">
                                <strong><?php esc_html_e('Next Scheduled Sync:', 'boonrisk-site-security-check-report'); ?></strong>
                                <?php 
                                printf(
                                    esc_html__('In %s', 'boonrisk-site-security-check-report'), 
                                    esc_html($sync_status['next_scheduled_formatted'])
                                ); 
                                ?>
                            </p>
                            <?php if ($sync_status['last_sync']): ?>
                            <p style="margin: 0;">
                                <strong><?php esc_html_e('Last Sync:', 'boonrisk-site-security-check-report'); ?></strong>
                                <?php echo esc_html($sync_status['last_sync_formatted']); ?>
                            </p>
                            <?php else: ?>
                            <p style="margin: 0;">
                                <strong><?php esc_html_e('Status:', 'boonrisk-site-security-check-report'); ?></strong>
                                <?php esc_html_e('Waiting for first scheduled sync', 'boonrisk-site-security-check-report'); ?>
                            </p>
                            <?php endif; ?>
                        </div>
                        
                        <p class="boonrisk-note" style="margin-top: 16px;">
                            <span class="dashicons dashicons-info"></span>
                            <?php esc_html_e('Want to see results now? Run a manual assessment or send data immediately using the buttons below.', 'boonrisk-site-security-check-report'); ?>
                        </p>
                    </div>
                </div>
                
            <?php elseif (!$has_api_key): ?>
                <!-- Step 1 Complete, Step 2: Need API Key -->
                <div class="boonrisk-card boonrisk-success-card">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <p><strong><?php esc_html_e('Local Assessment Complete', 'boonrisk-site-security-check-report'); ?></strong></p>
                    <p class="boonrisk-meta">
                        <?php 
                        printf(
                            /* translators: %s: Date and time of assessment */
                            esc_html__('Last run: %s', 'boonrisk-site-security-check-report'),
                            esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_local['timestamp']))
                        );
                        ?>
                    </p>
                </div>
                
                <div class="boonrisk-card boonrisk-requirement-card">
                    <div class="boonrisk-requirement-icon">
                        <span class="dashicons dashicons-admin-network"></span>
                    </div>
                    <h2><?php esc_html_e('Connect to Web Dashboard', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('To send your assessment to the dashboard, you need to connect with an API key.', 'boonrisk-site-security-check-report'); ?></p>
                    
                    <div class="boonrisk-requirement-benefits">
                        <p><strong><?php esc_html_e('Web Dashboard adds:', 'boonrisk-site-security-check-report'); ?></strong></p>
                        <ul>
                            <li><?php esc_html_e('Historical trend tracking', 'boonrisk-site-security-check-report'); ?></li>
                            <li><?php esc_html_e('Vulnerability intelligence', 'boonrisk-site-security-check-report'); ?></li>
                            <li><?php esc_html_e('PDF reports for stakeholders', 'boonrisk-site-security-check-report'); ?></li>
                            <li><?php esc_html_e('Multi-site overview', 'boonrisk-site-security-check-report'); ?></li>
                        </ul>
                    </div>
                    
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button button-primary button-hero">
                        <span class="dashicons dashicons-admin-network"></span>
                        <?php esc_html_e('Connect with API Key', 'boonrisk-site-security-check-report'); ?>
                    </a>
                    
                    <p class="boonrisk-note">
                        <span class="dashicons dashicons-external"></span>
                        <a href="https://app.boonrisk.com" target="_blank" rel="noopener">
                            <?php esc_html_e('Get your free API key at app.boonrisk.com', 'boonrisk-site-security-check-report'); ?>
                    </a>
                    </p>
                </div>
                
            <?php else: ?>
                <!-- Both Steps Complete - Show Sync Options -->
                <?php if (!empty($last_local)): ?>
                <div class="boonrisk-card boonrisk-success-card">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <div>
                        <p><strong><?php esc_html_e('Connected', 'boonrisk-site-security-check-report'); ?></strong></p>
                        <p class="boonrisk-meta">
                            <?php 
                            $local_summary = $last_local['results']['summary'] ?? null;
                            if ($local_summary):
                            ?>
                            <?php esc_html_e('Local posture:', 'boonrisk-site-security-check-report'); ?> 
                            <span class="posture-inline posture-<?php echo esc_attr($local_summary['posture']); ?>">
                                <?php echo esc_html($local_summary['posture_label']); ?>
                            </span>
                            • 
                            <?php endif; ?>
                            <?php 
                            printf(
                                /* translators: %s: Date and time of assessment */
                                esc_html__('Assessed: %s', 'boonrisk-site-security-check-report'),
                                esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_local['timestamp']))
                            );
                            ?>
                </p>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="boonrisk-app-analysis">
                    <!-- Manual Sync Section -->
                    <div class="boonrisk-card">
                        <h2><?php esc_html_e('Send Now', 'boonrisk-site-security-check-report'); ?></h2>
                <p><?php esc_html_e('Send your current assessment to the web dashboard immediately.', 'boonrisk-site-security-check-report'); ?></p>
                
                <form id="app-analysis-form">
                    <div class="boonrisk-consent-box">
                        <label>
                            <input type="checkbox" name="consent" id="app-consent" required>
                            <?php esc_html_e('I confirm sending site configuration data to BoonRisk servers.', 'boonrisk-site-security-check-report'); ?>
                        </label>
                    </div>
                    
                    <div class="boonrisk-data-disclosure">
                        <details>
                            <summary><?php esc_html_e('What data is sent?', 'boonrisk-site-security-check-report'); ?></summary>
                            <ul>
                                <li><?php esc_html_e('All local assessment findings and recommendations', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('PHP and WordPress versions', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Active plugin and theme names/versions', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Configuration flags (debug mode, file editor, etc.)', 'boonrisk-site-security-check-report'); ?></li>
                            </ul>
                            <p><?php esc_html_e('No personal data, content, or credentials are ever collected.', 'boonrisk-site-security-check-report'); ?></p>
                        </details>
                    </div>
                    
                    <button type="submit" class="button button-primary" id="run-app-analysis">
                        <span class="dashicons dashicons-upload"></span>
                        <?php esc_html_e('Send Now', 'boonrisk-site-security-check-report'); ?>
                    </button>
                    
                    <div id="app-analysis-status" class="boonrisk-status-message" style="display: none;"></div>
                </form>
                    </div>

                    <!-- Automatic Daily Sync Section -->
                    <div class="boonrisk-card boonrisk-auto-sync-card">
                        <h2>
                            <span class="dashicons dashicons-update"></span>
                            <?php esc_html_e('Automatic Daily Updates', 'boonrisk-site-security-check-report'); ?>
                        </h2>

                        <div class="boonrisk-auto-sync-status">
                            <?php if ($sync_status['enabled']): ?>
                                <div class="sync-status-badge sync-enabled">
                                    <span class="dashicons dashicons-yes-alt"></span>
                                    <?php esc_html_e('Enabled', 'boonrisk-site-security-check-report'); ?>
                                </div>
                                <p class="sync-status-details">
                                    <?php
                                    if ($sync_status['last_sync']):
                                        printf(
                                            /* translators: %s: human time diff */
                                            esc_html__('Last sent: %s', 'boonrisk-site-security-check-report'),
                                            esc_html($sync_status['last_sync_formatted'])
                                        );
                                    else:
                                        esc_html_e('Not sent yet', 'boonrisk-site-security-check-report');
                                    endif;
                                    ?>
                                    <br>
                                    <?php
                                    printf(
                                        /* translators: %s: human time diff */
                                        esc_html__('Next check: in %s', 'boonrisk-site-security-check-report'),
                                        esc_html($sync_status['next_scheduled_formatted'])
                                    );
                                    ?>
                                </p>
                            <?php else: ?>
                                <div class="sync-status-badge sync-disabled">
                                    <span class="dashicons dashicons-minus"></span>
                                    <?php esc_html_e('Disabled', 'boonrisk-site-security-check-report'); ?>
                                </div>
                                <p class="sync-status-details">
                                    <?php esc_html_e('The plugin will only send data when you click the button below.', 'boonrisk-site-security-check-report'); ?>
                                </p>
                            <?php endif; ?>
                        </div>

                        <div class="boonrisk-auto-sync-description">
                            <p><?php esc_html_e('When enabled, the plugin automatically checks your site once per day and sends the results to the web dashboard. Data is only sent if something has changed.', 'boonrisk-site-security-check-report'); ?></p>
                        </div>

                        <form id="auto-sync-form" class="boonrisk-auto-sync-form">
                            <?php if (!$sync_status['enabled']): ?>
                                <div class="boonrisk-consent-box">
                                    <label>
                                        <input type="checkbox" name="auto_sync_consent" id="auto-sync-consent">
                                        <?php esc_html_e('I consent to automatic daily data transmission to BoonRisk servers. No personal data is collected. I can disable this at any time.', 'boonrisk-site-security-check-report'); ?>
                                    </label>
                                </div>

                                <button type="button" class="button button-primary" id="enable-auto-sync">
                                    <span class="dashicons dashicons-yes"></span>
                                    <?php esc_html_e('Enable Daily Updates', 'boonrisk-site-security-check-report'); ?>
                                </button>
                            <?php else: ?>
                                <button type="button" class="button" id="disable-auto-sync">
                                    <span class="dashicons dashicons-no"></span>
                                    <?php esc_html_e('Disable Daily Updates', 'boonrisk-site-security-check-report'); ?>
                                </button>
                            <?php endif; ?>

                            <div id="auto-sync-status" class="boonrisk-status-message" style="display: none;"></div>
                        </form>
                    </div>

                    <?php
                    // Get sync history
                    $sync_history = $auto_sync->get_history();
                    if (!empty($sync_history)): 
                    ?>
                        <div class="boonrisk-card boonrisk-sync-history-card">
                            <h2><?php esc_html_e('Sync History', 'boonrisk-site-security-check-report'); ?></h2>
                            <p><?php esc_html_e('Recent synchronizations with the app dashboard.', 'boonrisk-site-security-check-report'); ?></p>
                            
                            <div class="boonrisk-sync-history-list">
                                <?php foreach ($sync_history as $entry): ?>
                                    <div class="boonrisk-sync-history-item <?php echo $entry['success'] ? 'sync-success' : 'sync-failed'; ?>">
                                        <div class="sync-icon">
                                            <?php if ($entry['success']): ?>
                                                <span class="dashicons dashicons-yes-alt" style="color: #22c55e;"></span>
                                            <?php else: ?>
                                                <span class="dashicons dashicons-dismiss" style="color: #ef4444;"></span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="sync-details">
                                            <div class="sync-meta">
                                                <strong>
                                                    <?php 
                                                    echo $entry['type'] === 'manual' 
                                                        ? esc_html__('Manual Sync', 'boonrisk-site-security-check-report') 
                                                        : esc_html__('Auto Sync', 'boonrisk-site-security-check-report');
                                                    ?>
                                                </strong>
                                                <span class="sync-time">
                                                    <?php echo esc_html(human_time_diff($entry['timestamp'], time()) . ' ' . __('ago', 'boonrisk-site-security-check-report')); ?>
                                                </span>
                                            </div>
                                            <?php if (!$entry['success'] && !empty($entry['error_message'])): ?>
                                                <p class="sync-error"><?php echo esc_html($entry['error_message']); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <a href="https://app.boonrisk.com" target="_blank" rel="noopener" class="button button-primary">
                                <?php esc_html_e('Open Web Dashboard', 'boonrisk-site-security-check-report'); ?>
                                <span class="dashicons dashicons-external"></span>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render Reports page
     */
    public static function render_reports() {
        $has_api_key = BoonRisk_Agent::has_api_key();
        $last_local = get_option('boonrisk_last_local_assessment', null);
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-media-document"></span>
                <?php esc_html_e('Reports', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <!-- Local Report Available -->
            <div class="boonrisk-card">
                <h2><?php esc_html_e('Local Report (Available Now)', 'boonrisk-site-security-check-report'); ?></h2>
                <p><?php esc_html_e('A printable HTML report is available from your local assessment. No account required.', 'boonrisk-site-security-check-report'); ?></p>
                <?php if ($last_local): ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-report')); ?>" class="button button-primary">
                        <span class="dashicons dashicons-media-document"></span>
                        <?php esc_html_e('View Local Report', 'boonrisk-site-security-check-report'); ?>
                    </a>
                <?php else: ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>" class="button">
                        <?php esc_html_e('Run Assessment First', 'boonrisk-site-security-check-report'); ?>
                    </a>
                <?php endif; ?>
            </div>
            
            <?php if (!$has_api_key): ?>
                <!-- Extended reports locked -->
                <div class="boonrisk-card boonrisk-optional-card">
                    <h2><?php esc_html_e('PDF Reports (Optional)', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('PDF reports and dashboard-based reports require an API key. These are optional enhancements.', 'boonrisk-site-security-check-report'); ?></p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button">
                        <?php esc_html_e('Connect to Enable', 'boonrisk-site-security-check-report'); ?>
                    </a>
                </div>
            <?php else: ?>
                <p class="boonrisk-description">
                    <?php esc_html_e('Generate professional security reports to share with clients, teams, or for compliance documentation.', 'boonrisk-site-security-check-report'); ?>
                </p>
                
                <div class="boonrisk-reports">
                    <div class="boonrisk-card">
                        <h2><?php esc_html_e('Generate Report', 'boonrisk-site-security-check-report'); ?></h2>
                        
                        <form id="generate-report-form">
                            <div class="boonrisk-consent-box">
                                <label>
                                    <input type="checkbox" name="consent" id="report-consent" required>
                                    <?php esc_html_e('I understand this will send site data to generate the report.', 'boonrisk-site-security-check-report'); ?>
                                </label>
                            </div>
                            
                            <div class="boonrisk-report-types">
                                <label class="report-type">
                                    <input type="radio" name="report_type" value="executive" checked>
                                    <span class="report-type-card">
                                        <strong><?php esc_html_e('Executive Summary', 'boonrisk-site-security-check-report'); ?></strong>
                                        <span><?php esc_html_e('High-level overview for business stakeholders', 'boonrisk-site-security-check-report'); ?></span>
                                    </span>
                                </label>
                                <label class="report-type">
                                    <input type="radio" name="report_type" value="technical">
                                    <span class="report-type-card">
                                        <strong><?php esc_html_e('Technical Report', 'boonrisk-site-security-check-report'); ?></strong>
                                        <span><?php esc_html_e('Detailed findings for security teams', 'boonrisk-site-security-check-report'); ?></span>
                                    </span>
                                </label>
                            </div>
                            
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Generate Report', 'boonrisk-site-security-check-report'); ?>
                            </button>
                        </form>
                    </div>
                    
                    <div class="boonrisk-card">
                        <h2><?php esc_html_e('Previous Reports', 'boonrisk-site-security-check-report'); ?></h2>
                        <p><?php esc_html_e('View and download previous reports from the web dashboard.', 'boonrisk-site-security-check-report'); ?></p>
                        <a href="https://app.boonrisk.io/reports" target="_blank" rel="noopener" class="button">
                            <?php esc_html_e('Open Reports Dashboard', 'boonrisk-site-security-check-report'); ?>
                            <span class="dashicons dashicons-external"></span>
                        </a>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render Settings page
     * Simplified - no duplication, just connection and about/privacy
     */
    public static function render_settings() {
        $has_api_key = BoonRisk_Agent::has_api_key();
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-admin-settings"></span>
                <?php esc_html_e('Settings', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <div class="boonrisk-settings">
                <!-- Connection Settings -->
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('Connection', 'boonrisk-site-security-check-report'); ?></h2>
                    
                    <?php if ($has_api_key): ?>
                        <div class="boonrisk-connection-status connected">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <strong><?php esc_html_e('Connected to BoonRisk Platform', 'boonrisk-site-security-check-report'); ?></strong>
                        </div>
                        <p><?php esc_html_e('Your site is connected to app.boonrisk.com. Assessments can be synced to your web dashboard.', 'boonrisk-site-security-check-report'); ?></p>
                        <div class="boonrisk-actions">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-sync')); ?>" class="button button-primary">
                                <span class="dashicons dashicons-chart-area"></span>
                                <?php esc_html_e('Data & Monitoring', 'boonrisk-site-security-check-report'); ?>
                            </a>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button">
                                <?php esc_html_e('Manage API Key', 'boonrisk-site-security-check-report'); ?>
                            </a>
                            <a href="https://app.boonrisk.com" target="_blank" class="button button-secondary">
                                <?php esc_html_e('Open Web Dashboard', 'boonrisk-site-security-check-report'); ?>
                                <span class="dashicons dashicons-external" style="margin-top: 3px;"></span>
                            </a>
                        </div>
                    <?php else: ?>
                        <div class="boonrisk-connection-status disconnected">
                            <span class="dashicons dashicons-info"></span>
                            <strong><?php esc_html_e('Not Connected', 'boonrisk-site-security-check-report'); ?></strong>
                        </div>
                        <p><?php esc_html_e('Local assessments are fully functional without connection. Connect to unlock advanced features like vulnerability intelligence, PDF reports, and trend tracking.', 'boonrisk-site-security-check-report'); ?></p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button button-primary">
                            <?php esc_html_e('Connect to Web Dashboard', 'boonrisk-site-security-check-report'); ?>
                        </a>
                    <?php endif; ?>
                </div>
                
                <!-- About & Privacy -->
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('About & Privacy', 'boonrisk-site-security-check-report'); ?></h2>
                    
                    <h3><?php esc_html_e('What BoonRisk Does', 'boonrisk-site-security-check-report'); ?></h3>
                    <ul style="margin-left: 20px;">
                        <li><?php esc_html_e('✓ Assesses Security, Trust & Readiness posture', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✓ Evaluates 28 controls across all domains', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✓ Explains risks in plain language', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✓ Provides actionable recommendations', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✓ Tracks posture over time (with connection)', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    
                    <h3><?php esc_html_e('What BoonRisk Does NOT Do', 'boonrisk-site-security-check-report'); ?></h3>
                    <ul style="margin-left: 20px;">
                        <li><?php esc_html_e('✗ Modify your site or configuration', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✗ Block traffic or attacks', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✗ Auto-fix issues', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('✗ Collect personal data or content', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    
                    <hr style="margin: 20px 0;">
                    
                    <h3><?php esc_html_e('Data Privacy', 'boonrisk-site-security-check-report'); ?></h3>
                    <p><strong><?php esc_html_e('Local Assessment:', 'boonrisk-site-security-check-report'); ?></strong> <?php esc_html_e('Runs entirely on your server. No data is sent externally.', 'boonrisk-site-security-check-report'); ?></p>
                    <p><strong><?php esc_html_e('Web Dashboard:', 'boonrisk-site-security-check-report'); ?></strong> <?php esc_html_e('Data is sent only when you explicitly connect and sync. Requires your consent.', 'boonrisk-site-security-check-report'); ?></p>
                    
                    <details style="margin-top: 15px;">
                        <summary style="cursor: pointer; font-weight: 600;"><?php esc_html_e('What data is collected? (Click to expand)', 'boonrisk-site-security-check-report'); ?></summary>
                        <div style="padding: 15px; background: #f5f5f5; margin-top: 10px; border-radius: 4px;">
                            <p><strong><?php esc_html_e('Collected (when connected):', 'boonrisk-site-security-check-report'); ?></strong></p>
                            <ul style="margin-left: 20px; margin-top: 5px;">
                                <li><?php esc_html_e('PHP and WordPress versions', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Active plugin and theme names/versions', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Configuration settings (debug mode, file editor, etc.)', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Server software type', 'boonrisk-site-security-check-report'); ?></li>
                            </ul>
                            <p style="margin-top: 10px;"><strong><?php esc_html_e('Never Collected:', 'boonrisk-site-security-check-report'); ?></strong></p>
                            <ul style="margin-left: 20px; margin-top: 5px;">
                                <li><?php esc_html_e('User data or personal information', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Passwords or credentials', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Post/page content', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Database contents', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('File contents', 'boonrisk-site-security-check-report'); ?></li>
                            </ul>
                        </div>
                    </details>
                </div>
                
                <!-- Reset Data -->
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('Reset', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('Clear all stored assessment data and start fresh. This does not disconnect your API key.', 'boonrisk-site-security-check-report'); ?></p>
                    <button type="button" class="button button-link-delete" id="reset-plugin-data">
                        <?php esc_html_e('Reset All Data', 'boonrisk-site-security-check-report'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Local Report page (printable HTML report)
     */
    public static function render_local_report() {
        $last_assessment = get_option('boonrisk_last_local_assessment', null);
        
        if (!$last_assessment) {
            ?>
            <div class="wrap boonrisk-admin">
                <h1><?php esc_html_e('Security Check Report', 'boonrisk-site-security-check-report'); ?></h1>
                <div class="boonrisk-card">
                    <p><?php esc_html_e('No assessment data available. Please run a local assessment first.', 'boonrisk-site-security-check-report'); ?></p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>" class="button button-primary">
                        <?php esc_html_e('Run Assessment', 'boonrisk-site-security-check-report'); ?>
                    </a>
                </div>
            </div>
            <?php
            return;
        }
        
        $results = $last_assessment['results'];
        $summary = $results['summary'];
        $checks = $results['checks'];
        $top_risks = $results['top_risks'] ?? [];
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');
        ?>
        <div class="wrap boonrisk-admin boonrisk-report-page">
            <div class="boonrisk-report-header">
                <h1><?php esc_html_e('Security Check Report', 'boonrisk-site-security-check-report'); ?></h1>
                <div class="boonrisk-report-actions no-print">
                    <button type="button" class="button" onclick="window.print();">
                        <span class="dashicons dashicons-printer"></span>
                        <?php esc_html_e('Print Report', 'boonrisk-site-security-check-report'); ?>
                    </button>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-local')); ?>" class="button">
                        <?php esc_html_e('Back to Assessment', 'boonrisk-site-security-check-report'); ?>
                    </a>
                </div>
            </div>
            
            <!-- Report Content -->
            <div class="boonrisk-report-content">
                <!-- Report Meta -->
                <div class="boonrisk-report-meta">
                    <table>
                        <tr>
                            <th><?php esc_html_e('Site', 'boonrisk-site-security-check-report'); ?></th>
                            <td><?php echo esc_html($site_name); ?> (<?php echo esc_url($site_url); ?>)</td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Report Date', 'boonrisk-site-security-check-report'); ?></th>
                            <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($results['generated_at']))); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Assessment Type', 'boonrisk-site-security-check-report'); ?></th>
                            <td><?php esc_html_e('Local Security Check', 'boonrisk-site-security-check-report'); ?></td>
                        </tr>
                    </table>
                </div>
                
                <!-- Security Check Summary -->
                <div class="boonrisk-report-section">
                    <h2><?php esc_html_e('Security Check Summary', 'boonrisk-site-security-check-report'); ?></h2>
                    
                    <div class="boonrisk-report-posture posture-<?php echo esc_attr($summary['posture']); ?>">
                        <div class="posture-badge">
                            <span class="posture-level"><?php echo esc_html($summary['posture_label']); ?></span>
                            <span class="posture-sublabel"><?php esc_html_e('Risk Level', 'boonrisk-site-security-check-report'); ?></span>
                        </div>
                        <div class="posture-explanation">
                            <p><?php echo esc_html($summary['posture_explanation']); ?></p>
                        </div>
                    </div>
                    
                    <div class="boonrisk-report-stats">
                        <div class="stat">
                            <span class="stat-value stat-high"><?php echo esc_html($summary['high_risk_count']); ?></span>
                            <span class="stat-label"><?php esc_html_e('High Risk', 'boonrisk-site-security-check-report'); ?></span>
                        </div>
                        <div class="stat">
                            <span class="stat-value stat-medium"><?php echo esc_html($summary['warning_count']); ?></span>
                            <span class="stat-label"><?php esc_html_e('Medium Risk', 'boonrisk-site-security-check-report'); ?></span>
                        </div>
                        <div class="stat">
                            <span class="stat-value stat-pass"><?php echo esc_html($summary['passed_count']); ?></span>
                            <span class="stat-label"><?php esc_html_e('Passed', 'boonrisk-site-security-check-report'); ?></span>
                        </div>
                    </div>
                </div>
                
                <!-- Top Risks -->
                <?php if (!empty($top_risks)): ?>
                <div class="boonrisk-report-section">
                    <h2><?php esc_html_e('Top Risks (Prioritized)', 'boonrisk-site-security-check-report'); ?></h2>
                    <p class="section-intro"><?php esc_html_e('The following issues are prioritized by their potential impact on your site security.', 'boonrisk-site-security-check-report'); ?></p>
                    
                    <?php foreach ($top_risks as $index => $risk): ?>
                    <div class="boonrisk-report-risk">
                        <h3><?php echo esc_html(($index + 1) . '. ' . $risk['name']); ?></h3>
                        <p class="risk-status status-<?php echo esc_attr($risk['status']); ?>">
                            <?php echo $risk['status'] === 'high_risk' ? esc_html__('High Risk', 'boonrisk-site-security-check-report') : esc_html__('Medium Risk', 'boonrisk-site-security-check-report'); ?>
                        </p>
                        <p><strong><?php esc_html_e('Finding:', 'boonrisk-site-security-check-report'); ?></strong> <?php echo esc_html($risk['message']); ?></p>
                        <p><strong><?php esc_html_e('Why it matters:', 'boonrisk-site-security-check-report'); ?></strong> <?php echo esc_html($risk['why_it_matters']); ?></p>
                        <?php if (!empty($risk['recommendation'])): ?>
                        <p><strong><?php esc_html_e('Recommendation:', 'boonrisk-site-security-check-report'); ?></strong> <?php echo esc_html($risk['recommendation']); ?></p>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <!-- All Findings -->
                <div class="boonrisk-report-section">
                    <h2><?php esc_html_e('All Findings', 'boonrisk-site-security-check-report'); ?></h2>
                    
                    <table class="boonrisk-report-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Check', 'boonrisk-site-security-check-report'); ?></th>
                                <th><?php esc_html_e('Status', 'boonrisk-site-security-check-report'); ?></th>
                                <th><?php esc_html_e('Value', 'boonrisk-site-security-check-report'); ?></th>
                                <th><?php esc_html_e('Details', 'boonrisk-site-security-check-report'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($checks as $check): ?>
                            <tr class="finding-<?php echo esc_attr($check['status']); ?>">
                                <td><strong><?php echo esc_html($check['name']); ?></strong></td>
                                <td>
                                    <?php 
                                    $status_labels = [
                                        'pass' => __('Pass', 'boonrisk-site-security-check-report'),
                                        'warning' => __('Medium', 'boonrisk-site-security-check-report'),
                                        'high_risk' => __('High', 'boonrisk-site-security-check-report'),
                                    ];
                                    echo esc_html($status_labels[$check['status']] ?? $check['status']);
                                    ?>
                                </td>
                                <td><code><?php echo esc_html($check['value']); ?></code></td>
                                <td>
                                    <?php echo esc_html($check['message']); ?>
                                    <?php if (!empty($check['recommendation'])): ?>
                                    <br><em><?php echo esc_html($check['recommendation']); ?></em>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Report Footer -->
                <div class="boonrisk-report-footer">
                    <p><?php esc_html_e('This report was generated locally by BoonRisk WordPress Security Check plugin.', 'boonrisk-site-security-check-report'); ?></p>
                    <p><?php esc_html_e('This assessment evaluates configuration and exposure risks. It does not scan for malware or actively test for vulnerabilities.', 'boonrisk-site-security-check-report'); ?></p>
                    <p class="report-cloud-note"><?php esc_html_e('Detailed vulnerability intelligence is available with an optional web connection. Local assessment focuses on configuration and exposure risks.', 'boonrisk-site-security-check-report'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Data & Monitoring page
     * Combines automatic daily updates + manual sync in one clean place
     */
    public static function render_sync($auto_sync) {
        $has_api_key = BoonRisk_Agent::has_api_key();
        $last_local = get_option('boonrisk_last_local_assessment', null);
        
        if (!$has_api_key) {
            // Not connected - redirect to connect page
            ?>
            <div class="wrap boonrisk-admin">
                <h1><?php esc_html_e('Data & Monitoring', 'boonrisk-site-security-check-report'); ?></h1>
                <div class="boonrisk-card">
                    <p><?php esc_html_e('You need to connect to the web dashboard first.', 'boonrisk-site-security-check-report'); ?></p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=boonrisk-connect')); ?>" class="button button-primary">
                        <?php esc_html_e('Connect Now', 'boonrisk-site-security-check-report'); ?>
                    </a>
                </div>
            </div>
            <?php
            return;
        }
        
        $sync_status = $auto_sync->get_status();
        $sync_history = $auto_sync->get_history();
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-chart-area"></span>
                <?php esc_html_e('Data & Monitoring', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <p class="description">
                <?php esc_html_e('Manage how your site data is synchronized with the BoonRisk web dashboard.', 'boonrisk-site-security-check-report'); ?>
            </p>
            
            <!-- Automatic Daily Updates -->
            <div class="boonrisk-card">
                <h2><?php esc_html_e('Automatic Daily Updates', 'boonrisk-site-security-check-report'); ?></h2>
                
                <div class="boonrisk-auto-sync-status">
                    <?php if ($sync_status['enabled']): ?>
                        <div class="sync-status-badge sync-enabled">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <?php esc_html_e('Enabled', 'boonrisk-site-security-check-report'); ?>
                        </div>
                        <p class="sync-status-details">
                            <?php 
                            if ($sync_status['last_sync']):
                                printf(
                                    /* translators: %s: human time diff */
                                    esc_html__('Last sent: %s', 'boonrisk-site-security-check-report'),
                                    esc_html($sync_status['last_sync_formatted'])
                                );
                            else:
                                esc_html_e('Not sent yet', 'boonrisk-site-security-check-report');
                            endif;
                            ?>
                            <br>
                            <?php 
                            printf(
                                /* translators: %s: human time diff */
                                esc_html__('Next check: in %s', 'boonrisk-site-security-check-report'),
                                esc_html($sync_status['next_scheduled_formatted'])
                            );
                            ?>
                        </p>
                    <?php else: ?>
                        <div class="sync-status-badge sync-disabled">
                            <span class="dashicons dashicons-minus"></span>
                            <?php esc_html_e('Disabled', 'boonrisk-site-security-check-report'); ?>
                        </div>
                        <p class="sync-status-details">
                            <?php esc_html_e('The plugin will only send data when you click "Send Now" below.', 'boonrisk-site-security-check-report'); ?>
                        </p>
                    <?php endif; ?>
                </div>
                
                <p><?php esc_html_e('When enabled, the plugin automatically checks your site once per day and sends the results to the web dashboard. Data is only sent if something has changed.', 'boonrisk-site-security-check-report'); ?></p>
                
                <form id="auto-sync-form" class="boonrisk-auto-sync-form">
                    <?php if (!$sync_status['enabled']): ?>
                        <div class="boonrisk-consent-box">
                            <label>
                                <input type="checkbox" name="auto_sync_consent" id="auto-sync-consent">
                                <?php esc_html_e('I consent to automatic daily data transmission to BoonRisk servers. No personal data is collected. I can disable this at any time.', 'boonrisk-site-security-check-report'); ?>
                            </label>
                        </div>
                        
                        <button type="button" class="button button-primary" id="enable-auto-sync">
                            <span class="dashicons dashicons-yes"></span>
                            <?php esc_html_e('Enable Daily Updates', 'boonrisk-site-security-check-report'); ?>
                        </button>
                    <?php else: ?>
                        <button type="button" class="button" id="disable-auto-sync">
                            <span class="dashicons dashicons-no"></span>
                            <?php esc_html_e('Disable Daily Updates', 'boonrisk-site-security-check-report'); ?>
                        </button>
                    <?php endif; ?>
                    
                    <div id="auto-sync-status" class="boonrisk-status-message" style="display: none;"></div>
                </form>
            </div>
            
            <!-- Manual Sync Section -->
            <div class="boonrisk-card">
                <h2><?php esc_html_e('Send Now', 'boonrisk-site-security-check-report'); ?></h2>
                <p><?php esc_html_e('Send your current assessment to the web dashboard immediately.', 'boonrisk-site-security-check-report'); ?></p>
                
                <form id="app-analysis-form">
                    <div class="boonrisk-consent-box">
                        <label>
                            <input type="checkbox" name="consent" id="app-consent" required>
                            <?php esc_html_e('I confirm sending site configuration data to BoonRisk servers.', 'boonrisk-site-security-check-report'); ?>
                        </label>
                    </div>
                    
                    <div class="boonrisk-data-disclosure">
                        <details>
                            <summary><?php esc_html_e('What data is sent?', 'boonrisk-site-security-check-report'); ?></summary>
                            <ul>
                                <li><?php esc_html_e('All local assessment findings and recommendations', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('PHP and WordPress versions', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Active plugin and theme names/versions', 'boonrisk-site-security-check-report'); ?></li>
                                <li><?php esc_html_e('Configuration flags (debug mode, file editor, etc.)', 'boonrisk-site-security-check-report'); ?></li>
                            </ul>
                            <p><?php esc_html_e('No personal data, content, or credentials are ever collected.', 'boonrisk-site-security-check-report'); ?></p>
                        </details>
                    </div>
                    
                    <button type="submit" class="button button-primary" id="run-app-analysis">
                        <span class="dashicons dashicons-upload"></span>
                        <?php esc_html_e('Send Now', 'boonrisk-site-security-check-report'); ?>
                    </button>
                    
                    <div id="app-analysis-status" class="boonrisk-status-message" style="display: none;"></div>
                </form>
            </div>
            
            <!-- Sync History -->
            <div class="boonrisk-card boonrisk-sync-history-card">
                <div class="boonrisk-card-header">
                    <h2>
                        <span class="dashicons dashicons-backup"></span>
                        <?php esc_html_e('Sync Journal', 'boonrisk-site-security-check-report'); ?>
                    </h2>
                    <?php if (!empty($sync_history)): ?>
                        <span class="boonrisk-badge">
                            <?php printf(esc_html__('%d entries', 'boonrisk-site-security-check-report'), count($sync_history)); ?>
                        </span>
                    <?php endif; ?>
                </div>
                
                <p class="boonrisk-card-description">
                    <?php esc_html_e('Complete log of all data transmissions to the web dashboard, both manual and automatic.', 'boonrisk-site-security-check-report'); ?>
                </p>
                
                <?php if (!empty($sync_history)): ?>
                    <div class="boonrisk-sync-history-table">
                        <table class="widefat striped">
                            <thead>
                                <tr>
                                    <th style="width: 40px;"><?php esc_html_e('Status', 'boonrisk-site-security-check-report'); ?></th>
                                    <th><?php esc_html_e('Type', 'boonrisk-site-security-check-report'); ?></th>
                                    <th><?php esc_html_e('Date & Time', 'boonrisk-site-security-check-report'); ?></th>
                                    <th><?php esc_html_e('Reason', 'boonrisk-site-security-check-report'); ?></th>
                                    <th><?php esc_html_e('Result', 'boonrisk-site-security-check-report'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($sync_history as $entry): ?>
                                    <tr class="<?php echo $entry['success'] ? '' : 'sync-failed-row'; ?>">
                                        <td class="sync-status-cell">
                                            <?php if ($entry['success']): ?>
                                                <span class="dashicons dashicons-yes-alt" style="color: #22c55e; font-size: 20px;"></span>
                                            <?php else: ?>
                                                <span class="dashicons dashicons-dismiss" style="color: #ef4444; font-size: 20px;"></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="sync-type-badge">
                                                <?php if ($entry['type'] === 'manual'): ?>
                                                    <span class="dashicons dashicons-admin-users" style="font-size: 14px; width: 14px; height: 14px;"></span>
                                                    <strong><?php esc_html_e('Manual', 'boonrisk-site-security-check-report'); ?></strong>
                                                <?php else: ?>
                                                    <span class="dashicons dashicons-update" style="font-size: 14px; width: 14px; height: 14px;"></span>
                                                    <strong><?php esc_html_e('Automatic', 'boonrisk-site-security-check-report'); ?></strong>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td>
                                            <div class="sync-datetime">
                                                <div class="sync-date">
                                                    <?php echo esc_html(date_i18n(get_option('date_format'), $entry['timestamp'])); ?>
                                                </div>
                                                <div class="sync-time-meta">
                                                    <?php echo esc_html(date_i18n(get_option('time_format'), $entry['timestamp'])); ?>
                                                    <span class="sync-relative">
                                                        (<?php echo esc_html(human_time_diff($entry['timestamp'], time())); ?> <?php esc_html_e('ago', 'boonrisk-site-security-check-report'); ?>)
                                                    </span>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <div class="sync-reason">
                                                <?php 
                                                $reason_labels = [
                                                    'user_initiated' => __('User clicked Send Now', 'boonrisk-site-security-check-report'),
                                                    'pending_request' => __('Dashboard requested scan', 'boonrisk-site-security-check-report'),
                                                    'changes_detected' => __('Configuration changed', 'boonrisk-site-security-check-report'),
                                                    'daily_heartbeat' => __('Daily scheduled check', 'boonrisk-site-security-check-report'),
                                                ];
                                                echo esc_html($reason_labels[$entry['reason']] ?? $entry['reason']);
                                                ?>
                                            </div>
                                        </td>
                                        <td>
                                            <?php if ($entry['success']): ?>
                                                <span class="sync-result-badge sync-success-badge">
                                                    <?php esc_html_e('✓ Sent successfully', 'boonrisk-site-security-check-report'); ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="sync-result-badge sync-error-badge">
                                                    <?php esc_html_e('✗ Failed', 'boonrisk-site-security-check-report'); ?>
                                                </span>
                                                <?php if (!empty($entry['error_message'])): ?>
                                                    <div class="sync-error-details">
                                                        <?php echo esc_html($entry['error_message']); ?>
                                                    </div>
                                                <?php endif; ?>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="boonrisk-card-footer">
                        <p class="boonrisk-help-text">
                            <span class="dashicons dashicons-info"></span>
                            <?php esc_html_e('Showing last 10 synchronization attempts. Only successful syncs update your dashboard data.', 'boonrisk-site-security-check-report'); ?>
                        </p>
                        <a href="https://app.boonrisk.com" target="_blank" rel="noopener" class="button button-primary">
                            <?php esc_html_e('Open Web Dashboard', 'boonrisk-site-security-check-report'); ?>
                            <span class="dashicons dashicons-external"></span>
                        </a>
                    </div>
                <?php else: ?>
                    <div class="boonrisk-empty-state">
                        <span class="dashicons dashicons-backup" style="font-size: 48px; color: #cbd5e1; margin-bottom: 16px;"></span>
                        <h3><?php esc_html_e('No sync history yet', 'boonrisk-site-security-check-report'); ?></h3>
                        <p><?php esc_html_e('When you send data to the dashboard (manually or automatically), all transmissions will be logged here.', 'boonrisk-site-security-check-report'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render About & Privacy page
     */
    public static function render_about() {
        ?>
        <div class="wrap boonrisk-admin">
            <h1>
                <span class="dashicons dashicons-info"></span>
                <?php esc_html_e('About & Privacy', 'boonrisk-site-security-check-report'); ?>
            </h1>
            
            <div class="boonrisk-about">
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('About BoonRisk', 'boonrisk-site-security-check-report'); ?></h2>
                    <p><?php esc_html_e('BoonRisk provides clear security checks and reports for WordPress sites. We help you understand your security risks through safe, read-only analysis – never scanning, blocking, or changing your site.', 'boonrisk-site-security-check-report'); ?></p>
                    
                    <h3><?php esc_html_e('What BoonRisk Does', 'boonrisk-site-security-check-report'); ?></h3>
                    <ul>
                        <li><?php esc_html_e('Assesses your security configuration', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Explains risks in plain language', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Provides recommendations', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Tracks posture over time (with API key)', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    
                    <h3><?php esc_html_e('What BoonRisk Does NOT Do', 'boonrisk-site-security-check-report'); ?></h3>
                    <ul>
                        <li><?php esc_html_e('Does not modify your site', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Does not block traffic', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Does not auto-fix issues', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Does not collect personal data', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                </div>
                
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('Privacy Policy', 'boonrisk-site-security-check-report'); ?></h2>
                    
                    <h3><?php esc_html_e('Data Collection', 'boonrisk-site-security-check-report'); ?></h3>
                    <p><?php esc_html_e('BoonRisk collects the following data ONLY when you explicitly send to the web dashboard:', 'boonrisk-site-security-check-report'); ?></p>
                    <ul>
                        <li><?php esc_html_e('PHP and WordPress versions', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Active plugin and theme names and versions', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Configuration flags (debug mode, file editor status, etc.)', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Server software type', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    
                    <h3><?php esc_html_e('What We Never Collect', 'boonrisk-site-security-check-report'); ?></h3>
                    <ul>
                        <li><?php esc_html_e('User data or personal information', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Passwords or credentials', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Post/page content', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Database contents', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('File contents', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    
                    <h3><?php esc_html_e('When Data is Sent', 'boonrisk-site-security-check-report'); ?></h3>
                    <p><?php esc_html_e('Data is sent to BoonRisk servers ONLY when you:', 'boonrisk-site-security-check-report'); ?></p>
                    <ul>
                        <li><?php esc_html_e('Click "Send to Dashboard" (requires consent checkbox)', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Click "Generate Report" (requires consent checkbox)', 'boonrisk-site-security-check-report'); ?></li>
                        <li><?php esc_html_e('Click "Test Connection"', 'boonrisk-site-security-check-report'); ?></li>
                    </ul>
                    <p><strong><?php esc_html_e('No background data transmission. No telemetry. No tracking.', 'boonrisk-site-security-check-report'); ?></strong></p>
                    
                    <p>
                        <a href="https://boonrisk.io/privacy" target="_blank" rel="noopener">
                            <?php esc_html_e('Read Full Privacy Policy', 'boonrisk-site-security-check-report'); ?>
                            <span class="dashicons dashicons-external"></span>
                        </a>
                    </p>
                </div>
                
                <div class="boonrisk-card">
                    <h2><?php esc_html_e('Plugin Information', 'boonrisk-site-security-check-report'); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e('Version', 'boonrisk-site-security-check-report'); ?></th>
                            <td><?php echo esc_html(BOONRISK_VERSION); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Website', 'boonrisk-site-security-check-report'); ?></th>
                            <td><a href="https://boonrisk.io" target="_blank" rel="noopener">boonrisk.io</a></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Support', 'boonrisk-site-security-check-report'); ?></th>
                            <td><a href="https://boonrisk.io/support" target="_blank" rel="noopener"><?php esc_html_e('Get Support', 'boonrisk-site-security-check-report'); ?></a></td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }
}

