<?php

/**
 * Functions that relate to individual boards.
 */
namespace Bouncingsprout_Boomerang;

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/** Getters **/
/**
 * Ghe the slug for a given board. Helper function as WordPress doesn't really provide a good method.
 *
 * @param $post
 *
 * @return string
 */
function boomerang_get_board_slug(  $post  ) {
    $post = get_post( $post );
    return $post->post_name;
}

/**
 * Get the board for a given post. This function intrusively searches for a board in the post's parent hierarchy.
 *
 * @param $post
 *
 * @return WP_Post|null
 */
function boomerang_get_board(  $post = false  ) {
    $post = get_post( $post );
    // If the post is a board, return it. That's the easy case.
    if ( $post && 'boomerang_board' === $post->post_type ) {
        return $post;
    }
    // If the post is a Boomerang, return the parent board.
    if ( $post && 'boomerang' === $post->post_type ) {
        return get_post( $post->post_parent );
    }
    // If the post is a page or post, check if it has a Boomerang shortcode, either as a Gutenberg block or a shortcode.
    if ( $post && ('page' === $post->post_type || 'post' === $post->post_type) ) {
        // If the post has Gutenberg blocks, check if it has a Boomerang shortcode as a Gutenberg block.
        $blocks = parse_blocks( $post->post_content );
        foreach ( $blocks as $block ) {
            if ( $block['blockName'] === 'boomerang-block/shortcode-gutenberg' ) {
                return get_post( $block['attrs']['board'] );
            }
        }
        // If the post has a shortcode, check if it's a Boomerang shortcode.
        if ( has_shortcode( $post->post_content, 'boomerang' ) ) {
            // Extract board ID from shortcode attributes
            $pattern = '/\\[boomerang\\s+board=["\']?(\\d+)["\']?\\]/';
            if ( preg_match( $pattern, $post->post_content, $matches ) ) {
                return get_post( $matches[1] );
            }
        }
    }
    // If we get here, we couldn't find a board.
    return null;
}

/** Conditionals **/
// ToDo: may be added at a later date.
// function is_board( $board_id = false ) {
// 	if ( $board_id ) {
// 		// We can only check blocks for individual Boards.
// 		global $post;
//
// 		if ( has_blocks() ) {
// 			$blocks = parse_blocks( $post->post_content );
// 			foreach( $blocks as $block ) {
// 				if( $block['blockName'] === 'boomerang-block/shortcode-gutenberg' ) {
//
// 				}
// 			}
// 			return true;
// 		}
// 	} else {
// 		global $post;
// 		if ( $post && ( has_shortcode( $post->post_content,
// 					'boomerang_board' ) || has_block( 'boomerang-block/shortcode-gutenberg',
// 					$post->post_content ) ) ) {
// 			return true;
// 		}
// 	}
//
// }
/**
 * Checks whether a given user can manage Boomerangs, or the current user if none specified.
 *
 * @return mixed|true|null
 */
function boomerang_can_manage(  $user = false  ) {
    if ( !$user ) {
        $user = wp_get_current_user();
    }
    // Site admins can always manage Boomerangs.
    if ( user_can( $user, 'manage_options' ) ) {
        return true;
    }
    // todo: More to be added soon.
    return apply_filters( 'boomerang_can_manage', false, $user );
}

/**
 * Checks if titles are displayed for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_title_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return false;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['show_title'] ?? false;
}

/**
 * Checks if featured images for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_image_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return false;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_image'] ?? false;
}

/**
 * Checks if comments are enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_comments_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return false;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_comments'] ?? false;
}

/**
 * Checks if votes are enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_votes_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return false;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_votes'] ?? false;
}

/**
 * Checks if bulk votes are enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_bulk_votes_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_bulk_votes'] ?? false;
}

/**
 * Checks if down-voting is enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_downvoting_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_downvoting'] ?? false;
}

/**
 * Checks if tags are enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_tags_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return false;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_tags'] ?? false;
}

/**
 * Checks if statuses are enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_statuses_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_statuses'] ?? false;
}

/**
 * Checks if filters are enabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_filters_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['show_filters'] ?? false;
}

/**
 * Gets the default ordering value for a given Board or Boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_default_ordering(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['default_ordering'] ?? false;
}

/**
 * Retrieves the ordering values for the board.
 *
 * This function retrieves the ordering values for the board by calling the boomerang_get_order_values() function. If the board has a default ordering value set, it rearranges the array
 * of values so that the default value is the first element.
 *
 * @param $post The Boomerang Board ID for which to retrieve the ordering values.
 *
 * @return array The ordering values for the board.
 * @see boomerang_get_order_values()
 *
 */
function boomerang_board_get_ordering_values(  $post  ) {
    $values = boomerang_get_order_values();
    if ( boomerang_board_default_ordering( $post ) ) {
        $key = boomerang_board_default_ordering( $post );
        $value = $values[$key];
        unset($values[$key]);
        $values = array(
            $key => $value,
        ) + $values;
    }
    return $values;
}

/**
 * Checks if authors are displayed for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_author_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['show_author'] ?? false;
}

/**
 * Checks if author avatars are displayed for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_author_avatar_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['show_author_avatar'] ?? false;
}

/**
 * Checks if published dates are displayed for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_date_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['show_date'] ?? true;
}

/**
 * Checks if published dates are displayed in a friendly way.
 *
 * @see human_time_diff()
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_friendly_date_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['show_friendly_date'] ?? false;
}

/**
 * Checks if pagination styling is disabled for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_pagination_styling_disabled(  $post = false  ) {
    $post = boomerang_get_board( $post );
    if ( !$post ) {
        return false;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['boomerang_disable_pagination_styling'] ?? false;
}

/**
 * Gets the pagination limit (items per page) for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_pagination_limit(  $post = false  ) {
    $post = boomerang_get_board( $post );
    if ( !$post ) {
        return 10;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['boomerang_pagination_limit'] ?? 10;
}

/**
 * Gets the pagination type for a given board or boomerang.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_pagination_type(  $post = false  ) {
    $post = boomerang_get_board( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['boomerang_pagination_type'] ?? 'numbered';
}

/**
 * Returns the default status for new Boomerangs.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_get_default_status(  $post  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return 'publish';
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    if ( !$meta['require_approval'] ) {
        return 'publish';
    } else {
        return 'pending';
    }
}

/**
 * Returns the container width for Boomerang pages, or 100%.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_get_container_width(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return;
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    if ( empty( $meta['container_width']['width'] ) || empty( $meta['container_width']['unit'] ) ) {
        return '100%';
    } else {
        return implode( $meta['container_width'] );
    }
}

function boomerang_get_layout(  $post = false  ) {
    $post = boomerang_get_post( $post );
    if ( !$post ) {
        return 'horizontal';
    }
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['archive_layout'] ?? 'horizontal';
}

/**
 * Helper function that retrieves the WP_Post object for either a Boomerang, or it's parent board,
 * or the current WP_Post if none is provided.
 *
 * @param $post
 *
 * @return array|WP_Post|null
 */
function boomerang_get_post(  $post = false  ) {
    if ( !$post ) {
        $post = get_post();
    } else {
        $post = get_post( $post );
    }
    if ( $post && 'boomerang' === $post->post_type ) {
        $post = get_post( $post->post_parent );
    }
    return $post;
}

/** Styles ************************************************************************************************************/
/**
 * Get styling for any Boomerang items.
 *
 * @param $board
 *
 * @return string|void
 */
function boomerang_get_styling(  $board = false  ) {
    if ( !$board ) {
        $post = boomerang_get_post();
        if ( empty( $post ) || 'boomerang_board' !== $post->post_type ) {
            return;
        } else {
            $board = $post->ID;
        }
    }
    $custom_css = '';
    $meta = get_post_meta( $board, 'boomerang_board_options', true );
    if ( empty( $meta['primary_color'] ) ) {
        $custom_css .= ':root {--boomerang-primary-color:#027AB0;}';
    } else {
        $custom_css .= ':root {--boomerang-primary-color:' . $meta['primary_color'] . ';}';
    }
    if ( empty( $meta['form_background_color'] ) ) {
        $custom_css .= ':root {--boomerang-form-background-color:#f3f4f4;}';
    } else {
        $custom_css .= ':root {--boomerang-form-background-color:' . $meta['form_background_color'] . ';}';
    }
    $custom_css .= ':root {--boomerang-team-color:#fab347;}';
    // Widths are generally handled by pages containing Boomerang shortcodes, so we defer to them
    $custom_css .= ':root {--boomerang-container-width:' . esc_attr( boomerang_get_container_width( $board ) ) . '}';
    return $custom_css;
}

/** Labels ************************************************************************************************************/
/**
 * Get the form labels from a boards settings screen.
 *
 * @param $board
 *
 * @return mixed
 */
function boomerang_get_labels(  $board  ) {
    $board = get_post( $board );
    $meta = get_post_meta( $board->ID, 'boomerang_board_options', true );
    return array(
        'singular'              => $meta['label_singular'] ?? 'feature request',
        'plural'                => $meta['label_plural'] ?? 'feature requests',
        'title'                 => $meta['label_title'] ?? 'Title',
        'content'               => $meta['label_content'] ?? 'Content',
        'tags'                  => $meta['label_tags'] ?? 'Tags',
        'submit'                => $meta['label_submit'] ?? 'Submit',
        'already_voted'         => $meta['message_already_voted'] ?? 'You have already voted',
        'message_vote_recorded' => $meta['message_vote_recorded'] ?? 'Thank you for your vote',
    );
}

/**
 * Gets the singular form of a board's name for a Boomerang.
 *
 * @param $board
 *
 * @return mixed
 */
function get_singular(  $board  ) {
    if ( boomerang_get_labels( $board )['singular'] ) {
        return boomerang_get_labels( $board )['singular'];
    } else {
        if ( get_singular_global() ) {
            return get_singular_global();
        } else {
            return 'boomerang';
        }
    }
}

/**
 * Gets the plural form of a board's name for a Boomerang.
 *
 * @param $board
 *
 * @return mixed
 */
function get_plural(  $board  ) {
    if ( boomerang_get_labels( $board )['plural'] ) {
        return boomerang_get_labels( $board )['plural'];
    } else {
        if ( get_plural_global() ) {
            return get_plural_global();
        } else {
            return 'boomerangs';
        }
    }
}

/** Boomerang Form ****************************************************************************************************/
/**
 * Checks to see if the current user can submit Boomerangs via the Boomerang form.
 *
 * @param $board
 *
 * @return mixed
 */
function boomerang_user_can_submit(  $board, $user_id  ) {
    if ( !is_user_logged_in() ) {
        $result = array(
            'message' => esc_html__( 'You must be logged in to post', 'boomerang' ),
        );
    } else {
        $result = true;
    }
    /**
     * Filter the result.
     *
     * @param true|array $result  The result to pass and filter
     * @param string     $board   The ID of the current board
     * @param int        $user_id The user ID of the current user trying to post a Boomerang
     */
    $result = apply_filters(
        'boomerang_user_can_submit',
        $result,
        $board,
        $user_id
    );
    return $result;
}

/**
 * Checks to see if a honeypot is enabled in our form.
 *
 * @param $board
 *
 * @return false|mixed
 */
function boomerang_board_honeypot_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['enable_honeypot'] ?? false;
}

/** Notifications *****************************************************************************************************/
/**
 * Checks if notifications should be sent when Boomerangs are created.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_send_email_new_boomerang(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['send_email_new_boomerang'] ?? false;
}

/**
 * Returns the emails set for notifications of new Boomerangs.
 *
 * @param $post
 *
 * @return mixed
 */
function boomerang_board_new_boomerang_email_addresses(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    return $meta['admin_email'] ?? false;
}

/**
 * Are new Boomerang emails enabled?
 *
 * @param $post
 *
 * @return bool
 */
function send_new_boomerang_email_enabled(  $post = false  ) {
    $post = boomerang_get_post( $post );
    $meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
    if ( $meta['notifications']['send_new_boomerang_email']['enabled'] ) {
        return true;
    }
    return false;
}

// /**
//  * Are new Boomerang emails enabled?
//  *
//  * @param $post
//  *
//  * @return bool
//  */
// function send_new_boomerang_email_data( $post = false ) {
//
// 	$post = boomerang_get_post( $post );
//
// 	$meta = get_post_meta( $post->ID, 'boomerang_board_options', true );
//
// 	if ( ! $meta['notifications']['send_email_new_boomerang'] ) {
// 		return false;
// 	}
//
// 	return false;
// }
/** Voting **/
/**
 * Checks to see if the current user can vote on Boomerangs.
 *
 * @param $board
 *
 * @return mixed
 */
function boomerang_user_can_vote(  $board, $user_id  ) {
    if ( !is_user_logged_in() ) {
        $result = array(
            'message' => esc_html__( 'You must be logged in to vote', 'boomerang' ),
        );
    } else {
        $result = true;
    }
    /**
     * Filter the result.
     *
     * @param true|array $result  The result to pass and filter
     * @param string     $board   The ID of the current board
     * @param int        $user_id The user ID of the current user trying to vote
     */
    $result = apply_filters(
        'boomerang_user_can_vote',
        $result,
        $board,
        $user_id
    );
    return $result;
}
