<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BSGCAL_Scheduler {
	public static function get_daily_slots( string $date_ymd ) : array {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: get_daily_slots called for date: ' . $date_ymd );
		}
		
		$settings = BSGCAL_Admin::get_settings();
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Settings loaded: ' . json_encode( $settings ) );
		}

		// Use plugin timezone setting
		$timezone_string = $settings['timezone'] ?? wp_timezone_string();
		$tz = new DateTimeZone( $timezone_string );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Using timezone: ' . $timezone_string );
		}
		
		// Get all Google Calendar events for the entire day once
		$day_start_for_google = new DateTimeImmutable( $date_ymd . ' 00:00:00', $tz );
		$day_end_for_google = new DateTimeImmutable( $date_ymd . ' 23:59:59', $tz );
		$all_google_events = self::get_google_calendar_events( 
			$day_start_for_google->format( 'Y-m-d H:i:s' ), 
			$day_end_for_google->format( 'Y-m-d H:i:s' ) 
		);
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Retrieved ' . count( $all_google_events ) . ' Google Calendar events for the day' );
		}
		$weekday_num = (int) wp_date( 'N', strtotime( $date_ymd ), $tz ); // 1 (Mon) .. 7 (Sun)
		$weekday_map = [ 1 => 'mon', 2 => 'tue', 3 => 'wed', 4 => 'thu', 5 => 'fri', 6 => 'sat', 7 => 'sun' ];
		$weekday_key = $weekday_map[ $weekday_num ] ?? 'mon';
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Weekday: ' . $weekday_key . ' (number: ' . $weekday_num . ')' );
		}

		// Check exceptions: unavailable date
		if ( ! empty( $settings['exceptions_unavailable_dates'] ) && in_array( $date_ymd, (array) $settings['exceptions_unavailable_dates'], true ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Scheduler Debug: Date marked as unavailable' );
			}
			return [];
		}

		$work_hours = $settings['work_hours'][ $weekday_key ] ?? [];
		$breaks = $settings['breaks'][ $weekday_key ] ?? [];
		$duration = (int) ( $settings['durations'][ $weekday_key ] ?? ( $settings['slot_duration_minutes'] ?? 30 ) );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Work hours: ' . json_encode( $work_hours ) );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Breaks: ' . json_encode( $breaks ) );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Duration: ' . $duration . ' minutes' );
		}

		// Per-date custom
		if ( ! empty( $settings['exceptions_custom_days'][ $date_ymd ] ) ) {
			$custom = $settings['exceptions_custom_days'][ $date_ymd ];
			$work_hours = $custom['work_hours'] ?? $work_hours;
			$breaks = $custom['breaks'] ?? $breaks;
			$duration = isset( $custom['duration'] ) ? (int) $custom['duration'] : $duration;
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Scheduler Debug: Using custom settings for this date' );
			}
		}

		if ( empty( $work_hours ) || empty( $work_hours['start'] ) || empty( $work_hours['end'] ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Scheduler Debug: No work hours configured for this day' );
			}
			return [];
		}

		$day_start = new DateTimeImmutable( $date_ymd . ' ' . $work_hours['start'], $tz );
		$day_end = new DateTimeImmutable( $date_ymd . ' ' . $work_hours['end'], $tz );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Day start: ' . $day_start->format( 'Y-m-d H:i:s' ) );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Day end: ' . $day_end->format( 'Y-m-d H:i:s' ) );
		}

		$slots = [];
		$current = $day_start;
		$interval = new DateInterval( 'PT' . max( 5, $duration ) . 'M' );

		$normalized_breaks = [];
		foreach ( (array) $breaks as $br ) {
			if ( empty( $br['start'] ) || empty( $br['end'] ) ) {
				continue;
			}
			$normalized_breaks[] = [
				'start' => new DateTimeImmutable( $date_ymd . ' ' . $br['start'], $tz ),
				'end' => new DateTimeImmutable( $date_ymd . ' ' . $br['end'], $tz ),
			];
		}
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Normalized breaks: ' . count( $normalized_breaks ) );
		}

		while ( $current < $day_end ) {
			$slot_end = $current->add( $interval );
			if ( $slot_end > $day_end ) {
				break;
			}

			// Skip if overlaps a break
			$overlaps_break = false;
			foreach ( $normalized_breaks as $br ) {
				if ( ! ( $slot_end <= $br['start'] || $current >= $br['end'] ) ) {
					$overlaps_break = true;
					break;
				}
			}
			if ( $overlaps_break ) {
				$current = $current->add( $interval );
				continue;
			}

			$start_str = $current->format( 'Y-m-d H:i:s' );
			$end_str = $slot_end->format( 'Y-m-d H:i:s' );

					// Check ONLY Google Calendar for conflicts (single source of truth)
		$conflicting_events = self::get_conflicting_google_events( $all_google_events, $current, $slot_end );
		$has_google_conflict = !empty( $conflicting_events );
		
		// Only use Google Calendar as source of truth
		$is_available = !$has_google_conflict;
		
		// Debug logging for each slot
		if ( $has_google_conflict && defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Slot ' . $current->format('H:i') . ' - ' . $slot_end->format('H:i') . ' has conflicts: ' . json_encode( $conflicting_events ) );
		}

			$slots[] = [
				'start' => $start_str,
				'end' => $end_str,
				'label' => $current->format( $settings['time_format'] === '12' ? 'g:i A' : 'H:i' ),
				'available' => $is_available,
				'google_conflicts' => $conflicting_events,
			];

			$current = $current->add( $interval );
		}
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Generated ' . count( $slots ) . ' total slots' );
		}

		return $slots;
	}

	/**
	 * Get Google Calendar events for a specific time slot
	 */
	private static function get_google_calendar_events( string $start_time, string $end_time ) : array {
		try {
			if ( ! class_exists( 'BSGCAL_Google_Client' ) ) {
				return [];
			}

			$google_client = new BSGCAL_Google_Client();
			$events = $google_client->get_events_for_timeslot( $start_time, $end_time );
			
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Scheduler Debug: Google Calendar events for ' . $start_time . ' - ' . $end_time . ': ' . count( $events ) );
			}
			
			return $events;
		} catch ( Exception $e ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Scheduler Error: Failed to get Google Calendar events: ' . $e->getMessage() );
			}
			return [];
		}
	}

	/**
	 * Get available slots for a specific date (alias for get_daily_slots)
	 */
	public static function get_available_slots( string $date_ymd ) : array {
		return self::get_daily_slots( $date_ymd );
	}

	/**
	 * Check if a time slot conflicts with any Google Calendar events
	 */
	private static function get_conflicting_google_events( array $google_events, DateTimeImmutable $slot_start, DateTimeImmutable $slot_end ) : array {
		$conflicts = [];
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Checking conflicts for slot ' . $slot_start->format('H:i') . ' - ' . $slot_end->format('H:i') . ' against ' . count( $google_events ) . ' events' );
		}
		
		foreach ( $google_events as $event ) {
			if ( empty( $event['start'] ) || empty( $event['end'] ) ) {
				continue;
			}
			
			try {
				// Parse event times
				$event_start = new DateTimeImmutable( $event['start'] );
				$event_end = new DateTimeImmutable( $event['end'] );
				
				// Check for overlap: events overlap if NOT (event_end <= slot_start OR event_start >= slot_end)
				$no_overlap = ( $event_end <= $slot_start || $event_start >= $slot_end );
				
				if ( ! $no_overlap ) {
					$conflicts[] = $event;
					if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
						// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
						error_log( 'BSGCAL Scheduler Debug: Conflict found - Event: ' . $event['summary'] . ' (' . $event['start'] . ' - ' . $event['end'] . ') overlaps with slot (' . $slot_start->format('H:i') . ' - ' . $slot_end->format('H:i') . ')' );
					}
				}
			} catch ( Exception $e ) {
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
					error_log( 'BSGCAL Scheduler Error: Failed to parse event time: ' . $e->getMessage() );
				}
				continue;
			}
		}
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Scheduler Debug: Found ' . count( $conflicts ) . ' conflicts for slot ' . $slot_start->format('H:i') . ' - ' . $slot_end->format('H:i') );
		}
		return $conflicts;
	}
} 