<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BSGCAL_REST {
	public static function init() : void {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Initializing REST API' );
		}
		
		// Check if REST API is available
		if ( ! function_exists( 'rest_url' ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: REST API functions not available' );
			}
			return;
		}
		
		// Check if we can register routes
		if ( ! function_exists( 'register_rest_route' ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: register_rest_route function not available' );
			}
			return;
		}
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: REST API functions available, adding hook' );
		}
		add_action( 'rest_api_init', [ __CLASS__, 'register_routes' ] );
		
		// Also try to register routes immediately if REST API is already initialized
		if ( did_action( 'rest_api_init' ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Debug: rest_api_init already fired, registering routes immediately' );
			}
			self::register_routes();
		}
	}

	public static function test_endpoint() : WP_REST_Response {
		return new WP_REST_Response( [ 'status' => 'ok', 'message' => 'BSGCAL REST API is working' ] );
	}
	
	public static function register_routes() : void {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Registering REST routes' );
		}
		
		$test_result = register_rest_route( 'bsgcal/v1', '/test', [
			'methods' => WP_REST_Server::READABLE,
			'callback' => [ __CLASS__, 'test_endpoint' ],
			'permission_callback' => '__return_true',
		] );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Test route registration result: ' . ( $test_result ? 'SUCCESS' : 'FAILED' ) );
		}
		
		$slots_result = register_rest_route( 'bsgcal/v1', '/slots', [
			'methods' => WP_REST_Server::READABLE,
			'callback' => [ __CLASS__, 'get_slots' ],
			'args' => [
				'date' => [ 'required' => true, 'type' => 'string' ],
			],
			'permission_callback' => '__return_true',
		] );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Slots route registration result: ' . ( $slots_result ? 'SUCCESS' : 'FAILED' ) );
		}

		$book_result = register_rest_route( 'bsgcal/v1', '/book', [
			'methods' => WP_REST_Server::CREATABLE,
			'callback' => [ __CLASS__, 'create_booking' ],
			'args' => [
				'date' => [ 'required' => true, 'type' => 'string' ],
				'start_time' => [ 'required' => true, 'type' => 'string' ],
				'name' => [ 'required' => true, 'type' => 'string' ],
				'phone' => [ 'required' => true, 'type' => 'string' ],
			],
			'permission_callback' => '__return_true',
		] );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Book route registration result: ' . ( $book_result ? 'SUCCESS' : 'FAILED' ) );
		}
		
		// Debug: Check final routes
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			$routes = rest_get_server()->get_routes();
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Final available routes: ' . json_encode( array_keys( $routes ) ) );
			
			// Debug: Check if routes are accessible
			foreach ( $routes as $route => $handlers ) {
				if ( strpos( $route, 'bsgcal' ) !== false ) {
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
					error_log( 'BSGCAL REST Debug: Found BSGCAL route: ' . $route . ' with handlers: ' . json_encode( array_keys( $handlers ) ) );
				}
			}
			
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: REST routes registered' );
		}
	}

	public static function get_slots( WP_REST_Request $req ) : WP_REST_Response {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: get_slots called - METHOD WORKS!' );
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : 'unknown';
			$request_method = isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : 'unknown';
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Request URL: ' . $request_uri );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Request Method: ' . $request_method );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Request object class: ' . get_class( $req ) );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Request data: ' . json_encode( $req->get_params() ) );
		}
		
		$date = sanitize_text_field( (string) $req['date'] );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Requested date: ' . $date );
		}
		
		if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: Invalid date format: ' . $date );
			}
			return new WP_REST_Response( [ 'error' => 'invalid_date' ], 400 );
		}
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Getting slots from scheduler' );
		}
		$slots = BSGCAL_Scheduler::get_daily_slots( $date );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Scheduler returned ' . count( $slots ) . ' slots' );
			
			if ( ! empty( $slots ) ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Debug: First slot: ' . json_encode( $slots[0] ) );
			}
		}
		
		return new WP_REST_Response( [ 'slots' => $slots ] );
	}

	public static function create_booking( WP_REST_Request $req ) : WP_REST_Response {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: create_booking called - METHOD WORKS!' );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Request object class: ' . get_class( $req ) );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Request data: ' . json_encode( $req->get_params() ) );
		}
		
		$settings = BSGCAL_Admin::get_settings();
		$date = sanitize_text_field( (string) $req['date'] );
		$start_time = sanitize_text_field( (string) $req['start_time'] );
		$name = sanitize_text_field( (string) $req['name'] );
		$phone = sanitize_text_field( (string) $req['phone'] );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Sanitized data - date: ' . $date . ', start_time: ' . $start_time . ', name: ' . $name . ', phone: ' . $phone );
		}

		if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: Invalid date format: ' . $date );
			}
			return new WP_REST_Response( [ 'error' => 'invalid_date' ], 400 );
		}
		if ( ! preg_match( '/^\d{2}:\d{2}$/', $start_time ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: Invalid time format: ' . $start_time );
			}
			return new WP_REST_Response( [ 'error' => 'invalid_time' ], 400 );
		}
		if ( $name === '' || $phone === '' ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: Missing fields - name: ' . $name . ', phone: ' . $phone );
			}
			return new WP_REST_Response( [ 'error' => 'missing_fields' ], 400 );
		}

		$weekday_num = (int) wp_date( 'N', strtotime( $date ), wp_timezone() );
		$weekday_map = [ 1 => 'mon', 2 => 'tue', 3 => 'wed', 4 => 'thu', 5 => 'fri', 6 => 'sat', 7 => 'sun' ];
		$weekday_key = $weekday_map[ $weekday_num ] ?? 'mon';
		$duration = (int) ( $settings['durations'][ $weekday_key ] ?? ( $settings['slot_duration_minutes'] ?? 30 ) );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Weekday: ' . $weekday_key . ', Duration: ' . $duration );
		}

		// Use plugin timezone setting
		$timezone_string = $settings['timezone'] ?? wp_timezone_string();
		$tz = new DateTimeZone( $timezone_string );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Using timezone: ' . $timezone_string );
		}
		
		$start_dt = new DateTimeImmutable( $date . ' ' . $start_time, $tz );
		$end_dt = $start_dt->add( new DateInterval( 'PT' . max( 5, $duration ) . 'M' ) );
		$start_str = $start_dt->format( 'Y-m-d H:i:s' );
		$end_str = $end_dt->format( 'Y-m-d H:i:s' );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Calculated times - start: ' . $start_str . ', end: ' . $end_str );
		}

		// Check if slot is available in Google Calendar
		$slots = BSGCAL_Scheduler::get_daily_slots( $date );
		$slot_available = false;
		
		foreach ( $slots as $slot ) {
			if ( $slot['start'] === $start_str && $slot['available'] ) {
				$slot_available = true;
				break;
			}
		}
		
		if ( ! $slot_available ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: Slot not available according to Google Calendar' );
			}
			return new WP_REST_Response( [ 'error' => 'slot_not_available' ], 409 );
		}

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Inserting booking into database' );
		}
		$booking_id = BSGCAL_DB::insert_booking( [
			'meeting_start' => $start_str,
			'meeting_end' => $end_str,
			'client_name' => $name,
			'client_phone' => $phone,
		] );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Booking inserted with ID: ' . $booking_id );
		}

		// Try to create Google event
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Creating Google Calendar event' );
		}
		$google_event_id = BSGCAL_Google_Client::create_event( [
			'summary' => sprintf(
				// translators: %s: client's name.
				__( 'Meeting with %s', 'booksync-to-google-calendar' ),
				$name
			),
			'description' => sprintf( "Name: %s\nPhone: %s", $name, $phone ),
			'start' => $start_dt,
			'end' => $end_dt,
		] );
		
		if ( is_string( $google_event_id ) && $google_event_id !== '' ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Debug: Google event created with ID: ' . $google_event_id );
			}
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->update( BSGCAL_DB::table_name(), [ 'google_event_id' => $google_event_id ], [ 'id' => $booking_id ] );
		} else {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL REST Error: Failed to create Google event: ' . ( is_wp_error( $google_event_id ) ? $google_event_id->get_error_message() : 'Unknown error' ) );
			}
		}

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL REST Debug: Returning success response' );
		}
		return new WP_REST_Response( [ 'id' => $booking_id, 'google_event_id' => $google_event_id ] );
	}
} 
