<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BSGCAL_Google_Client {
	
	/**
	 * Check if we can use OAuth or need to fall back to Service Account
	 */
	public static function can_use_oauth() : bool {
		// Check if we have OAuth credentials configured
		$settings = BSGCAL_Admin::get_settings();
		if ( empty( $settings['google_client_id'] ) || empty( $settings['google_client_secret'] ) ) {
			return false;
		}
		
		// Check if we have stored tokens
		$tokens = get_option( 'bsgcal_tokens', [] );
		if ( ! empty( $tokens['access_token'] ) || ! empty( $tokens['refresh_token'] ) ) {
			return true;
		}
		
		// If no tokens, we can still try OAuth flow
		return true;
	}
	
	/**
	 * Get access token using Service Account
	 */
	public static function get_service_account_token() {
		$settings = BSGCAL_Admin::get_settings();
		$service_account_json = $settings['service_account_json'] ?? '';
		
		if ( empty( $service_account_json ) ) {
			return new WP_Error( 'no_service_account', __( 'Service Account JSON not configured.', 'booksync-to-google-calendar' ) );
		}
		
		$credentials = json_decode( $service_account_json, true );
		if ( ! is_array( $credentials ) || empty( $credentials['private_key'] ) ) {
			return new WP_Error( 'invalid_service_account', __( 'Invalid Service Account JSON.', 'booksync-to-google-calendar' ) );
		}
		
		// Create JWT token
		$header = [
			'alg' => 'RS256',
			'typ' => 'JWT'
		];
		
		$now = time();
		$payload = [
			'iss' => $credentials['client_email'],
			'scope' => 'https://www.googleapis.com/auth/calendar.events',
			'aud' => 'https://oauth2.googleapis.com/token',
			'exp' => $now + 3600,
			'iat' => $now
		];
		
		$jwt = self::create_jwt( $header, $payload, $credentials['private_key'] );
		
		// Exchange JWT for access token
		$response = wp_remote_post( 'https://oauth2.googleapis.com/token', [
			'timeout' => 20,
			'headers' => [ 'Content-Type' => 'application/x-www-form-urlencoded' ],
			'body' => [
				'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
				'assertion' => $jwt
			],
		] );
		
		if ( is_wp_error( $response ) ) {
			return $response;
		}
		
		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		
		if ( $code !== 200 || ! is_array( $body ) || empty( $body['access_token'] ) ) {
			return new WP_Error( 'service_account_failed', __( 'Failed to get service account token.', 'booksync-to-google-calendar' ) );
		}
		
		// Save the token
		$tokens = [
			'access_token' => (string) $body['access_token'],
			'refresh_token' => '',
			'token_expires' => time() + (int) ( $body['expires_in'] ?? 3600 ) - 60,
			'token_type' => 'service_account',
		];
		update_option( 'bsgcal_tokens', $tokens );
		return $tokens['access_token'];
	}
	
	/**
	 * Create JWT token
	 */
	private static function create_jwt( array $header, array $payload, string $private_key ) : string {
		$header_encoded = self::base64url_encode( json_encode( $header ) );
		$payload_encoded = self::base64url_encode( json_encode( $payload ) );
		
		$data = $header_encoded . '.' . $payload_encoded;
		$signature = '';
		
		if ( openssl_sign( $data, $signature, $private_key, OPENSSL_ALGO_SHA256 ) ) {
			$signature_encoded = self::base64url_encode( $signature );
			return $data . '.' . $signature_encoded;
		}
		
		return '';
	}
	
	/**
	 * Base64URL encode
	 */
	private static function base64url_encode( string $data ) : string {
		return rtrim( strtr( base64_encode( $data ), '+/', '-_' ), '=' );
	}

	public static function get_auth_url() : string {
		$settings = BSGCAL_Admin::get_settings();
		$client_id = $settings['google_client_id'];
		
		// Use custom redirect URI if set, otherwise use the Google settings page URL
		// Note: Google Console requires exact URL match, so we use the full Google settings page URL
		$redirect = $settings['google_redirect_uri'] ?: admin_url( 'admin.php?page=bsgcal-settings' );
		
		if ( empty( $client_id ) ) {
			return '';
		}
		
		$params = [
			'client_id' => $client_id,
			'redirect_uri' => $redirect,
			'response_type' => 'code',
			'scope' => 'https://www.googleapis.com/auth/calendar.events',
			'access_type' => 'offline',
			'prompt' => 'consent',
			'state' => rawurlencode( add_query_arg( [ 'page' => 'bsgcal-settings' ], admin_url( 'admin.php' ) ) ),
		];
		
		$auth_url = add_query_arg( $params, 'https://accounts.google.com/o/oauth2/v2/auth' );
		
		return $auth_url;
	}

	public static function exchange_code_for_tokens( string $code ) {
		$settings = BSGCAL_Admin::get_settings();
		$client_id = $settings['google_client_id'];
		$client_secret = $settings['google_client_secret'];
		
		// Use custom redirect URI if set, otherwise use the Google settings page URL
		// Note: Google Console requires exact URL match, so we use the full Google settings page URL
		$redirect = $settings['google_redirect_uri'] ?: admin_url( 'admin.php?page=bsgcal-settings' );
		
		if ( empty( $client_id ) || empty( $client_secret ) ) {
			return new WP_Error( 'missing_client', __( 'Client ID/Secret not configured.', 'booksync-to-google-calendar' ) );
		}
		
		$request_body = [
			'code' => $code,
			'client_id' => $client_id,
			'client_secret' => $client_secret,
			'redirect_uri' => $redirect,
			'grant_type' => 'authorization_code',
		];
		
		// Try alternative endpoints if main one fails
		$endpoints = [
			'https://oauth2.googleapis.com/token',
			'https://accounts.google.com/oauth/token',
			'https://www.googleapis.com/oauth2/v4/token'
		];
		
		// Add custom endpoint if defined
		if ( defined( 'BSGCAL_OAUTH_ENDPOINT' ) ) {
			array_unshift( $endpoints, BSGCAL_OAUTH_ENDPOINT );
		}
		
		$last_error = null;
		
		foreach ( $endpoints as $endpoint ) {
			$request_args = [
				'timeout' => 20,
				'headers' => [ 'Content-Type' => 'application/x-www-form-urlencoded' ],
				'body' => $request_body,
			];
			
			// Force IPv4 if configured
			if ( defined( 'BSGCAL_FORCE_IPV4' ) && BSGCAL_FORCE_IPV4 ) {
				$request_args['httpversion'] = '1.1';
				$request_args['user-agent'] = 'WordPress/' . get_bloginfo( 'version' ) . '; ' . get_bloginfo( 'url' );
			}
			
			$response = wp_remote_post( $endpoint, $request_args );
			
			if ( is_wp_error( $response ) ) {
				$last_error = $response;
				continue;
			}
			
			$response_code = wp_remote_retrieve_response_code( $response );
			$response_body = wp_remote_retrieve_body( $response );
			
			if ( $response_code === 200 ) {
				break; // Success, exit loop
			} else {
				$last_error = new WP_Error( 'exchange_failed', 'Endpoint ' . $endpoint . ' returned code: ' . $response_code );
			}
		}
		
		// If we get here and no successful response, return last error
		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			if ( $last_error ) {
				return $last_error;
			}
			return new WP_Error( 'all_exchange_endpoints_failed', __( 'All exchange endpoints failed.', 'booksync-to-google-calendar' ) );
		}
		
		$body = json_decode( $response_body, true );
		if ( ! is_array( $body ) || empty( $body['access_token'] ) ) {
			return new WP_Error( 'exchange_failed', __( 'Failed to exchange code for tokens.', 'booksync-to-google-calendar' ) );
		}
		
		// Save tokens
		$tokens = [
			'access_token' => (string) $body['access_token'],
			'refresh_token' => (string) ( $body['refresh_token'] ?? '' ),
			'token_expires' => time() + (int) ( $body['expires_in'] ?? 3600 ) - 60,
			'token_type' => 'oauth',
		];
		
		update_option( 'bsgcal_tokens', $tokens );
		return true;
	}

	public static function refresh_access_token() {
		$tokens = get_option( 'bsgcal_tokens', [] );
		$settings = BSGCAL_Admin::get_settings();
		if ( empty( $tokens['refresh_token'] ) ) {
			return new WP_Error( 'no_refresh_token', __( 'No refresh token available.', 'booksync-to-google-calendar' ) );
		}
		
		$endpoints = [
			'https://oauth2.googleapis.com/token',
			'https://accounts.google.com/oauth/token',
			'https://www.googleapis.com/oauth2/v4/token'
		];
		
		// Add custom endpoint if defined
		if ( defined( 'BSGCAL_OAUTH_ENDPOINT' ) ) {
			array_unshift( $endpoints, BSGCAL_OAUTH_ENDPOINT );
		}
		
		$last_error = null;
		
		foreach ( $endpoints as $endpoint ) {
			$request_args = [
				'timeout' => 20,
				'headers' => [ 'Content-Type' => 'application/x-www-form-urlencoded' ],
				'body' => [
					'refresh_token' => $tokens['refresh_token'],
					'client_id' => $settings['google_client_id'],
					'client_secret' => $settings['google_client_secret'],
					'grant_type' => 'refresh_token',
				],
			];
			
			// Force IPv4 if configured
			if ( defined( 'BSGCAL_FORCE_IPV4' ) && BSGCAL_FORCE_IPV4 ) {
				$request_args['httpversion'] = '1.1';
				$request_args['user-agent'] = 'WordPress/' . get_bloginfo( 'version' ) . '; ' . get_bloginfo( 'url' );
			}
			
			$response = wp_remote_post( $endpoint, $request_args );
			
			if ( is_wp_error( $response ) ) {
				$last_error = $response;
				continue;
			}
			
			$response_code = wp_remote_retrieve_response_code( $response );
			$response_body = wp_remote_retrieve_body( $response );
			
			if ( $response_code === 200 ) {
				break; // Success, exit loop
			} else {
				$last_error = new WP_Error( 'refresh_failed', 'Endpoint ' . $endpoint . ' returned code: ' . $response_code );
			}
		}
		
		// If we get here and no successful response, return last error
		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			if ( $last_error ) {
				return $last_error;
			}
			return new WP_Error( 'all_refresh_endpoints_failed', __( 'All refresh endpoints failed.', 'booksync-to-google-calendar' ) );
		}
		
		$body = json_decode( $response_body, true );
		if ( ! is_array( $body ) || empty( $body['access_token'] ) ) {
			return new WP_Error( 'refresh_failed', __( 'Failed to refresh access token.', 'booksync-to-google-calendar' ) );
		}
		
		$tokens['access_token'] = (string) $body['access_token'];
		$tokens['token_expires'] = time() + (int) ( $body['expires_in'] ?? 3600 ) - 60;
		$tokens['token_type'] = 'oauth';
		
		update_option( 'bsgcal_tokens', $tokens );
		
		return $tokens['access_token'];
	}

	private static function ensure_access_token() {
		$tokens = get_option( 'bsgcal_tokens', [] );
		$settings = BSGCAL_Admin::get_settings();
		
		// Check if we have a valid access token
		if ( ! empty( $tokens['access_token'] ) && time() < (int) $tokens['token_expires'] ) {
			return $tokens['access_token'];
		}
		
		// If OAuth is blocked, try Service Account
		if ( ! self::can_use_oauth() ) {
			return self::get_service_account_token();
		}
		
		// Try OAuth refresh if available
		if ( ! empty( $tokens['refresh_token'] ) ) {
			return self::refresh_access_token();
		}
		
		// If no refresh token, try Service Account as fallback
		if ( ! empty( $settings['service_account_json'] ) ) {
			return self::get_service_account_token();
		}
		
		return new WP_Error( 'no_token', __( 'Not authenticated with Google. OAuth blocked and no Service Account configured.', 'booksync-to-google-calendar' ) );
	}

	public static function create_event( array $event ) {
		$token = self::ensure_access_token();
		if ( is_wp_error( $token ) ) {
			return '';
		}
		
		$settings = BSGCAL_Admin::get_settings();
		$calendar_id = $settings['calendar_id'] ?: 'primary';
		$tz = $settings['timezone'] ?? wp_timezone_string();
		
		$payload = [
			'summary' => $event['summary'] ?? 'Meeting',
			'description' => $event['description'] ?? '',
			'start' => [ 'dateTime' => $event['start']->format( DATE_RFC3339 ), 'timeZone' => $tz ],
			'end' => [ 'dateTime' => $event['end']->format( DATE_RFC3339 ), 'timeZone' => $tz ],
		];
		
		$url = 'https://www.googleapis.com/calendar/v3/calendars/' . rawurlencode( $calendar_id ) . '/events';
		
		$response = wp_remote_post( $url, [
			'timeout' => 20,
			'headers' => [
				'Authorization' => 'Bearer ' . $token,
				'Content-Type' => 'application/json',
			],
			'body' => wp_json_encode( $payload ),
		] );
		
		if ( is_wp_error( $response ) ) {
			return '';
		}
		
		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		
		$body_data = json_decode( $body, true );
		
		if ( $code >= 200 && $code < 300 && is_array( $body_data ) && ! empty( $body_data['id'] ) ) {
			return (string) $body_data['id'];
		}
		
		return '';
	}

	/**
	 * Get Google Calendar events for a specific time slot
	 */
	public static function get_events_for_timeslot( string $start_time, string $end_time ) : array {
		$token = self::ensure_access_token();
		if ( is_wp_error( $token ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Google Client Error: Cannot get events - no valid token: ' . $token->get_error_message() );
			}
			return [];
		}

		$settings = BSGCAL_Admin::get_settings();
		$calendar_id = $settings['calendar_id'] ?: 'primary';
		
		// Convert to RFC3339 format for Google API with proper timezone
		$timezone_string = $settings['timezone'] ?? wp_timezone_string();
		$timezone = new DateTimeZone( $timezone_string );
		
		$start_dt = new DateTime( $start_time, $timezone );
		$end_dt = new DateTime( $end_time, $timezone );
		
		$start_rfc = $start_dt->format( DATE_RFC3339 );
		$end_rfc = $end_dt->format( DATE_RFC3339 );
		
		$query_params = [
			'timeMin' => $start_rfc,
			'timeMax' => $end_rfc,
			'singleEvents' => 'true',
			'orderBy' => 'startTime',
			'maxResults' => 100,
		];
		
		$url = 'https://www.googleapis.com/calendar/v3/calendars/' . rawurlencode( $calendar_id ) . '/events';
				// Build URL manually to avoid encoding issues
		$url_with_params = $url . '?' . http_build_query( $query_params );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Fetching events from ' . $start_rfc . ' to ' . $end_rfc );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Using timezone: ' . $timezone_string );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Calendar ID: ' . $calendar_id );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Request URL: ' . $url_with_params );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Query params: ' . json_encode( $query_params ) );
		}
		
		$response = wp_remote_get( $url_with_params, [
			'timeout' => 20,
			'headers' => [
				'Authorization' => 'Bearer ' . $token,
				'Content-Type' => 'application/json',
			],
		] );

		if ( is_wp_error( $response ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Google Client Error: Failed to fetch events: ' . $response->get_error_message() );
			}
			return [];
		}

		$code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_headers = wp_remote_retrieve_headers( $response );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Response code: ' . $code );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Response headers: ' . json_encode( $response_headers ) );
		}
		
		if ( $code !== 200 ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Google Client Error: Google API returned code ' . $code . ' for events request' );
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Google Client Error: Response body: ' . $response_body );
			}
			return [];
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		if ( ! is_array( $body ) || ! isset( $body['items'] ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'BSGCAL Google Client Error: Invalid response from Google API for events' );
			}
			return [];
		}

		$events = [];
		foreach ( $body['items'] as $item ) {
			$start_time = null;
			$end_time = null;
			
			// Handle both dateTime (timed events) and date (all-day events)
			if ( ! empty( $item['start']['dateTime'] ) ) {
				$start_time = $item['start']['dateTime'];
				$end_time = $item['end']['dateTime'];
			} elseif ( ! empty( $item['start']['date'] ) ) {
				// All-day events: treat as blocking the entire day
				$start_time = $item['start']['date'] . 'T00:00:00';
				$end_time = $item['end']['date'] . 'T23:59:59';
				
				// If end date is provided, use it
				if ( ! empty( $item['end']['date'] ) ) {
					$end_time = $item['end']['date'] . 'T00:00:00';
				}
			}
			
			if ( $start_time && $end_time ) {
				$events[] = [
					'id' => $item['id'] ?? '',
					'summary' => $item['summary'] ?? 'No title',
					'start' => $start_time,
					'end' => $end_time,
					'is_all_day' => !empty( $item['start']['date'] ),
				];
				
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
					error_log( 'BSGCAL Google Client Debug: Found event: ' . ($item['summary'] ?? 'No title') . ' (' . $start_time . ' - ' . $end_time . ')' );
				}
			}
		}

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL Google Client Debug: Found ' . count( $events ) . ' events in time slot' );
		}
		return $events;
	}

	public static function verify_access_token() : bool {
		$tokens = get_option( 'bsgcal_tokens', [] );
		if ( empty( $tokens['access_token'] ) ) {
			return false;
		}
		
		// Check if token is expired
		if ( time() >= (int) $tokens['token_expires'] ) {
			return false;
		}
		
		// Verify token with Google
		$response = wp_remote_get( 'https://www.googleapis.com/oauth2/v1/tokeninfo?access_token=' . $tokens['access_token'], [
			'timeout' => 10,
		] );
		
		if ( is_wp_error( $response ) ) {
			return false;
		}
		
		$code = wp_remote_retrieve_response_code( $response );
		if ( $code !== 200 ) {
			return false;
		}
		
		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		if ( ! is_array( $body ) || empty( $body['scope'] ) ) {
			return false;
		}
		
		// Check if token has required scope
		if ( strpos( $body['scope'], 'https://www.googleapis.com/auth/calendar.events' ) === false ) {
			return false;
		}
		
		return true;
	}
} 
