<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BSGCAL_Admin {
	public const OPTION_KEY = 'bsgcal_settings';

	public static function init() : void {
		add_action( 'admin_menu', [ __CLASS__, 'register_menus' ] );
		add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );
		add_action( 'admin_init', [ __CLASS__, 'maybe_handle_oauth' ] );
		add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_admin_assets' ] );
	}

	public static function get_settings() : array {
		$defaults = [
					'name_label' => __( 'Name', 'booksync-to-google-calendar' ),
		'phone_label' => __( 'Phone number', 'booksync-to-google-calendar' ),
		'no_slots_label' => __( 'No slots for this date', 'booksync-to-google-calendar' ),
					'loading_slots_label' => __( 'Loading slots...', 'booksync-to-google-calendar' ),
		'booked_message' => __( 'Booked!', 'booksync-to-google-calendar' ),
			'calendar_language' => get_locale(),
			'time_format' => '24', // 24-hour format by default
			'weekday_abbreviations' => [
				'mon' => __( 'Mon', 'booksync-to-google-calendar' ),
				'tue' => __( 'Tue', 'booksync-to-google-calendar' ),
				'wed' => __( 'Wed', 'booksync-to-google-calendar' ),
				'thu' => __( 'Thu', 'booksync-to-google-calendar' ),
				'fri' => __( 'Fri', 'booksync-to-google-calendar' ),
				'sat' => __( 'Sat', 'booksync-to-google-calendar' ),
				'sun' => __( 'Sun', 'booksync-to-google-calendar' ),
			],
			'slot_duration_minutes' => 30,
			'timezone' => wp_timezone_string(),
			'time_buffer_minutes' => 15, // Buffer time to prevent booking very recent slots
			'durations' => [
				'mon' => 30, 'tue' => 30, 'wed' => 30, 'thu' => 30, 'fri' => 30, 'sat' => 30, 'sun' => 30,
			],
			'work_hours' => [
				'mon' => [ 'start' => '09:00', 'end' => '17:00' ],
				'tue' => [ 'start' => '09:00', 'end' => '17:00' ],
				'wed' => [ 'start' => '09:00', 'end' => '17:00' ],
				'thu' => [ 'start' => '09:00', 'end' => '17:00' ],
				'fri' => [ 'start' => '09:00', 'end' => '17:00' ],
				'sat' => [ 'start' => '', 'end' => '' ],
				'sun' => [ 'start' => '', 'end' => '' ],
			],

			'google_client_id' => '',
			'google_client_secret' => '',
			'google_redirect_uri' => admin_url( 'admin.php?page=bsgcal-settings' ),
			'calendar_id' => 'primary',
			'access_token' => '',
			'refresh_token' => '',
			'token_expires' => 0,
			'token_type' => '',
		];
		$settings = get_option( self::OPTION_KEY, [] );
		if ( ! is_array( $settings ) ) {
			$settings = [];
		}
		return wp_parse_args( $settings, $defaults );
	}

	public static function register_settings() : void {
		register_setting( 'bsgcal_settings_group', self::OPTION_KEY, [ __CLASS__, 'sanitize_settings' ] );
	}

	public static function sanitize_settings( array $settings ) : array {
		$old = self::get_settings();

		$settings['name_label'] = sanitize_text_field( $settings['name_label'] ?? $old['name_label'] );
		$settings['phone_label'] = sanitize_text_field( $settings['phone_label'] ?? $old['phone_label'] );
		$settings['no_slots_label'] = sanitize_text_field( $settings['no_slots_label'] ?? $old['no_slots_label'] );
		$settings['loading_slots_label'] = sanitize_text_field( $settings['loading_slots_label'] ?? $old['loading_slots_label'] );
		$settings['booked_message'] = sanitize_text_field( $settings['booked_message'] ?? $old['booked_message'] );
		$settings['calendar_language'] = sanitize_text_field( $settings['calendar_language'] ?? $old['calendar_language'] );
		$settings['time_format'] = sanitize_text_field( $settings['time_format'] ?? $old['time_format'] );
		
		// Validate weekday abbreviations
		$weekdays = ['mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun'];
		$default_abbreviations = [
			'mon' => __( 'Mon', 'booksync-to-google-calendar' ),
			'tue' => __( 'Tue', 'booksync-to-google-calendar' ),
			'wed' => __( 'Wed', 'booksync-to-google-calendar' ),
			'thu' => __( 'Thu', 'booksync-to-google-calendar' ),
			'fri' => __( 'Fri', 'booksync-to-google-calendar' ),
			'sat' => __( 'Sat', 'booksync-to-google-calendar' ),
			'sun' => __( 'Sun', 'booksync-to-google-calendar' ),
		];
		foreach ($weekdays as $day) {
			if (isset($settings['weekday_abbreviations'][$day])) {
				$settings['weekday_abbreviations'][$day] = sanitize_text_field($settings['weekday_abbreviations'][$day]);
			} else {
				$settings['weekday_abbreviations'][$day] = $old['weekday_abbreviations'][$day] ?? $default_abbreviations[$day];
			}
		}
		$settings['slot_duration_minutes'] = max( 5, (int) ( $settings['slot_duration_minutes'] ?? $old['slot_duration_minutes'] ) );
		
		// Validate time buffer
		$settings['time_buffer_minutes'] = max( 0, (int) ( $settings['time_buffer_minutes'] ?? $old['time_buffer_minutes'] ) );
		
		// Validate durations by day
		$default_duration = $settings['slot_duration_minutes'];
		$weekdays = ['mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun'];
		foreach ($weekdays as $day) {
			if (isset($settings['durations'][$day])) {
				$duration = (int) $settings['durations'][$day];
				$settings['durations'][$day] = max(5, $duration);
			} else {
				$settings['durations'][$day] = $default_duration;
			}
		}
		
		// Validate work hours by day
		$default_work_hours = ['start' => '09:00', 'end' => '17:00'];
		foreach ($weekdays as $day) {
			if (isset($settings['work_hours'][$day])) {
				$work_hours = $settings['work_hours'][$day];
				
				// Validate start time
				if (isset($work_hours['start']) && !empty($work_hours['start'])) {
					if (preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $work_hours['start'])) {
						$settings['work_hours'][$day]['start'] = $work_hours['start'];
					} else {
						$settings['work_hours'][$day]['start'] = $default_work_hours['start'];
					}
				} else {
					$settings['work_hours'][$day]['start'] = '';
				}
				
				// Validate end time
				if (isset($work_hours['end']) && !empty($work_hours['end'])) {
					if (preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $work_hours['end'])) {
						$settings['work_hours'][$day]['end'] = $work_hours['end'];
					} else {
						$settings['work_hours'][$day]['end'] = $default_work_hours['end'];
					}
				} else {
					$settings['work_hours'][$day]['end'] = '';
				}
				
				// Validate that end time is after start time
				if (!empty($settings['work_hours'][$day]['start']) && !empty($settings['work_hours'][$day]['end'])) {
					if ($settings['work_hours'][$day]['start'] >= $settings['work_hours'][$day]['end']) {
						// Reset to default if invalid
						$settings['work_hours'][$day] = $default_work_hours;
						add_settings_error( self::OPTION_KEY, 'invalid_work_hours_' . $day, __( 'Invalid work hours: end time must be after start time.', 'booksync-to-google-calendar' ), 'error' );
					}
				}
			} else {
				$settings['work_hours'][$day] = $default_work_hours;
			}
		}
		
		// Validate timezone
		$timezone = sanitize_text_field( $settings['timezone'] ?? $old['timezone'] );
		if ( in_array( $timezone, timezone_identifiers_list() ) ) {
			$settings['timezone'] = $timezone;
		} else {
			$settings['timezone'] = wp_timezone_string();
		}

		$settings['google_client_id'] = sanitize_text_field( $settings['google_client_id'] ?? $old['google_client_id'] );
		$settings['google_client_secret'] = sanitize_text_field( $settings['google_client_secret'] ?? $old['google_client_secret'] );
		$settings['google_redirect_uri'] = esc_url_raw( $settings['google_redirect_uri'] ?? $old['google_redirect_uri'] );
		$settings['calendar_id'] = sanitize_text_field( $settings['calendar_id'] ?? $old['calendar_id'] );
		




		// Preserve tokens
		$settings['access_token'] = $old['access_token'];
		$settings['refresh_token'] = $old['refresh_token'];
		$settings['token_expires'] = $old['token_expires'];

		return $settings;
	}

	public static function register_menus() : void {
		add_menu_page(
					__( 'BookSync to Google Calendar', 'booksync-to-google-calendar' ),
		__( 'BookSync to Google Calendar', 'booksync-to-google-calendar' ),
			'manage_options',
			'bsgcal-settings',
			[ __CLASS__, 'render_settings_page' ],
			'dashicons-calendar',
			56
		);

		add_submenu_page( 'bsgcal-settings', __( 'BookSync to Google Calendar', 'booksync-to-google-calendar' ), __( 'BookSync to Google Calendar', 'booksync-to-google-calendar' ), 'manage_options', 'bsgcal-settings', [ __CLASS__, 'render_settings_page' ] );
		add_submenu_page( 'bsgcal-settings', __( 'Settings', 'booksync-to-google-calendar' ), __( 'Settings', 'booksync-to-google-calendar' ), 'manage_options', 'bsgcal-google-settings', [ __CLASS__, 'render_google_settings_page' ] );
		add_submenu_page( 'bsgcal-settings', __( 'Bookings', 'booksync-to-google-calendar' ), __( 'Bookings', 'booksync-to-google-calendar' ), 'manage_options', 'bsgcal-bookings', [ __CLASS__, 'render_bookings_page' ] );
	}

	/**
	 * Enqueue admin scripts and styles
	 */
	public static function enqueue_admin_assets( string $hook ) : void {
		// Only load on plugin admin pages
		$plugin_pages = [ 'toplevel_page_bsgcal-settings', 'booksync-to-google-calendar_page_bsgcal-google-settings', 'booksync-to-google-calendar_page_bsgcal-bookings' ];
		if ( ! in_array( $hook, $plugin_pages, true ) && strpos( $hook, 'bsgcal-' ) === false ) {
			return;
		}

		// Register and enqueue jQuery (already included in WordPress, but ensure it's loaded)
		wp_enqueue_script( 'jquery' );

		// Register admin script handle (empty URL means inline scripts only)
		wp_register_script( 'bsgcal-admin-script', '', [ 'jquery' ], BSGCAL_VERSION, true );
		wp_enqueue_script( 'bsgcal-admin-script' );

		// Register admin style handle (empty URL means inline styles only)
		wp_register_style( 'bsgcal-admin-style', '', [], BSGCAL_VERSION );
		wp_enqueue_style( 'bsgcal-admin-style' );
	}



	public static function render_settings_page() : void {
		if ( ! current_user_can( 'manage_options' ) ) return;
		$settings = self::get_settings();
		$tokens = get_option( 'bsgcal_tokens', [] );
		$authed = ! empty( $tokens['access_token'] );
		$auth_url = BSGCAL_Google_Client::get_auth_url();
		$oauth_available = BSGCAL_Google_Client::can_use_oauth();
		?>
		<div class="wrap">
			<h1><?php echo esc_html__( 'BookSync to Google Calendar', 'booksync-to-google-calendar' ); ?></h1>
			
			<!-- Shortcode Information -->
			<div class="notice notice-info">
				<h3><?php echo esc_html__( 'How to use this plugin', 'booksync-to-google-calendar' ); ?></h3>
				<p><strong><?php echo esc_html__( 'Shortcode:', 'booksync-to-google-calendar' ); ?></strong> <code>[event_to_calendar]</code></p>
				<p><?php echo esc_html__( 'Use this shortcode on any page or post to display the booking calendar. You can also use it in Gutenberg blocks or widgets that support shortcodes.', 'booksync-to-google-calendar' ); ?></p>
				<p><strong><?php echo esc_html__( 'Example:', 'booksync-to-google-calendar' ); ?></strong></p>
				<pre><code>Book an appointment:
[event_to_calendar]</code></pre>
				<p><strong><?php echo esc_html__( 'Need help?', 'booksync-to-google-calendar' ); ?></strong> <a href="<?php echo esc_url( plugin_dir_url( BSGCAL_PLUGIN_FILE ) . 'SETUP-INSTRUCTIONS.md' ); ?>" target="_blank"><?php echo esc_html__( 'View detailed setup instructions', 'booksync-to-google-calendar' ); ?></a></p>
			</div>
			<?php
		// Debug: Check if settings errors exist
		$settings_errors = get_settings_errors( self::OPTION_KEY );
		if ( ! empty( $settings_errors ) ) {
			echo '<div class="notice notice-info"><p>Debug: Found ' . count( $settings_errors ) . ' settings messages</p></div>';
		}
		
		// Display any settings errors
		settings_errors( self::OPTION_KEY );
		
		// Add inline styles
		$inline_css = '
			/* Work hours validation styles */
			.form-table input[type="time"].error {
				border-color: #dc3232;
				box-shadow: 0 0 0 1px #dc3232;
			}
			
			.form-table input[type="time"].error:focus {
				border-color: #dc3232;
				box-shadow: 0 0 0 1px #dc3232;
			}
			
			.form-table .work-hours-row td {
				vertical-align: middle;
			}
			
			.form-table .work-hours-row input[type="time"] {
				width: 80px;
				margin: 0 5px;
			}
			
			.form-table .work-hours-row span {
				margin: 0 5px;
				color: #666;
			}
			
			.form-table input[type="time"] {
				width: 80px;
				margin: 0 5px;
			}
		';
		wp_add_inline_style( 'bsgcal-admin-style', $inline_css );
		?>
		
		<form method="post" action="options.php">
				<?php settings_fields( 'bsgcal_settings_group' ); ?>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php echo esc_html__( 'Name label', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[name_label]" value="<?php echo esc_attr( $settings['name_label'] ); ?>" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Phone label', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[phone_label]" value="<?php echo esc_attr( $settings['phone_label'] ); ?>" class="regular-text" /></td>
					</tr>
				</table>
				
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php echo esc_html__( 'No slots message', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[no_slots_label]" value="<?php echo esc_attr( $settings['no_slots_label'] ); ?>" class="regular-text" />
							<p class="description"><?php echo esc_html__( 'Message shown when no time slots are available for a selected date.', 'booksync-to-google-calendar' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Loading message', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[loading_slots_label]" value="<?php echo esc_attr( $settings['loading_slots_label'] ); ?>" class="regular-text" />
							<p class="description"><?php echo esc_html__( 'Message shown while loading available time slots.', 'booksync-to-google-calendar' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Success message', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[booked_message]" value="<?php echo esc_attr( $settings['booked_message'] ); ?>" class="regular-text" />
							<p class="description"><?php echo esc_html__( 'Message shown when booking is successful.', 'booksync-to-google-calendar' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Calendar language', 'booksync-to-google-calendar' ); ?></th>
						<td>
							<select name="<?php echo esc_attr( self::OPTION_KEY ); ?>[calendar_language]" class="regular-text">
								<?php
								$available_locales = [
									'en_US' => 'English (US)',
									'ru_RU' => 'Русский (Russian)',
									'de_DE' => 'Deutsch (German)',
									'fr_FR' => 'Français (French)',
									'es_ES' => 'Español (Spanish)',
									'it_IT' => 'Italiano (Italian)',
									'pt_BR' => 'Português (Brazil)',
									'ja_JP' => '日本語 (Japanese)',
									'zh_CN' => '中文 (Chinese Simplified)',
									'ko_KR' => '한국어 (Korean)',
									'ar' => 'العربية (Arabic)',
									'hi_IN' => 'हिन्दी (Hindi)',
									'tr_TR' => 'Türkçe (Turkish)',
									'pl_PL' => 'Polski (Polish)',
									'nl_NL' => 'Nederlands (Dutch)',
								];
								$selected_locale = $settings['calendar_language'] ?? get_locale();
								foreach ($available_locales as $locale => $name) {
									$selected = ($locale === $selected_locale) ? 'selected' : '';
									echo '<option value="' . esc_attr($locale) . '" ' . esc_attr($selected) . '>' . esc_html($name) . '</option>';
								}
								?>
							</select>
							<p class="description"><?php echo esc_html__( 'Select the language for calendar display. This affects month names and day abbreviations.', 'booksync-to-google-calendar' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Time format', 'booksync-to-google-calendar' ); ?></th>
						<td>
							<select name="<?php echo esc_attr( self::OPTION_KEY ); ?>[time_format]" class="regular-text">
								<option value="24" <?php selected( $settings['time_format'] ?? '24', '24' ); ?>><?php echo esc_html__( '24-hour format (14:30)', 'booksync-to-google-calendar' ); ?></option>
								<option value="12" <?php selected( $settings['time_format'] ?? '24', '12' ); ?>><?php echo esc_html__( '12-hour format (2:30 PM)', 'booksync-to-google-calendar' ); ?></option>
							</select>
							<p class="description"><?php echo esc_html__( 'Select the time format for displaying time slots.', 'booksync-to-google-calendar' ); ?></p>
						</td>
					</tr>

				</table>
				
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php echo esc_html__( 'Weekday abbreviations', 'booksync-to-google-calendar' ); ?></th>
						<td>
							<table class="form-table" style="margin: 0; width: auto;">
								<tr>
									<td style="padding-right: 15px;"><strong>Monday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][mon]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['mon'] ?? 'Mon' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Tuesday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][tue]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['tue'] ?? 'Tue' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Wednesday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][wed]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['wed'] ?? 'Wed' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Thursday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][thu]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['thu'] ?? 'Thu' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Friday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][fri]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['fri'] ?? 'Fri' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Saturday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][sat]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['sat'] ?? 'Sat' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Sunday:</strong></td>
									<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[weekday_abbreviations][sun]" value="<?php echo esc_attr( $settings['weekday_abbreviations']['sun'] ?? 'Sun' ); ?>" class="small-text" maxlength="3" /></td>
								</tr>
							</table>
							<p class="description"><?php echo esc_html__( 'Customize weekday abbreviations (max 3 characters). Leave empty to use automatic language detection.', 'booksync-to-google-calendar' ); ?></p>
							<p style="margin-top: 10px;">
								<button type="button" id="bsgcal-auto-detect-language" class="button button-secondary">Auto-detect from Language</button>
								<button type="button" id="bsgcal-reset-weekdays" class="button button-secondary">Reset to Default</button>
							</p>
						</td>
					</tr>
				</table>
				
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php echo esc_html__( 'Default slot duration (min)', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[slot_duration_minutes]" value="<?php echo esc_attr( $settings['slot_duration_minutes'] ); ?>" class="small-text" /></td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Time buffer (min)', 'booksync-to-google-calendar' ); ?></th>
						<td><input type="number" min="0" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[time_buffer_minutes]" value="<?php echo esc_attr( $settings['time_buffer_minutes'] ); ?>" class="small-text" />
							<p class="description"><?php echo esc_html__( 'Minimum time buffer before current time to prevent booking very recent slots. Set to 0 to allow immediate booking.', 'booksync-to-google-calendar' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo esc_html__( 'Slot duration by day (min)', 'booksync-to-google-calendar' ); ?></th>
						<td>
							<table class="form-table" style="margin: 0; width: auto;">
								<tr>
									<td style="padding-right: 15px;"><strong>Monday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][mon]" value="<?php echo esc_attr( $settings['durations']['mon'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Tuesday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][tue]" value="<?php echo esc_attr( $settings['durations']['tue'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Wednesday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][wed]" value="<?php echo esc_attr( $settings['durations']['wed'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Thursday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][thu]" value="<?php echo esc_attr( $settings['durations']['thu'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Friday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][fri]" value="<?php echo esc_attr( $settings['durations']['fri'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Saturday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][sat]" value="<?php echo esc_attr( $settings['durations']['sat'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Sunday:</strong></td>
									<td><input type="number" min="5" step="5" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[durations][sun]" value="<?php echo esc_attr( $settings['durations']['sun'] ?? $settings['slot_duration_minutes'] ); ?>" class="small-text" /> min</td>
								</tr>
							</table>
							<p class="description"><?php echo esc_html__( 'Set different slot durations for each day. Leave empty to use default duration.', 'booksync-to-google-calendar' ); ?></p>
							<p style="margin-top: 10px;">
								<button type="button" id="bsgcal-apply-all-durations" class="button button-secondary">Apply Default to All Days</button>
								<button type="button" id="bsgcal-reset-all-durations" class="button button-secondary">Reset All to Default</button>
							</p>
							<?php
							$inline_js_durations = "
							jQuery(document).ready(function($) {
								// Auto-update day durations when default duration changes
								$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').on('change', function() {
									var defaultDuration = parseInt($(this).val()) || 30;
									$('input[name*=\"[durations][\"]').each(function() {
										if ($(this).val() === '') {
											$(this).val(defaultDuration);
										}
									});
								});
								
								// Add \"Reset to Default\" buttons for each day
								$('input[name*=\"[durations][\"]').each(function() {
									var \$input = $(this);
									var \$td = \$input.closest('td');
									var \$resetBtn = $('<button type=\"button\" class=\"button button-small\" style=\"margin-left: 5px;\">Reset</button>');
									
									\$resetBtn.on('click', function() {
										var defaultDuration = parseInt($('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').val()) || 30;
										\$input.val(defaultDuration);
									});
									
									\$td.append(\$resetBtn);
								});
								
								// Apply default to all days
								$('#bsgcal-apply-all-durations').on('click', function() {
									var defaultDuration = parseInt($('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').val()) || 30;
									$('input[name*=\"[durations][\"]').val(defaultDuration);
								});
								
								// Reset all to default
								$('#bsgcal-reset-all-durations').on('click', function() {
									var defaultDuration = parseInt($('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').val()) || 30;
									$('input[name*=\"[durations][\"]').val(defaultDuration);
								});
							});
							";
							wp_add_inline_script( 'bsgcal-admin-script', $inline_js_durations );
							?>
						</td>
					</tr>
				</table>
				
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php echo esc_html__( 'Work hours by day', 'booksync-to-google-calendar' ); ?></th>
						<td>
							<table class="form-table" style="margin: 0; width: auto;">
								<tr>
									<td style="padding-right: 15px;"><strong>Monday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][mon][start]" value="<?php echo esc_attr( $settings['work_hours']['mon']['start'] ?? '09:00' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][mon][end]" value="<?php echo esc_attr( $settings['work_hours']['mon']['end'] ?? '17:00' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="mon" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
								<tr>
									<td style="padding-right: 15px;"><strong>Tuesday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][tue][start]" value="<?php echo esc_attr( $settings['work_hours']['tue']['start'] ?? '09:00' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][tue][end]" value="<?php echo esc_attr( $settings['work_hours']['tue']['end'] ?? '17:00' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="tue" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
								<tr>
									<td><strong>Wednesday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][wed][start]" value="<?php echo esc_attr( $settings['work_hours']['wed']['start'] ?? '09:00' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][wed][end]" value="<?php echo esc_attr( $settings['work_hours']['wed']['end'] ?? '17:00' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="wed" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
								<tr>
									<td><strong>Thursday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][thu][start]" value="<?php echo esc_attr( $settings['work_hours']['thu']['start'] ?? '09:00' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][thu][end]" value="<?php echo esc_attr( $settings['work_hours']['thu']['end'] ?? '17:00' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="thu" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
								<tr>
									<td><strong>Friday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][fri][start]" value="<?php echo esc_attr( $settings['work_hours']['fri']['start'] ?? '09:00' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][fri][end]" value="<?php echo esc_attr( $settings['work_hours']['fri']['end'] ?? '17:00' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="fri" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
								<tr>
									<td><strong>Saturday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][sat][start]" value="<?php echo esc_attr( $settings['work_hours']['sat']['start'] ?? '' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][sat][end]" value="<?php echo esc_attr( $settings['work_hours']['sat']['end'] ?? '' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="sat" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
								<tr>
									<td><strong>Sunday:</strong></td>
									<td>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][sun][start]" value="<?php echo esc_attr( $settings['work_hours']['sun']['start'] ?? '' ); ?>" class="small-text" />
										<span>to</span>
										<input type="time" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[work_hours][sun][end]" value="<?php echo esc_attr( $settings['work_hours']['sun']['end'] ?? '' ); ?>" class="small-text" />
										<button type="button" class="button button-small bsgcal-reset-day" data-day="sun" style="margin-left: 5px;">Reset</button>
									</td>
								</tr>
							</table>
							<p class="description"><?php echo esc_html__( 'Set work hours for each day. Leave empty to mark day as non-working. Use 24-hour format (HH:MM).', 'booksync-to-google-calendar' ); ?></p>
							<p style="margin-top: 10px;">
								<button type="button" id="bsgcal-apply-9-17-weekdays" class="button button-secondary">Apply 9:00-17:00 to All Weekdays</button>
								<button type="button" id="bsgcal-clear-weekend-hours" class="button button-secondary">Clear Weekend Hours</button>
							</p>
						</td>
					</tr>
				</table>
				

				<?php
				$inline_js_workhours = "
				jQuery(document).ready(function($) {
					// Work hours management
					$('.bsgcal-reset-day').on('click', function() {
						var day = $(this).data('day');
						var isWeekday = ['mon', 'tue', 'wed', 'thu', 'fri'].indexOf(day) !== -1;
						
						// Reset weekday abbreviation
						if (isWeekday) {
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[weekday_abbreviations][' + day + ']\"]').val(['Mon', 'Tue', 'Wed', 'Thu', 'Fri'][['mon', 'tue', 'wed', 'thu', 'fri'].indexOf(day)]);
						} else {
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[weekday_abbreviations][' + day + ']\"]').val(['Sat', 'Sun'][['sat', 'sun'].indexOf(day)]);
						}
						
						// Reset duration to default
						var defaultDuration = parseInt($('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').val()) || 30;
						$('input[name*=\"[durations][' + day + ']\"]').val(defaultDuration);
						
						// Reset work hours
						if (isWeekday) {
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][start]\"]').val('09:00');
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][end]\"]').val('17:00');
						} else {
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][start]\"]').val('');
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][end]\"]').val('');
						}
					});
					
					// Auto-update day durations when default duration changes
					$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').on('change', function() {
						var defaultDuration = parseInt($(this).val()) || 30;
						$('input[name*=\"[durations][\"]').each(function() {
							if ($(this).val() === '') {
								$(this).val(defaultDuration);
							}
						});
					});
					
					// Apply default to all days
					$('#bsgcal-apply-all-durations').on('click', function() {
						var defaultDuration = parseInt($('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').val()) || 30;
						$('input[name*=\"[durations][\"]').val(defaultDuration);
					});
					
					// Reset all to default
					$('#bsgcal-reset-all-durations').on('click', function() {
						var defaultDuration = parseInt($('input[name=\"" . esc_js( self::OPTION_KEY ) . "[slot_duration_minutes]\"]').val()) || 30;
						$('input[name*=\"[durations][\"]').val(defaultDuration);
					});
					
					// Apply 9:00-17:00 to all weekdays
					$('#bsgcal-apply-9-17-weekdays').on('click', function() {
						['mon', 'tue', 'wed', 'thu', 'fri'].forEach(function(day) {
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][start]\"]').val('09:00');
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][end]\"]').val('17:00');
						});
					});
					
					// Clear weekend hours
					$('#bsgcal-clear-weekend-hours').on('click', function() {
						['sat', 'sun'].forEach(function(day) {
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][start]\"]').val('');
							$('input[name=\"" . esc_js( self::OPTION_KEY ) . "[work_hours][' + day + '][end]\"]').val('');
						});
					});
				});
				";
				wp_add_inline_script( 'bsgcal-admin-script', $inline_js_workhours );
				?>

				<?php submit_button(); ?>
			</form>
			

			
		</div>
		<?php
	}

	public static function render_google_settings_page() : void {
		if ( ! current_user_can( 'manage_options' ) ) return;
		$settings = self::get_settings();
		$tokens = get_option( 'bsgcal_tokens', [] );
		$authed = ! empty( $tokens['access_token'] );
		$auth_url = BSGCAL_Google_Client::get_auth_url();
		$oauth_available = BSGCAL_Google_Client::can_use_oauth();
		?>
		<div class="wrap">
			<h1><?php echo esc_html__( 'Settings', 'booksync-to-google-calendar' ); ?></h1>
			
			<form method="post" action="options.php">
				<?php settings_fields( 'bsgcal_settings_group' ); ?>
				
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php echo esc_html__( 'Timezone', 'booksync-to-google-calendar' ); ?></th>
						<td>
							<select name="<?php echo esc_attr( self::OPTION_KEY ); ?>[timezone]" class="regular-text">
								<?php
								$selected_timezone = $settings['timezone'] ?? wp_timezone_string();
								$timezones = timezone_identifiers_list();
								foreach ( $timezones as $tz ) {
									$selected = ( $tz === $selected_timezone ) ? 'selected' : '';
									echo '<option value="' . esc_attr( $tz ) . '" ' . esc_attr($selected) . '>' . esc_html( $tz ) . '</option>';
								}
								?>
							</select>
							<p class="description">
								<?php echo esc_html__( 'Select your local timezone. Current WordPress timezone: ', 'booksync-to-google-calendar' ); ?>
								<strong><?php echo esc_html( wp_timezone_string() ); ?></strong>
								<br>
								<?php echo esc_html__( 'Current time in selected timezone: ', 'booksync-to-google-calendar' ); ?>
								<strong id="bsgcal-current-time"><?php echo esc_html( wp_date( 'Y-m-d H:i:s', null, new DateTimeZone( $selected_timezone ) ) ); ?></strong>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">Google Client ID</th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[google_client_id]" value="<?php echo esc_attr( $settings['google_client_id'] ); ?>" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row">Google Client Secret</th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[google_client_secret]" value="<?php echo esc_attr( $settings['google_client_secret'] ); ?>" class="regular-text" /></td>
					</tr>
					<tr>
						<th scope="row">Redirect URI</th>
						<td><input type="url" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[google_redirect_uri]" value="<?php echo esc_attr( $settings['google_redirect_uri'] ); ?>" class="regular-text" />
							<p class="description">
								<strong><?php echo esc_html__( 'Default (recommended):', 'booksync-to-google-calendar' ); ?></strong> <?php echo esc_html__( 'Your site URL + /wp-admin/admin.php?page=bsgcal-settings', 'booksync-to-google-calendar' ); ?><br>
								<strong><?php echo esc_html__( 'Example:', 'booksync-to-google-calendar' ); ?></strong> <?php echo esc_html__( 'https://yoursite.com/wp-admin/admin.php?page=bsgcal-settings', 'booksync-to-google-calendar' ); ?><br>
								<strong><?php echo esc_html__( 'Custom:', 'booksync-to-google-calendar' ); ?></strong> <?php echo esc_html__( 'Leave empty to use default, or specify custom redirect URI if needed for your setup.', 'booksync-to-google-calendar' ); ?>
							</p></td>
					</tr>
					<tr>
						<th scope="row">Calendar ID</th>
						<td><input type="text" name="<?php echo esc_attr( self::OPTION_KEY ); ?>[calendar_id]" value="<?php echo esc_attr( $settings['calendar_id'] ); ?>" class="regular-text" />
							<p class="description"><?php echo esc_html__( 'Use primary or specific calendar ID', 'booksync-to-google-calendar' ); ?></p></td>
					</tr>
				</table>
				
				<?php submit_button(); ?>
			</form>
			
			<?php if ( $oauth_available && $auth_url ) : ?>
				<p><a class="button button-primary" href="<?php echo esc_url( $auth_url ); ?>"><?php echo esc_html__( 'Connect Google Account (OAuth)', 'booksync-to-google-calendar' ); ?></a></p>
			<?php endif; ?>
			
		</div>
		<?php
	}

	public static function render_bookings_page() : void {
		if ( ! current_user_can( 'manage_options' ) ) return;
		
		// Verify nonce for pagination parameters
		if ( isset( $_GET['per_page'] ) || isset( $_GET['paged'] ) ) {
			$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'bsgcal_bookings_nonce' ) ) {
				wp_die( esc_html__( 'Security check failed', 'booksync-to-google-calendar' ) );
			}
		}
		
		// Get pagination parameters
		$per_page = isset( $_GET['per_page'] ) ? (int) $_GET['per_page'] : 20;
		$current_page = isset( $_GET['paged'] ) ? max( 1, (int) $_GET['paged'] ) : 1;
		
		// Validate per_page options
		$per_page_options = [20, 50, -1]; // -1 means show all
		if ( ! in_array( $per_page, $per_page_options ) ) {
			$per_page = 20;
		}
		
		// Calculate offset
		$offset = ( $current_page - 1 ) * $per_page;
		
		// Get total count and bookings
		$total_bookings = BSGCAL_DB::count_bookings();
		
		if ( $per_page === -1 ) {
			// Show all bookings
			$bookings = BSGCAL_DB::list_bookings( $total_bookings, 0 );
			$total_pages = 1;
		} else {
			// Show paginated bookings
			$bookings = BSGCAL_DB::list_bookings( $per_page, $offset );
			$total_pages = ceil( $total_bookings / $per_page );
		}
		
		// Build current URL for pagination
		$current_url = remove_query_arg( ['paged', 'per_page'] );
		?>
		<div class="wrap">
			<h1><?php echo esc_html__( 'Bookings', 'booksync-to-google-calendar' ); ?></h1>
			
			<!-- Per Page Selector -->
			<div class="tablenav top">
				<div class="alignleft actions">
					<label for="per-page-selector"><?php echo esc_html__( 'Show:', 'booksync-to-google-calendar' ); ?></label>
					<select id="per-page-selector" onchange="changePerPage(this.value)">
						<option value="20" <?php selected( $per_page, 20 ); ?>><?php echo esc_html__( '20 items', 'booksync-to-google-calendar' ); ?></option>
						<option value="50" <?php selected( $per_page, 50 ); ?>><?php echo esc_html__( '50 items', 'booksync-to-google-calendar' ); ?></option>
						<option value="-1" <?php selected( $per_page, -1 ); ?>><?php echo esc_html__( 'All items', 'booksync-to-google-calendar' ); ?></option>
					</select>
				</div>
				
				<?php if ( $per_page !== -1 && $total_pages > 1 ) : ?>
					<div class="tablenav-pages">
						<span class="displaying-num">
							<?php 
							$start_item = $offset + 1;
							$end_item = min( $offset + $per_page, $total_bookings );
							printf(
								// translators: %1$d: start item number, %2$d: end item number, %3$d: total number of items.
								esc_html__( 'Showing %1$d-%2$d of %3$d items', 'booksync-to-google-calendar' ),
								esc_html( $start_item ),
								esc_html( $end_item ),
								esc_html( $total_bookings )
							);
							?>
						</span>
						
						<?php if ( $total_pages > 1 ) : ?>
							<span class="pagination-links">
								<?php
								// Previous page link
								if ( $current_page > 1 ) {
									$prev_url = add_query_arg( 'paged', $current_page - 1, $current_url );
									echo '<a class="prev-page" href="' . esc_url( $prev_url ) . '">‹</a>';
								}
								
								// Page numbers
								$start_page = max( 1, $current_page - 2 );
								$end_page = min( $total_pages, $current_page + 2 );
								
								if ( $start_page > 1 ) {
									echo '<a class="first-page" href="' . esc_url( add_query_arg( 'paged', 1, $current_url ) ) . '">1</a>';
									if ( $start_page > 2 ) {
										echo '<span class="paging-input">…</span>';
									}
								}
								
								for ( $i = $start_page; $i <= $end_page; $i++ ) {
									if ( $i === $current_page ) {
										echo '<span class="paging-input"><span class="tablenav-paging-text">' . esc_html($i) . '</span></span>';
									} else {
										echo '<a class="paging-input" href="' . esc_url( add_query_arg( 'paged', $i, $current_url ) ) . '">' . esc_html($i) . '</a>';
									}
								}
								
								if ( $end_page < $total_pages ) {
									if ( $end_page < $total_pages - 1 ) {
										echo '<span class="paging-input">…</span>';
									}
									echo '<a class="last-page" href="' . esc_url( add_query_arg( 'paged', $total_pages, $current_url ) ) . '">' . esc_html($total_pages) . '</a>';
								}
								
								// Next page link
								if ( $current_page < $total_pages ) {
									$next_url = add_query_arg( 'paged', $current_page + 1, $current_url );
									echo '<a class="next-page" href="' . esc_url( $next_url ) . '">›</a>';
								}
								?>
							</span>
						<?php endif; ?>
					</div>
				<?php endif; ?>
			</div>
			
			<table class="widefat striped">
				<thead>
					<tr>
						<th>ID</th>
						<th><?php echo esc_html__( 'Start', 'booksync-to-google-calendar' ); ?></th>
						<th><?php echo esc_html__( 'End', 'booksync-to-google-calendar' ); ?></th>
						<th><?php echo esc_html__( 'Name', 'booksync-to-google-calendar' ); ?></th>
						<th><?php echo esc_html__( 'Phone', 'booksync-to-google-calendar' ); ?></th>
						<th><?php echo esc_html__( 'Status', 'booksync-to-google-calendar' ); ?></th>
					</tr>
				</thead>
				<tbody>
				<?php if ( empty( $bookings ) ) : ?>
					<tr>
						<td colspan="6" style="text-align: center; padding: 20px;">
							<?php echo esc_html__( 'No bookings found.', 'booksync-to-google-calendar' ); ?>
						</td>
					</tr>
				<?php else : ?>
					<?php foreach ( $bookings as $b ) : ?>
						<tr>
							<td><?php echo (int) $b['id']; ?></td>
							<td><?php echo esc_html( $b['meeting_start'] ); ?></td>
							<td><?php echo esc_html( $b['meeting_end'] ); ?></td>
							<td><?php echo esc_html( $b['client_name'] ); ?></td>
							<td><?php echo esc_html( $b['client_phone'] ); ?></td>
							<td><?php echo esc_html( $b['status'] ); ?></td>
						</tr>
					<?php endforeach; ?>
				<?php endif; ?>
				</tbody>
			</table>
			
			<!-- Bottom pagination -->
			<?php if ( $per_page !== -1 && $total_pages > 1 ) : ?>
				<div class="tablenav bottom">
					<div class="tablenav-pages">
						<span class="displaying-num">
							<?php 
							printf(
								// translators: %1$d: start item number, %2$d: end item number, %3$d: total number of items.
								esc_html__( 'Showing %1$d-%2$d of %3$d items', 'booksync-to-google-calendar' ),
								esc_html( $start_item ),
								esc_html( $end_item ),
								esc_html( $total_bookings )
							);
							?>
						</span>
						
						<span class="pagination-links">
							<?php
							// Previous page link
							if ( $current_page > 1 ) {
								$prev_url = add_query_arg( 'paged', $current_page - 1, $current_url );
								echo '<a class="prev-page" href="' . esc_url( $prev_url ) . '">‹</a>';
							}
							
							// Page numbers
							$start_page = max( 1, $current_page - 2 );
							$end_page = min( $total_pages, $current_page + 2 );
							
							if ( $start_page > 1 ) {
								echo '<a class="first-page" href="' . esc_url( add_query_arg( 'paged', 1, $current_url ) ) . '">1</a>';
								if ( $start_page > 2 ) {
									echo '<span class="paging-input">…</span>';
								}
							}
							
							for ( $i = $start_page; $i <= $end_page; $i++ ) {
								if ( $i === $current_page ) {
									echo '<span class="paging-input"><span class="tablenav-paging-text">' . esc_html($i) . '</span></span>';
								} else {
									echo '<a class="paging-input" href="' . esc_url( add_query_arg( 'paged', $i, $current_url ) ) . '">' . esc_html($i) . '</a>';
								}
							}
							
							if ( $end_page < $total_pages ) {
								if ( $end_page < $total_pages - 1 ) {
									echo '<span class="paging-input">…</span>';
								}
								echo '<a class="last-page" href="' . esc_url( add_query_arg( 'paged', $total_pages, $current_url ) ) . '">' . esc_html($total_pages) . '</a>';
							}
							
							// Next page link
							if ( $current_page < $total_pages ) {
								$next_url = add_query_arg( 'paged', $current_page + 1, $current_url );
								echo '<a class="next-page" href="' . esc_url( $next_url ) . '">›</a>';
							}
							?>
						</span>
					</div>
				</div>
			<?php endif; ?>
		</div>
		
		<?php
		$inline_js_pagination = "
		function changePerPage(value) {
			var url = new URL(window.location);
			url.searchParams.set('per_page', value);
			url.searchParams.delete('paged'); // Reset to first page
			window.location.href = url.toString();
		}
		";
		wp_add_inline_script( 'bsgcal-admin-script', $inline_js_pagination );
		?>
		<?php
	}

	public static function maybe_handle_oauth() : void {
		if ( ! is_admin() ) return;
		if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'bsgcal-settings' ) return;
		
		if ( isset( $_GET['code'] ) && is_string( $_GET['code'] ) ) {
			// Verify nonce for OAuth callback
			$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'bsgcal_oauth_nonce' ) ) {
				wp_die( esc_html__( 'Security check failed', 'booksync-to-google-calendar' ) );
			}
			
			$code = sanitize_text_field( wp_unslash( $_GET['code'] ) );
			
			$result = BSGCAL_Google_Client::exchange_code_for_tokens( $code );
			if ( is_wp_error( $result ) ) {
				add_settings_error( self::OPTION_KEY, 'bsgcal_oauth_error', $result->get_error_message(), 'error' );
			} else {
				add_settings_error( self::OPTION_KEY, 'bsgcal_oauth_ok', __( 'Google account connected successfully.', 'booksync-to-google-calendar' ), 'updated' );
				wp_safe_redirect( admin_url( 'admin.php?page=bsgcal-settings' ) );
				exit;
			}
		}
	}

} 
