<?php
/**
 * Plugin Name: BookSync to Google Calendar
 * Description: Interactive booking calendar with Google Calendar sync.
 * Version: 1.1.3
 * Author: Mykola Zitser
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: booksync-to-google-calendar
 * Requires at least: 6.0
 * Requires PHP: 8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define constants
if ( ! defined( 'BSGCAL_PLUGIN_FILE' ) ) {
	define( 'BSGCAL_PLUGIN_FILE', __FILE__ );
}
if ( ! defined( 'BSGCAL_PLUGIN_DIR' ) ) {
	define( 'BSGCAL_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'BSGCAL_PLUGIN_URL' ) ) {
	define( 'BSGCAL_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}
if ( ! defined( 'BSGCAL_VERSION' ) ) {
	define( 'BSGCAL_VERSION', '1.1.2' );
}

// System compatibility checks
function bsgcal_check_system_compatibility() : array {
	$errors = [];
	
	// Check PHP version
		if ( version_compare( PHP_VERSION, '8.0.0', '<' ) ) {
		$errors[] = sprintf(
			// translators: %s: current PHP version.
			__( 'BookSync to Google Calendar requires PHP 8.0 or higher. Current version: %s', 'booksync-to-google-calendar' ),
			PHP_VERSION
		);
	}
	
	// Check WordPress version
	if ( version_compare( get_bloginfo( 'version' ), '6.0.0', '<' ) ) {
		$errors[] = sprintf(
			// translators: %s: current WordPress version.
			__( 'BookSync to Google Calendar requires WordPress 6.0 or higher. Current version: %s', 'booksync-to-google-calendar' ),
			get_bloginfo( 'version' )
		);
	}
	
	// Check MySQL/MariaDB version
	global $wpdb;
	if ( $wpdb->db_version() ) {
		$db_version = $wpdb->db_version();
		if ( version_compare( $db_version, '5.7.0', '<' ) ) {
			$errors[] = sprintf(
				// translators: %s: current database version.
				__( 'BookSync to Google Calendar requires MySQL 5.7+ or MariaDB 10.2+. Current version: %s', 'booksync-to-google-calendar' ),
				$db_version
			);
		}
	}
	
	// Check required PHP extensions
	$required_extensions = [ 'curl', 'json', 'openssl' ];
	foreach ( $required_extensions as $ext ) {
		if ( ! extension_loaded( $ext ) ) {
			$errors[] = sprintf(
				// translators: %s: required PHP extension name.
				__( 'BookSync to Google Calendar requires PHP extension: %s', 'booksync-to-google-calendar' ),
				$ext
			);
		}
	}
	
	return $errors;
}

// Display compatibility errors in admin
function bsgcal_admin_notices() {
	if ( ! current_user_can( 'activate_plugins' ) ) {
		return;
	}
	
	$errors = bsgcal_check_system_compatibility();
	if ( ! empty( $errors ) ) {
		echo '<div class="notice notice-error"><p>';
		echo '<strong>' . esc_html__( 'BookSync to Google Calendar - System Compatibility Issues:', 'booksync-to-google-calendar' ) . '</strong><br>';
		foreach ( $errors as $error ) {
			echo '• ' . esc_html( $error ) . '<br>';
		}
		echo '</p></div>';
	}
}
add_action( 'admin_notices', 'bsgcal_admin_notices' );

// Includes
require_once BSGCAL_PLUGIN_DIR . 'includes/class-i18n.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-activator.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-db.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-scheduler.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-admin.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-frontend.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-rest.php';
require_once BSGCAL_PLUGIN_DIR . 'includes/class-google-client.php';
// Support module temporarily disabled - will be developed later
// require_once BSGCAL_PLUGIN_DIR . 'includes/class-support.php';

// Activation / deactivation
// Note: register_activation_hook must be called from the main plugin file
// The class will be loaded when the hook fires
register_activation_hook( __FILE__, 'bsgcal_activate_plugin' );
register_deactivation_hook( __FILE__, 'bsgcal_deactivate_plugin' );

/**
 * Plugin activation callback
 */
function bsgcal_activate_plugin() : void {
	// Ensure constants are defined
	if ( ! defined( 'BSGCAL_PLUGIN_DIR' ) ) {
		define( 'BSGCAL_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
	}
	
	// Load activator class
	if ( ! class_exists( 'BSGCAL_Activator' ) ) {
		$activator_file = BSGCAL_PLUGIN_DIR . 'includes/class-activator.php';
		if ( file_exists( $activator_file ) ) {
			require_once $activator_file;
		}
	}
	
	// Activate plugin
	if ( class_exists( 'BSGCAL_Activator' ) ) {
		BSGCAL_Activator::activate();
	}
}

/**
 * Plugin deactivation callback
 */
function bsgcal_deactivate_plugin() : void {
	// Ensure constants are defined
	if ( ! defined( 'BSGCAL_PLUGIN_DIR' ) ) {
		define( 'BSGCAL_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
	}
	
	// Load activator class
	if ( ! class_exists( 'BSGCAL_Activator' ) ) {
		$activator_file = BSGCAL_PLUGIN_DIR . 'includes/class-activator.php';
		if ( file_exists( $activator_file ) ) {
			require_once $activator_file;
		}
	}
	
	// Deactivate plugin
	if ( class_exists( 'BSGCAL_Activator' ) ) {
		BSGCAL_Activator::deactivate();
	}
}

// Bootstrap plugin
function bsgcal_init_plugin() : void {
	if ( ! class_exists( 'BSGCAL_Admin' ) ) {
		return;
	}
	
	try {
		// Initialize i18n
		if ( class_exists( 'BSGCAL_I18n' ) ) {
			BSGCAL_I18n::load_textdomain();
		}
		
		BSGCAL_Admin::init();
		if ( class_exists( 'BSGCAL_Frontend' ) ) {
			BSGCAL_Frontend::init();
		}
		// Support module temporarily disabled - will be developed later
		// if ( class_exists( 'BSGCAL_Support' ) ) {
		// 	BSGCAL_Support::init();
		// }
	} catch ( Exception $e ) {
		// Silent fail in production
	}
}

function bsgcal_init_rest_api() : void {
	if ( ! class_exists( 'BSGCAL_REST' ) ) {
		return;
	}
	
	try {
		BSGCAL_REST::init();
	} catch ( Exception $e ) {
		// Silent fail in production
	}
}

// AJAX handlers for frontend booking
function bsgcal_ajax_get_slots() : void {
	// Ensure we're in AJAX context
	if ( ! wp_doing_ajax() ) {
		wp_die( 'Not AJAX request' );
	}
	
	// Verify nonce
	$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : ( isset( $_GET['nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['nonce'] ) ) : '' );
	if ( ! wp_verify_nonce( $nonce, 'bsgcal_ajax_nonce' ) ) {
		wp_die( 'Security check failed' );
	}
	
	// Set proper content type
	header( 'Content-Type: application/json' );
	
	$date = sanitize_text_field( wp_unslash( $_POST['date'] ?? $_GET['date'] ?? '' ) );
	
	if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
		wp_send_json_error( [ 'error' => 'invalid_date' ] );
		return;
	}
	
	try {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL AJAX Debug: Getting slots for date: ' . $date );
		}
		$slots = BSGCAL_Scheduler::get_available_slots( $date );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL AJAX Debug: Got ' . count( $slots ) . ' slots: ' . json_encode( $slots ) );
		}
		wp_send_json_success( $slots );
	} catch ( Exception $e ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'BSGCAL AJAX Debug: Exception: ' . $e->getMessage() );
		}
		wp_send_json_error( [ 'error' => 'exception', 'message' => $e->getMessage() ] );
	}
}

function bsgcal_ajax_create_booking() : void {
	// Ensure we're in AJAX context
	if ( ! wp_doing_ajax() ) {
		wp_die( 'Not AJAX request' );
	}
	
	// Verify nonce
	$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
	if ( ! wp_verify_nonce( $nonce, 'bsgcal_ajax_nonce' ) ) {
		wp_die( 'Security check failed' );
	}
	
	// Set proper content type
	header( 'Content-Type: application/json' );
	
	$date = sanitize_text_field( wp_unslash( $_POST['date'] ?? '' ) );
	$start_time = sanitize_text_field( wp_unslash( $_POST['start_time'] ?? '' ) );
	$name = sanitize_text_field( wp_unslash( $_POST['name'] ?? '' ) );
	$phone = sanitize_text_field( wp_unslash( $_POST['phone'] ?? '' ) );
	
	if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
		wp_send_json_error( [ 'error' => 'invalid_date' ] );
		return;
	}
	
	if ( ! preg_match( '/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $start_time ) ) {
		wp_send_json_error( [ 'error' => 'invalid_time' ] );
		return;
	}
	
	if ( empty( $name ) || empty( $phone ) ) {
		wp_send_json_error( [ 'error' => 'missing_fields' ] );
		return;
	}
	
	try {
		$settings = BSGCAL_Admin::get_settings();
		$timezone_string = $settings['timezone'] ?? wp_timezone_string();
		$start_datetime = new DateTime( $date . ' ' . $start_time, new DateTimeZone( $timezone_string ) );
		$end_datetime = clone $start_datetime;
		$end_datetime->add( new DateInterval( 'PT30M' ) );
		
		// Check if slot is available
		$slots = BSGCAL_Scheduler::get_available_slots( $date );
		$slot_available = false;
		
		foreach ( $slots as $slot ) {
			if ( $slot['start'] === $start_datetime->format( 'Y-m-d H:i:s' ) && $slot['available'] ) {
				$slot_available = true;
				break;
			}
		}
		
		if ( ! $slot_available ) {
			wp_send_json_error( [ 'error' => 'slot_not_available' ] );
			return;
		}
		
		// Create Google Calendar event
		$event_result = '';
		if ( class_exists( 'BSGCAL_Google_Client' ) ) {
			$event_result = BSGCAL_Google_Client::create_event( [
				'summary' => 'Meeting with ' . $name,
				'description' => 'Phone: ' . $phone,
				'start' => $start_datetime,
				'end' => $end_datetime,
			] );
		}
		
		if ( empty( $event_result ) ) {
			wp_send_json_error( [ 'error' => 'google_calendar_failed', 'message' => 'Failed to create event in Google Calendar' ] );
			return;
		}
		
		// Create backup booking in WordPress
		$booking_id = BSGCAL_DB::insert_booking( [
			'meeting_start' => $start_datetime->format( 'Y-m-d H:i:s' ),
			'meeting_end' => $end_datetime->format( 'Y-m-d H:i:s' ),
			'client_name' => $name,
			'client_phone' => $phone,
			'google_event_id' => $event_result,
			'status' => 'confirmed',
		] );
		
		if ( $booking_id ) {
			wp_send_json_success( [ 'message' => 'Booking created successfully', 'booking_id' => $booking_id ] );
		} else {
			wp_send_json_error( [ 'error' => 'database_error', 'message' => 'Failed to save booking' ] );
		}
		
	} catch ( Exception $e ) {
		wp_send_json_error( [ 'error' => 'exception', 'message' => $e->getMessage() ] );
	}
}

add_action( 'plugins_loaded', 'bsgcal_init_plugin' );
add_action( 'rest_api_init', 'bsgcal_init_rest_api' );

// Register AJAX handlers
add_action( 'wp_ajax_bsgcal_get_slots', 'bsgcal_ajax_get_slots' );
add_action( 'wp_ajax_nopriv_bsgcal_get_slots', 'bsgcal_ajax_get_slots' );
add_action( 'wp_ajax_bsgcal_create_booking', 'bsgcal_ajax_create_booking' );
add_action( 'wp_ajax_nopriv_bsgcal_create_booking', 'bsgcal_ajax_create_booking' );

// Gutenberg block registration (simple wrapper for shortcode)
add_action( 'init', function() {
	if ( function_exists( 'register_block_type' ) ) {
		register_block_type( __DIR__ . '/block' );
	}
} );

// Shortcode
add_shortcode( 'event_to_calendar', [ 'BSGCAL_Frontend', 'render_shortcode' ] ); 