(function(){
	/**
	 * Google Event Sync Frontend
	 * Interactive booking calendar with Google Calendar sync
	 */
	
	function h(tag, attrs, children){
		var el = document.createElement(tag);
		if(attrs){
			Object.keys(attrs).forEach(function(k){
				if(k === 'class') el.className = attrs[k];
				else if(k === 'for') el.htmlFor = attrs[k];
				else el.setAttribute(k, attrs[k]);
			});
		}
		(children||[]).forEach(function(ch){
			if(typeof ch === 'string') el.appendChild(document.createTextNode(ch));
			else if(ch) el.appendChild(ch);
		});
		return el;
	}



	function pad(n){ return (n < 10 ? '0' : '') + n; }
	function ymd(d){ return d.getFullYear() + '-' + pad(d.getMonth()+1) + '-' + pad(d.getDate()); }
	function monthLabel(d){
		var calendarConfig = (window.BSGCAL_FRONTEND && BSGCAL_FRONTEND.calendar) || {};
		var language = calendarConfig.language || 'en-US';
		
		try {
			return d.toLocaleString(language, { month: 'long', year: 'numeric' });
		} catch (e) {
			// Fallback to browser default if language is not supported
			return d.toLocaleString(undefined, { month: 'long', year: 'numeric' });
		}
	}
	function restBase(){
		var b = (window.BSGCAL_FRONTEND && BSGCAL_FRONTEND.rest_base) || '';
		return b.replace(/\/+$/, '');
	}

	// Helper functions for date/time validation
	function isDateInPast(dateStr) {
		var today = new Date();
		var todayStr = ymd(today);
		return dateStr < todayStr;
	}

	function isTimeInPast(dateStr, timeStr) {
		var now = new Date();
		var today = ymd(now);
		
		// If date is not today, it's not in the past
		if (dateStr !== today) {
			return false;
		}
		
		// Parse time string (HH:MM format)
		var timeParts = timeStr.split(':');
		var slotHour = parseInt(timeParts[0], 10);
		var slotMinute = parseInt(timeParts[1], 10);
		
		// Compare with current time
		var currentHour = now.getHours();
		var currentMinute = now.getMinutes();
		
		// Add buffer time (e.g., 15 minutes) to prevent booking very recent slots
		var bufferMinutes = parseInt((window.BSGCAL_FRONTEND && window.BSGCAL_FRONTEND.time_buffer) || 15, 10);
		var adjustedHour = currentHour;
		var adjustedMinute = currentMinute + bufferMinutes;
		
		// Adjust hours if minutes overflow
		if (adjustedMinute >= 60) {
			adjustedHour += Math.floor(adjustedMinute / 60);
			adjustedMinute = adjustedMinute % 60;
		}
		
		// Convert both times to minutes for easier comparison
		var slotTotalMinutes = slotHour * 60 + slotMinute;
		var cutoffTotalMinutes = adjustedHour * 60 + adjustedMinute;
		
		return slotTotalMinutes < cutoffTotalMinutes;
	}

	function isSlotAvailable(slot, selectedDate) {
		// Check if slot is marked as unavailable
		var originalAvailable = slot.available !== undefined ? slot.available : true;
		
		// For now, just return the original availability
		// The past time check will be done separately by the caller
		return originalAvailable;
	}

	function buildCalendar(container, state, onPick){
		container.innerHTML = '';
		var header = h('div', { class:'bsgcal-cal-header' });
		var prevBtn = h('button', { type:'button', class:'bsgcal-cal-nav' }, ['‹']);
		var title = h('div', { class:'bsgcal-cal-title' }, [ monthLabel(state.currentMonth) ]);
		var nextBtn = h('button', { type:'button', class:'bsgcal-cal-nav' }, ['›']);
		header.appendChild(prevBtn); header.appendChild(title); header.appendChild(nextBtn);

		var grid = h('div', { class:'bsgcal-cal-grid' });
		var calendarConfig = (window.BSGCAL_FRONTEND && BSGCAL_FRONTEND.calendar) || {};
		var weekdays = calendarConfig.weekdays || { mon: 'Mon', tue: 'Tue', wed: 'Wed', thu: 'Thu', fri: 'Fri', sat: 'Sat', sun: 'Sun' };
		
		// Use weekday abbreviations from settings or fallback to defaults
		var weekdayLabels = [
			weekdays.mon || 'Mon',
			weekdays.tue || 'Tue', 
			weekdays.wed || 'Wed',
			weekdays.thu || 'Thu',
			weekdays.fri || 'Fri',
			weekdays.sat || 'Sat',
			weekdays.sun || 'Sun'
		];
		
		weekdayLabels.forEach(function(w){ 
			grid.appendChild(h('div', { class:'bsgcal-cal-wd' }, [w])); 
		});

		var first = new Date(state.currentMonth.getFullYear(), state.currentMonth.getMonth(), 1);
		var firstWeekday = (first.getDay()+6)%7; // 0=Mon .. 6=Sun
		var daysInMonth = new Date(state.currentMonth.getFullYear(), state.currentMonth.getMonth()+1, 0).getDate();
		var daysPrev = firstWeekday;
		var totalCells = 42; // 6 weeks
		var startDate = new Date(first);
		startDate.setDate(first.getDate() - daysPrev);
		var todayStr = ymd(new Date());

		for(var i=0;i<totalCells;i++){
			var d = new Date(startDate);
			d.setDate(startDate.getDate()+i);
			var inMonth = d.getMonth() === state.currentMonth.getMonth();
			var dd = ymd(d);
			var isPast = isDateInPast(dd);
			
			var cellClasses = ['bsgcal-cal-day'];
			if (!inMonth) cellClasses.push('is-out');
			if (dd === todayStr) cellClasses.push('is-today');
			if (dd === state.date) cellClasses.push('is-selected');
			if (isPast) cellClasses.push('is-past');
			
			var cell = h('button', { type:'button', class: cellClasses.join(' ') }, [ String(d.getDate()) ]);
			
			// Only allow clicking on non-past dates
			if (!isPast) {
				(function(dateStr){
					cell.addEventListener('click', function(){ onPick(dateStr); });
				})(dd);
			} else {
				cell.disabled = true;
				cell.title = 'This date is in the past';
			}
			
			grid.appendChild(cell);
		}

		prevBtn.addEventListener('click', function(){
			state.currentMonth = new Date(state.currentMonth.getFullYear(), state.currentMonth.getMonth()-1, 1);
			buildCalendar(container, state, onPick);
		});
		nextBtn.addEventListener('click', function(){
			state.currentMonth = new Date(state.currentMonth.getFullYear(), state.currentMonth.getMonth()+1, 1);
			buildCalendar(container, state, onPick);
		});

		container.appendChild(header);
		container.appendChild(grid);
	}

	function render(){
		var root = document.getElementById('bsgcal-app');
		if(!root) return;
		var labels = (window.BSGCAL_FRONTEND && BSGCAL_FRONTEND.labels) || {};
		var today = new Date();
		var state = { date: ymd(today), slots: [], pickedTime: '', name: '', phone: '', busy:false, message:'', currentMonth: new Date(today.getFullYear(), today.getMonth(), 1) };

		var calendarWrap = h('div', { class:'bsgcal-calendar' });
		var slotList = h('div', { class:'bsgcal-slots' });
		var nameInput = h('input', { type:'text', placeholder: labels.name || 'Name' });
		var phoneInput = h('input', { type:'tel', placeholder: labels.phone || 'Phone' });
		var submitBtn = h('button', { type:'button' }, [ labels.submit || 'Book' ]);
		var msg = h('div', { class:'bsgcal-message' });
		
		// AbortController for canceling previous requests
		var currentSlotRequest = null;
		var loadSlotsTimeout = null;
		
		// Global click handler for slot interactions
		slotList.addEventListener('click', function(event) {
			// Handle slot clicks
		});

		function highlightSelected(){
			Array.prototype.forEach.call(calendarWrap.querySelectorAll('.bsgcal-cal-day'), function(b){ b.classList.remove('is-selected'); });
			var sel = Array.prototype.find.call(calendarWrap.querySelectorAll('.bsgcal-cal-day'), function(b){ return b.textContent === String(parseInt(state.date.slice(8,10),10)) && !b.classList.contains('is-out'); });
			if(sel) sel.classList.add('is-selected');
		}

		function updateForm(){
			if(state.pickedTime && state.date){
				nameInput.style.display = 'block';
				phoneInput.style.display = 'block';
				submitBtn.style.display = 'block';
			}else{
				nameInput.style.display = 'none';
				phoneInput.style.display = 'none';
				submitBtn.style.display = 'none';
			}
		}

		function loadSlotsDebounced(){
			if(loadSlotsTimeout) clearTimeout(loadSlotsTimeout);
			loadSlotsTimeout = setTimeout(loadSlots, 150);
		}

		function loadSlots(){
			// Cancel previous request if exists
			if (currentSlotRequest) {
				currentSlotRequest.abort();
			}
			
			// Create new AbortController for this request
			currentSlotRequest = new AbortController();
			var requestId = Date.now(); // Unique ID for this request
			var requestDate = state.date; // Store the date for this specific request
			
			slotList.innerHTML = '<div class="bsgcal-loading">' + (labels.loading_slots || 'Loading slots...') + '</div>';
			state.slots = [];
			state.pickedTime = '';
			if(!state.date) {
				return;
			}
			
			// Call AJAX function directly
			loadSlotsViaAjax(requestId, requestDate);
			return;
		}

		// Function to load slots via AJAX (replaces REST API)
		function loadSlotsViaAjax(requestId, requestDate) {
			var formData = new FormData();
			formData.append('action', 'bsgcal_get_slots');
			formData.append('date', requestDate);
			
			fetch(window.BSGCAL_FRONTEND.ajax_url || '/wp-admin/admin-ajax.php', {
				method: 'POST',
				credentials: 'same-origin',
				signal: currentSlotRequest.signal,
				body: formData
			})
			.then(function(r) {
				// Check if response is JSON
				const contentType = r.headers.get('content-type');
				if (!contentType || !contentType.includes('application/json')) {
					throw new Error('Response is not JSON. Content-Type: ' + contentType);
				}
				return r.json();
			})
			.then(function(ajaxRes) {
				// Check if this response is still relevant
				if (requestDate !== state.date) {
					return;
				}
				
				if (ajaxRes.success) {
					// Fix: data is directly the slots array, not data.slots
					var slots = ajaxRes.data || [];
					state.slots = slots;
					
					// Clear loading message
					slotList.innerHTML = '';
					if(slots.length === 0){
						slotList.appendChild(h('div', { class:'bsgcal-no-slots' }, [labels.no_slots || 'No slots for this date']));
						return;
					}
					slots.forEach(function(s, index){
						// Check if slot is available using our new validation function
						var originalAvailable = isSlotAvailable(s, state.date);
						var isPastTime = isTimeInPast(state.date, s.start.slice(11,16));
						
						// Final availability: must be originally available AND not in the past
						var isAvailable = originalAvailable && !isPastTime;
						
						var btnClasses = ['bsgcal-slot'];
						if (!isAvailable) {
							btnClasses.push('unavailable');
							if (isPastTime) {
								btnClasses.push('past-time');
							}
						}
						
						var btnAttrs = { 
							type: 'button', 
							class: btnClasses.join(' '),
							'data-time': s.start.slice(11,16),
							'data-label': s.label
						};
						
						// Only add disabled if actually disabled
						if (!isAvailable) {
							btnAttrs.disabled = true;
							
							// Add tooltip explaining why slot is unavailable
							var reason = '';
							if (s.google_conflicts && s.google_conflicts.length > 0) {
								var eventTitles = s.google_conflicts.map(function(event) {
									return event.summary || 'Untitled event';
								}).join(', ');
								reason = 'Conflicts with Google Calendar: ' + eventTitles;
							} else if (isPastTime) {
								reason = 'This time slot is in the past';
							} else {
								reason = 'Not available';
							}
							btnAttrs.title = reason;
						}
						
						var btn = h('button', btnAttrs, [ s.label ]);
						
						// Add direct click handler to button
						
						if (isAvailable) {
							// Use closure to capture slot data
							btn.onclick = (function(slot) {
								return function(e) {
									// Set the picked time
									state.pickedTime = slot.start.slice(11,16);
									
									// Remove active class from all slots
									Array.from(slotList.children).forEach(function(child) {
										if (child.classList.contains('bsgcal-slot')) {
											child.classList.remove('active');
										}
									});
									
									// Add active class to clicked slot
									btn.classList.add('active');
									
									// Update form
									updateForm();
								};
							})(s);
						}
						
						slotList.appendChild(btn);
					});
				} else {
					throw new Error('AJAX failed: ' + (ajaxRes.data || 'Unknown error'));
				}
			})
			.catch(function(ajaxError) {
				// Check if request was aborted
				if (ajaxError.name === 'AbortError') {
					return;
				}
				
				console.error('BSGCAL Frontend: AJAX request failed:', ajaxError);
				slotList.innerHTML = '<div class="bsgcal-error">Failed to load slots</div>';
			});
		}

		submitBtn.addEventListener('click', function(){
			
			if(state.busy) return;
			msg.textContent = '';
			msg.className = 'bsgcal-message';
			
			// Detailed validation
			var validationErrors = [];
			if(!state.date) validationErrors.push('No date selected');
			if(!state.pickedTime) validationErrors.push('No time slot selected');
			if(!nameInput.value) validationErrors.push('Name is empty');
			if(!phoneInput.value) validationErrors.push('Phone is empty');
			
			if(validationErrors.length > 0){
				msg.textContent = 'Please fill all fields and pick a slot. Errors: ' + validationErrors.join(', ');
				return;
			}
			state.busy = true;
			submitBtn.disabled = true;
			
			// Use AJAX directly for booking
			var formData = new FormData();
			formData.append('action', 'bsgcal_create_booking');
			formData.append('date', state.date);
			formData.append('start_time', state.pickedTime);
			formData.append('name', nameInput.value);
			formData.append('phone', phoneInput.value);
			
			fetch(window.BSGCAL_FRONTEND.ajax_url || '/wp-admin/admin-ajax.php', {
				method: 'POST',
				credentials: 'same-origin',
				body: formData
			})
			.then(function(r) {
				// Check if response is JSON
				const contentType = r.headers.get('content-type');
				if (!contentType || !contentType.includes('application/json')) {
					throw new Error('Response is not JSON. Content-Type: ' + contentType);
				}
				return r.json();
			})
			.then(function(ajaxRes) {
				if (ajaxRes.success) {
					msg.textContent = labels.booked || 'Booked!';
					msg.className = 'bsgcal-message success';
					nameInput.value = '';
					phoneInput.value = '';
					
					// Скрываем форму
					nameInput.style.display = 'none';
					phoneInput.style.display = 'none';
					submitBtn.style.display = 'none';
					
					// Очищаем временные слоты
					slotList.innerHTML = '';
					
					// Сбрасываем состояние
					state.pickedTime = '';
					state.slots = [];
					

				} else {
					msg.textContent = ajaxRes.data || 'Booking failed';
					msg.className = 'bsgcal-message error';
				}
			})
			.catch(function(ajaxError) {
				console.error('BSGCAL Frontend: AJAX booking failed:', ajaxError);
				msg.textContent = 'Error';
				msg.className = 'bsgcal-message error';
			})
			.finally(function() {
				state.busy = false;
				submitBtn.disabled = false;
			});
		});

		function onPick(dateStr){
			state.date = dateStr;
			
			// Проверяем, нужно ли переключить месяц календаря
			var selectedDate = new Date(dateStr);
			var currentMonth = state.currentMonth;
			
			// Если выбранная дата в другом месяце, переключаем календарь
			if (selectedDate.getMonth() !== currentMonth.getMonth() || 
				selectedDate.getFullYear() !== currentMonth.getFullYear()) {
				state.currentMonth = new Date(selectedDate.getFullYear(), selectedDate.getMonth(), 1);
				buildCalendar(calendarWrap, state, onPick);
			}
			
			highlightSelected();
			
			// Очищаем сообщение при выборе новой даты
			msg.textContent = '';
			msg.className = 'bsgcal-message';
			
			// Показываем форму при выборе новой даты
			nameInput.style.display = 'block';
			phoneInput.style.display = 'block';
			submitBtn.style.display = 'block';
			

			
			updateForm();
			loadSlotsDebounced();
		}

		// Initial setup
		buildCalendar(calendarWrap, state, onPick);
		highlightSelected();
		updateForm();
		loadSlots();

		// Append elements
		root.appendChild(calendarWrap);
		root.appendChild(slotList);
		root.appendChild(nameInput);
		root.appendChild(phoneInput);
		root.appendChild(submitBtn);
		root.appendChild(msg);

		// Initial form state
		updateForm();
	}

	// Initialize when DOM is ready
	if(document.readyState === 'loading'){
		document.addEventListener('DOMContentLoaded', render);
	}else{
		render();
	}
})(); 