<?php
/**
 * Plugin Name: Vigore Widget
 * Description: The Vigore Widget Plugin allows you to embed the vigore.app widget service as an iframe on your WordPress website, enabling customers to book services directly from your site.
 * Version: 2.1.0
 * Author: devvigore
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 6.0
 * Tested up to: 6.8
 * Requires PHP: 7.0
 * Text Domain: vigore-widget
 * Contributors: devvigore
 */

if (!defined('ABSPATH'))
    exit;

// Hooks
add_action('wp_footer', 'vigore_enqueue_assets');
add_action('wp_footer', 'vigore_add_widget');
add_action('admin_menu', 'vigore_create_menu');
add_action('admin_init', 'vigore_register_settings');
add_action('admin_menu', 'vigore_add_welcome_page');

// Load SVG icon as base64
function vigore_get_custom_icon_svg()
{
    $icon_path = plugin_dir_path(__FILE__) . 'assets/icons/vigore-icon.svg';
    if (file_exists($icon_path)) {
        return 'data:image/svg+xml;base64,' . base64_encode(file_get_contents($icon_path));
    }
    return 'dashicons-admin-generic';
}

// Create admin menu
function vigore_create_menu()
{
    add_menu_page(
        'Vigore Widget',
        'Vigore Widget',
        'manage_options',
        'vigore-widget',
        'vigore_main_page',
        vigore_get_custom_icon_svg(),
        56
    );
}

// Register settings
function vigore_register_settings()
{
    register_setting('vigore-settings-group', 'vigore_organization_id', 'sanitize_text_field');
    register_setting('vigore-settings-group', 'vigore_professional_id', 'sanitize_text_field');
    register_setting('vigore-settings-group', 'vigore_language', 'sanitize_text_field');
    register_setting('vigore-settings-group', 'vigore_access_token', 'sanitize_text_field');
    register_setting('vigore-settings-group', 'vigore_user_email', 'sanitize_email');
    register_setting('vigore-settings-group', 'vigore_user_first_name', 'sanitize_text_field');
}

// Activation hook
register_activation_hook(__FILE__, 'vigore_plugin_activate');
function vigore_plugin_activate()
{
    add_option('vigore_do_activation_redirect', true);
}

// Handle activation redirect
add_action('admin_init', 'vigore_plugin_redirect');
function vigore_plugin_redirect()
{
    if (get_option('vigore_do_activation_redirect', false)) {
        delete_option('vigore_do_activation_redirect');
        if (!isset($_GET['activate-multi'])) {
            // Redirect to the dedicated welcome page
            wp_redirect(admin_url('admin.php?page=vigore-welcome'));
            exit;
        }
    }
}

// Add AJAX handler for fetching instructors
add_action('wp_ajax_vigore_fetch_instructors', 'vigore_fetch_instructors');
function vigore_fetch_instructors()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    check_ajax_referer('vigore_auth_nonce', 'nonce');

    // Get access token from current user's meta
    $current_user_id = get_current_user_id();
    $access_token = get_user_meta($current_user_id, 'vigore_access_token', true);
    if (!$access_token) {
        wp_send_json_error('Not authenticated');
    }

    $organization_id = get_option('vigore_organization_id');
    if (!$organization_id) {
        wp_send_json_error('Organization ID not set');
    }

    $response = wp_remote_post("https://api.vigore.app/api/organization/instructors/{$organization_id}", array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $access_token
        ),
        'body' => json_encode(array(
            'page' => 0,
            'size' => 100,
            'sort' => 'email',
            'sortDirection' => 'desc',
            'filter' => ''
        )),
        'timeout' => 30
    ));

    if (is_wp_error($response)) {
        wp_send_json_error($response->get_error_message());
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (isset($data['error']) && $data['error']) {
        wp_send_json_error($data['responseMessage'] ?? 'Failed to fetch instructors');
    }

    wp_send_json_success($data);
}

// Add AJAX handler for storing selected instructor
add_action('wp_ajax_vigore_store_instructor', 'vigore_store_instructor');
function vigore_store_instructor()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    if (!isset($_POST['vigore_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['vigore_nonce'])), 'vigore_save_settings')) {
        wp_send_json_error('Invalid nonce');
    }

    $instructor_id = isset($_POST['instructor_id']) ? sanitize_text_field(wp_unslash($_POST['instructor_id'])) : '';
    if (empty($instructor_id)) {
        wp_send_json_error('Instructor ID is required');
    }

    update_option('vigore_professional_id', $instructor_id);
    wp_send_json_success();
}

// Unified main page for Vigore Widget (login/settings)
function vigore_main_page()
{
    // Check for current user's Vigore access token
    $current_user_id = get_current_user_id();
    $access_token = get_user_meta($current_user_id, 'vigore_access_token', true);

    if ($access_token) {
        // User is logged in - show settings
        vigore_render_settings_page();
    } else {
        // User is not logged in - show login
        vigore_render_login_page();
    }
}

// Render settings page content
function vigore_render_settings_page()
{
    $icon_svg_path = plugin_dir_path(__FILE__) . 'assets/icons/vigore-icon.svg';
    $icon_svg = '';
    if (file_exists($icon_svg_path)) {
        $icon_svg = file_get_contents($icon_svg_path);
        $icon_svg = preg_replace(
            '/<svg([^>]+)>/',
            '<svg$1 width="20" height="34" style="width:20px;height:34px;margin-right:10px;vertical-align:middle;">',
            $icon_svg
        );
    }
    // Get user data from current user's meta
    $current_user_id = get_current_user_id();
    $user_email = get_user_meta($current_user_id, 'vigore_user_email', true);
    $user_first_name = get_user_meta($current_user_id, 'vigore_user_first_name', true);
    ?>
    <div class="wrap">
        <div id="vigore_header">
            <div class="header-title">
                <img src="<?php echo esc_url(plugins_url('assets/images/vigore_logo_img.png', __FILE__)); ?>"
                    alt="Vigore Logo" style="height: 36px; margin-right: 14px;" />
                <div class="header-title-text"><?php esc_html_e('Vigore Widget Settings', 'vigore-widget'); ?></div>
            </div>
            <div class="header-actions">
                <?php if ($user_first_name): ?>
                    <span class="vigore-user-name"><?php echo esc_html($user_first_name); ?></span>
                <?php endif; ?>
                <button id="vigore-logout-js"
                    style="background: transparent; border: none; padding: 0; display: flex; align-items: center; cursor: pointer;"
                    type="button" alt="Logout">
                    <img src="<?php echo esc_url(plugins_url('assets/icons/logout.svg', __FILE__)); ?>" alt="Logout"
                        style="width: 18px; height: 18px; display: inline-block; vertical-align: middle;" />
                </button>
            </div>
        </div>
        <div class="vigore-settings-container">
            <form method="post" action="options.php" id="vigore-settings-form">
                <?php
                settings_fields('vigore-settings-group');
                do_settings_sections('vigore-settings-group');
                wp_nonce_field('vigore_save_settings', 'vigore_nonce');
                ?>
                <div class="form-group">
                    <label><?php esc_html_e('Organization ID', 'vigore-widget'); ?> <span
                            style="color:red;">*</span></label>
                    <input type="text" name="vigore_organization_id"
                        value="<?php echo esc_attr(get_option('vigore_organization_id')); ?>" />
                </div>
                <div class="form-group">
                    <label><?php esc_html_e('Professional ID', 'vigore-widget'); ?></label>
                    <div class="vigore-professional-selector vigore-selector-row">
                        <select name="vigore_professional_id" id="vigore_professional_id">
                            <option value=""><?php esc_html_e('Select a professional', 'vigore-widget'); ?></option>
                        </select>
                    </div>
                    <div class="text-align-right">
                        <?php
                        $has_professional = get_option('vigore_professional_id');
                        if (!empty($has_professional)):
                            ?>
                            <a href="#" id="vigore-remove-instructor" class="vigore-remove-instructor-btn">
                                <?php esc_html_e('Remove Instructor', 'vigore-widget'); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                    <input type="hidden" id="current_professional_id"
                        value="<?php echo esc_attr(get_option('vigore_professional_id')); ?>" />
                </div>
                <div class="form-group">
                    <label><?php esc_html_e('Language', 'vigore-widget'); ?></label>
                    <select name="vigore_language">
                        <option value="nl" <?php selected(get_option('vigore_language'), 'nl'); ?>>
                            <?php esc_html_e('Dutch', 'vigore-widget'); ?>
                        </option>
                        <option value="en" <?php selected(get_option('vigore_language'), 'en'); ?>>
                            <?php esc_html_e('English', 'vigore-widget'); ?>
                        </option>
                    </select>
                </div>
                <div class="padding-y-20">
                    <p class="description" style="color:#d63638; font-size:12px; margin-top: -10px;">
                        <strong><?php esc_html_e('Note:', 'vigore-widget'); ?></strong>
                        <?php esc_html_e('Organization ID is a required field.', 'vigore-widget'); ?>
                    </p>
                </div>
                <div>
                    <input type="submit" class="button button-primary"
                        value="<?php esc_attr_e('Save Settings', 'vigore-widget'); ?>" />
                </div>
            </form>

        </div>
        <script type="text/javascript">
            document.addEventListener('DOMContentLoaded', function () {
                var logoutBtn = document.getElementById('vigore-logout-js');
                if (logoutBtn) {
                    logoutBtn.addEventListener('click', function (e) {
                        e.preventDefault();
                        fetch(ajaxurl, {
                            method: 'POST',
                            credentials: 'same-origin',
                            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                            body: 'action=vigore_logout_js&_ajax_nonce=<?php echo esc_attr(wp_create_nonce('vigore_logout_js')); ?>'
                        })
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    window.location.href = '<?php echo esc_url(admin_url('admin.php?page=vigore-widget')); ?>';
                                } else {
                                    alert('Logout failed: ' + (data.data || 'Unknown error'));
                                }
                            });
                    });
                }
            });
        </script>
    </div>
    <?php
}

// Render login page content
function vigore_render_login_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }
    $user_email = get_option('vigore_user_email');
    ?>
    <div class="wrap">
        <div class="vigore-auth-container">
            <div class="vigore-auth-box">
                <div class="vigore-header">
                    <img src="<?php echo esc_url(plugins_url('assets/images/vigore_logo_img.png', __FILE__)); ?>"
                        alt="Google Icon" class="vigore-icon">
                    <h3 class="vigore-title"><?php esc_html_e('Vigore Authentication', 'vigore-widget'); ?></h3>
                </div>
                <p class="vigore-login-subtitle">
                    <?php esc_html_e('Login to setup and connect the widget with your Vigore account.', 'vigore-widget'); ?>
                </p>
                <form id="vigore-login-form" method="post" action="javascript:void(0);">
                    <div class="form-group">
                        <label for="vigore-email"><?php esc_html_e('Email', 'vigore-widget'); ?></label>
                        <input type="email" id="vigore-email" name="email" placeholder="Enter your email address" required>
                    </div>
                    <div class="form-group">
                        <label for="vigore-password"><?php esc_html_e('Password', 'vigore-widget'); ?></label>
                        <input type="password" id="vigore-password" name="password" placeholder="Enter your password"
                            required>
                    </div>
                    <div class="padding-y-20 text-align-right">
                        <a id="forgot-password-text" href="https://dashboard.vigore.app/forgot-password" target="_blank">
                            <?php esc_html_e('Forgot your password?', 'vigore-widget'); ?>
                        </a>
                    </div>
                    <div id="notice-error-place"></div>
                    <div>
                        <button type="submit"
                            class="button button-primary"><?php esc_html_e('Login', 'vigore-widget'); ?></button>
                    </div>
                    <div class="vigore-separator"><span>or</span></div>
                    <div>
                        <button type="button" id="google-login-btn" class="vigore-google-button">
                            <img src="<?php echo esc_url(plugins_url('assets/icons/google-icon.svg', __FILE__)); ?>"
                                alt="Google Icon">
                            Continue with Google
                        </button>
                    </div>
                </form>
            </div>
            <div class="vigore-auth-box" style="display: none;">
                <h3><?php esc_html_e('Sign Up', 'vigore-widget'); ?></h3>
                <form id="vigore-signup-form" method="post" action="javascript:void(0);">
                    <p>
                        <label for="vigore-signup-email"><?php esc_html_e('Email', 'vigore-widget'); ?></label>
                        <input type="email" id="vigore-signup-email" name="email" required>
                    </p>
                    <p>
                        <label for="vigore-signup-password"><?php esc_html_e('Password', 'vigore-widget'); ?></label>
                        <input type="password" id="vigore-signup-password" name="password" required>
                    </p>
                    <p>
                        <button type="submit"
                            class="button button-primary"><?php esc_html_e('Sign Up', 'vigore-widget'); ?></button>
                    </p>
                    <p class="vigore-auth-switch">
                        <?php esc_html_e('Already have an account?', 'vigore-widget'); ?>
                        <a href="#" class="vigore-switch-to-login"><?php esc_html_e('Login', 'vigore-widget'); ?></a>
                    </p>
                </form>
            </div>
        </div>

    </div>
    <?php
}

// Settings page content (legacy - now redirects to main page)
function vigore_settings_page()
{
    $access_token = get_option('vigore_access_token');
    if (!$access_token) {
        wp_redirect(admin_url('admin.php?page=vigore-widget'));
        exit;
    }
    vigore_render_settings_page();
}

// Authentication page content (legacy - now redirects to main page)
function vigore_auth_page()
{
    $access_token = get_option('vigore_access_token');
    if ($access_token) {
        wp_redirect(admin_url('admin.php?page=vigore-widget'));
        exit;
    }
    vigore_render_login_page();
}

// Add AJAX handlers for authentication
add_action('wp_ajax_vigore_store_auth', 'vigore_store_auth');
function vigore_store_auth()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    check_ajax_referer('vigore_auth_nonce', 'nonce');

    $access_token = isset($_POST['access_token']) ? sanitize_text_field(wp_unslash($_POST['access_token'])) : '';
    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
    $organization_id = isset($_POST['organization_id']) ? sanitize_text_field(wp_unslash($_POST['organization_id'])) : '';
    $professional_id = isset($_POST['professional_id']) ? sanitize_text_field(wp_unslash($_POST['professional_id'])) : '';
    $first_name = isset($_POST['first_name']) ? sanitize_text_field(wp_unslash($_POST['first_name'])) : '';

    // Store access token and user info in user meta
    $current_user_id = get_current_user_id();
    update_user_meta($current_user_id, 'vigore_access_token', $access_token);
    update_user_meta($current_user_id, 'vigore_user_email', $email);
    if ($first_name) {
        update_user_meta($current_user_id, 'vigore_user_first_name', $first_name);
    }

    // Store organization and professional IDs in database (these are settings, not user-specific)
    if ($organization_id) {
        update_option('vigore_organization_id', $organization_id);
    }
    if ($professional_id) {
        update_option('vigore_professional_id', $professional_id);
    }

    wp_send_json_success();
}

// Handle logout
add_action('wp_ajax_vigore_logout', 'vigore_handle_logout');
function vigore_handle_logout()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    check_ajax_referer('vigore_auth_nonce', 'nonce');

    // Clear current user's Vigore data
    $current_user_id = get_current_user_id();
    delete_user_meta($current_user_id, 'vigore_access_token');
    delete_user_meta($current_user_id, 'vigore_user_email');
    delete_user_meta($current_user_id, 'vigore_user_first_name');

    wp_send_json_success();
}

// Add AJAX handlers for authentication
add_action('wp_ajax_vigore_api_login', 'vigore_handle_api_login');
function vigore_handle_api_login()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    check_ajax_referer('vigore_auth_nonce', 'nonce');

    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
    $password = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : '';

    // Make API request
    $response = wp_remote_post('https://api.vigore.app/auth/login', array(
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body' => json_encode(array(
            'email' => $email,
            'password' => $password
        )),
        'timeout' => 30
    ));

    if (is_wp_error($response)) {
        wp_send_json_error($response->get_error_message());
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if ($status_code === 401 || (isset($data['error']) && $data['error'])) {
        wp_send_json_error($data['responseMessage'] ?? 'Invalid credentials');
    }

    if (empty($data) || !isset($data['token'])) {
        wp_send_json_error('Invalid response from server');
    }

    // Transform the response to match our expected format
    $transformed_data = array(
        'access_token' => $data['token'],
        'email' => $data['email'],
        'organization_id' => $data['roles'][0]['organizationId'] ?? '',
        'professional_id' => $data['userId'] ?? '',
        'first_name' => $data['firstName'] ?? ''
    );

    wp_send_json_success($transformed_data);
}

// Floating widget on all pages
function vigore_add_widget()
{
    $organization_id = esc_attr(get_option('vigore_organization_id'));
    $professional_id = esc_attr(get_option('vigore_professional_id'));
    $language = get_option('vigore_language', 'nl');

    $base_url = "https://widget.vigore.app/$language";

    $iframe_url = (!empty($professional_id))
        ? "$base_url/team/{$professional_id}/services?organization_id={$organization_id}&instructorId={$professional_id}"
        : "$base_url/?organization_id={$organization_id}";
    ?>
    <div id="vigore-widget" style="position: fixed; bottom: 0; right: 0; width: 431px; height: 90%; z-index: 9999;">
        <iframe src="<?php echo esc_url($iframe_url); ?>" width="100%" height="100%" style="border: 0;"
            title="<?php esc_attr_e('Vigore Widget', 'vigore-widget'); ?>"></iframe>
    </div>
    <?php
}

function getVigore_widget_url()
{
    $organization_id = esc_attr(get_option('vigore_organization_id'));
    $professional_id = esc_attr(get_option('vigore_professional_id'));
    $language = get_option('vigore_language', 'nl');

    $base_url = "https://widget.vigore.app/$language";

    return (!empty($professional_id))
        ? "$base_url/team/{$professional_id}/services?organization_id={$organization_id}&instructorId={$professional_id}"
        : "$base_url/?organization_id={$organization_id}";
}

// Enqueue scripts and styles
function vigore_enqueue_assets()
{
    // Register and enqueue CSS
    wp_register_style(
        'vigore-widget-style',
        plugins_url('assets/css/vigore-widget.css', __FILE__),
        array(),
        '1.0'
    );
    wp_enqueue_style('vigore-widget-style');

    // Register and enqueue JavaScript
    wp_register_script(
        'vigore-widget-script',
        plugins_url('assets/js/vigore-widget.js', __FILE__),
        array(),
        '1.0',
        true
    );
    wp_enqueue_script('vigore-widget-script');
}

// Add AJAX handler for saving settings
add_action('wp_ajax_vigore_save_settings', 'vigore_save_settings');
function vigore_save_settings()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    $vigore_nonce = isset($_POST['vigore_nonce']) ? sanitize_text_field(wp_unslash($_POST['vigore_nonce'])) : '';
    if (!wp_verify_nonce($vigore_nonce, 'vigore_save_settings')) {
        wp_send_json_error('Invalid nonce');
    }

    $vigore_organization_id = isset($_POST['vigore_organization_id']) ? sanitize_text_field(wp_unslash($_POST['vigore_organization_id'])) : '';
    $vigore_professional_id = isset($_POST['vigore_professional_id']) ? sanitize_text_field(wp_unslash($_POST['vigore_professional_id'])) : '';
    $vigore_language = isset($_POST['vigore_language']) ? sanitize_text_field(wp_unslash($_POST['vigore_language'])) : '';

    update_option('vigore_organization_id', $vigore_organization_id);
    update_option('vigore_professional_id', $vigore_professional_id);
    update_option('vigore_language', $vigore_language);

    wp_send_json_success();
}

// Add admin styles and scripts
add_action('admin_enqueue_scripts', 'vigore_admin_styles');
function vigore_admin_styles($hook)
{
    // Only load on our plugin's pages
    if (!in_array($hook, array('toplevel_page_vigore-widget')) && (!isset($_GET['page']) || $_GET['page'] !== 'vigore-welcome')) {
        return;
    }

    // Enqueue admin styles
    wp_enqueue_style(
        'vigore-admin-style',
        plugins_url('assets/css/vigore-widget.css', __FILE__),
        array(),
        '1.0'
    );

    // Enqueue admin scripts
    wp_enqueue_script(
        'vigore-auth-script',
        plugins_url('assets/js/vigore-auth.js', __FILE__),
        array(),
        '1.0',
        true
    );

    wp_enqueue_script(
        'vigore-instructors-script',
        plugins_url('assets/js/vigore-instructors.js', __FILE__),
        array('vigore-auth-script'),
        '1.0',
        true
    );

    // Add debugging
    wp_localize_script('vigore-auth-script', 'vigoreAuth', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('vigore_auth_nonce'),
        'settingsNonce' => wp_create_nonce('vigore_save_settings'),
        'settingsUrl' => admin_url('admin.php?page=vigore-widget'),
        'debug' => true
    ));
}

// Add AJAX handler for removing instructor
add_action('wp_ajax_vigore_remove_instructor', 'vigore_remove_instructor');
function vigore_remove_instructor()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    check_ajax_referer('vigore_auth_nonce', 'nonce');

    delete_option('vigore_professional_id');
    wp_send_json_success();
}

// Welcome page content (not in menu, only for redirect)
function vigore_welcome_page()
{
    ?>
    <div class="wrap">
        <div class="vigore-welcome-container">
            <div>
                <img id="welcome-banner"
                    src="<?php echo esc_url(plugins_url('assets/images/banner-772x250.png', __FILE__)); ?>"
                    alt="Vigore Banner">
                <div class="vigore-welcome-text-container">
                    <div class="vigore-welcome-text">
                        <h1>Welcome to Vigore Widget!</h1>
                        <p>The Vigore Widget transforms your WordPress website into a booking platform. Your customers can
                            now book services directly from your site using a beautiful, floating widget.</p>

                        <h3>Ready to get started?</h3>
                        <p>Simply connect your Vigore account and the widget will appear on all your pages, allowing
                            customers to browse services and book appointments without leaving your website.</p>

                        <div class="vigore-welcome-buttons">
                            <a href="https://vigore.app/en/sign-up" target="_blank" class="vigore-signup-button">Sign
                                Up</a>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=vigore-widget')); ?>"
                                class="vigore-connect-link">Login</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
}

// AJAX handler for logout
add_action('wp_ajax_vigore_logout_js', 'vigore_logout_js_handler');
function vigore_logout_js_handler()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    // Clear current user's Vigore data
    $current_user_id = get_current_user_id();
    delete_user_meta($current_user_id, 'vigore_access_token');
    delete_user_meta($current_user_id, 'vigore_user_email');
    delete_user_meta($current_user_id, 'vigore_user_first_name');
    wp_send_json_success();
}

// Add hidden welcome page
function vigore_add_welcome_page()
{
    add_submenu_page(
        null, // No parent menu
        'Vigore Welcome', // Page title
        'Vigore Welcome', // Menu title (hidden)
        'manage_options', // Capability
        'vigore-welcome', // Menu slug
        'vigore_welcome_page' // Callback function
    );
}
